/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.groundstation.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.groundstation.GroundStationClient;
import software.amazon.awssdk.services.groundstation.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.groundstation.model.DescribeContactRequest;
import software.amazon.awssdk.services.groundstation.model.DescribeContactResponse;
import software.amazon.awssdk.services.groundstation.model.GroundStationRequest;
import software.amazon.awssdk.services.groundstation.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultGroundStationWaiter implements GroundStationWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final GroundStationClient client;

    private final AttributeMap managedResources;

    private final Waiter<DescribeContactResponse> contactScheduledWaiter;

    private DefaultGroundStationWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = GroundStationClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.contactScheduledWaiter = Waiter.builder(DescribeContactResponse.class).acceptors(contactScheduledWaiterAcceptors())
                .overrideConfiguration(contactScheduledWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeContactResponse> waitUntilContactScheduled(DescribeContactRequest describeContactRequest) {
        return contactScheduledWaiter.run(() -> client.describeContact(applyWaitersUserAgent(describeContactRequest)));
    }

    @Override
    public WaiterResponse<DescribeContactResponse> waitUntilContactScheduled(DescribeContactRequest describeContactRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return contactScheduledWaiter.run(() -> client.describeContact(applyWaitersUserAgent(describeContactRequest)),
                contactScheduledWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeContactResponse>> contactScheduledWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeContactResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("contactStatus").value(), "FAILED_TO_SCHEDULE");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (contactStatus=FAILED_TO_SCHEDULE) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("contactStatus").value(), "SCHEDULED");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration contactScheduledWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(180);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static GroundStationWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends GroundStationRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements GroundStationWaiter.Builder {
        private GroundStationClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public GroundStationWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public GroundStationWaiter.Builder client(GroundStationClient client) {
            this.client = client;
            return this;
        }

        public GroundStationWaiter build() {
            return new DefaultGroundStationWaiter(this);
        }
    }
}
