/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.guardduty;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.guardduty.model.AcceptInvitationRequest;
import software.amazon.awssdk.services.guardduty.model.AcceptInvitationResponse;
import software.amazon.awssdk.services.guardduty.model.ArchiveFindingsRequest;
import software.amazon.awssdk.services.guardduty.model.ArchiveFindingsResponse;
import software.amazon.awssdk.services.guardduty.model.BadRequestException;
import software.amazon.awssdk.services.guardduty.model.CreateDetectorRequest;
import software.amazon.awssdk.services.guardduty.model.CreateDetectorResponse;
import software.amazon.awssdk.services.guardduty.model.CreateFilterRequest;
import software.amazon.awssdk.services.guardduty.model.CreateFilterResponse;
import software.amazon.awssdk.services.guardduty.model.CreateIpSetRequest;
import software.amazon.awssdk.services.guardduty.model.CreateIpSetResponse;
import software.amazon.awssdk.services.guardduty.model.CreateMembersRequest;
import software.amazon.awssdk.services.guardduty.model.CreateMembersResponse;
import software.amazon.awssdk.services.guardduty.model.CreatePublishingDestinationRequest;
import software.amazon.awssdk.services.guardduty.model.CreatePublishingDestinationResponse;
import software.amazon.awssdk.services.guardduty.model.CreateSampleFindingsRequest;
import software.amazon.awssdk.services.guardduty.model.CreateSampleFindingsResponse;
import software.amazon.awssdk.services.guardduty.model.CreateThreatIntelSetRequest;
import software.amazon.awssdk.services.guardduty.model.CreateThreatIntelSetResponse;
import software.amazon.awssdk.services.guardduty.model.DeclineInvitationsRequest;
import software.amazon.awssdk.services.guardduty.model.DeclineInvitationsResponse;
import software.amazon.awssdk.services.guardduty.model.DeleteDetectorRequest;
import software.amazon.awssdk.services.guardduty.model.DeleteDetectorResponse;
import software.amazon.awssdk.services.guardduty.model.DeleteFilterRequest;
import software.amazon.awssdk.services.guardduty.model.DeleteFilterResponse;
import software.amazon.awssdk.services.guardduty.model.DeleteInvitationsRequest;
import software.amazon.awssdk.services.guardduty.model.DeleteInvitationsResponse;
import software.amazon.awssdk.services.guardduty.model.DeleteIpSetRequest;
import software.amazon.awssdk.services.guardduty.model.DeleteIpSetResponse;
import software.amazon.awssdk.services.guardduty.model.DeleteMembersRequest;
import software.amazon.awssdk.services.guardduty.model.DeleteMembersResponse;
import software.amazon.awssdk.services.guardduty.model.DeletePublishingDestinationRequest;
import software.amazon.awssdk.services.guardduty.model.DeletePublishingDestinationResponse;
import software.amazon.awssdk.services.guardduty.model.DeleteThreatIntelSetRequest;
import software.amazon.awssdk.services.guardduty.model.DeleteThreatIntelSetResponse;
import software.amazon.awssdk.services.guardduty.model.DescribeOrganizationConfigurationRequest;
import software.amazon.awssdk.services.guardduty.model.DescribeOrganizationConfigurationResponse;
import software.amazon.awssdk.services.guardduty.model.DescribePublishingDestinationRequest;
import software.amazon.awssdk.services.guardduty.model.DescribePublishingDestinationResponse;
import software.amazon.awssdk.services.guardduty.model.DisableOrganizationAdminAccountRequest;
import software.amazon.awssdk.services.guardduty.model.DisableOrganizationAdminAccountResponse;
import software.amazon.awssdk.services.guardduty.model.DisassociateFromMasterAccountRequest;
import software.amazon.awssdk.services.guardduty.model.DisassociateFromMasterAccountResponse;
import software.amazon.awssdk.services.guardduty.model.DisassociateMembersRequest;
import software.amazon.awssdk.services.guardduty.model.DisassociateMembersResponse;
import software.amazon.awssdk.services.guardduty.model.EnableOrganizationAdminAccountRequest;
import software.amazon.awssdk.services.guardduty.model.EnableOrganizationAdminAccountResponse;
import software.amazon.awssdk.services.guardduty.model.GetDetectorRequest;
import software.amazon.awssdk.services.guardduty.model.GetDetectorResponse;
import software.amazon.awssdk.services.guardduty.model.GetFilterRequest;
import software.amazon.awssdk.services.guardduty.model.GetFilterResponse;
import software.amazon.awssdk.services.guardduty.model.GetFindingsRequest;
import software.amazon.awssdk.services.guardduty.model.GetFindingsResponse;
import software.amazon.awssdk.services.guardduty.model.GetFindingsStatisticsRequest;
import software.amazon.awssdk.services.guardduty.model.GetFindingsStatisticsResponse;
import software.amazon.awssdk.services.guardduty.model.GetInvitationsCountRequest;
import software.amazon.awssdk.services.guardduty.model.GetInvitationsCountResponse;
import software.amazon.awssdk.services.guardduty.model.GetIpSetRequest;
import software.amazon.awssdk.services.guardduty.model.GetIpSetResponse;
import software.amazon.awssdk.services.guardduty.model.GetMasterAccountRequest;
import software.amazon.awssdk.services.guardduty.model.GetMasterAccountResponse;
import software.amazon.awssdk.services.guardduty.model.GetMemberDetectorsRequest;
import software.amazon.awssdk.services.guardduty.model.GetMemberDetectorsResponse;
import software.amazon.awssdk.services.guardduty.model.GetMembersRequest;
import software.amazon.awssdk.services.guardduty.model.GetMembersResponse;
import software.amazon.awssdk.services.guardduty.model.GetThreatIntelSetRequest;
import software.amazon.awssdk.services.guardduty.model.GetThreatIntelSetResponse;
import software.amazon.awssdk.services.guardduty.model.GetUsageStatisticsRequest;
import software.amazon.awssdk.services.guardduty.model.GetUsageStatisticsResponse;
import software.amazon.awssdk.services.guardduty.model.GuardDutyException;
import software.amazon.awssdk.services.guardduty.model.GuardDutyRequest;
import software.amazon.awssdk.services.guardduty.model.InternalServerErrorException;
import software.amazon.awssdk.services.guardduty.model.InviteMembersRequest;
import software.amazon.awssdk.services.guardduty.model.InviteMembersResponse;
import software.amazon.awssdk.services.guardduty.model.ListDetectorsRequest;
import software.amazon.awssdk.services.guardduty.model.ListDetectorsResponse;
import software.amazon.awssdk.services.guardduty.model.ListFiltersRequest;
import software.amazon.awssdk.services.guardduty.model.ListFiltersResponse;
import software.amazon.awssdk.services.guardduty.model.ListFindingsRequest;
import software.amazon.awssdk.services.guardduty.model.ListFindingsResponse;
import software.amazon.awssdk.services.guardduty.model.ListInvitationsRequest;
import software.amazon.awssdk.services.guardduty.model.ListInvitationsResponse;
import software.amazon.awssdk.services.guardduty.model.ListIpSetsRequest;
import software.amazon.awssdk.services.guardduty.model.ListIpSetsResponse;
import software.amazon.awssdk.services.guardduty.model.ListMembersRequest;
import software.amazon.awssdk.services.guardduty.model.ListMembersResponse;
import software.amazon.awssdk.services.guardduty.model.ListOrganizationAdminAccountsRequest;
import software.amazon.awssdk.services.guardduty.model.ListOrganizationAdminAccountsResponse;
import software.amazon.awssdk.services.guardduty.model.ListPublishingDestinationsRequest;
import software.amazon.awssdk.services.guardduty.model.ListPublishingDestinationsResponse;
import software.amazon.awssdk.services.guardduty.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.guardduty.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.guardduty.model.ListThreatIntelSetsRequest;
import software.amazon.awssdk.services.guardduty.model.ListThreatIntelSetsResponse;
import software.amazon.awssdk.services.guardduty.model.StartMonitoringMembersRequest;
import software.amazon.awssdk.services.guardduty.model.StartMonitoringMembersResponse;
import software.amazon.awssdk.services.guardduty.model.StopMonitoringMembersRequest;
import software.amazon.awssdk.services.guardduty.model.StopMonitoringMembersResponse;
import software.amazon.awssdk.services.guardduty.model.TagResourceRequest;
import software.amazon.awssdk.services.guardduty.model.TagResourceResponse;
import software.amazon.awssdk.services.guardduty.model.UnarchiveFindingsRequest;
import software.amazon.awssdk.services.guardduty.model.UnarchiveFindingsResponse;
import software.amazon.awssdk.services.guardduty.model.UntagResourceRequest;
import software.amazon.awssdk.services.guardduty.model.UntagResourceResponse;
import software.amazon.awssdk.services.guardduty.model.UpdateDetectorRequest;
import software.amazon.awssdk.services.guardduty.model.UpdateDetectorResponse;
import software.amazon.awssdk.services.guardduty.model.UpdateFilterRequest;
import software.amazon.awssdk.services.guardduty.model.UpdateFilterResponse;
import software.amazon.awssdk.services.guardduty.model.UpdateFindingsFeedbackRequest;
import software.amazon.awssdk.services.guardduty.model.UpdateFindingsFeedbackResponse;
import software.amazon.awssdk.services.guardduty.model.UpdateIpSetRequest;
import software.amazon.awssdk.services.guardduty.model.UpdateIpSetResponse;
import software.amazon.awssdk.services.guardduty.model.UpdateMemberDetectorsRequest;
import software.amazon.awssdk.services.guardduty.model.UpdateMemberDetectorsResponse;
import software.amazon.awssdk.services.guardduty.model.UpdateOrganizationConfigurationRequest;
import software.amazon.awssdk.services.guardduty.model.UpdateOrganizationConfigurationResponse;
import software.amazon.awssdk.services.guardduty.model.UpdatePublishingDestinationRequest;
import software.amazon.awssdk.services.guardduty.model.UpdatePublishingDestinationResponse;
import software.amazon.awssdk.services.guardduty.model.UpdateThreatIntelSetRequest;
import software.amazon.awssdk.services.guardduty.model.UpdateThreatIntelSetResponse;
import software.amazon.awssdk.services.guardduty.paginators.GetUsageStatisticsPublisher;
import software.amazon.awssdk.services.guardduty.paginators.ListDetectorsPublisher;
import software.amazon.awssdk.services.guardduty.paginators.ListFiltersPublisher;
import software.amazon.awssdk.services.guardduty.paginators.ListFindingsPublisher;
import software.amazon.awssdk.services.guardduty.paginators.ListIPSetsPublisher;
import software.amazon.awssdk.services.guardduty.paginators.ListInvitationsPublisher;
import software.amazon.awssdk.services.guardduty.paginators.ListMembersPublisher;
import software.amazon.awssdk.services.guardduty.paginators.ListOrganizationAdminAccountsPublisher;
import software.amazon.awssdk.services.guardduty.paginators.ListPublishingDestinationsPublisher;
import software.amazon.awssdk.services.guardduty.paginators.ListThreatIntelSetsPublisher;
import software.amazon.awssdk.services.guardduty.transform.AcceptInvitationRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.ArchiveFindingsRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.CreateDetectorRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.CreateFilterRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.CreateIpSetRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.CreateMembersRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.CreatePublishingDestinationRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.CreateSampleFindingsRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.CreateThreatIntelSetRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.DeclineInvitationsRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.DeleteDetectorRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.DeleteFilterRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.DeleteInvitationsRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.DeleteIpSetRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.DeleteMembersRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.DeletePublishingDestinationRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.DeleteThreatIntelSetRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.DescribeOrganizationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.DescribePublishingDestinationRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.DisableOrganizationAdminAccountRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.DisassociateFromMasterAccountRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.DisassociateMembersRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.EnableOrganizationAdminAccountRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.GetDetectorRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.GetFilterRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.GetFindingsRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.GetFindingsStatisticsRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.GetInvitationsCountRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.GetIpSetRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.GetMasterAccountRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.GetMemberDetectorsRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.GetMembersRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.GetThreatIntelSetRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.GetUsageStatisticsRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.InviteMembersRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.ListDetectorsRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.ListFiltersRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.ListFindingsRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.ListInvitationsRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.ListIpSetsRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.ListMembersRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.ListOrganizationAdminAccountsRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.ListPublishingDestinationsRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.ListThreatIntelSetsRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.StartMonitoringMembersRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.StopMonitoringMembersRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.UnarchiveFindingsRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.UpdateDetectorRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.UpdateFilterRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.UpdateFindingsFeedbackRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.UpdateIpSetRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.UpdateMemberDetectorsRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.UpdateOrganizationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.UpdatePublishingDestinationRequestMarshaller;
import software.amazon.awssdk.services.guardduty.transform.UpdateThreatIntelSetRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link GuardDutyAsyncClient}.
 *
 * @see GuardDutyAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultGuardDutyAsyncClient implements GuardDutyAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultGuardDutyAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultGuardDutyAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Accepts the invitation to be monitored by a GuardDuty administrator account.
     * </p>
     *
     * @param acceptInvitationRequest
     * @return A Java Future containing the result of the AcceptInvitation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.AcceptInvitation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/AcceptInvitation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<AcceptInvitationResponse> acceptInvitation(AcceptInvitationRequest acceptInvitationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acceptInvitationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptInvitation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AcceptInvitationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AcceptInvitationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AcceptInvitationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AcceptInvitationRequest, AcceptInvitationResponse>()
                            .withOperationName("AcceptInvitation")
                            .withMarshaller(new AcceptInvitationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(acceptInvitationRequest));
            CompletableFuture<AcceptInvitationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Archives GuardDuty findings that are specified by the list of finding IDs.
     * </p>
     * <note>
     * <p>
     * Only the administrator account can archive findings. Member accounts don't have permission to archive findings
     * from their accounts.
     * </p>
     * </note>
     *
     * @param archiveFindingsRequest
     * @return A Java Future containing the result of the ArchiveFindings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.ArchiveFindings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/ArchiveFindings" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ArchiveFindingsResponse> archiveFindings(ArchiveFindingsRequest archiveFindingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, archiveFindingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ArchiveFindings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ArchiveFindingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ArchiveFindingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ArchiveFindingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ArchiveFindingsRequest, ArchiveFindingsResponse>()
                            .withOperationName("ArchiveFindings")
                            .withMarshaller(new ArchiveFindingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(archiveFindingsRequest));
            CompletableFuture<ArchiveFindingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a single Amazon GuardDuty detector. A detector is a resource that represents the GuardDuty service. To
     * start using GuardDuty, you must create a detector in each Region where you enable the service. You can have only
     * one detector per account per Region. All data sources are enabled in a new detector by default.
     * </p>
     *
     * @param createDetectorRequest
     * @return A Java Future containing the result of the CreateDetector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.CreateDetector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/CreateDetector" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDetectorResponse> createDetector(CreateDetectorRequest createDetectorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDetectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDetector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDetectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDetectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDetectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDetectorRequest, CreateDetectorResponse>()
                            .withOperationName("CreateDetector")
                            .withMarshaller(new CreateDetectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createDetectorRequest));
            CompletableFuture<CreateDetectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a filter using the specified finding criteria.
     * </p>
     *
     * @param createFilterRequest
     * @return A Java Future containing the result of the CreateFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.CreateFilter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/CreateFilter" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateFilterResponse> createFilter(CreateFilterRequest createFilterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFilterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFilter");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateFilterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateFilterRequest, CreateFilterResponse>()
                            .withOperationName("CreateFilter").withMarshaller(new CreateFilterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createFilterRequest));
            CompletableFuture<CreateFilterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new IPSet, which is called a trusted IP list in the console user interface. An IPSet is a list of IP
     * addresses that are trusted for secure communication with Amazon Web Services infrastructure and applications.
     * GuardDuty doesn't generate findings for IP addresses that are included in IPSets. Only users from the
     * administrator account can use this operation.
     * </p>
     *
     * @param createIpSetRequest
     * @return A Java Future containing the result of the CreateIPSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.CreateIPSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/CreateIPSet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateIpSetResponse> createIPSet(CreateIpSetRequest createIpSetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createIpSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateIPSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateIpSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateIpSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateIpSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateIpSetRequest, CreateIpSetResponse>()
                            .withOperationName("CreateIPSet").withMarshaller(new CreateIpSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createIpSetRequest));
            CompletableFuture<CreateIpSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates member accounts of the current Amazon Web Services account by specifying a list of Amazon Web Services
     * account IDs. This step is a prerequisite for managing the associated member accounts either by invitation or
     * through an organization.
     * </p>
     * <p>
     * When using <code>Create Members</code> as an organizations delegated administrator this action will enable
     * GuardDuty in the added member accounts, with the exception of the organization delegated administrator account,
     * which must enable GuardDuty prior to being added as a member.
     * </p>
     * <p>
     * If you are adding accounts by invitation use this action after GuardDuty has been enabled in potential member
     * accounts and before using <a
     * href="https://docs.aws.amazon.com/guardduty/latest/APIReference/API_InviteMembers.html">
     * <code>Invite Members</code> </a>.
     * </p>
     *
     * @param createMembersRequest
     * @return A Java Future containing the result of the CreateMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.CreateMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/CreateMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMembersResponse> createMembers(CreateMembersRequest createMembersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMembersRequest, CreateMembersResponse>()
                            .withOperationName("CreateMembers")
                            .withMarshaller(new CreateMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createMembersRequest));
            CompletableFuture<CreateMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a publishing destination to export findings to. The resource to export findings to must exist before you
     * use this operation.
     * </p>
     *
     * @param createPublishingDestinationRequest
     * @return A Java Future containing the result of the CreatePublishingDestination operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.CreatePublishingDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/CreatePublishingDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePublishingDestinationResponse> createPublishingDestination(
            CreatePublishingDestinationRequest createPublishingDestinationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPublishingDestinationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePublishingDestination");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePublishingDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePublishingDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePublishingDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePublishingDestinationRequest, CreatePublishingDestinationResponse>()
                            .withOperationName("CreatePublishingDestination")
                            .withMarshaller(new CreatePublishingDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createPublishingDestinationRequest));
            CompletableFuture<CreatePublishingDestinationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Generates example findings of types specified by the list of finding types. If 'NULL' is specified for
     * <code>findingTypes</code>, the API generates example findings of all supported finding types.
     * </p>
     *
     * @param createSampleFindingsRequest
     * @return A Java Future containing the result of the CreateSampleFindings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.CreateSampleFindings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/CreateSampleFindings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSampleFindingsResponse> createSampleFindings(
            CreateSampleFindingsRequest createSampleFindingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSampleFindingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSampleFindings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSampleFindingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSampleFindingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSampleFindingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSampleFindingsRequest, CreateSampleFindingsResponse>()
                            .withOperationName("CreateSampleFindings")
                            .withMarshaller(new CreateSampleFindingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createSampleFindingsRequest));
            CompletableFuture<CreateSampleFindingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new ThreatIntelSet. ThreatIntelSets consist of known malicious IP addresses. GuardDuty generates
     * findings based on ThreatIntelSets. Only users of the administrator account can use this operation.
     * </p>
     *
     * @param createThreatIntelSetRequest
     * @return A Java Future containing the result of the CreateThreatIntelSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.CreateThreatIntelSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/CreateThreatIntelSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateThreatIntelSetResponse> createThreatIntelSet(
            CreateThreatIntelSetRequest createThreatIntelSetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createThreatIntelSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateThreatIntelSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateThreatIntelSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateThreatIntelSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateThreatIntelSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateThreatIntelSetRequest, CreateThreatIntelSetResponse>()
                            .withOperationName("CreateThreatIntelSet")
                            .withMarshaller(new CreateThreatIntelSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createThreatIntelSetRequest));
            CompletableFuture<CreateThreatIntelSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Declines invitations sent to the current member account by Amazon Web Services accounts specified by their
     * account IDs.
     * </p>
     *
     * @param declineInvitationsRequest
     * @return A Java Future containing the result of the DeclineInvitations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.DeclineInvitations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/DeclineInvitations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeclineInvitationsResponse> declineInvitations(DeclineInvitationsRequest declineInvitationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, declineInvitationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeclineInvitations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeclineInvitationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeclineInvitationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeclineInvitationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeclineInvitationsRequest, DeclineInvitationsResponse>()
                            .withOperationName("DeclineInvitations")
                            .withMarshaller(new DeclineInvitationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(declineInvitationsRequest));
            CompletableFuture<DeclineInvitationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Amazon GuardDuty detector that is specified by the detector ID.
     * </p>
     *
     * @param deleteDetectorRequest
     * @return A Java Future containing the result of the DeleteDetector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.DeleteDetector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/DeleteDetector" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDetectorResponse> deleteDetector(DeleteDetectorRequest deleteDetectorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDetectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDetector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDetectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDetectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDetectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDetectorRequest, DeleteDetectorResponse>()
                            .withOperationName("DeleteDetector")
                            .withMarshaller(new DeleteDetectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDetectorRequest));
            CompletableFuture<DeleteDetectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the filter specified by the filter name.
     * </p>
     *
     * @param deleteFilterRequest
     * @return A Java Future containing the result of the DeleteFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.DeleteFilter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/DeleteFilter" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFilterResponse> deleteFilter(DeleteFilterRequest deleteFilterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFilterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFilter");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteFilterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteFilterRequest, DeleteFilterResponse>()
                            .withOperationName("DeleteFilter").withMarshaller(new DeleteFilterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteFilterRequest));
            CompletableFuture<DeleteFilterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the IPSet specified by the <code>ipSetId</code>. IPSets are called trusted IP lists in the console user
     * interface.
     * </p>
     *
     * @param deleteIpSetRequest
     * @return A Java Future containing the result of the DeleteIPSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.DeleteIPSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/DeleteIPSet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteIpSetResponse> deleteIPSet(DeleteIpSetRequest deleteIpSetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteIpSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteIPSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteIpSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteIpSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteIpSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteIpSetRequest, DeleteIpSetResponse>()
                            .withOperationName("DeleteIPSet").withMarshaller(new DeleteIpSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteIpSetRequest));
            CompletableFuture<DeleteIpSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes invitations sent to the current member account by Amazon Web Services accounts specified by their account
     * IDs.
     * </p>
     *
     * @param deleteInvitationsRequest
     * @return A Java Future containing the result of the DeleteInvitations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.DeleteInvitations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/DeleteInvitations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteInvitationsResponse> deleteInvitations(DeleteInvitationsRequest deleteInvitationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteInvitationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteInvitations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteInvitationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteInvitationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteInvitationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteInvitationsRequest, DeleteInvitationsResponse>()
                            .withOperationName("DeleteInvitations")
                            .withMarshaller(new DeleteInvitationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteInvitationsRequest));
            CompletableFuture<DeleteInvitationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes GuardDuty member accounts (to the current GuardDuty administrator account) specified by the account IDs.
     * </p>
     *
     * @param deleteMembersRequest
     * @return A Java Future containing the result of the DeleteMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.DeleteMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/DeleteMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMembersResponse> deleteMembers(DeleteMembersRequest deleteMembersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMembersRequest, DeleteMembersResponse>()
                            .withOperationName("DeleteMembers")
                            .withMarshaller(new DeleteMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteMembersRequest));
            CompletableFuture<DeleteMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the publishing definition with the specified <code>destinationId</code>.
     * </p>
     *
     * @param deletePublishingDestinationRequest
     * @return A Java Future containing the result of the DeletePublishingDestination operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.DeletePublishingDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/DeletePublishingDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePublishingDestinationResponse> deletePublishingDestination(
            DeletePublishingDestinationRequest deletePublishingDestinationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePublishingDestinationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePublishingDestination");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePublishingDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePublishingDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePublishingDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePublishingDestinationRequest, DeletePublishingDestinationResponse>()
                            .withOperationName("DeletePublishingDestination")
                            .withMarshaller(new DeletePublishingDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deletePublishingDestinationRequest));
            CompletableFuture<DeletePublishingDestinationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the ThreatIntelSet specified by the ThreatIntelSet ID.
     * </p>
     *
     * @param deleteThreatIntelSetRequest
     * @return A Java Future containing the result of the DeleteThreatIntelSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.DeleteThreatIntelSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/DeleteThreatIntelSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteThreatIntelSetResponse> deleteThreatIntelSet(
            DeleteThreatIntelSetRequest deleteThreatIntelSetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteThreatIntelSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteThreatIntelSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteThreatIntelSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteThreatIntelSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteThreatIntelSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteThreatIntelSetRequest, DeleteThreatIntelSetResponse>()
                            .withOperationName("DeleteThreatIntelSet")
                            .withMarshaller(new DeleteThreatIntelSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteThreatIntelSetRequest));
            CompletableFuture<DeleteThreatIntelSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the account selected as the delegated administrator for GuardDuty.
     * </p>
     *
     * @param describeOrganizationConfigurationRequest
     * @return A Java Future containing the result of the DescribeOrganizationConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.DescribeOrganizationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/DescribeOrganizationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeOrganizationConfigurationResponse> describeOrganizationConfiguration(
            DescribeOrganizationConfigurationRequest describeOrganizationConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeOrganizationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeOrganizationConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeOrganizationConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeOrganizationConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeOrganizationConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeOrganizationConfigurationRequest, DescribeOrganizationConfigurationResponse>()
                            .withOperationName("DescribeOrganizationConfiguration")
                            .withMarshaller(new DescribeOrganizationConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeOrganizationConfigurationRequest));
            CompletableFuture<DescribeOrganizationConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the publishing destination specified by the provided <code>destinationId</code>.
     * </p>
     *
     * @param describePublishingDestinationRequest
     * @return A Java Future containing the result of the DescribePublishingDestination operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.DescribePublishingDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/DescribePublishingDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePublishingDestinationResponse> describePublishingDestination(
            DescribePublishingDestinationRequest describePublishingDestinationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describePublishingDestinationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePublishingDestination");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePublishingDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribePublishingDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribePublishingDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePublishingDestinationRequest, DescribePublishingDestinationResponse>()
                            .withOperationName("DescribePublishingDestination")
                            .withMarshaller(new DescribePublishingDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describePublishingDestinationRequest));
            CompletableFuture<DescribePublishingDestinationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disables an Amazon Web Services account within the Organization as the GuardDuty delegated administrator.
     * </p>
     *
     * @param disableOrganizationAdminAccountRequest
     * @return A Java Future containing the result of the DisableOrganizationAdminAccount operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.DisableOrganizationAdminAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/DisableOrganizationAdminAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisableOrganizationAdminAccountResponse> disableOrganizationAdminAccount(
            DisableOrganizationAdminAccountRequest disableOrganizationAdminAccountRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disableOrganizationAdminAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableOrganizationAdminAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisableOrganizationAdminAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisableOrganizationAdminAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisableOrganizationAdminAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisableOrganizationAdminAccountRequest, DisableOrganizationAdminAccountResponse>()
                            .withOperationName("DisableOrganizationAdminAccount")
                            .withMarshaller(new DisableOrganizationAdminAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disableOrganizationAdminAccountRequest));
            CompletableFuture<DisableOrganizationAdminAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates the current GuardDuty member account from its administrator account.
     * </p>
     *
     * @param disassociateFromMasterAccountRequest
     * @return A Java Future containing the result of the DisassociateFromMasterAccount operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.DisassociateFromMasterAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/DisassociateFromMasterAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateFromMasterAccountResponse> disassociateFromMasterAccount(
            DisassociateFromMasterAccountRequest disassociateFromMasterAccountRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateFromMasterAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateFromMasterAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateFromMasterAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateFromMasterAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateFromMasterAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateFromMasterAccountRequest, DisassociateFromMasterAccountResponse>()
                            .withOperationName("DisassociateFromMasterAccount")
                            .withMarshaller(new DisassociateFromMasterAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disassociateFromMasterAccountRequest));
            CompletableFuture<DisassociateFromMasterAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates GuardDuty member accounts (to the current GuardDuty administrator account) specified by the account
     * IDs. Member accounts added through <a
     * href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_invitations.html">Invitation</a> get deleted from
     * the current GuardDuty administrator account after 30 days of disassociation.
     * </p>
     *
     * @param disassociateMembersRequest
     * @return A Java Future containing the result of the DisassociateMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.DisassociateMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/DisassociateMembers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateMembersResponse> disassociateMembers(
            DisassociateMembersRequest disassociateMembersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateMembersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateMembersRequest, DisassociateMembersResponse>()
                            .withOperationName("DisassociateMembers")
                            .withMarshaller(new DisassociateMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disassociateMembersRequest));
            CompletableFuture<DisassociateMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables an Amazon Web Services account within the organization as the GuardDuty delegated administrator.
     * </p>
     *
     * @param enableOrganizationAdminAccountRequest
     * @return A Java Future containing the result of the EnableOrganizationAdminAccount operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.EnableOrganizationAdminAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/EnableOrganizationAdminAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<EnableOrganizationAdminAccountResponse> enableOrganizationAdminAccount(
            EnableOrganizationAdminAccountRequest enableOrganizationAdminAccountRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                enableOrganizationAdminAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableOrganizationAdminAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableOrganizationAdminAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, EnableOrganizationAdminAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EnableOrganizationAdminAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableOrganizationAdminAccountRequest, EnableOrganizationAdminAccountResponse>()
                            .withOperationName("EnableOrganizationAdminAccount")
                            .withMarshaller(new EnableOrganizationAdminAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(enableOrganizationAdminAccountRequest));
            CompletableFuture<EnableOrganizationAdminAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves an Amazon GuardDuty detector specified by the detectorId.
     * </p>
     *
     * @param getDetectorRequest
     * @return A Java Future containing the result of the GetDetector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.GetDetector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/GetDetector" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetDetectorResponse> getDetector(GetDetectorRequest getDetectorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDetectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDetector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDetectorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetDetectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDetectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDetectorRequest, GetDetectorResponse>()
                            .withOperationName("GetDetector").withMarshaller(new GetDetectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getDetectorRequest));
            CompletableFuture<GetDetectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the details of the filter specified by the filter name.
     * </p>
     *
     * @param getFilterRequest
     * @return A Java Future containing the result of the GetFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.GetFilter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/GetFilter" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetFilterResponse> getFilter(GetFilterRequest getFilterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getFilterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetFilter");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetFilterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetFilterRequest, GetFilterResponse>().withOperationName("GetFilter")
                            .withMarshaller(new GetFilterRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getFilterRequest));
            CompletableFuture<GetFilterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes Amazon GuardDuty findings specified by finding IDs.
     * </p>
     *
     * @param getFindingsRequest
     * @return A Java Future containing the result of the GetFindings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.GetFindings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/GetFindings" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetFindingsResponse> getFindings(GetFindingsRequest getFindingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getFindingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetFindings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetFindingsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetFindingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetFindingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetFindingsRequest, GetFindingsResponse>()
                            .withOperationName("GetFindings").withMarshaller(new GetFindingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getFindingsRequest));
            CompletableFuture<GetFindingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists Amazon GuardDuty findings statistics for the specified detector ID.
     * </p>
     *
     * @param getFindingsStatisticsRequest
     * @return A Java Future containing the result of the GetFindingsStatistics operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.GetFindingsStatistics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/GetFindingsStatistics"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetFindingsStatisticsResponse> getFindingsStatistics(
            GetFindingsStatisticsRequest getFindingsStatisticsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getFindingsStatisticsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetFindingsStatistics");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetFindingsStatisticsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetFindingsStatisticsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetFindingsStatisticsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetFindingsStatisticsRequest, GetFindingsStatisticsResponse>()
                            .withOperationName("GetFindingsStatistics")
                            .withMarshaller(new GetFindingsStatisticsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getFindingsStatisticsRequest));
            CompletableFuture<GetFindingsStatisticsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the IPSet specified by the <code>ipSetId</code>.
     * </p>
     *
     * @param getIpSetRequest
     * @return A Java Future containing the result of the GetIPSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.GetIPSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/GetIPSet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetIpSetResponse> getIPSet(GetIpSetRequest getIpSetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getIpSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetIPSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetIpSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetIpSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetIpSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetIpSetRequest, GetIpSetResponse>().withOperationName("GetIPSet")
                            .withMarshaller(new GetIpSetRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getIpSetRequest));
            CompletableFuture<GetIpSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the count of all GuardDuty membership invitations that were sent to the current member account except the
     * currently accepted invitation.
     * </p>
     *
     * @param getInvitationsCountRequest
     * @return A Java Future containing the result of the GetInvitationsCount operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.GetInvitationsCount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/GetInvitationsCount" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetInvitationsCountResponse> getInvitationsCount(
            GetInvitationsCountRequest getInvitationsCountRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInvitationsCountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInvitationsCount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetInvitationsCountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetInvitationsCountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetInvitationsCountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetInvitationsCountRequest, GetInvitationsCountResponse>()
                            .withOperationName("GetInvitationsCount")
                            .withMarshaller(new GetInvitationsCountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getInvitationsCountRequest));
            CompletableFuture<GetInvitationsCountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides the details for the GuardDuty administrator account associated with the current GuardDuty member
     * account.
     * </p>
     *
     * @param getMasterAccountRequest
     * @return A Java Future containing the result of the GetMasterAccount operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.GetMasterAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/GetMasterAccount" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMasterAccountResponse> getMasterAccount(GetMasterAccountRequest getMasterAccountRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMasterAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMasterAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMasterAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetMasterAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMasterAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMasterAccountRequest, GetMasterAccountResponse>()
                            .withOperationName("GetMasterAccount")
                            .withMarshaller(new GetMasterAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getMasterAccountRequest));
            CompletableFuture<GetMasterAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes which data sources are enabled for the member account's detector.
     * </p>
     *
     * @param getMemberDetectorsRequest
     * @return A Java Future containing the result of the GetMemberDetectors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.GetMemberDetectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/GetMemberDetectors" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMemberDetectorsResponse> getMemberDetectors(GetMemberDetectorsRequest getMemberDetectorsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMemberDetectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMemberDetectors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMemberDetectorsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetMemberDetectorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMemberDetectorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMemberDetectorsRequest, GetMemberDetectorsResponse>()
                            .withOperationName("GetMemberDetectors")
                            .withMarshaller(new GetMemberDetectorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getMemberDetectorsRequest));
            CompletableFuture<GetMemberDetectorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves GuardDuty member accounts (of the current GuardDuty administrator account) specified by the account
     * IDs.
     * </p>
     *
     * @param getMembersRequest
     * @return A Java Future containing the result of the GetMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.GetMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/GetMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetMembersResponse> getMembers(GetMembersRequest getMembersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMembersRequest, GetMembersResponse>().withOperationName("GetMembers")
                            .withMarshaller(new GetMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getMembersRequest));
            CompletableFuture<GetMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the ThreatIntelSet that is specified by the ThreatIntelSet ID.
     * </p>
     *
     * @param getThreatIntelSetRequest
     * @return A Java Future containing the result of the GetThreatIntelSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.GetThreatIntelSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/GetThreatIntelSet" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetThreatIntelSetResponse> getThreatIntelSet(GetThreatIntelSetRequest getThreatIntelSetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getThreatIntelSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetThreatIntelSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetThreatIntelSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetThreatIntelSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetThreatIntelSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetThreatIntelSetRequest, GetThreatIntelSetResponse>()
                            .withOperationName("GetThreatIntelSet")
                            .withMarshaller(new GetThreatIntelSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getThreatIntelSetRequest));
            CompletableFuture<GetThreatIntelSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists Amazon GuardDuty usage statistics over the last 30 days for the specified detector ID. For newly enabled
     * detectors or data sources the cost returned will include only the usage so far under 30 days, this may differ
     * from the cost metrics in the console, which projects usage over 30 days to provide a monthly cost estimate. For
     * more information see <a
     * href="https://docs.aws.amazon.com/guardduty/latest/ug/monitoring_costs.html#usage-calculations">Understanding How
     * Usage Costs are Calculated</a>.
     * </p>
     *
     * @param getUsageStatisticsRequest
     * @return A Java Future containing the result of the GetUsageStatistics operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.GetUsageStatistics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/GetUsageStatistics" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetUsageStatisticsResponse> getUsageStatistics(GetUsageStatisticsRequest getUsageStatisticsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getUsageStatisticsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetUsageStatistics");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetUsageStatisticsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetUsageStatisticsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetUsageStatisticsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetUsageStatisticsRequest, GetUsageStatisticsResponse>()
                            .withOperationName("GetUsageStatistics")
                            .withMarshaller(new GetUsageStatisticsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getUsageStatisticsRequest));
            CompletableFuture<GetUsageStatisticsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists Amazon GuardDuty usage statistics over the last 30 days for the specified detector ID. For newly enabled
     * detectors or data sources the cost returned will include only the usage so far under 30 days, this may differ
     * from the cost metrics in the console, which projects usage over 30 days to provide a monthly cost estimate. For
     * more information see <a
     * href="https://docs.aws.amazon.com/guardduty/latest/ug/monitoring_costs.html#usage-calculations">Understanding How
     * Usage Costs are Calculated</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getUsageStatistics(software.amazon.awssdk.services.guardduty.model.GetUsageStatisticsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.guardduty.paginators.GetUsageStatisticsPublisher publisher = client.getUsageStatisticsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.guardduty.paginators.GetUsageStatisticsPublisher publisher = client.getUsageStatisticsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.guardduty.model.GetUsageStatisticsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.guardduty.model.GetUsageStatisticsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getUsageStatistics(software.amazon.awssdk.services.guardduty.model.GetUsageStatisticsRequest)}
     * operation.</b>
     * </p>
     *
     * @param getUsageStatisticsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.GetUsageStatistics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/GetUsageStatistics" target="_top">AWS
     *      API Documentation</a>
     */
    public GetUsageStatisticsPublisher getUsageStatisticsPaginator(GetUsageStatisticsRequest getUsageStatisticsRequest) {
        return new GetUsageStatisticsPublisher(this, applyPaginatorUserAgent(getUsageStatisticsRequest));
    }

    /**
     * <p>
     * Invites other Amazon Web Services accounts (created as members of the current Amazon Web Services account by
     * CreateMembers) to enable GuardDuty, and allow the current Amazon Web Services account to view and manage these
     * accounts' findings on their behalf as the GuardDuty administrator account.
     * </p>
     *
     * @param inviteMembersRequest
     * @return A Java Future containing the result of the InviteMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.InviteMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/InviteMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<InviteMembersResponse> inviteMembers(InviteMembersRequest inviteMembersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, inviteMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InviteMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<InviteMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    InviteMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<InviteMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<InviteMembersRequest, InviteMembersResponse>()
                            .withOperationName("InviteMembers")
                            .withMarshaller(new InviteMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(inviteMembersRequest));
            CompletableFuture<InviteMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists detectorIds of all the existing Amazon GuardDuty detector resources.
     * </p>
     *
     * @param listDetectorsRequest
     * @return A Java Future containing the result of the ListDetectors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.ListDetectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/ListDetectors" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDetectorsResponse> listDetectors(ListDetectorsRequest listDetectorsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDetectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDetectors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDetectorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDetectorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDetectorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDetectorsRequest, ListDetectorsResponse>()
                            .withOperationName("ListDetectors")
                            .withMarshaller(new ListDetectorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDetectorsRequest));
            CompletableFuture<ListDetectorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists detectorIds of all the existing Amazon GuardDuty detector resources.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listDetectors(software.amazon.awssdk.services.guardduty.model.ListDetectorsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.guardduty.paginators.ListDetectorsPublisher publisher = client.listDetectorsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.guardduty.paginators.ListDetectorsPublisher publisher = client.listDetectorsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.guardduty.model.ListDetectorsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.guardduty.model.ListDetectorsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDetectors(software.amazon.awssdk.services.guardduty.model.ListDetectorsRequest)} operation.</b>
     * </p>
     *
     * @param listDetectorsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.ListDetectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/ListDetectors" target="_top">AWS API
     *      Documentation</a>
     */
    public ListDetectorsPublisher listDetectorsPaginator(ListDetectorsRequest listDetectorsRequest) {
        return new ListDetectorsPublisher(this, applyPaginatorUserAgent(listDetectorsRequest));
    }

    /**
     * <p>
     * Returns a paginated list of the current filters.
     * </p>
     *
     * @param listFiltersRequest
     * @return A Java Future containing the result of the ListFilters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.ListFilters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/ListFilters" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListFiltersResponse> listFilters(ListFiltersRequest listFiltersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFiltersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFilters");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFiltersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListFiltersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFiltersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFiltersRequest, ListFiltersResponse>()
                            .withOperationName("ListFilters").withMarshaller(new ListFiltersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listFiltersRequest));
            CompletableFuture<ListFiltersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a paginated list of the current filters.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listFilters(software.amazon.awssdk.services.guardduty.model.ListFiltersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.guardduty.paginators.ListFiltersPublisher publisher = client.listFiltersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.guardduty.paginators.ListFiltersPublisher publisher = client.listFiltersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.guardduty.model.ListFiltersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.guardduty.model.ListFiltersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listFilters(software.amazon.awssdk.services.guardduty.model.ListFiltersRequest)} operation.</b>
     * </p>
     *
     * @param listFiltersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.ListFilters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/ListFilters" target="_top">AWS API
     *      Documentation</a>
     */
    public ListFiltersPublisher listFiltersPaginator(ListFiltersRequest listFiltersRequest) {
        return new ListFiltersPublisher(this, applyPaginatorUserAgent(listFiltersRequest));
    }

    /**
     * <p>
     * Lists Amazon GuardDuty findings for the specified detector ID.
     * </p>
     *
     * @param listFindingsRequest
     * @return A Java Future containing the result of the ListFindings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.ListFindings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/ListFindings" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListFindingsResponse> listFindings(ListFindingsRequest listFindingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFindingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFindings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFindingsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListFindingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFindingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFindingsRequest, ListFindingsResponse>()
                            .withOperationName("ListFindings").withMarshaller(new ListFindingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listFindingsRequest));
            CompletableFuture<ListFindingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists Amazon GuardDuty findings for the specified detector ID.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listFindings(software.amazon.awssdk.services.guardduty.model.ListFindingsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.guardduty.paginators.ListFindingsPublisher publisher = client.listFindingsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.guardduty.paginators.ListFindingsPublisher publisher = client.listFindingsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.guardduty.model.ListFindingsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.guardduty.model.ListFindingsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listFindings(software.amazon.awssdk.services.guardduty.model.ListFindingsRequest)} operation.</b>
     * </p>
     *
     * @param listFindingsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.ListFindings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/ListFindings" target="_top">AWS API
     *      Documentation</a>
     */
    public ListFindingsPublisher listFindingsPaginator(ListFindingsRequest listFindingsRequest) {
        return new ListFindingsPublisher(this, applyPaginatorUserAgent(listFindingsRequest));
    }

    /**
     * <p>
     * Lists the IPSets of the GuardDuty service specified by the detector ID. If you use this operation from a member
     * account, the IPSets returned are the IPSets from the associated administrator account.
     * </p>
     *
     * @param listIpSetsRequest
     * @return A Java Future containing the result of the ListIPSets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.ListIPSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/ListIPSets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListIpSetsResponse> listIPSets(ListIpSetsRequest listIpSetsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listIpSetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListIPSets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListIpSetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListIpSetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListIpSetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListIpSetsRequest, ListIpSetsResponse>().withOperationName("ListIPSets")
                            .withMarshaller(new ListIpSetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listIpSetsRequest));
            CompletableFuture<ListIpSetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the IPSets of the GuardDuty service specified by the detector ID. If you use this operation from a member
     * account, the IPSets returned are the IPSets from the associated administrator account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listIPSets(software.amazon.awssdk.services.guardduty.model.ListIpSetsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.guardduty.paginators.ListIPSetsPublisher publisher = client.listIPSetsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.guardduty.paginators.ListIPSetsPublisher publisher = client.listIPSetsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.guardduty.model.ListIpSetsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.guardduty.model.ListIpSetsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listIPSets(software.amazon.awssdk.services.guardduty.model.ListIpSetsRequest)} operation.</b>
     * </p>
     *
     * @param listIpSetsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.ListIPSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/ListIPSets" target="_top">AWS API
     *      Documentation</a>
     */
    public ListIPSetsPublisher listIPSetsPaginator(ListIpSetsRequest listIpSetsRequest) {
        return new ListIPSetsPublisher(this, applyPaginatorUserAgent(listIpSetsRequest));
    }

    /**
     * <p>
     * Lists all GuardDuty membership invitations that were sent to the current Amazon Web Services account.
     * </p>
     *
     * @param listInvitationsRequest
     * @return A Java Future containing the result of the ListInvitations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.ListInvitations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/ListInvitations" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListInvitationsResponse> listInvitations(ListInvitationsRequest listInvitationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInvitationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInvitations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInvitationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListInvitationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListInvitationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInvitationsRequest, ListInvitationsResponse>()
                            .withOperationName("ListInvitations")
                            .withMarshaller(new ListInvitationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listInvitationsRequest));
            CompletableFuture<ListInvitationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all GuardDuty membership invitations that were sent to the current Amazon Web Services account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listInvitations(software.amazon.awssdk.services.guardduty.model.ListInvitationsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.guardduty.paginators.ListInvitationsPublisher publisher = client.listInvitationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.guardduty.paginators.ListInvitationsPublisher publisher = client.listInvitationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.guardduty.model.ListInvitationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.guardduty.model.ListInvitationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listInvitations(software.amazon.awssdk.services.guardduty.model.ListInvitationsRequest)} operation.</b>
     * </p>
     *
     * @param listInvitationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.ListInvitations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/ListInvitations" target="_top">AWS API
     *      Documentation</a>
     */
    public ListInvitationsPublisher listInvitationsPaginator(ListInvitationsRequest listInvitationsRequest) {
        return new ListInvitationsPublisher(this, applyPaginatorUserAgent(listInvitationsRequest));
    }

    /**
     * <p>
     * Lists details about all member accounts for the current GuardDuty administrator account.
     * </p>
     *
     * @param listMembersRequest
     * @return A Java Future containing the result of the ListMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.ListMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/ListMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListMembersResponse> listMembers(ListMembersRequest listMembersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMembersRequest, ListMembersResponse>()
                            .withOperationName("ListMembers").withMarshaller(new ListMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listMembersRequest));
            CompletableFuture<ListMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists details about all member accounts for the current GuardDuty administrator account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listMembers(software.amazon.awssdk.services.guardduty.model.ListMembersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.guardduty.paginators.ListMembersPublisher publisher = client.listMembersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.guardduty.paginators.ListMembersPublisher publisher = client.listMembersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.guardduty.model.ListMembersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.guardduty.model.ListMembersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMembers(software.amazon.awssdk.services.guardduty.model.ListMembersRequest)} operation.</b>
     * </p>
     *
     * @param listMembersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.ListMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/ListMembers" target="_top">AWS API
     *      Documentation</a>
     */
    public ListMembersPublisher listMembersPaginator(ListMembersRequest listMembersRequest) {
        return new ListMembersPublisher(this, applyPaginatorUserAgent(listMembersRequest));
    }

    /**
     * <p>
     * Lists the accounts configured as GuardDuty delegated administrators.
     * </p>
     *
     * @param listOrganizationAdminAccountsRequest
     * @return A Java Future containing the result of the ListOrganizationAdminAccounts operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.ListOrganizationAdminAccounts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/ListOrganizationAdminAccounts"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListOrganizationAdminAccountsResponse> listOrganizationAdminAccounts(
            ListOrganizationAdminAccountsRequest listOrganizationAdminAccountsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listOrganizationAdminAccountsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOrganizationAdminAccounts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListOrganizationAdminAccountsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListOrganizationAdminAccountsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListOrganizationAdminAccountsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListOrganizationAdminAccountsRequest, ListOrganizationAdminAccountsResponse>()
                            .withOperationName("ListOrganizationAdminAccounts")
                            .withMarshaller(new ListOrganizationAdminAccountsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listOrganizationAdminAccountsRequest));
            CompletableFuture<ListOrganizationAdminAccountsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the accounts configured as GuardDuty delegated administrators.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listOrganizationAdminAccounts(software.amazon.awssdk.services.guardduty.model.ListOrganizationAdminAccountsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.guardduty.paginators.ListOrganizationAdminAccountsPublisher publisher = client.listOrganizationAdminAccountsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.guardduty.paginators.ListOrganizationAdminAccountsPublisher publisher = client.listOrganizationAdminAccountsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.guardduty.model.ListOrganizationAdminAccountsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.guardduty.model.ListOrganizationAdminAccountsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listOrganizationAdminAccounts(software.amazon.awssdk.services.guardduty.model.ListOrganizationAdminAccountsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listOrganizationAdminAccountsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.ListOrganizationAdminAccounts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/ListOrganizationAdminAccounts"
     *      target="_top">AWS API Documentation</a>
     */
    public ListOrganizationAdminAccountsPublisher listOrganizationAdminAccountsPaginator(
            ListOrganizationAdminAccountsRequest listOrganizationAdminAccountsRequest) {
        return new ListOrganizationAdminAccountsPublisher(this, applyPaginatorUserAgent(listOrganizationAdminAccountsRequest));
    }

    /**
     * <p>
     * Returns a list of publishing destinations associated with the specified <code>detectorId</code>.
     * </p>
     *
     * @param listPublishingDestinationsRequest
     * @return A Java Future containing the result of the ListPublishingDestinations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.ListPublishingDestinations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/ListPublishingDestinations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPublishingDestinationsResponse> listPublishingDestinations(
            ListPublishingDestinationsRequest listPublishingDestinationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPublishingDestinationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPublishingDestinations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPublishingDestinationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPublishingDestinationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPublishingDestinationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPublishingDestinationsRequest, ListPublishingDestinationsResponse>()
                            .withOperationName("ListPublishingDestinations")
                            .withMarshaller(new ListPublishingDestinationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPublishingDestinationsRequest));
            CompletableFuture<ListPublishingDestinationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of publishing destinations associated with the specified <code>detectorId</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPublishingDestinations(software.amazon.awssdk.services.guardduty.model.ListPublishingDestinationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.guardduty.paginators.ListPublishingDestinationsPublisher publisher = client.listPublishingDestinationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.guardduty.paginators.ListPublishingDestinationsPublisher publisher = client.listPublishingDestinationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.guardduty.model.ListPublishingDestinationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.guardduty.model.ListPublishingDestinationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPublishingDestinations(software.amazon.awssdk.services.guardduty.model.ListPublishingDestinationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPublishingDestinationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.ListPublishingDestinations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/ListPublishingDestinations"
     *      target="_top">AWS API Documentation</a>
     */
    public ListPublishingDestinationsPublisher listPublishingDestinationsPaginator(
            ListPublishingDestinationsRequest listPublishingDestinationsRequest) {
        return new ListPublishingDestinationsPublisher(this, applyPaginatorUserAgent(listPublishingDestinationsRequest));
    }

    /**
     * <p>
     * Lists tags for a resource. Tagging is currently supported for detectors, finding filters, IP sets, and threat
     * intel sets, with a limit of 50 tags per resource. When invoked, this operation returns all assigned tags for a
     * given resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the ThreatIntelSets of the GuardDuty service specified by the detector ID. If you use this operation from a
     * member account, the ThreatIntelSets associated with the administrator account are returned.
     * </p>
     *
     * @param listThreatIntelSetsRequest
     * @return A Java Future containing the result of the ListThreatIntelSets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.ListThreatIntelSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/ListThreatIntelSets" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListThreatIntelSetsResponse> listThreatIntelSets(
            ListThreatIntelSetsRequest listThreatIntelSetsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listThreatIntelSetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListThreatIntelSets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListThreatIntelSetsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListThreatIntelSetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListThreatIntelSetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListThreatIntelSetsRequest, ListThreatIntelSetsResponse>()
                            .withOperationName("ListThreatIntelSets")
                            .withMarshaller(new ListThreatIntelSetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listThreatIntelSetsRequest));
            CompletableFuture<ListThreatIntelSetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the ThreatIntelSets of the GuardDuty service specified by the detector ID. If you use this operation from a
     * member account, the ThreatIntelSets associated with the administrator account are returned.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listThreatIntelSets(software.amazon.awssdk.services.guardduty.model.ListThreatIntelSetsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.guardduty.paginators.ListThreatIntelSetsPublisher publisher = client.listThreatIntelSetsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.guardduty.paginators.ListThreatIntelSetsPublisher publisher = client.listThreatIntelSetsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.guardduty.model.ListThreatIntelSetsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.guardduty.model.ListThreatIntelSetsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listThreatIntelSets(software.amazon.awssdk.services.guardduty.model.ListThreatIntelSetsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listThreatIntelSetsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.ListThreatIntelSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/ListThreatIntelSets" target="_top">AWS
     *      API Documentation</a>
     */
    public ListThreatIntelSetsPublisher listThreatIntelSetsPaginator(ListThreatIntelSetsRequest listThreatIntelSetsRequest) {
        return new ListThreatIntelSetsPublisher(this, applyPaginatorUserAgent(listThreatIntelSetsRequest));
    }

    /**
     * <p>
     * Turns on GuardDuty monitoring of the specified member accounts. Use this operation to restart monitoring of
     * accounts that you stopped monitoring with the <code>StopMonitoringMembers</code> operation.
     * </p>
     *
     * @param startMonitoringMembersRequest
     * @return A Java Future containing the result of the StartMonitoringMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.StartMonitoringMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/StartMonitoringMembers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartMonitoringMembersResponse> startMonitoringMembers(
            StartMonitoringMembersRequest startMonitoringMembersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startMonitoringMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartMonitoringMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartMonitoringMembersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartMonitoringMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartMonitoringMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartMonitoringMembersRequest, StartMonitoringMembersResponse>()
                            .withOperationName("StartMonitoringMembers")
                            .withMarshaller(new StartMonitoringMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startMonitoringMembersRequest));
            CompletableFuture<StartMonitoringMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops GuardDuty monitoring for the specified member accounts. Use the <code>StartMonitoringMembers</code>
     * operation to restart monitoring for those accounts.
     * </p>
     *
     * @param stopMonitoringMembersRequest
     * @return A Java Future containing the result of the StopMonitoringMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.StopMonitoringMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/StopMonitoringMembers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopMonitoringMembersResponse> stopMonitoringMembers(
            StopMonitoringMembersRequest stopMonitoringMembersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopMonitoringMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopMonitoringMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopMonitoringMembersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopMonitoringMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopMonitoringMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopMonitoringMembersRequest, StopMonitoringMembersResponse>()
                            .withOperationName("StopMonitoringMembers")
                            .withMarshaller(new StopMonitoringMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopMonitoringMembersRequest));
            CompletableFuture<StopMonitoringMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds tags to a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Unarchives GuardDuty findings specified by the <code>findingIds</code>.
     * </p>
     *
     * @param unarchiveFindingsRequest
     * @return A Java Future containing the result of the UnarchiveFindings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.UnarchiveFindings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/UnarchiveFindings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UnarchiveFindingsResponse> unarchiveFindings(UnarchiveFindingsRequest unarchiveFindingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, unarchiveFindingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UnarchiveFindings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UnarchiveFindingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UnarchiveFindingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UnarchiveFindingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UnarchiveFindingsRequest, UnarchiveFindingsResponse>()
                            .withOperationName("UnarchiveFindings")
                            .withMarshaller(new UnarchiveFindingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(unarchiveFindingsRequest));
            CompletableFuture<UnarchiveFindingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the Amazon GuardDuty detector specified by the detectorId.
     * </p>
     *
     * @param updateDetectorRequest
     * @return A Java Future containing the result of the UpdateDetector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.UpdateDetector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/UpdateDetector" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDetectorResponse> updateDetector(UpdateDetectorRequest updateDetectorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDetectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDetector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDetectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDetectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDetectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDetectorRequest, UpdateDetectorResponse>()
                            .withOperationName("UpdateDetector")
                            .withMarshaller(new UpdateDetectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateDetectorRequest));
            CompletableFuture<UpdateDetectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the filter specified by the filter name.
     * </p>
     *
     * @param updateFilterRequest
     * @return A Java Future containing the result of the UpdateFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.UpdateFilter
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/UpdateFilter" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFilterResponse> updateFilter(UpdateFilterRequest updateFilterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFilterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFilter");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFilterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFilterRequest, UpdateFilterResponse>()
                            .withOperationName("UpdateFilter").withMarshaller(new UpdateFilterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateFilterRequest));
            CompletableFuture<UpdateFilterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Marks the specified GuardDuty findings as useful or not useful.
     * </p>
     *
     * @param updateFindingsFeedbackRequest
     * @return A Java Future containing the result of the UpdateFindingsFeedback operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.UpdateFindingsFeedback
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/UpdateFindingsFeedback"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFindingsFeedbackResponse> updateFindingsFeedback(
            UpdateFindingsFeedbackRequest updateFindingsFeedbackRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFindingsFeedbackRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFindingsFeedback");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFindingsFeedbackResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateFindingsFeedbackResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFindingsFeedbackResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFindingsFeedbackRequest, UpdateFindingsFeedbackResponse>()
                            .withOperationName("UpdateFindingsFeedback")
                            .withMarshaller(new UpdateFindingsFeedbackRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateFindingsFeedbackRequest));
            CompletableFuture<UpdateFindingsFeedbackResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the IPSet specified by the IPSet ID.
     * </p>
     *
     * @param updateIpSetRequest
     * @return A Java Future containing the result of the UpdateIPSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.UpdateIPSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/UpdateIPSet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateIpSetResponse> updateIPSet(UpdateIpSetRequest updateIpSetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateIpSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateIPSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateIpSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateIpSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateIpSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateIpSetRequest, UpdateIpSetResponse>()
                            .withOperationName("UpdateIPSet").withMarshaller(new UpdateIpSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateIpSetRequest));
            CompletableFuture<UpdateIpSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Contains information on member accounts to be updated.
     * </p>
     *
     * @param updateMemberDetectorsRequest
     * @return A Java Future containing the result of the UpdateMemberDetectors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.UpdateMemberDetectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/UpdateMemberDetectors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateMemberDetectorsResponse> updateMemberDetectors(
            UpdateMemberDetectorsRequest updateMemberDetectorsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateMemberDetectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMemberDetectors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateMemberDetectorsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateMemberDetectorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateMemberDetectorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateMemberDetectorsRequest, UpdateMemberDetectorsResponse>()
                            .withOperationName("UpdateMemberDetectors")
                            .withMarshaller(new UpdateMemberDetectorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateMemberDetectorsRequest));
            CompletableFuture<UpdateMemberDetectorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the delegated administrator account with the values provided.
     * </p>
     *
     * @param updateOrganizationConfigurationRequest
     * @return A Java Future containing the result of the UpdateOrganizationConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.UpdateOrganizationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/UpdateOrganizationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateOrganizationConfigurationResponse> updateOrganizationConfiguration(
            UpdateOrganizationConfigurationRequest updateOrganizationConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateOrganizationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateOrganizationConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateOrganizationConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateOrganizationConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateOrganizationConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateOrganizationConfigurationRequest, UpdateOrganizationConfigurationResponse>()
                            .withOperationName("UpdateOrganizationConfiguration")
                            .withMarshaller(new UpdateOrganizationConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateOrganizationConfigurationRequest));
            CompletableFuture<UpdateOrganizationConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates information about the publishing destination specified by the <code>destinationId</code>.
     * </p>
     *
     * @param updatePublishingDestinationRequest
     * @return A Java Future containing the result of the UpdatePublishingDestination operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.UpdatePublishingDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/UpdatePublishingDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePublishingDestinationResponse> updatePublishingDestination(
            UpdatePublishingDestinationRequest updatePublishingDestinationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePublishingDestinationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePublishingDestination");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePublishingDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePublishingDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePublishingDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePublishingDestinationRequest, UpdatePublishingDestinationResponse>()
                            .withOperationName("UpdatePublishingDestination")
                            .withMarshaller(new UpdatePublishingDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updatePublishingDestinationRequest));
            CompletableFuture<UpdatePublishingDestinationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the ThreatIntelSet specified by the ThreatIntelSet ID.
     * </p>
     *
     * @param updateThreatIntelSetRequest
     * @return A Java Future containing the result of the UpdateThreatIntelSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A bad request exception object.</li>
     *         <li>InternalServerErrorException An internal server error exception object.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GuardDutyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GuardDutyAsyncClient.UpdateThreatIntelSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/guardduty-2017-11-28/UpdateThreatIntelSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateThreatIntelSetResponse> updateThreatIntelSet(
            UpdateThreatIntelSetRequest updateThreatIntelSetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateThreatIntelSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GuardDuty");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateThreatIntelSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateThreatIntelSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateThreatIntelSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateThreatIntelSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateThreatIntelSetRequest, UpdateThreatIntelSetResponse>()
                            .withOperationName("UpdateThreatIntelSet")
                            .withMarshaller(new UpdateThreatIntelSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateThreatIntelSetRequest));
            CompletableFuture<UpdateThreatIntelSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(GuardDutyException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends GuardDutyRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
