/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.guardduty.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about a malware scan.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Scan implements SdkPojo, Serializable, ToCopyableBuilder<Scan.Builder, Scan> {
    private static final SdkField<String> DETECTOR_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DetectorId").getter(getter(Scan::detectorId)).setter(setter(Builder::detectorId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("detectorId").build()).build();

    private static final SdkField<String> ADMIN_DETECTOR_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AdminDetectorId").getter(getter(Scan::adminDetectorId)).setter(setter(Builder::adminDetectorId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("adminDetectorId").build()).build();

    private static final SdkField<String> SCAN_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("ScanId")
            .getter(getter(Scan::scanId)).setter(setter(Builder::scanId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("scanId").build()).build();

    private static final SdkField<String> SCAN_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScanStatus").getter(getter(Scan::scanStatusAsString)).setter(setter(Builder::scanStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("scanStatus").build()).build();

    private static final SdkField<String> FAILURE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FailureReason").getter(getter(Scan::failureReason)).setter(setter(Builder::failureReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("failureReason").build()).build();

    private static final SdkField<Instant> SCAN_START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ScanStartTime").getter(getter(Scan::scanStartTime)).setter(setter(Builder::scanStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("scanStartTime").build()).build();

    private static final SdkField<Instant> SCAN_END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ScanEndTime").getter(getter(Scan::scanEndTime)).setter(setter(Builder::scanEndTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("scanEndTime").build()).build();

    private static final SdkField<TriggerDetails> TRIGGER_DETAILS_FIELD = SdkField
            .<TriggerDetails> builder(MarshallingType.SDK_POJO).memberName("TriggerDetails").getter(getter(Scan::triggerDetails))
            .setter(setter(Builder::triggerDetails)).constructor(TriggerDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("triggerDetails").build()).build();

    private static final SdkField<ResourceDetails> RESOURCE_DETAILS_FIELD = SdkField
            .<ResourceDetails> builder(MarshallingType.SDK_POJO).memberName("ResourceDetails")
            .getter(getter(Scan::resourceDetails)).setter(setter(Builder::resourceDetails)).constructor(ResourceDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceDetails").build()).build();

    private static final SdkField<ScanResultDetails> SCAN_RESULT_DETAILS_FIELD = SdkField
            .<ScanResultDetails> builder(MarshallingType.SDK_POJO).memberName("ScanResultDetails")
            .getter(getter(Scan::scanResultDetails)).setter(setter(Builder::scanResultDetails))
            .constructor(ScanResultDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("scanResultDetails").build()).build();

    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccountId").getter(getter(Scan::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accountId").build()).build();

    private static final SdkField<Long> TOTAL_BYTES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("TotalBytes").getter(getter(Scan::totalBytes)).setter(setter(Builder::totalBytes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("totalBytes").build()).build();

    private static final SdkField<Long> FILE_COUNT_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("FileCount")
            .getter(getter(Scan::fileCount)).setter(setter(Builder::fileCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fileCount").build()).build();

    private static final SdkField<List<VolumeDetail>> ATTACHED_VOLUMES_FIELD = SdkField
            .<List<VolumeDetail>> builder(MarshallingType.LIST)
            .memberName("AttachedVolumes")
            .getter(getter(Scan::attachedVolumes))
            .setter(setter(Builder::attachedVolumes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("attachedVolumes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<VolumeDetail> builder(MarshallingType.SDK_POJO)
                                            .constructor(VolumeDetail::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DETECTOR_ID_FIELD,
            ADMIN_DETECTOR_ID_FIELD, SCAN_ID_FIELD, SCAN_STATUS_FIELD, FAILURE_REASON_FIELD, SCAN_START_TIME_FIELD,
            SCAN_END_TIME_FIELD, TRIGGER_DETAILS_FIELD, RESOURCE_DETAILS_FIELD, SCAN_RESULT_DETAILS_FIELD, ACCOUNT_ID_FIELD,
            TOTAL_BYTES_FIELD, FILE_COUNT_FIELD, ATTACHED_VOLUMES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String detectorId;

    private final String adminDetectorId;

    private final String scanId;

    private final String scanStatus;

    private final String failureReason;

    private final Instant scanStartTime;

    private final Instant scanEndTime;

    private final TriggerDetails triggerDetails;

    private final ResourceDetails resourceDetails;

    private final ScanResultDetails scanResultDetails;

    private final String accountId;

    private final Long totalBytes;

    private final Long fileCount;

    private final List<VolumeDetail> attachedVolumes;

    private Scan(BuilderImpl builder) {
        this.detectorId = builder.detectorId;
        this.adminDetectorId = builder.adminDetectorId;
        this.scanId = builder.scanId;
        this.scanStatus = builder.scanStatus;
        this.failureReason = builder.failureReason;
        this.scanStartTime = builder.scanStartTime;
        this.scanEndTime = builder.scanEndTime;
        this.triggerDetails = builder.triggerDetails;
        this.resourceDetails = builder.resourceDetails;
        this.scanResultDetails = builder.scanResultDetails;
        this.accountId = builder.accountId;
        this.totalBytes = builder.totalBytes;
        this.fileCount = builder.fileCount;
        this.attachedVolumes = builder.attachedVolumes;
    }

    /**
     * <p>
     * The unique ID of the detector that the request is associated with.
     * </p>
     * 
     * @return The unique ID of the detector that the request is associated with.
     */
    public final String detectorId() {
        return detectorId;
    }

    /**
     * <p>
     * The unique detector ID of the administrator account that the request is associated with. Note that this value
     * will be the same as the one used for <code>DetectorId</code> if the account is an administrator.
     * </p>
     * 
     * @return The unique detector ID of the administrator account that the request is associated with. Note that this
     *         value will be the same as the one used for <code>DetectorId</code> if the account is an administrator.
     */
    public final String adminDetectorId() {
        return adminDetectorId;
    }

    /**
     * <p>
     * The unique scan ID associated with a scan entry.
     * </p>
     * 
     * @return The unique scan ID associated with a scan entry.
     */
    public final String scanId() {
        return scanId;
    }

    /**
     * <p>
     * An enum value representing possible scan statuses.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scanStatus} will
     * return {@link ScanStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #scanStatusAsString}.
     * </p>
     * 
     * @return An enum value representing possible scan statuses.
     * @see ScanStatus
     */
    public final ScanStatus scanStatus() {
        return ScanStatus.fromValue(scanStatus);
    }

    /**
     * <p>
     * An enum value representing possible scan statuses.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scanStatus} will
     * return {@link ScanStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #scanStatusAsString}.
     * </p>
     * 
     * @return An enum value representing possible scan statuses.
     * @see ScanStatus
     */
    public final String scanStatusAsString() {
        return scanStatus;
    }

    /**
     * <p>
     * Represents the reason for FAILED scan status.
     * </p>
     * 
     * @return Represents the reason for FAILED scan status.
     */
    public final String failureReason() {
        return failureReason;
    }

    /**
     * <p>
     * The timestamp of when the scan was triggered.
     * </p>
     * 
     * @return The timestamp of when the scan was triggered.
     */
    public final Instant scanStartTime() {
        return scanStartTime;
    }

    /**
     * <p>
     * The timestamp of when the scan was finished.
     * </p>
     * 
     * @return The timestamp of when the scan was finished.
     */
    public final Instant scanEndTime() {
        return scanEndTime;
    }

    /**
     * <p>
     * Represents the reason the scan was triggered.
     * </p>
     * 
     * @return Represents the reason the scan was triggered.
     */
    public final TriggerDetails triggerDetails() {
        return triggerDetails;
    }

    /**
     * <p>
     * Represents the resources that were scanned in the scan entry.
     * </p>
     * 
     * @return Represents the resources that were scanned in the scan entry.
     */
    public final ResourceDetails resourceDetails() {
        return resourceDetails;
    }

    /**
     * <p>
     * Represents the result of the scan.
     * </p>
     * 
     * @return Represents the result of the scan.
     */
    public final ScanResultDetails scanResultDetails() {
        return scanResultDetails;
    }

    /**
     * <p>
     * The ID for the account that belongs to the scan.
     * </p>
     * 
     * @return The ID for the account that belongs to the scan.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * Represents total bytes that were scanned.
     * </p>
     * 
     * @return Represents total bytes that were scanned.
     */
    public final Long totalBytes() {
        return totalBytes;
    }

    /**
     * <p>
     * Represents the number of files that were scanned.
     * </p>
     * 
     * @return Represents the number of files that were scanned.
     */
    public final Long fileCount() {
        return fileCount;
    }

    /**
     * For responses, this returns true if the service returned a value for the AttachedVolumes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAttachedVolumes() {
        return attachedVolumes != null && !(attachedVolumes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of volumes that were attached to the original instance to be scanned.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttachedVolumes} method.
     * </p>
     * 
     * @return List of volumes that were attached to the original instance to be scanned.
     */
    public final List<VolumeDetail> attachedVolumes() {
        return attachedVolumes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(detectorId());
        hashCode = 31 * hashCode + Objects.hashCode(adminDetectorId());
        hashCode = 31 * hashCode + Objects.hashCode(scanId());
        hashCode = 31 * hashCode + Objects.hashCode(scanStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(failureReason());
        hashCode = 31 * hashCode + Objects.hashCode(scanStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(scanEndTime());
        hashCode = 31 * hashCode + Objects.hashCode(triggerDetails());
        hashCode = 31 * hashCode + Objects.hashCode(resourceDetails());
        hashCode = 31 * hashCode + Objects.hashCode(scanResultDetails());
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(totalBytes());
        hashCode = 31 * hashCode + Objects.hashCode(fileCount());
        hashCode = 31 * hashCode + Objects.hashCode(hasAttachedVolumes() ? attachedVolumes() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Scan)) {
            return false;
        }
        Scan other = (Scan) obj;
        return Objects.equals(detectorId(), other.detectorId()) && Objects.equals(adminDetectorId(), other.adminDetectorId())
                && Objects.equals(scanId(), other.scanId()) && Objects.equals(scanStatusAsString(), other.scanStatusAsString())
                && Objects.equals(failureReason(), other.failureReason())
                && Objects.equals(scanStartTime(), other.scanStartTime()) && Objects.equals(scanEndTime(), other.scanEndTime())
                && Objects.equals(triggerDetails(), other.triggerDetails())
                && Objects.equals(resourceDetails(), other.resourceDetails())
                && Objects.equals(scanResultDetails(), other.scanResultDetails())
                && Objects.equals(accountId(), other.accountId()) && Objects.equals(totalBytes(), other.totalBytes())
                && Objects.equals(fileCount(), other.fileCount()) && hasAttachedVolumes() == other.hasAttachedVolumes()
                && Objects.equals(attachedVolumes(), other.attachedVolumes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Scan").add("DetectorId", detectorId()).add("AdminDetectorId", adminDetectorId())
                .add("ScanId", scanId()).add("ScanStatus", scanStatusAsString()).add("FailureReason", failureReason())
                .add("ScanStartTime", scanStartTime()).add("ScanEndTime", scanEndTime()).add("TriggerDetails", triggerDetails())
                .add("ResourceDetails", resourceDetails()).add("ScanResultDetails", scanResultDetails())
                .add("AccountId", accountId()).add("TotalBytes", totalBytes()).add("FileCount", fileCount())
                .add("AttachedVolumes", hasAttachedVolumes() ? attachedVolumes() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DetectorId":
            return Optional.ofNullable(clazz.cast(detectorId()));
        case "AdminDetectorId":
            return Optional.ofNullable(clazz.cast(adminDetectorId()));
        case "ScanId":
            return Optional.ofNullable(clazz.cast(scanId()));
        case "ScanStatus":
            return Optional.ofNullable(clazz.cast(scanStatusAsString()));
        case "FailureReason":
            return Optional.ofNullable(clazz.cast(failureReason()));
        case "ScanStartTime":
            return Optional.ofNullable(clazz.cast(scanStartTime()));
        case "ScanEndTime":
            return Optional.ofNullable(clazz.cast(scanEndTime()));
        case "TriggerDetails":
            return Optional.ofNullable(clazz.cast(triggerDetails()));
        case "ResourceDetails":
            return Optional.ofNullable(clazz.cast(resourceDetails()));
        case "ScanResultDetails":
            return Optional.ofNullable(clazz.cast(scanResultDetails()));
        case "AccountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "TotalBytes":
            return Optional.ofNullable(clazz.cast(totalBytes()));
        case "FileCount":
            return Optional.ofNullable(clazz.cast(fileCount()));
        case "AttachedVolumes":
            return Optional.ofNullable(clazz.cast(attachedVolumes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Scan, T> g) {
        return obj -> g.apply((Scan) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Scan> {
        /**
         * <p>
         * The unique ID of the detector that the request is associated with.
         * </p>
         * 
         * @param detectorId
         *        The unique ID of the detector that the request is associated with.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder detectorId(String detectorId);

        /**
         * <p>
         * The unique detector ID of the administrator account that the request is associated with. Note that this value
         * will be the same as the one used for <code>DetectorId</code> if the account is an administrator.
         * </p>
         * 
         * @param adminDetectorId
         *        The unique detector ID of the administrator account that the request is associated with. Note that
         *        this value will be the same as the one used for <code>DetectorId</code> if the account is an
         *        administrator.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder adminDetectorId(String adminDetectorId);

        /**
         * <p>
         * The unique scan ID associated with a scan entry.
         * </p>
         * 
         * @param scanId
         *        The unique scan ID associated with a scan entry.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scanId(String scanId);

        /**
         * <p>
         * An enum value representing possible scan statuses.
         * </p>
         * 
         * @param scanStatus
         *        An enum value representing possible scan statuses.
         * @see ScanStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScanStatus
         */
        Builder scanStatus(String scanStatus);

        /**
         * <p>
         * An enum value representing possible scan statuses.
         * </p>
         * 
         * @param scanStatus
         *        An enum value representing possible scan statuses.
         * @see ScanStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScanStatus
         */
        Builder scanStatus(ScanStatus scanStatus);

        /**
         * <p>
         * Represents the reason for FAILED scan status.
         * </p>
         * 
         * @param failureReason
         *        Represents the reason for FAILED scan status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReason(String failureReason);

        /**
         * <p>
         * The timestamp of when the scan was triggered.
         * </p>
         * 
         * @param scanStartTime
         *        The timestamp of when the scan was triggered.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scanStartTime(Instant scanStartTime);

        /**
         * <p>
         * The timestamp of when the scan was finished.
         * </p>
         * 
         * @param scanEndTime
         *        The timestamp of when the scan was finished.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scanEndTime(Instant scanEndTime);

        /**
         * <p>
         * Represents the reason the scan was triggered.
         * </p>
         * 
         * @param triggerDetails
         *        Represents the reason the scan was triggered.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder triggerDetails(TriggerDetails triggerDetails);

        /**
         * <p>
         * Represents the reason the scan was triggered.
         * </p>
         * This is a convenience method that creates an instance of the {@link TriggerDetails.Builder} avoiding the need
         * to create one manually via {@link TriggerDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TriggerDetails.Builder#build()} is called immediately and its
         * result is passed to {@link #triggerDetails(TriggerDetails)}.
         * 
         * @param triggerDetails
         *        a consumer that will call methods on {@link TriggerDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #triggerDetails(TriggerDetails)
         */
        default Builder triggerDetails(Consumer<TriggerDetails.Builder> triggerDetails) {
            return triggerDetails(TriggerDetails.builder().applyMutation(triggerDetails).build());
        }

        /**
         * <p>
         * Represents the resources that were scanned in the scan entry.
         * </p>
         * 
         * @param resourceDetails
         *        Represents the resources that were scanned in the scan entry.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceDetails(ResourceDetails resourceDetails);

        /**
         * <p>
         * Represents the resources that were scanned in the scan entry.
         * </p>
         * This is a convenience method that creates an instance of the {@link ResourceDetails.Builder} avoiding the
         * need to create one manually via {@link ResourceDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResourceDetails.Builder#build()} is called immediately and its
         * result is passed to {@link #resourceDetails(ResourceDetails)}.
         * 
         * @param resourceDetails
         *        a consumer that will call methods on {@link ResourceDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceDetails(ResourceDetails)
         */
        default Builder resourceDetails(Consumer<ResourceDetails.Builder> resourceDetails) {
            return resourceDetails(ResourceDetails.builder().applyMutation(resourceDetails).build());
        }

        /**
         * <p>
         * Represents the result of the scan.
         * </p>
         * 
         * @param scanResultDetails
         *        Represents the result of the scan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scanResultDetails(ScanResultDetails scanResultDetails);

        /**
         * <p>
         * Represents the result of the scan.
         * </p>
         * This is a convenience method that creates an instance of the {@link ScanResultDetails.Builder} avoiding the
         * need to create one manually via {@link ScanResultDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ScanResultDetails.Builder#build()} is called immediately and its
         * result is passed to {@link #scanResultDetails(ScanResultDetails)}.
         * 
         * @param scanResultDetails
         *        a consumer that will call methods on {@link ScanResultDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scanResultDetails(ScanResultDetails)
         */
        default Builder scanResultDetails(Consumer<ScanResultDetails.Builder> scanResultDetails) {
            return scanResultDetails(ScanResultDetails.builder().applyMutation(scanResultDetails).build());
        }

        /**
         * <p>
         * The ID for the account that belongs to the scan.
         * </p>
         * 
         * @param accountId
         *        The ID for the account that belongs to the scan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * Represents total bytes that were scanned.
         * </p>
         * 
         * @param totalBytes
         *        Represents total bytes that were scanned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalBytes(Long totalBytes);

        /**
         * <p>
         * Represents the number of files that were scanned.
         * </p>
         * 
         * @param fileCount
         *        Represents the number of files that were scanned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileCount(Long fileCount);

        /**
         * <p>
         * List of volumes that were attached to the original instance to be scanned.
         * </p>
         * 
         * @param attachedVolumes
         *        List of volumes that were attached to the original instance to be scanned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachedVolumes(Collection<VolumeDetail> attachedVolumes);

        /**
         * <p>
         * List of volumes that were attached to the original instance to be scanned.
         * </p>
         * 
         * @param attachedVolumes
         *        List of volumes that were attached to the original instance to be scanned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachedVolumes(VolumeDetail... attachedVolumes);

        /**
         * <p>
         * List of volumes that were attached to the original instance to be scanned.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.guardduty.model.VolumeDetail.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.guardduty.model.VolumeDetail#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.guardduty.model.VolumeDetail.Builder#build()} is called immediately
         * and its result is passed to {@link #attachedVolumes(List<VolumeDetail>)}.
         * 
         * @param attachedVolumes
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.guardduty.model.VolumeDetail.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #attachedVolumes(java.util.Collection<VolumeDetail>)
         */
        Builder attachedVolumes(Consumer<VolumeDetail.Builder>... attachedVolumes);
    }

    static final class BuilderImpl implements Builder {
        private String detectorId;

        private String adminDetectorId;

        private String scanId;

        private String scanStatus;

        private String failureReason;

        private Instant scanStartTime;

        private Instant scanEndTime;

        private TriggerDetails triggerDetails;

        private ResourceDetails resourceDetails;

        private ScanResultDetails scanResultDetails;

        private String accountId;

        private Long totalBytes;

        private Long fileCount;

        private List<VolumeDetail> attachedVolumes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Scan model) {
            detectorId(model.detectorId);
            adminDetectorId(model.adminDetectorId);
            scanId(model.scanId);
            scanStatus(model.scanStatus);
            failureReason(model.failureReason);
            scanStartTime(model.scanStartTime);
            scanEndTime(model.scanEndTime);
            triggerDetails(model.triggerDetails);
            resourceDetails(model.resourceDetails);
            scanResultDetails(model.scanResultDetails);
            accountId(model.accountId);
            totalBytes(model.totalBytes);
            fileCount(model.fileCount);
            attachedVolumes(model.attachedVolumes);
        }

        public final String getDetectorId() {
            return detectorId;
        }

        public final void setDetectorId(String detectorId) {
            this.detectorId = detectorId;
        }

        @Override
        public final Builder detectorId(String detectorId) {
            this.detectorId = detectorId;
            return this;
        }

        public final String getAdminDetectorId() {
            return adminDetectorId;
        }

        public final void setAdminDetectorId(String adminDetectorId) {
            this.adminDetectorId = adminDetectorId;
        }

        @Override
        public final Builder adminDetectorId(String adminDetectorId) {
            this.adminDetectorId = adminDetectorId;
            return this;
        }

        public final String getScanId() {
            return scanId;
        }

        public final void setScanId(String scanId) {
            this.scanId = scanId;
        }

        @Override
        public final Builder scanId(String scanId) {
            this.scanId = scanId;
            return this;
        }

        public final String getScanStatus() {
            return scanStatus;
        }

        public final void setScanStatus(String scanStatus) {
            this.scanStatus = scanStatus;
        }

        @Override
        public final Builder scanStatus(String scanStatus) {
            this.scanStatus = scanStatus;
            return this;
        }

        @Override
        public final Builder scanStatus(ScanStatus scanStatus) {
            this.scanStatus(scanStatus == null ? null : scanStatus.toString());
            return this;
        }

        public final String getFailureReason() {
            return failureReason;
        }

        public final void setFailureReason(String failureReason) {
            this.failureReason = failureReason;
        }

        @Override
        public final Builder failureReason(String failureReason) {
            this.failureReason = failureReason;
            return this;
        }

        public final Instant getScanStartTime() {
            return scanStartTime;
        }

        public final void setScanStartTime(Instant scanStartTime) {
            this.scanStartTime = scanStartTime;
        }

        @Override
        public final Builder scanStartTime(Instant scanStartTime) {
            this.scanStartTime = scanStartTime;
            return this;
        }

        public final Instant getScanEndTime() {
            return scanEndTime;
        }

        public final void setScanEndTime(Instant scanEndTime) {
            this.scanEndTime = scanEndTime;
        }

        @Override
        public final Builder scanEndTime(Instant scanEndTime) {
            this.scanEndTime = scanEndTime;
            return this;
        }

        public final TriggerDetails.Builder getTriggerDetails() {
            return triggerDetails != null ? triggerDetails.toBuilder() : null;
        }

        public final void setTriggerDetails(TriggerDetails.BuilderImpl triggerDetails) {
            this.triggerDetails = triggerDetails != null ? triggerDetails.build() : null;
        }

        @Override
        public final Builder triggerDetails(TriggerDetails triggerDetails) {
            this.triggerDetails = triggerDetails;
            return this;
        }

        public final ResourceDetails.Builder getResourceDetails() {
            return resourceDetails != null ? resourceDetails.toBuilder() : null;
        }

        public final void setResourceDetails(ResourceDetails.BuilderImpl resourceDetails) {
            this.resourceDetails = resourceDetails != null ? resourceDetails.build() : null;
        }

        @Override
        public final Builder resourceDetails(ResourceDetails resourceDetails) {
            this.resourceDetails = resourceDetails;
            return this;
        }

        public final ScanResultDetails.Builder getScanResultDetails() {
            return scanResultDetails != null ? scanResultDetails.toBuilder() : null;
        }

        public final void setScanResultDetails(ScanResultDetails.BuilderImpl scanResultDetails) {
            this.scanResultDetails = scanResultDetails != null ? scanResultDetails.build() : null;
        }

        @Override
        public final Builder scanResultDetails(ScanResultDetails scanResultDetails) {
            this.scanResultDetails = scanResultDetails;
            return this;
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final Long getTotalBytes() {
            return totalBytes;
        }

        public final void setTotalBytes(Long totalBytes) {
            this.totalBytes = totalBytes;
        }

        @Override
        public final Builder totalBytes(Long totalBytes) {
            this.totalBytes = totalBytes;
            return this;
        }

        public final Long getFileCount() {
            return fileCount;
        }

        public final void setFileCount(Long fileCount) {
            this.fileCount = fileCount;
        }

        @Override
        public final Builder fileCount(Long fileCount) {
            this.fileCount = fileCount;
            return this;
        }

        public final List<VolumeDetail.Builder> getAttachedVolumes() {
            List<VolumeDetail.Builder> result = VolumeDetailsCopier.copyToBuilder(this.attachedVolumes);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAttachedVolumes(Collection<VolumeDetail.BuilderImpl> attachedVolumes) {
            this.attachedVolumes = VolumeDetailsCopier.copyFromBuilder(attachedVolumes);
        }

        @Override
        public final Builder attachedVolumes(Collection<VolumeDetail> attachedVolumes) {
            this.attachedVolumes = VolumeDetailsCopier.copy(attachedVolumes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attachedVolumes(VolumeDetail... attachedVolumes) {
            attachedVolumes(Arrays.asList(attachedVolumes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attachedVolumes(Consumer<VolumeDetail.Builder>... attachedVolumes) {
            attachedVolumes(Stream.of(attachedVolumes).map(c -> VolumeDetail.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public Scan build() {
            return new Scan(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
