/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.guardduty.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains files infected with the given threat providing details of malware name and severity.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ScanThreatName implements SdkPojo, Serializable, ToCopyableBuilder<ScanThreatName.Builder, ScanThreatName> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(ScanThreatName::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> SEVERITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Severity").getter(getter(ScanThreatName::severity)).setter(setter(Builder::severity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("severity").build()).build();

    private static final SdkField<Integer> ITEM_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ItemCount").getter(getter(ScanThreatName::itemCount)).setter(setter(Builder::itemCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("itemCount").build()).build();

    private static final SdkField<List<ScanFilePath>> FILE_PATHS_FIELD = SdkField
            .<List<ScanFilePath>> builder(MarshallingType.LIST)
            .memberName("FilePaths")
            .getter(getter(ScanThreatName::filePaths))
            .setter(setter(Builder::filePaths))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("filePaths").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ScanFilePath> builder(MarshallingType.SDK_POJO)
                                            .constructor(ScanFilePath::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, SEVERITY_FIELD,
            ITEM_COUNT_FIELD, FILE_PATHS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String severity;

    private final Integer itemCount;

    private final List<ScanFilePath> filePaths;

    private ScanThreatName(BuilderImpl builder) {
        this.name = builder.name;
        this.severity = builder.severity;
        this.itemCount = builder.itemCount;
        this.filePaths = builder.filePaths;
    }

    /**
     * <p>
     * The name of the identified threat.
     * </p>
     * 
     * @return The name of the identified threat.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * Severity of threat identified as part of the malware scan.
     * </p>
     * 
     * @return Severity of threat identified as part of the malware scan.
     */
    public final String severity() {
        return severity;
    }

    /**
     * <p>
     * Total number of files infected with given threat.
     * </p>
     * 
     * @return Total number of files infected with given threat.
     */
    public final Integer itemCount() {
        return itemCount;
    }

    /**
     * For responses, this returns true if the service returned a value for the FilePaths property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasFilePaths() {
        return filePaths != null && !(filePaths instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of infected files in EBS volume with details.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFilePaths} method.
     * </p>
     * 
     * @return List of infected files in EBS volume with details.
     */
    public final List<ScanFilePath> filePaths() {
        return filePaths;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(severity());
        hashCode = 31 * hashCode + Objects.hashCode(itemCount());
        hashCode = 31 * hashCode + Objects.hashCode(hasFilePaths() ? filePaths() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ScanThreatName)) {
            return false;
        }
        ScanThreatName other = (ScanThreatName) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(severity(), other.severity())
                && Objects.equals(itemCount(), other.itemCount()) && hasFilePaths() == other.hasFilePaths()
                && Objects.equals(filePaths(), other.filePaths());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ScanThreatName").add("Name", name()).add("Severity", severity()).add("ItemCount", itemCount())
                .add("FilePaths", hasFilePaths() ? filePaths() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Severity":
            return Optional.ofNullable(clazz.cast(severity()));
        case "ItemCount":
            return Optional.ofNullable(clazz.cast(itemCount()));
        case "FilePaths":
            return Optional.ofNullable(clazz.cast(filePaths()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ScanThreatName, T> g) {
        return obj -> g.apply((ScanThreatName) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ScanThreatName> {
        /**
         * <p>
         * The name of the identified threat.
         * </p>
         * 
         * @param name
         *        The name of the identified threat.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Severity of threat identified as part of the malware scan.
         * </p>
         * 
         * @param severity
         *        Severity of threat identified as part of the malware scan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder severity(String severity);

        /**
         * <p>
         * Total number of files infected with given threat.
         * </p>
         * 
         * @param itemCount
         *        Total number of files infected with given threat.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder itemCount(Integer itemCount);

        /**
         * <p>
         * List of infected files in EBS volume with details.
         * </p>
         * 
         * @param filePaths
         *        List of infected files in EBS volume with details.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filePaths(Collection<ScanFilePath> filePaths);

        /**
         * <p>
         * List of infected files in EBS volume with details.
         * </p>
         * 
         * @param filePaths
         *        List of infected files in EBS volume with details.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filePaths(ScanFilePath... filePaths);

        /**
         * <p>
         * List of infected files in EBS volume with details.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.guardduty.model.ScanFilePath.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.guardduty.model.ScanFilePath#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.guardduty.model.ScanFilePath.Builder#build()} is called immediately
         * and its result is passed to {@link #filePaths(List<ScanFilePath>)}.
         * 
         * @param filePaths
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.guardduty.model.ScanFilePath.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #filePaths(java.util.Collection<ScanFilePath>)
         */
        Builder filePaths(Consumer<ScanFilePath.Builder>... filePaths);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String severity;

        private Integer itemCount;

        private List<ScanFilePath> filePaths = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ScanThreatName model) {
            name(model.name);
            severity(model.severity);
            itemCount(model.itemCount);
            filePaths(model.filePaths);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getSeverity() {
            return severity;
        }

        public final void setSeverity(String severity) {
            this.severity = severity;
        }

        @Override
        public final Builder severity(String severity) {
            this.severity = severity;
            return this;
        }

        public final Integer getItemCount() {
            return itemCount;
        }

        public final void setItemCount(Integer itemCount) {
            this.itemCount = itemCount;
        }

        @Override
        public final Builder itemCount(Integer itemCount) {
            this.itemCount = itemCount;
            return this;
        }

        public final List<ScanFilePath.Builder> getFilePaths() {
            List<ScanFilePath.Builder> result = FilePathsCopier.copyToBuilder(this.filePaths);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setFilePaths(Collection<ScanFilePath.BuilderImpl> filePaths) {
            this.filePaths = FilePathsCopier.copyFromBuilder(filePaths);
        }

        @Override
        public final Builder filePaths(Collection<ScanFilePath> filePaths) {
            this.filePaths = FilePathsCopier.copy(filePaths);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filePaths(ScanFilePath... filePaths) {
            filePaths(Arrays.asList(filePaths));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filePaths(Consumer<ScanFilePath.Builder>... filePaths) {
            filePaths(Stream.of(filePaths).map(c -> ScanFilePath.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public ScanThreatName build() {
            return new ScanThreatName(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
