/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.guardduty.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the indicators that include a set of signals observed in an attack sequence.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Indicator implements SdkPojo, Serializable, ToCopyableBuilder<Indicator.Builder, Indicator> {
    private static final SdkField<String> KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Key")
            .getter(getter(Indicator::keyAsString)).setter(setter(Builder::key))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("key").build()).build();

    private static final SdkField<List<String>> VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Values")
            .getter(getter(Indicator::values))
            .setter(setter(Builder::values))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("values").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Title")
            .getter(getter(Indicator::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("title").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(KEY_FIELD, VALUES_FIELD,
            TITLE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String key;

    private final List<String> values;

    private final String title;

    private Indicator(BuilderImpl builder) {
        this.key = builder.key;
        this.values = builder.values;
        this.title = builder.title;
    }

    /**
     * <p>
     * Specific indicator keys observed in the attack sequence.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #key} will return
     * {@link IndicatorType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyAsString}.
     * </p>
     * 
     * @return Specific indicator keys observed in the attack sequence.
     * @see IndicatorType
     */
    public final IndicatorType key() {
        return IndicatorType.fromValue(key);
    }

    /**
     * <p>
     * Specific indicator keys observed in the attack sequence.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #key} will return
     * {@link IndicatorType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyAsString}.
     * </p>
     * 
     * @return Specific indicator keys observed in the attack sequence.
     * @see IndicatorType
     */
    public final String keyAsString() {
        return key;
    }

    /**
     * For responses, this returns true if the service returned a value for the Values property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasValues() {
        return values != null && !(values instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Values associated with each indicator key. For example, if the indicator key is <code>SUSPICIOUS_NETWORK</code>,
     * then the value will be the name of the network. If the indicator key is <code>ATTACK_TACTIC</code>, then the
     * value will be one of the MITRE tactics.
     * </p>
     * <p>
     * For more information about the values associated with the key, see GuardDuty Extended Threat Detection in the
     * <i>GuardDuty User Guide.</i>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasValues} method.
     * </p>
     * 
     * @return Values associated with each indicator key. For example, if the indicator key is
     *         <code>SUSPICIOUS_NETWORK</code>, then the value will be the name of the network. If the indicator key is
     *         <code>ATTACK_TACTIC</code>, then the value will be one of the MITRE tactics. </p>
     *         <p>
     *         For more information about the values associated with the key, see GuardDuty Extended Threat Detection in
     *         the <i>GuardDuty User Guide.</i>
     */
    public final List<String> values() {
        return values;
    }

    /**
     * <p>
     * Title describing the indicator.
     * </p>
     * 
     * @return Title describing the indicator.
     */
    public final String title() {
        return title;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(keyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasValues() ? values() : null);
        hashCode = 31 * hashCode + Objects.hashCode(title());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Indicator)) {
            return false;
        }
        Indicator other = (Indicator) obj;
        return Objects.equals(keyAsString(), other.keyAsString()) && hasValues() == other.hasValues()
                && Objects.equals(values(), other.values()) && Objects.equals(title(), other.title());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Indicator").add("Key", keyAsString()).add("Values", hasValues() ? values() : null)
                .add("Title", title()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Key":
            return Optional.ofNullable(clazz.cast(keyAsString()));
        case "Values":
            return Optional.ofNullable(clazz.cast(values()));
        case "Title":
            return Optional.ofNullable(clazz.cast(title()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("key", KEY_FIELD);
        map.put("values", VALUES_FIELD);
        map.put("title", TITLE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Indicator, T> g) {
        return obj -> g.apply((Indicator) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Indicator> {
        /**
         * <p>
         * Specific indicator keys observed in the attack sequence.
         * </p>
         * 
         * @param key
         *        Specific indicator keys observed in the attack sequence.
         * @see IndicatorType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IndicatorType
         */
        Builder key(String key);

        /**
         * <p>
         * Specific indicator keys observed in the attack sequence.
         * </p>
         * 
         * @param key
         *        Specific indicator keys observed in the attack sequence.
         * @see IndicatorType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IndicatorType
         */
        Builder key(IndicatorType key);

        /**
         * <p>
         * Values associated with each indicator key. For example, if the indicator key is
         * <code>SUSPICIOUS_NETWORK</code>, then the value will be the name of the network. If the indicator key is
         * <code>ATTACK_TACTIC</code>, then the value will be one of the MITRE tactics.
         * </p>
         * <p>
         * For more information about the values associated with the key, see GuardDuty Extended Threat Detection in the
         * <i>GuardDuty User Guide.</i>
         * </p>
         * 
         * @param values
         *        Values associated with each indicator key. For example, if the indicator key is
         *        <code>SUSPICIOUS_NETWORK</code>, then the value will be the name of the network. If the indicator key
         *        is <code>ATTACK_TACTIC</code>, then the value will be one of the MITRE tactics. </p>
         *        <p>
         *        For more information about the values associated with the key, see GuardDuty Extended Threat Detection
         *        in the <i>GuardDuty User Guide.</i>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(Collection<String> values);

        /**
         * <p>
         * Values associated with each indicator key. For example, if the indicator key is
         * <code>SUSPICIOUS_NETWORK</code>, then the value will be the name of the network. If the indicator key is
         * <code>ATTACK_TACTIC</code>, then the value will be one of the MITRE tactics.
         * </p>
         * <p>
         * For more information about the values associated with the key, see GuardDuty Extended Threat Detection in the
         * <i>GuardDuty User Guide.</i>
         * </p>
         * 
         * @param values
         *        Values associated with each indicator key. For example, if the indicator key is
         *        <code>SUSPICIOUS_NETWORK</code>, then the value will be the name of the network. If the indicator key
         *        is <code>ATTACK_TACTIC</code>, then the value will be one of the MITRE tactics. </p>
         *        <p>
         *        For more information about the values associated with the key, see GuardDuty Extended Threat Detection
         *        in the <i>GuardDuty User Guide.</i>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(String... values);

        /**
         * <p>
         * Title describing the indicator.
         * </p>
         * 
         * @param title
         *        Title describing the indicator.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);
    }

    static final class BuilderImpl implements Builder {
        private String key;

        private List<String> values = DefaultSdkAutoConstructList.getInstance();

        private String title;

        private BuilderImpl() {
        }

        private BuilderImpl(Indicator model) {
            key(model.key);
            values(model.values);
            title(model.title);
        }

        public final String getKey() {
            return key;
        }

        public final void setKey(String key) {
            this.key = key;
        }

        @Override
        public final Builder key(String key) {
            this.key = key;
            return this;
        }

        @Override
        public final Builder key(IndicatorType key) {
            this.key(key == null ? null : key.toString());
            return this;
        }

        public final Collection<String> getValues() {
            if (values instanceof SdkAutoConstructList) {
                return null;
            }
            return values;
        }

        public final void setValues(Collection<String> values) {
            this.values = IndicatorValuesCopier.copy(values);
        }

        @Override
        public final Builder values(Collection<String> values) {
            this.values = IndicatorValuesCopier.copy(values);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder values(String... values) {
            values(Arrays.asList(values));
            return this;
        }

        public final String getTitle() {
            return title;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        @Override
        public Indicator build() {
            return new Indicator(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
