/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.guardduty.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the signals involved in the attack sequence.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Signal implements SdkPojo, Serializable, ToCopyableBuilder<Signal.Builder, Signal> {
    private static final SdkField<String> UID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Uid")
            .getter(getter(Signal::uid)).setter(setter(Builder::uid))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("uid").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(Signal::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(Signal::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Signal::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedAt").getter(getter(Signal::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("UpdatedAt").getter(getter(Signal::updatedAt)).setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updatedAt").build()).build();

    private static final SdkField<Instant> FIRST_SEEN_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("FirstSeenAt").getter(getter(Signal::firstSeenAt)).setter(setter(Builder::firstSeenAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("firstSeenAt").build()).build();

    private static final SdkField<Instant> LAST_SEEN_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastSeenAt").getter(getter(Signal::lastSeenAt)).setter(setter(Builder::lastSeenAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastSeenAt").build()).build();

    private static final SdkField<Double> SEVERITY_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("Severity").getter(getter(Signal::severity)).setter(setter(Builder::severity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("severity").build()).build();

    private static final SdkField<Integer> COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Count")
            .getter(getter(Signal::count)).setter(setter(Builder::count))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("count").build()).build();

    private static final SdkField<List<String>> RESOURCE_UIDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ResourceUids")
            .getter(getter(Signal::resourceUids))
            .setter(setter(Builder::resourceUids))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceUids").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> ACTOR_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ActorIds")
            .getter(getter(Signal::actorIds))
            .setter(setter(Builder::actorIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("actorIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> ENDPOINT_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("EndpointIds")
            .getter(getter(Signal::endpointIds))
            .setter(setter(Builder::endpointIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endpointIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Indicator>> SIGNAL_INDICATORS_FIELD = SdkField
            .<List<Indicator>> builder(MarshallingType.LIST)
            .memberName("SignalIndicators")
            .getter(getter(Signal::signalIndicators))
            .setter(setter(Builder::signalIndicators))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("signalIndicators").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Indicator> builder(MarshallingType.SDK_POJO)
                                            .constructor(Indicator::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(UID_FIELD, TYPE_FIELD,
            DESCRIPTION_FIELD, NAME_FIELD, CREATED_AT_FIELD, UPDATED_AT_FIELD, FIRST_SEEN_AT_FIELD, LAST_SEEN_AT_FIELD,
            SEVERITY_FIELD, COUNT_FIELD, RESOURCE_UIDS_FIELD, ACTOR_IDS_FIELD, ENDPOINT_IDS_FIELD, SIGNAL_INDICATORS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String uid;

    private final String type;

    private final String description;

    private final String name;

    private final Instant createdAt;

    private final Instant updatedAt;

    private final Instant firstSeenAt;

    private final Instant lastSeenAt;

    private final Double severity;

    private final Integer count;

    private final List<String> resourceUids;

    private final List<String> actorIds;

    private final List<String> endpointIds;

    private final List<Indicator> signalIndicators;

    private Signal(BuilderImpl builder) {
        this.uid = builder.uid;
        this.type = builder.type;
        this.description = builder.description;
        this.name = builder.name;
        this.createdAt = builder.createdAt;
        this.updatedAt = builder.updatedAt;
        this.firstSeenAt = builder.firstSeenAt;
        this.lastSeenAt = builder.lastSeenAt;
        this.severity = builder.severity;
        this.count = builder.count;
        this.resourceUids = builder.resourceUids;
        this.actorIds = builder.actorIds;
        this.endpointIds = builder.endpointIds;
        this.signalIndicators = builder.signalIndicators;
    }

    /**
     * <p>
     * The unique identifier of the signal.
     * </p>
     * 
     * @return The unique identifier of the signal.
     */
    public final String uid() {
        return uid;
    }

    /**
     * <p>
     * The type of the signal used to identify an attack sequence.
     * </p>
     * <p>
     * Signals can be GuardDuty findings or activities observed in data sources that GuardDuty monitors. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_data-sources.html">Foundational data sources</a>
     * in the <i>GuardDuty User Guide</i>.
     * </p>
     * <p>
     * A signal type can be one of the valid values listed in this API. Here are the related descriptions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>FINDING</code> - Individually generated GuardDuty finding.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CLOUD_TRAIL</code> - Activity observed from CloudTrail logs
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>S3_DATA_EVENTS</code> - Activity observed from CloudTrail data events for S3. Activities associated with
     * this type will show up only when you have enabled GuardDuty S3 Protection feature in your account. For more
     * information about S3 Protection and steps to enable it, see <a
     * href="https://docs.aws.amazon.com/guardduty/latest/ug/s3-protection.html">S3 Protection</a> in the <i>GuardDuty
     * User Guide</i>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link SignalType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the signal used to identify an attack sequence.</p>
     *         <p>
     *         Signals can be GuardDuty findings or activities observed in data sources that GuardDuty monitors. For
     *         more information, see <a
     *         href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_data-sources.html">Foundational data
     *         sources</a> in the <i>GuardDuty User Guide</i>.
     *         </p>
     *         <p>
     *         A signal type can be one of the valid values listed in this API. Here are the related descriptions:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>FINDING</code> - Individually generated GuardDuty finding.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CLOUD_TRAIL</code> - Activity observed from CloudTrail logs
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>S3_DATA_EVENTS</code> - Activity observed from CloudTrail data events for S3. Activities associated
     *         with this type will show up only when you have enabled GuardDuty S3 Protection feature in your account.
     *         For more information about S3 Protection and steps to enable it, see <a
     *         href="https://docs.aws.amazon.com/guardduty/latest/ug/s3-protection.html">S3 Protection</a> in the
     *         <i>GuardDuty User Guide</i>.
     *         </p>
     *         </li>
     * @see SignalType
     */
    public final SignalType type() {
        return SignalType.fromValue(type);
    }

    /**
     * <p>
     * The type of the signal used to identify an attack sequence.
     * </p>
     * <p>
     * Signals can be GuardDuty findings or activities observed in data sources that GuardDuty monitors. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_data-sources.html">Foundational data sources</a>
     * in the <i>GuardDuty User Guide</i>.
     * </p>
     * <p>
     * A signal type can be one of the valid values listed in this API. Here are the related descriptions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>FINDING</code> - Individually generated GuardDuty finding.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CLOUD_TRAIL</code> - Activity observed from CloudTrail logs
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>S3_DATA_EVENTS</code> - Activity observed from CloudTrail data events for S3. Activities associated with
     * this type will show up only when you have enabled GuardDuty S3 Protection feature in your account. For more
     * information about S3 Protection and steps to enable it, see <a
     * href="https://docs.aws.amazon.com/guardduty/latest/ug/s3-protection.html">S3 Protection</a> in the <i>GuardDuty
     * User Guide</i>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link SignalType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the signal used to identify an attack sequence.</p>
     *         <p>
     *         Signals can be GuardDuty findings or activities observed in data sources that GuardDuty monitors. For
     *         more information, see <a
     *         href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_data-sources.html">Foundational data
     *         sources</a> in the <i>GuardDuty User Guide</i>.
     *         </p>
     *         <p>
     *         A signal type can be one of the valid values listed in this API. Here are the related descriptions:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>FINDING</code> - Individually generated GuardDuty finding.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CLOUD_TRAIL</code> - Activity observed from CloudTrail logs
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>S3_DATA_EVENTS</code> - Activity observed from CloudTrail data events for S3. Activities associated
     *         with this type will show up only when you have enabled GuardDuty S3 Protection feature in your account.
     *         For more information about S3 Protection and steps to enable it, see <a
     *         href="https://docs.aws.amazon.com/guardduty/latest/ug/s3-protection.html">S3 Protection</a> in the
     *         <i>GuardDuty User Guide</i>.
     *         </p>
     *         </li>
     * @see SignalType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The description of the signal.
     * </p>
     * 
     * @return The description of the signal.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The name of the signal. For example, when signal type is <code>FINDING</code>, the signal name is the name of the
     * finding.
     * </p>
     * 
     * @return The name of the signal. For example, when signal type is <code>FINDING</code>, the signal name is the
     *         name of the finding.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The timestamp when the first finding or activity related to this signal was observed.
     * </p>
     * 
     * @return The timestamp when the first finding or activity related to this signal was observed.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The timestamp when this signal was last observed.
     * </p>
     * 
     * @return The timestamp when this signal was last observed.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    /**
     * <p>
     * The timestamp when the first finding or activity related to this signal was observed.
     * </p>
     * 
     * @return The timestamp when the first finding or activity related to this signal was observed.
     */
    public final Instant firstSeenAt() {
        return firstSeenAt;
    }

    /**
     * <p>
     * The timestamp when the last finding or activity related to this signal was observed.
     * </p>
     * 
     * @return The timestamp when the last finding or activity related to this signal was observed.
     */
    public final Instant lastSeenAt() {
        return lastSeenAt;
    }

    /**
     * <p>
     * The severity associated with the signal. For more information about severity, see <a
     * href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_findings-severity.html">Findings severity
     * levels</a> in the <i>GuardDuty User Guide</i>.
     * </p>
     * 
     * @return The severity associated with the signal. For more information about severity, see <a
     *         href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_findings-severity.html">Findings severity
     *         levels</a> in the <i>GuardDuty User Guide</i>.
     */
    public final Double severity() {
        return severity;
    }

    /**
     * <p>
     * The number of times this signal was observed.
     * </p>
     * 
     * @return The number of times this signal was observed.
     */
    public final Integer count() {
        return count;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceUids property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceUids() {
        return resourceUids != null && !(resourceUids instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Information about the unique identifiers of the resources involved in the signal.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceUids} method.
     * </p>
     * 
     * @return Information about the unique identifiers of the resources involved in the signal.
     */
    public final List<String> resourceUids() {
        return resourceUids;
    }

    /**
     * For responses, this returns true if the service returned a value for the ActorIds property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasActorIds() {
        return actorIds != null && !(actorIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Information about the IDs of the threat actors involved in the signal.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasActorIds} method.
     * </p>
     * 
     * @return Information about the IDs of the threat actors involved in the signal.
     */
    public final List<String> actorIds() {
        return actorIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the EndpointIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEndpointIds() {
        return endpointIds != null && !(endpointIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Information about the endpoint IDs associated with this signal.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEndpointIds} method.
     * </p>
     * 
     * @return Information about the endpoint IDs associated with this signal.
     */
    public final List<String> endpointIds() {
        return endpointIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the SignalIndicators property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSignalIndicators() {
        return signalIndicators != null && !(signalIndicators instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Contains information about the indicators associated with the signals.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSignalIndicators} method.
     * </p>
     * 
     * @return Contains information about the indicators associated with the signals.
     */
    public final List<Indicator> signalIndicators() {
        return signalIndicators;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(uid());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(firstSeenAt());
        hashCode = 31 * hashCode + Objects.hashCode(lastSeenAt());
        hashCode = 31 * hashCode + Objects.hashCode(severity());
        hashCode = 31 * hashCode + Objects.hashCode(count());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceUids() ? resourceUids() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasActorIds() ? actorIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEndpointIds() ? endpointIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSignalIndicators() ? signalIndicators() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Signal)) {
            return false;
        }
        Signal other = (Signal) obj;
        return Objects.equals(uid(), other.uid()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(description(), other.description()) && Objects.equals(name(), other.name())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(updatedAt(), other.updatedAt())
                && Objects.equals(firstSeenAt(), other.firstSeenAt()) && Objects.equals(lastSeenAt(), other.lastSeenAt())
                && Objects.equals(severity(), other.severity()) && Objects.equals(count(), other.count())
                && hasResourceUids() == other.hasResourceUids() && Objects.equals(resourceUids(), other.resourceUids())
                && hasActorIds() == other.hasActorIds() && Objects.equals(actorIds(), other.actorIds())
                && hasEndpointIds() == other.hasEndpointIds() && Objects.equals(endpointIds(), other.endpointIds())
                && hasSignalIndicators() == other.hasSignalIndicators()
                && Objects.equals(signalIndicators(), other.signalIndicators());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Signal").add("Uid", uid()).add("Type", typeAsString()).add("Description", description())
                .add("Name", name()).add("CreatedAt", createdAt()).add("UpdatedAt", updatedAt())
                .add("FirstSeenAt", firstSeenAt()).add("LastSeenAt", lastSeenAt()).add("Severity", severity())
                .add("Count", count()).add("ResourceUids", hasResourceUids() ? resourceUids() : null)
                .add("ActorIds", hasActorIds() ? actorIds() : null).add("EndpointIds", hasEndpointIds() ? endpointIds() : null)
                .add("SignalIndicators", hasSignalIndicators() ? signalIndicators() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Uid":
            return Optional.ofNullable(clazz.cast(uid()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "CreatedAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "UpdatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        case "FirstSeenAt":
            return Optional.ofNullable(clazz.cast(firstSeenAt()));
        case "LastSeenAt":
            return Optional.ofNullable(clazz.cast(lastSeenAt()));
        case "Severity":
            return Optional.ofNullable(clazz.cast(severity()));
        case "Count":
            return Optional.ofNullable(clazz.cast(count()));
        case "ResourceUids":
            return Optional.ofNullable(clazz.cast(resourceUids()));
        case "ActorIds":
            return Optional.ofNullable(clazz.cast(actorIds()));
        case "EndpointIds":
            return Optional.ofNullable(clazz.cast(endpointIds()));
        case "SignalIndicators":
            return Optional.ofNullable(clazz.cast(signalIndicators()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("uid", UID_FIELD);
        map.put("type", TYPE_FIELD);
        map.put("description", DESCRIPTION_FIELD);
        map.put("name", NAME_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("updatedAt", UPDATED_AT_FIELD);
        map.put("firstSeenAt", FIRST_SEEN_AT_FIELD);
        map.put("lastSeenAt", LAST_SEEN_AT_FIELD);
        map.put("severity", SEVERITY_FIELD);
        map.put("count", COUNT_FIELD);
        map.put("resourceUids", RESOURCE_UIDS_FIELD);
        map.put("actorIds", ACTOR_IDS_FIELD);
        map.put("endpointIds", ENDPOINT_IDS_FIELD);
        map.put("signalIndicators", SIGNAL_INDICATORS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Signal, T> g) {
        return obj -> g.apply((Signal) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Signal> {
        /**
         * <p>
         * The unique identifier of the signal.
         * </p>
         * 
         * @param uid
         *        The unique identifier of the signal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder uid(String uid);

        /**
         * <p>
         * The type of the signal used to identify an attack sequence.
         * </p>
         * <p>
         * Signals can be GuardDuty findings or activities observed in data sources that GuardDuty monitors. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_data-sources.html">Foundational data
         * sources</a> in the <i>GuardDuty User Guide</i>.
         * </p>
         * <p>
         * A signal type can be one of the valid values listed in this API. Here are the related descriptions:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>FINDING</code> - Individually generated GuardDuty finding.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CLOUD_TRAIL</code> - Activity observed from CloudTrail logs
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>S3_DATA_EVENTS</code> - Activity observed from CloudTrail data events for S3. Activities associated
         * with this type will show up only when you have enabled GuardDuty S3 Protection feature in your account. For
         * more information about S3 Protection and steps to enable it, see <a
         * href="https://docs.aws.amazon.com/guardduty/latest/ug/s3-protection.html">S3 Protection</a> in the
         * <i>GuardDuty User Guide</i>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of the signal used to identify an attack sequence.</p>
         *        <p>
         *        Signals can be GuardDuty findings or activities observed in data sources that GuardDuty monitors. For
         *        more information, see <a
         *        href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_data-sources.html">Foundational data
         *        sources</a> in the <i>GuardDuty User Guide</i>.
         *        </p>
         *        <p>
         *        A signal type can be one of the valid values listed in this API. Here are the related descriptions:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>FINDING</code> - Individually generated GuardDuty finding.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CLOUD_TRAIL</code> - Activity observed from CloudTrail logs
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>S3_DATA_EVENTS</code> - Activity observed from CloudTrail data events for S3. Activities
         *        associated with this type will show up only when you have enabled GuardDuty S3 Protection feature in
         *        your account. For more information about S3 Protection and steps to enable it, see <a
         *        href="https://docs.aws.amazon.com/guardduty/latest/ug/s3-protection.html">S3 Protection</a> in the
         *        <i>GuardDuty User Guide</i>.
         *        </p>
         *        </li>
         * @see SignalType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SignalType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of the signal used to identify an attack sequence.
         * </p>
         * <p>
         * Signals can be GuardDuty findings or activities observed in data sources that GuardDuty monitors. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_data-sources.html">Foundational data
         * sources</a> in the <i>GuardDuty User Guide</i>.
         * </p>
         * <p>
         * A signal type can be one of the valid values listed in this API. Here are the related descriptions:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>FINDING</code> - Individually generated GuardDuty finding.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CLOUD_TRAIL</code> - Activity observed from CloudTrail logs
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>S3_DATA_EVENTS</code> - Activity observed from CloudTrail data events for S3. Activities associated
         * with this type will show up only when you have enabled GuardDuty S3 Protection feature in your account. For
         * more information about S3 Protection and steps to enable it, see <a
         * href="https://docs.aws.amazon.com/guardduty/latest/ug/s3-protection.html">S3 Protection</a> in the
         * <i>GuardDuty User Guide</i>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of the signal used to identify an attack sequence.</p>
         *        <p>
         *        Signals can be GuardDuty findings or activities observed in data sources that GuardDuty monitors. For
         *        more information, see <a
         *        href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_data-sources.html">Foundational data
         *        sources</a> in the <i>GuardDuty User Guide</i>.
         *        </p>
         *        <p>
         *        A signal type can be one of the valid values listed in this API. Here are the related descriptions:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>FINDING</code> - Individually generated GuardDuty finding.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CLOUD_TRAIL</code> - Activity observed from CloudTrail logs
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>S3_DATA_EVENTS</code> - Activity observed from CloudTrail data events for S3. Activities
         *        associated with this type will show up only when you have enabled GuardDuty S3 Protection feature in
         *        your account. For more information about S3 Protection and steps to enable it, see <a
         *        href="https://docs.aws.amazon.com/guardduty/latest/ug/s3-protection.html">S3 Protection</a> in the
         *        <i>GuardDuty User Guide</i>.
         *        </p>
         *        </li>
         * @see SignalType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SignalType
         */
        Builder type(SignalType type);

        /**
         * <p>
         * The description of the signal.
         * </p>
         * 
         * @param description
         *        The description of the signal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The name of the signal. For example, when signal type is <code>FINDING</code>, the signal name is the name of
         * the finding.
         * </p>
         * 
         * @param name
         *        The name of the signal. For example, when signal type is <code>FINDING</code>, the signal name is the
         *        name of the finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The timestamp when the first finding or activity related to this signal was observed.
         * </p>
         * 
         * @param createdAt
         *        The timestamp when the first finding or activity related to this signal was observed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The timestamp when this signal was last observed.
         * </p>
         * 
         * @param updatedAt
         *        The timestamp when this signal was last observed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);

        /**
         * <p>
         * The timestamp when the first finding or activity related to this signal was observed.
         * </p>
         * 
         * @param firstSeenAt
         *        The timestamp when the first finding or activity related to this signal was observed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder firstSeenAt(Instant firstSeenAt);

        /**
         * <p>
         * The timestamp when the last finding or activity related to this signal was observed.
         * </p>
         * 
         * @param lastSeenAt
         *        The timestamp when the last finding or activity related to this signal was observed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastSeenAt(Instant lastSeenAt);

        /**
         * <p>
         * The severity associated with the signal. For more information about severity, see <a
         * href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_findings-severity.html">Findings severity
         * levels</a> in the <i>GuardDuty User Guide</i>.
         * </p>
         * 
         * @param severity
         *        The severity associated with the signal. For more information about severity, see <a
         *        href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_findings-severity.html">Findings
         *        severity levels</a> in the <i>GuardDuty User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder severity(Double severity);

        /**
         * <p>
         * The number of times this signal was observed.
         * </p>
         * 
         * @param count
         *        The number of times this signal was observed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder count(Integer count);

        /**
         * <p>
         * Information about the unique identifiers of the resources involved in the signal.
         * </p>
         * 
         * @param resourceUids
         *        Information about the unique identifiers of the resources involved in the signal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceUids(Collection<String> resourceUids);

        /**
         * <p>
         * Information about the unique identifiers of the resources involved in the signal.
         * </p>
         * 
         * @param resourceUids
         *        Information about the unique identifiers of the resources involved in the signal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceUids(String... resourceUids);

        /**
         * <p>
         * Information about the IDs of the threat actors involved in the signal.
         * </p>
         * 
         * @param actorIds
         *        Information about the IDs of the threat actors involved in the signal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actorIds(Collection<String> actorIds);

        /**
         * <p>
         * Information about the IDs of the threat actors involved in the signal.
         * </p>
         * 
         * @param actorIds
         *        Information about the IDs of the threat actors involved in the signal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actorIds(String... actorIds);

        /**
         * <p>
         * Information about the endpoint IDs associated with this signal.
         * </p>
         * 
         * @param endpointIds
         *        Information about the endpoint IDs associated with this signal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointIds(Collection<String> endpointIds);

        /**
         * <p>
         * Information about the endpoint IDs associated with this signal.
         * </p>
         * 
         * @param endpointIds
         *        Information about the endpoint IDs associated with this signal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointIds(String... endpointIds);

        /**
         * <p>
         * Contains information about the indicators associated with the signals.
         * </p>
         * 
         * @param signalIndicators
         *        Contains information about the indicators associated with the signals.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signalIndicators(Collection<Indicator> signalIndicators);

        /**
         * <p>
         * Contains information about the indicators associated with the signals.
         * </p>
         * 
         * @param signalIndicators
         *        Contains information about the indicators associated with the signals.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signalIndicators(Indicator... signalIndicators);

        /**
         * <p>
         * Contains information about the indicators associated with the signals.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.guardduty.model.Indicator.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.guardduty.model.Indicator#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.guardduty.model.Indicator.Builder#build()} is called immediately and
         * its result is passed to {@link #signalIndicators(List<Indicator>)}.
         * 
         * @param signalIndicators
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.guardduty.model.Indicator.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #signalIndicators(java.util.Collection<Indicator>)
         */
        Builder signalIndicators(Consumer<Indicator.Builder>... signalIndicators);
    }

    static final class BuilderImpl implements Builder {
        private String uid;

        private String type;

        private String description;

        private String name;

        private Instant createdAt;

        private Instant updatedAt;

        private Instant firstSeenAt;

        private Instant lastSeenAt;

        private Double severity;

        private Integer count;

        private List<String> resourceUids = DefaultSdkAutoConstructList.getInstance();

        private List<String> actorIds = DefaultSdkAutoConstructList.getInstance();

        private List<String> endpointIds = DefaultSdkAutoConstructList.getInstance();

        private List<Indicator> signalIndicators = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Signal model) {
            uid(model.uid);
            type(model.type);
            description(model.description);
            name(model.name);
            createdAt(model.createdAt);
            updatedAt(model.updatedAt);
            firstSeenAt(model.firstSeenAt);
            lastSeenAt(model.lastSeenAt);
            severity(model.severity);
            count(model.count);
            resourceUids(model.resourceUids);
            actorIds(model.actorIds);
            endpointIds(model.endpointIds);
            signalIndicators(model.signalIndicators);
        }

        public final String getUid() {
            return uid;
        }

        public final void setUid(String uid) {
            this.uid = uid;
        }

        @Override
        public final Builder uid(String uid) {
            this.uid = uid;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(SignalType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        public final Instant getFirstSeenAt() {
            return firstSeenAt;
        }

        public final void setFirstSeenAt(Instant firstSeenAt) {
            this.firstSeenAt = firstSeenAt;
        }

        @Override
        public final Builder firstSeenAt(Instant firstSeenAt) {
            this.firstSeenAt = firstSeenAt;
            return this;
        }

        public final Instant getLastSeenAt() {
            return lastSeenAt;
        }

        public final void setLastSeenAt(Instant lastSeenAt) {
            this.lastSeenAt = lastSeenAt;
        }

        @Override
        public final Builder lastSeenAt(Instant lastSeenAt) {
            this.lastSeenAt = lastSeenAt;
            return this;
        }

        public final Double getSeverity() {
            return severity;
        }

        public final void setSeverity(Double severity) {
            this.severity = severity;
        }

        @Override
        public final Builder severity(Double severity) {
            this.severity = severity;
            return this;
        }

        public final Integer getCount() {
            return count;
        }

        public final void setCount(Integer count) {
            this.count = count;
        }

        @Override
        public final Builder count(Integer count) {
            this.count = count;
            return this;
        }

        public final Collection<String> getResourceUids() {
            if (resourceUids instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceUids;
        }

        public final void setResourceUids(Collection<String> resourceUids) {
            this.resourceUids = ResourceUidsCopier.copy(resourceUids);
        }

        @Override
        public final Builder resourceUids(Collection<String> resourceUids) {
            this.resourceUids = ResourceUidsCopier.copy(resourceUids);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceUids(String... resourceUids) {
            resourceUids(Arrays.asList(resourceUids));
            return this;
        }

        public final Collection<String> getActorIds() {
            if (actorIds instanceof SdkAutoConstructList) {
                return null;
            }
            return actorIds;
        }

        public final void setActorIds(Collection<String> actorIds) {
            this.actorIds = ActorIdsCopier.copy(actorIds);
        }

        @Override
        public final Builder actorIds(Collection<String> actorIds) {
            this.actorIds = ActorIdsCopier.copy(actorIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actorIds(String... actorIds) {
            actorIds(Arrays.asList(actorIds));
            return this;
        }

        public final Collection<String> getEndpointIds() {
            if (endpointIds instanceof SdkAutoConstructList) {
                return null;
            }
            return endpointIds;
        }

        public final void setEndpointIds(Collection<String> endpointIds) {
            this.endpointIds = EndpointIdsCopier.copy(endpointIds);
        }

        @Override
        public final Builder endpointIds(Collection<String> endpointIds) {
            this.endpointIds = EndpointIdsCopier.copy(endpointIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder endpointIds(String... endpointIds) {
            endpointIds(Arrays.asList(endpointIds));
            return this;
        }

        public final List<Indicator.Builder> getSignalIndicators() {
            List<Indicator.Builder> result = IndicatorsCopier.copyToBuilder(this.signalIndicators);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSignalIndicators(Collection<Indicator.BuilderImpl> signalIndicators) {
            this.signalIndicators = IndicatorsCopier.copyFromBuilder(signalIndicators);
        }

        @Override
        public final Builder signalIndicators(Collection<Indicator> signalIndicators) {
            this.signalIndicators = IndicatorsCopier.copy(signalIndicators);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder signalIndicators(Indicator... signalIndicators) {
            signalIndicators(Arrays.asList(signalIndicators));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder signalIndicators(Consumer<Indicator.Builder>... signalIndicators) {
            signalIndicators(Stream.of(signalIndicators).map(c -> Indicator.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public Signal build() {
            return new Signal(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
