/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.health.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Detailed information about an event. A combination of an <a
 * href="https://docs.aws.amazon.com/health/latest/APIReference/API_Event.html">Event</a> object, an <a
 * href="https://docs.aws.amazon.com/health/latest/APIReference/API_EventDescription.html">EventDescription</a> object,
 * and additional metadata about the event. Returned by the <a
 * href="https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventDetails.html">DescribeEventDetails</a>
 * operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EventDetails implements SdkPojo, Serializable, ToCopyableBuilder<EventDetails.Builder, EventDetails> {
    private static final SdkField<Event> EVENT_FIELD = SdkField.<Event> builder(MarshallingType.SDK_POJO).memberName("event")
            .getter(getter(EventDetails::event)).setter(setter(Builder::event)).constructor(Event::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("event").build()).build();

    private static final SdkField<EventDescription> EVENT_DESCRIPTION_FIELD = SdkField
            .<EventDescription> builder(MarshallingType.SDK_POJO).memberName("eventDescription")
            .getter(getter(EventDetails::eventDescription)).setter(setter(Builder::eventDescription))
            .constructor(EventDescription::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventDescription").build()).build();

    private static final SdkField<Map<String, String>> EVENT_METADATA_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("eventMetadata")
            .getter(getter(EventDetails::eventMetadata))
            .setter(setter(Builder::eventMetadata))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventMetadata").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EVENT_FIELD,
            EVENT_DESCRIPTION_FIELD, EVENT_METADATA_FIELD));

    private static final long serialVersionUID = 1L;

    private final Event event;

    private final EventDescription eventDescription;

    private final Map<String, String> eventMetadata;

    private EventDetails(BuilderImpl builder) {
        this.event = builder.event;
        this.eventDescription = builder.eventDescription;
        this.eventMetadata = builder.eventMetadata;
    }

    /**
     * <p>
     * Summary information about the event.
     * </p>
     * 
     * @return Summary information about the event.
     */
    public final Event event() {
        return event;
    }

    /**
     * <p>
     * The most recent description of the event.
     * </p>
     * 
     * @return The most recent description of the event.
     */
    public final EventDescription eventDescription() {
        return eventDescription;
    }

    /**
     * For responses, this returns true if the service returned a value for the EventMetadata property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEventMetadata() {
        return eventMetadata != null && !(eventMetadata instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Additional metadata about the event.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEventMetadata} method.
     * </p>
     * 
     * @return Additional metadata about the event.
     */
    public final Map<String, String> eventMetadata() {
        return eventMetadata;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(event());
        hashCode = 31 * hashCode + Objects.hashCode(eventDescription());
        hashCode = 31 * hashCode + Objects.hashCode(hasEventMetadata() ? eventMetadata() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EventDetails)) {
            return false;
        }
        EventDetails other = (EventDetails) obj;
        return Objects.equals(event(), other.event()) && Objects.equals(eventDescription(), other.eventDescription())
                && hasEventMetadata() == other.hasEventMetadata() && Objects.equals(eventMetadata(), other.eventMetadata());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EventDetails").add("Event", event()).add("EventDescription", eventDescription())
                .add("EventMetadata", hasEventMetadata() ? eventMetadata() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "event":
            return Optional.ofNullable(clazz.cast(event()));
        case "eventDescription":
            return Optional.ofNullable(clazz.cast(eventDescription()));
        case "eventMetadata":
            return Optional.ofNullable(clazz.cast(eventMetadata()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EventDetails, T> g) {
        return obj -> g.apply((EventDetails) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EventDetails> {
        /**
         * <p>
         * Summary information about the event.
         * </p>
         * 
         * @param event
         *        Summary information about the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder event(Event event);

        /**
         * <p>
         * Summary information about the event.
         * </p>
         * This is a convenience that creates an instance of the {@link Event.Builder} avoiding the need to create one
         * manually via {@link Event#builder()}.
         *
         * When the {@link Consumer} completes, {@link Event.Builder#build()} is called immediately and its result is
         * passed to {@link #event(Event)}.
         * 
         * @param event
         *        a consumer that will call methods on {@link Event.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #event(Event)
         */
        default Builder event(Consumer<Event.Builder> event) {
            return event(Event.builder().applyMutation(event).build());
        }

        /**
         * <p>
         * The most recent description of the event.
         * </p>
         * 
         * @param eventDescription
         *        The most recent description of the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventDescription(EventDescription eventDescription);

        /**
         * <p>
         * The most recent description of the event.
         * </p>
         * This is a convenience that creates an instance of the {@link EventDescription.Builder} avoiding the need to
         * create one manually via {@link EventDescription#builder()}.
         *
         * When the {@link Consumer} completes, {@link EventDescription.Builder#build()} is called immediately and its
         * result is passed to {@link #eventDescription(EventDescription)}.
         * 
         * @param eventDescription
         *        a consumer that will call methods on {@link EventDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #eventDescription(EventDescription)
         */
        default Builder eventDescription(Consumer<EventDescription.Builder> eventDescription) {
            return eventDescription(EventDescription.builder().applyMutation(eventDescription).build());
        }

        /**
         * <p>
         * Additional metadata about the event.
         * </p>
         * 
         * @param eventMetadata
         *        Additional metadata about the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventMetadata(Map<String, String> eventMetadata);
    }

    static final class BuilderImpl implements Builder {
        private Event event;

        private EventDescription eventDescription;

        private Map<String, String> eventMetadata = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(EventDetails model) {
            event(model.event);
            eventDescription(model.eventDescription);
            eventMetadata(model.eventMetadata);
        }

        public final Event.Builder getEvent() {
            return event != null ? event.toBuilder() : null;
        }

        public final void setEvent(Event.BuilderImpl event) {
            this.event = event != null ? event.build() : null;
        }

        @Override
        @Transient
        public final Builder event(Event event) {
            this.event = event;
            return this;
        }

        public final EventDescription.Builder getEventDescription() {
            return eventDescription != null ? eventDescription.toBuilder() : null;
        }

        public final void setEventDescription(EventDescription.BuilderImpl eventDescription) {
            this.eventDescription = eventDescription != null ? eventDescription.build() : null;
        }

        @Override
        @Transient
        public final Builder eventDescription(EventDescription eventDescription) {
            this.eventDescription = eventDescription;
            return this;
        }

        public final Map<String, String> getEventMetadata() {
            if (eventMetadata instanceof SdkAutoConstructMap) {
                return null;
            }
            return eventMetadata;
        }

        public final void setEventMetadata(Map<String, String> eventMetadata) {
            this.eventMetadata = _eventMetadataCopier.copy(eventMetadata);
        }

        @Override
        @Transient
        public final Builder eventMetadata(Map<String, String> eventMetadata) {
            this.eventMetadata = _eventMetadataCopier.copy(eventMetadata);
            return this;
        }

        @Override
        public EventDetails build() {
            return new EventDetails(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
