/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.health.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Metadata about a type of event that is reported by AWS Health. Data consists of the category (for example,
 * <code>issue</code>), the service (for example, <code>EC2</code>), and the event type code (for example,
 * <code>AWS_EC2_SYSTEM_MAINTENANCE_EVENT</code>).
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EventType implements SdkPojo, Serializable, ToCopyableBuilder<EventType.Builder, EventType> {
    private static final SdkField<String> SERVICE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(EventType::service)).setter(setter(Builder::service))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("service").build()).build();

    private static final SdkField<String> CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(EventType::code)).setter(setter(Builder::code))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("code").build()).build();

    private static final SdkField<String> CATEGORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(EventType::categoryAsString)).setter(setter(Builder::category))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("category").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SERVICE_FIELD, CODE_FIELD,
            CATEGORY_FIELD));

    private static final long serialVersionUID = 1L;

    private final String service;

    private final String code;

    private final String category;

    private EventType(BuilderImpl builder) {
        this.service = builder.service;
        this.code = builder.code;
        this.category = builder.category;
    }

    /**
     * <p>
     * The AWS service that is affected by the event. For example, <code>EC2</code>, <code>RDS</code>.
     * </p>
     * 
     * @return The AWS service that is affected by the event. For example, <code>EC2</code>, <code>RDS</code>.
     */
    public String service() {
        return service;
    }

    /**
     * <p>
     * The unique identifier for the event type. The format is <code>AWS_<i>SERVICE</i>_<i>DESCRIPTION</i> </code>; for
     * example, <code>AWS_EC2_SYSTEM_MAINTENANCE_EVENT</code>.
     * </p>
     * 
     * @return The unique identifier for the event type. The format is
     *         <code>AWS_<i>SERVICE</i>_<i>DESCRIPTION</i> </code>; for example,
     *         <code>AWS_EC2_SYSTEM_MAINTENANCE_EVENT</code>.
     */
    public String code() {
        return code;
    }

    /**
     * <p>
     * A list of event type category codes (<code>issue</code>, <code>scheduledChange</code>, or
     * <code>accountNotification</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #category} will
     * return {@link EventTypeCategory#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #categoryAsString}.
     * </p>
     * 
     * @return A list of event type category codes (<code>issue</code>, <code>scheduledChange</code>, or
     *         <code>accountNotification</code>).
     * @see EventTypeCategory
     */
    public EventTypeCategory category() {
        return EventTypeCategory.fromValue(category);
    }

    /**
     * <p>
     * A list of event type category codes (<code>issue</code>, <code>scheduledChange</code>, or
     * <code>accountNotification</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #category} will
     * return {@link EventTypeCategory#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #categoryAsString}.
     * </p>
     * 
     * @return A list of event type category codes (<code>issue</code>, <code>scheduledChange</code>, or
     *         <code>accountNotification</code>).
     * @see EventTypeCategory
     */
    public String categoryAsString() {
        return category;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(service());
        hashCode = 31 * hashCode + Objects.hashCode(code());
        hashCode = 31 * hashCode + Objects.hashCode(categoryAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EventType)) {
            return false;
        }
        EventType other = (EventType) obj;
        return Objects.equals(service(), other.service()) && Objects.equals(code(), other.code())
                && Objects.equals(categoryAsString(), other.categoryAsString());
    }

    @Override
    public String toString() {
        return ToString.builder("EventType").add("Service", service()).add("Code", code()).add("Category", categoryAsString())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "service":
            return Optional.ofNullable(clazz.cast(service()));
        case "code":
            return Optional.ofNullable(clazz.cast(code()));
        case "category":
            return Optional.ofNullable(clazz.cast(categoryAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EventType, T> g) {
        return obj -> g.apply((EventType) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EventType> {
        /**
         * <p>
         * The AWS service that is affected by the event. For example, <code>EC2</code>, <code>RDS</code>.
         * </p>
         * 
         * @param service
         *        The AWS service that is affected by the event. For example, <code>EC2</code>, <code>RDS</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder service(String service);

        /**
         * <p>
         * The unique identifier for the event type. The format is <code>AWS_<i>SERVICE</i>_<i>DESCRIPTION</i> </code>;
         * for example, <code>AWS_EC2_SYSTEM_MAINTENANCE_EVENT</code>.
         * </p>
         * 
         * @param code
         *        The unique identifier for the event type. The format is
         *        <code>AWS_<i>SERVICE</i>_<i>DESCRIPTION</i> </code>; for example,
         *        <code>AWS_EC2_SYSTEM_MAINTENANCE_EVENT</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder code(String code);

        /**
         * <p>
         * A list of event type category codes (<code>issue</code>, <code>scheduledChange</code>, or
         * <code>accountNotification</code>).
         * </p>
         * 
         * @param category
         *        A list of event type category codes (<code>issue</code>, <code>scheduledChange</code>, or
         *        <code>accountNotification</code>).
         * @see EventTypeCategory
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventTypeCategory
         */
        Builder category(String category);

        /**
         * <p>
         * A list of event type category codes (<code>issue</code>, <code>scheduledChange</code>, or
         * <code>accountNotification</code>).
         * </p>
         * 
         * @param category
         *        A list of event type category codes (<code>issue</code>, <code>scheduledChange</code>, or
         *        <code>accountNotification</code>).
         * @see EventTypeCategory
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventTypeCategory
         */
        Builder category(EventTypeCategory category);
    }

    static final class BuilderImpl implements Builder {
        private String service;

        private String code;

        private String category;

        private BuilderImpl() {
        }

        private BuilderImpl(EventType model) {
            service(model.service);
            code(model.code);
            category(model.category);
        }

        public final String getService() {
            return service;
        }

        @Override
        public final Builder service(String service) {
            this.service = service;
            return this;
        }

        public final void setService(String service) {
            this.service = service;
        }

        public final String getCode() {
            return code;
        }

        @Override
        public final Builder code(String code) {
            this.code = code;
            return this;
        }

        public final void setCode(String code) {
            this.code = code;
        }

        public final String getCategoryAsString() {
            return category;
        }

        @Override
        public final Builder category(String category) {
            this.category = category;
            return this;
        }

        @Override
        public final Builder category(EventTypeCategory category) {
            this.category(category.toString());
            return this;
        }

        public final void setCategory(String category) {
            this.category = category;
        }

        @Override
        public EventType build() {
            return new EventType(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
