/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.healthlake;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.healthlake.model.AccessDeniedException;
import software.amazon.awssdk.services.healthlake.model.ConflictException;
import software.amazon.awssdk.services.healthlake.model.CreateFhirDatastoreRequest;
import software.amazon.awssdk.services.healthlake.model.CreateFhirDatastoreResponse;
import software.amazon.awssdk.services.healthlake.model.DeleteFhirDatastoreRequest;
import software.amazon.awssdk.services.healthlake.model.DeleteFhirDatastoreResponse;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirDatastoreRequest;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirDatastoreResponse;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirExportJobRequest;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirExportJobResponse;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirImportJobRequest;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirImportJobResponse;
import software.amazon.awssdk.services.healthlake.model.HealthLakeException;
import software.amazon.awssdk.services.healthlake.model.HealthLakeRequest;
import software.amazon.awssdk.services.healthlake.model.InternalServerException;
import software.amazon.awssdk.services.healthlake.model.ListFhirDatastoresRequest;
import software.amazon.awssdk.services.healthlake.model.ListFhirDatastoresResponse;
import software.amazon.awssdk.services.healthlake.model.ListFhirExportJobsRequest;
import software.amazon.awssdk.services.healthlake.model.ListFhirExportJobsResponse;
import software.amazon.awssdk.services.healthlake.model.ListFhirImportJobsRequest;
import software.amazon.awssdk.services.healthlake.model.ListFhirImportJobsResponse;
import software.amazon.awssdk.services.healthlake.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.healthlake.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.healthlake.model.ResourceNotFoundException;
import software.amazon.awssdk.services.healthlake.model.StartFhirExportJobRequest;
import software.amazon.awssdk.services.healthlake.model.StartFhirExportJobResponse;
import software.amazon.awssdk.services.healthlake.model.StartFhirImportJobRequest;
import software.amazon.awssdk.services.healthlake.model.StartFhirImportJobResponse;
import software.amazon.awssdk.services.healthlake.model.TagResourceRequest;
import software.amazon.awssdk.services.healthlake.model.TagResourceResponse;
import software.amazon.awssdk.services.healthlake.model.ThrottlingException;
import software.amazon.awssdk.services.healthlake.model.UntagResourceRequest;
import software.amazon.awssdk.services.healthlake.model.UntagResourceResponse;
import software.amazon.awssdk.services.healthlake.model.ValidationException;
import software.amazon.awssdk.services.healthlake.paginators.ListFHIRDatastoresIterable;
import software.amazon.awssdk.services.healthlake.paginators.ListFHIRExportJobsIterable;
import software.amazon.awssdk.services.healthlake.paginators.ListFHIRImportJobsIterable;
import software.amazon.awssdk.services.healthlake.transform.CreateFhirDatastoreRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.DeleteFhirDatastoreRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.DescribeFhirDatastoreRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.DescribeFhirExportJobRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.DescribeFhirImportJobRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.ListFhirDatastoresRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.ListFhirExportJobsRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.ListFhirImportJobsRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.StartFhirExportJobRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.StartFhirImportJobRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link HealthLakeClient}.
 *
 * @see HealthLakeClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultHealthLakeClient implements HealthLakeClient {
    private static final Logger log = Logger.loggerFor(DefaultHealthLakeClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultHealthLakeClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a Data Store that can ingest and export FHIR formatted data.
     * </p>
     *
     * @param createFhirDatastoreRequest
     * @return Result of the CreateFHIRDatastore operation returned by the service.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ThrottlingException
     *         The user has exceeded their maximum number of allowed calls to the given API.
     * @throws AccessDeniedException
     *         Access is denied. Your account is not authorized to perform this operation.
     * @throws InternalServerException
     *         Unknown error occurs in the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.CreateFHIRDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/CreateFHIRDatastore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateFhirDatastoreResponse createFHIRDatastore(CreateFhirDatastoreRequest createFhirDatastoreRequest)
            throws ValidationException, ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateFhirDatastoreResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateFhirDatastoreResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFhirDatastoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFHIRDatastore");

            return clientHandler.execute(new ClientExecutionParams<CreateFhirDatastoreRequest, CreateFhirDatastoreResponse>()
                    .withOperationName("CreateFHIRDatastore").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createFhirDatastoreRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateFhirDatastoreRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a Data Store.
     * </p>
     *
     * @param deleteFhirDatastoreRequest
     * @return Result of the DeleteFHIRDatastore operation returned by the service.
     * @throws AccessDeniedException
     *         Access is denied. Your account is not authorized to perform this operation.
     * @throws ConflictException
     *         The Data Store is in a transition state and the user requested action can not be performed.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ResourceNotFoundException
     *         The requested Data Store was not found.
     * @throws ThrottlingException
     *         The user has exceeded their maximum number of allowed calls to the given API.
     * @throws InternalServerException
     *         Unknown error occurs in the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.DeleteFHIRDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/DeleteFHIRDatastore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteFhirDatastoreResponse deleteFHIRDatastore(DeleteFhirDatastoreRequest deleteFhirDatastoreRequest)
            throws AccessDeniedException, ConflictException, ValidationException, ResourceNotFoundException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteFhirDatastoreResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteFhirDatastoreResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFhirDatastoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFHIRDatastore");

            return clientHandler.execute(new ClientExecutionParams<DeleteFhirDatastoreRequest, DeleteFhirDatastoreResponse>()
                    .withOperationName("DeleteFHIRDatastore").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteFhirDatastoreRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteFhirDatastoreRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the properties associated with the FHIR Data Store, including the Data Store ID, Data Store ARN, Data Store
     * name, Data Store status, created at, Data Store type version, and Data Store endpoint.
     * </p>
     *
     * @param describeFhirDatastoreRequest
     * @return Result of the DescribeFHIRDatastore operation returned by the service.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ResourceNotFoundException
     *         The requested Data Store was not found.
     * @throws ThrottlingException
     *         The user has exceeded their maximum number of allowed calls to the given API.
     * @throws InternalServerException
     *         Unknown error occurs in the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.DescribeFHIRDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/DescribeFHIRDatastore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeFhirDatastoreResponse describeFHIRDatastore(DescribeFhirDatastoreRequest describeFhirDatastoreRequest)
            throws ValidationException, ResourceNotFoundException, ThrottlingException, InternalServerException,
            AwsServiceException, SdkClientException, HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeFhirDatastoreResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeFhirDatastoreResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFhirDatastoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFHIRDatastore");

            return clientHandler.execute(new ClientExecutionParams<DescribeFhirDatastoreRequest, DescribeFhirDatastoreResponse>()
                    .withOperationName("DescribeFHIRDatastore").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeFhirDatastoreRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeFhirDatastoreRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Displays the properties of a FHIR export job, including the ID, ARN, name, and the status of the job.
     * </p>
     *
     * @param describeFhirExportJobRequest
     * @return Result of the DescribeFHIRExportJob operation returned by the service.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ResourceNotFoundException
     *         The requested Data Store was not found.
     * @throws ThrottlingException
     *         The user has exceeded their maximum number of allowed calls to the given API.
     * @throws InternalServerException
     *         Unknown error occurs in the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.DescribeFHIRExportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/DescribeFHIRExportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeFhirExportJobResponse describeFHIRExportJob(DescribeFhirExportJobRequest describeFhirExportJobRequest)
            throws ValidationException, ResourceNotFoundException, ThrottlingException, InternalServerException,
            AwsServiceException, SdkClientException, HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeFhirExportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeFhirExportJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFhirExportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFHIRExportJob");

            return clientHandler.execute(new ClientExecutionParams<DescribeFhirExportJobRequest, DescribeFhirExportJobResponse>()
                    .withOperationName("DescribeFHIRExportJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeFhirExportJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeFhirExportJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Displays the properties of a FHIR import job, including the ID, ARN, name, and the status of the job.
     * </p>
     *
     * @param describeFhirImportJobRequest
     * @return Result of the DescribeFHIRImportJob operation returned by the service.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ResourceNotFoundException
     *         The requested Data Store was not found.
     * @throws ThrottlingException
     *         The user has exceeded their maximum number of allowed calls to the given API.
     * @throws InternalServerException
     *         Unknown error occurs in the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.DescribeFHIRImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/DescribeFHIRImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeFhirImportJobResponse describeFHIRImportJob(DescribeFhirImportJobRequest describeFhirImportJobRequest)
            throws ValidationException, ResourceNotFoundException, ThrottlingException, InternalServerException,
            AwsServiceException, SdkClientException, HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeFhirImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeFhirImportJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFhirImportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFHIRImportJob");

            return clientHandler.execute(new ClientExecutionParams<DescribeFhirImportJobRequest, DescribeFhirImportJobResponse>()
                    .withOperationName("DescribeFHIRImportJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeFhirImportJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeFhirImportJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all FHIR Data Stores that are in the user’s account, regardless of Data Store status.
     * </p>
     *
     * @param listFhirDatastoresRequest
     * @return Result of the ListFHIRDatastores operation returned by the service.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ThrottlingException
     *         The user has exceeded their maximum number of allowed calls to the given API.
     * @throws InternalServerException
     *         Unknown error occurs in the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.ListFHIRDatastores
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/ListFHIRDatastores" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListFhirDatastoresResponse listFHIRDatastores(ListFhirDatastoresRequest listFhirDatastoresRequest)
            throws ValidationException, ThrottlingException, InternalServerException, AwsServiceException, SdkClientException,
            HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListFhirDatastoresResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListFhirDatastoresResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFhirDatastoresRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFHIRDatastores");

            return clientHandler.execute(new ClientExecutionParams<ListFhirDatastoresRequest, ListFhirDatastoresResponse>()
                    .withOperationName("ListFHIRDatastores").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listFhirDatastoresRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListFhirDatastoresRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all FHIR Data Stores that are in the user’s account, regardless of Data Store status.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listFHIRDatastores(software.amazon.awssdk.services.healthlake.model.ListFhirDatastoresRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.healthlake.paginators.ListFHIRDatastoresIterable responses = client.listFHIRDatastoresPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.healthlake.paginators.ListFHIRDatastoresIterable responses = client
     *             .listFHIRDatastoresPaginator(request);
     *     for (software.amazon.awssdk.services.healthlake.model.ListFhirDatastoresResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.healthlake.paginators.ListFHIRDatastoresIterable responses = client.listFHIRDatastoresPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listFHIRDatastores(software.amazon.awssdk.services.healthlake.model.ListFhirDatastoresRequest)}
     * operation.</b>
     * </p>
     *
     * @param listFhirDatastoresRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ThrottlingException
     *         The user has exceeded their maximum number of allowed calls to the given API.
     * @throws InternalServerException
     *         Unknown error occurs in the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.ListFHIRDatastores
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/ListFHIRDatastores" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListFHIRDatastoresIterable listFHIRDatastoresPaginator(ListFhirDatastoresRequest listFhirDatastoresRequest)
            throws ValidationException, ThrottlingException, InternalServerException, AwsServiceException, SdkClientException,
            HealthLakeException {
        return new ListFHIRDatastoresIterable(this, applyPaginatorUserAgent(listFhirDatastoresRequest));
    }

    /**
     * <p>
     * Lists all FHIR export jobs associated with an account and their statuses.
     * </p>
     *
     * @param listFhirExportJobsRequest
     * @return Result of the ListFHIRExportJobs operation returned by the service.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ResourceNotFoundException
     *         The requested Data Store was not found.
     * @throws AccessDeniedException
     *         Access is denied. Your account is not authorized to perform this operation.
     * @throws ThrottlingException
     *         The user has exceeded their maximum number of allowed calls to the given API.
     * @throws InternalServerException
     *         Unknown error occurs in the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.ListFHIRExportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/ListFHIRExportJobs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListFhirExportJobsResponse listFHIRExportJobs(ListFhirExportJobsRequest listFhirExportJobsRequest)
            throws ValidationException, ResourceNotFoundException, AccessDeniedException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListFhirExportJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListFhirExportJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFhirExportJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFHIRExportJobs");

            return clientHandler.execute(new ClientExecutionParams<ListFhirExportJobsRequest, ListFhirExportJobsResponse>()
                    .withOperationName("ListFHIRExportJobs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listFhirExportJobsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListFhirExportJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all FHIR export jobs associated with an account and their statuses.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listFHIRExportJobs(software.amazon.awssdk.services.healthlake.model.ListFhirExportJobsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.healthlake.paginators.ListFHIRExportJobsIterable responses = client.listFHIRExportJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.healthlake.paginators.ListFHIRExportJobsIterable responses = client
     *             .listFHIRExportJobsPaginator(request);
     *     for (software.amazon.awssdk.services.healthlake.model.ListFhirExportJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.healthlake.paginators.ListFHIRExportJobsIterable responses = client.listFHIRExportJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listFHIRExportJobs(software.amazon.awssdk.services.healthlake.model.ListFhirExportJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listFhirExportJobsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ResourceNotFoundException
     *         The requested Data Store was not found.
     * @throws AccessDeniedException
     *         Access is denied. Your account is not authorized to perform this operation.
     * @throws ThrottlingException
     *         The user has exceeded their maximum number of allowed calls to the given API.
     * @throws InternalServerException
     *         Unknown error occurs in the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.ListFHIRExportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/ListFHIRExportJobs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListFHIRExportJobsIterable listFHIRExportJobsPaginator(ListFhirExportJobsRequest listFhirExportJobsRequest)
            throws ValidationException, ResourceNotFoundException, AccessDeniedException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, HealthLakeException {
        return new ListFHIRExportJobsIterable(this, applyPaginatorUserAgent(listFhirExportJobsRequest));
    }

    /**
     * <p>
     * Lists all FHIR import jobs associated with an account and their statuses.
     * </p>
     *
     * @param listFhirImportJobsRequest
     * @return Result of the ListFHIRImportJobs operation returned by the service.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ResourceNotFoundException
     *         The requested Data Store was not found.
     * @throws AccessDeniedException
     *         Access is denied. Your account is not authorized to perform this operation.
     * @throws ThrottlingException
     *         The user has exceeded their maximum number of allowed calls to the given API.
     * @throws InternalServerException
     *         Unknown error occurs in the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.ListFHIRImportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/ListFHIRImportJobs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListFhirImportJobsResponse listFHIRImportJobs(ListFhirImportJobsRequest listFhirImportJobsRequest)
            throws ValidationException, ResourceNotFoundException, AccessDeniedException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListFhirImportJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListFhirImportJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFhirImportJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFHIRImportJobs");

            return clientHandler.execute(new ClientExecutionParams<ListFhirImportJobsRequest, ListFhirImportJobsResponse>()
                    .withOperationName("ListFHIRImportJobs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listFhirImportJobsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListFhirImportJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all FHIR import jobs associated with an account and their statuses.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listFHIRImportJobs(software.amazon.awssdk.services.healthlake.model.ListFhirImportJobsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.healthlake.paginators.ListFHIRImportJobsIterable responses = client.listFHIRImportJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.healthlake.paginators.ListFHIRImportJobsIterable responses = client
     *             .listFHIRImportJobsPaginator(request);
     *     for (software.amazon.awssdk.services.healthlake.model.ListFhirImportJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.healthlake.paginators.ListFHIRImportJobsIterable responses = client.listFHIRImportJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listFHIRImportJobs(software.amazon.awssdk.services.healthlake.model.ListFhirImportJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listFhirImportJobsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ResourceNotFoundException
     *         The requested Data Store was not found.
     * @throws AccessDeniedException
     *         Access is denied. Your account is not authorized to perform this operation.
     * @throws ThrottlingException
     *         The user has exceeded their maximum number of allowed calls to the given API.
     * @throws InternalServerException
     *         Unknown error occurs in the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.ListFHIRImportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/ListFHIRImportJobs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListFHIRImportJobsIterable listFHIRImportJobsPaginator(ListFhirImportJobsRequest listFhirImportJobsRequest)
            throws ValidationException, ResourceNotFoundException, AccessDeniedException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, HealthLakeException {
        return new ListFHIRImportJobsIterable(this, applyPaginatorUserAgent(listFhirImportJobsRequest));
    }

    /**
     * <p>
     * Returns a list of all existing tags associated with a Data Store.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ResourceNotFoundException
     *         The requested Data Store was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Begins a FHIR export job.
     * </p>
     *
     * @param startFhirExportJobRequest
     * @return Result of the StartFHIRExportJob operation returned by the service.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ThrottlingException
     *         The user has exceeded their maximum number of allowed calls to the given API.
     * @throws AccessDeniedException
     *         Access is denied. Your account is not authorized to perform this operation.
     * @throws ResourceNotFoundException
     *         The requested Data Store was not found.
     * @throws InternalServerException
     *         Unknown error occurs in the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.StartFHIRExportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/StartFHIRExportJob" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StartFhirExportJobResponse startFHIRExportJob(StartFhirExportJobRequest startFhirExportJobRequest)
            throws ValidationException, ThrottlingException, AccessDeniedException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartFhirExportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartFhirExportJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startFhirExportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartFHIRExportJob");

            return clientHandler.execute(new ClientExecutionParams<StartFhirExportJobRequest, StartFhirExportJobResponse>()
                    .withOperationName("StartFHIRExportJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startFhirExportJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartFhirExportJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Begins a FHIR Import job.
     * </p>
     *
     * @param startFhirImportJobRequest
     * @return Result of the StartFHIRImportJob operation returned by the service.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ThrottlingException
     *         The user has exceeded their maximum number of allowed calls to the given API.
     * @throws AccessDeniedException
     *         Access is denied. Your account is not authorized to perform this operation.
     * @throws ResourceNotFoundException
     *         The requested Data Store was not found.
     * @throws InternalServerException
     *         Unknown error occurs in the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.StartFHIRImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/StartFHIRImportJob" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StartFhirImportJobResponse startFHIRImportJob(StartFhirImportJobRequest startFhirImportJobRequest)
            throws ValidationException, ThrottlingException, AccessDeniedException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartFhirImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartFhirImportJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startFhirImportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartFHIRImportJob");

            return clientHandler.execute(new ClientExecutionParams<StartFhirImportJobRequest, StartFhirImportJobResponse>()
                    .withOperationName("StartFHIRImportJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startFhirImportJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartFhirImportJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds a user specifed key and value tag to a Data Store.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ResourceNotFoundException
     *         The requested Data Store was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from a Data Store.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ResourceNotFoundException
     *         The requested Data Store was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private <T extends HealthLakeRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(HealthLakeException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
