/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.healthlake;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.healthlake.internal.HealthLakeServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.healthlake.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.healthlake.model.AccessDeniedException;
import software.amazon.awssdk.services.healthlake.model.ConflictException;
import software.amazon.awssdk.services.healthlake.model.CreateFhirDatastoreRequest;
import software.amazon.awssdk.services.healthlake.model.CreateFhirDatastoreResponse;
import software.amazon.awssdk.services.healthlake.model.DeleteFhirDatastoreRequest;
import software.amazon.awssdk.services.healthlake.model.DeleteFhirDatastoreResponse;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirDatastoreRequest;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirDatastoreResponse;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirExportJobRequest;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirExportJobResponse;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirImportJobRequest;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirImportJobResponse;
import software.amazon.awssdk.services.healthlake.model.HealthLakeException;
import software.amazon.awssdk.services.healthlake.model.InternalServerException;
import software.amazon.awssdk.services.healthlake.model.ListFhirDatastoresRequest;
import software.amazon.awssdk.services.healthlake.model.ListFhirDatastoresResponse;
import software.amazon.awssdk.services.healthlake.model.ListFhirExportJobsRequest;
import software.amazon.awssdk.services.healthlake.model.ListFhirExportJobsResponse;
import software.amazon.awssdk.services.healthlake.model.ListFhirImportJobsRequest;
import software.amazon.awssdk.services.healthlake.model.ListFhirImportJobsResponse;
import software.amazon.awssdk.services.healthlake.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.healthlake.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.healthlake.model.ResourceNotFoundException;
import software.amazon.awssdk.services.healthlake.model.StartFhirExportJobRequest;
import software.amazon.awssdk.services.healthlake.model.StartFhirExportJobResponse;
import software.amazon.awssdk.services.healthlake.model.StartFhirImportJobRequest;
import software.amazon.awssdk.services.healthlake.model.StartFhirImportJobResponse;
import software.amazon.awssdk.services.healthlake.model.TagResourceRequest;
import software.amazon.awssdk.services.healthlake.model.TagResourceResponse;
import software.amazon.awssdk.services.healthlake.model.ThrottlingException;
import software.amazon.awssdk.services.healthlake.model.UntagResourceRequest;
import software.amazon.awssdk.services.healthlake.model.UntagResourceResponse;
import software.amazon.awssdk.services.healthlake.model.ValidationException;
import software.amazon.awssdk.services.healthlake.transform.CreateFhirDatastoreRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.DeleteFhirDatastoreRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.DescribeFhirDatastoreRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.DescribeFhirExportJobRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.DescribeFhirImportJobRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.ListFhirDatastoresRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.ListFhirExportJobsRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.ListFhirImportJobsRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.StartFhirExportJobRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.StartFhirImportJobRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.healthlake.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.healthlake.waiters.HealthLakeWaiter;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link HealthLakeClient}.
 *
 * @see HealthLakeClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultHealthLakeClient implements HealthLakeClient {
    private static final Logger log = Logger.loggerFor(DefaultHealthLakeClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultHealthLakeClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "HealthLake" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Create a FHIR-enabled data store.
     * </p>
     *
     * @param createFhirDatastoreRequest
     * @return Result of the CreateFHIRDatastore operation returned by the service.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ThrottlingException
     *         The user has exceeded their maximum number of allowed calls to the given API.
     * @throws AccessDeniedException
     *         Access is denied. Your account is not authorized to perform this operation.
     * @throws InternalServerException
     *         An unknown internal error occurred in the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.CreateFHIRDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/CreateFHIRDatastore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateFhirDatastoreResponse createFHIRDatastore(CreateFhirDatastoreRequest createFhirDatastoreRequest)
            throws ValidationException, ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateFhirDatastoreResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateFhirDatastoreResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createFhirDatastoreRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFhirDatastoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFHIRDatastore");

            return clientHandler.execute(new ClientExecutionParams<CreateFhirDatastoreRequest, CreateFhirDatastoreResponse>()
                    .withOperationName("CreateFHIRDatastore").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createFhirDatastoreRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateFhirDatastoreRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delete a FHIR-enabled data store.
     * </p>
     *
     * @param deleteFhirDatastoreRequest
     * @return Result of the DeleteFHIRDatastore operation returned by the service.
     * @throws AccessDeniedException
     *         Access is denied. Your account is not authorized to perform this operation.
     * @throws ConflictException
     *         The data store is in a transition state and the user requested action cannot be performed.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ResourceNotFoundException
     *         The requested data store was not found.
     * @throws ThrottlingException
     *         The user has exceeded their maximum number of allowed calls to the given API.
     * @throws InternalServerException
     *         An unknown internal error occurred in the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.DeleteFHIRDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/DeleteFHIRDatastore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteFhirDatastoreResponse deleteFHIRDatastore(DeleteFhirDatastoreRequest deleteFhirDatastoreRequest)
            throws AccessDeniedException, ConflictException, ValidationException, ResourceNotFoundException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteFhirDatastoreResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteFhirDatastoreResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteFhirDatastoreRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFhirDatastoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFHIRDatastore");

            return clientHandler.execute(new ClientExecutionParams<DeleteFhirDatastoreRequest, DeleteFhirDatastoreResponse>()
                    .withOperationName("DeleteFHIRDatastore").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteFhirDatastoreRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteFhirDatastoreRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get properties for a FHIR-enabled data store.
     * </p>
     *
     * @param describeFhirDatastoreRequest
     * @return Result of the DescribeFHIRDatastore operation returned by the service.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ResourceNotFoundException
     *         The requested data store was not found.
     * @throws ThrottlingException
     *         The user has exceeded their maximum number of allowed calls to the given API.
     * @throws InternalServerException
     *         An unknown internal error occurred in the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.DescribeFHIRDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/DescribeFHIRDatastore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeFhirDatastoreResponse describeFHIRDatastore(DescribeFhirDatastoreRequest describeFhirDatastoreRequest)
            throws ValidationException, ResourceNotFoundException, ThrottlingException, InternalServerException,
            AwsServiceException, SdkClientException, HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeFhirDatastoreResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeFhirDatastoreResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeFhirDatastoreRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFhirDatastoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFHIRDatastore");

            return clientHandler.execute(new ClientExecutionParams<DescribeFhirDatastoreRequest, DescribeFhirDatastoreResponse>()
                    .withOperationName("DescribeFHIRDatastore").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeFhirDatastoreRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeFhirDatastoreRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get FHIR export job properties.
     * </p>
     *
     * @param describeFhirExportJobRequest
     * @return Result of the DescribeFHIRExportJob operation returned by the service.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ResourceNotFoundException
     *         The requested data store was not found.
     * @throws ThrottlingException
     *         The user has exceeded their maximum number of allowed calls to the given API.
     * @throws InternalServerException
     *         An unknown internal error occurred in the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.DescribeFHIRExportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/DescribeFHIRExportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeFhirExportJobResponse describeFHIRExportJob(DescribeFhirExportJobRequest describeFhirExportJobRequest)
            throws ValidationException, ResourceNotFoundException, ThrottlingException, InternalServerException,
            AwsServiceException, SdkClientException, HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeFhirExportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeFhirExportJobResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeFhirExportJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFhirExportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFHIRExportJob");

            return clientHandler.execute(new ClientExecutionParams<DescribeFhirExportJobRequest, DescribeFhirExportJobResponse>()
                    .withOperationName("DescribeFHIRExportJob").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeFhirExportJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeFhirExportJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get the import job properties to learn more about the job or job progress.
     * </p>
     *
     * @param describeFhirImportJobRequest
     * @return Result of the DescribeFHIRImportJob operation returned by the service.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ResourceNotFoundException
     *         The requested data store was not found.
     * @throws ThrottlingException
     *         The user has exceeded their maximum number of allowed calls to the given API.
     * @throws InternalServerException
     *         An unknown internal error occurred in the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.DescribeFHIRImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/DescribeFHIRImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeFhirImportJobResponse describeFHIRImportJob(DescribeFhirImportJobRequest describeFhirImportJobRequest)
            throws ValidationException, ResourceNotFoundException, ThrottlingException, InternalServerException,
            AwsServiceException, SdkClientException, HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeFhirImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeFhirImportJobResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeFhirImportJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFhirImportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFHIRImportJob");

            return clientHandler.execute(new ClientExecutionParams<DescribeFhirImportJobRequest, DescribeFhirImportJobResponse>()
                    .withOperationName("DescribeFHIRImportJob").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeFhirImportJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeFhirImportJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List all FHIR-enabled data stores in a user’s account, regardless of data store status.
     * </p>
     *
     * @param listFhirDatastoresRequest
     * @return Result of the ListFHIRDatastores operation returned by the service.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ThrottlingException
     *         The user has exceeded their maximum number of allowed calls to the given API.
     * @throws InternalServerException
     *         An unknown internal error occurred in the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.ListFHIRDatastores
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/ListFHIRDatastores" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListFhirDatastoresResponse listFHIRDatastores(ListFhirDatastoresRequest listFhirDatastoresRequest)
            throws ValidationException, ThrottlingException, InternalServerException, AwsServiceException, SdkClientException,
            HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListFhirDatastoresResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListFhirDatastoresResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listFhirDatastoresRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFhirDatastoresRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFHIRDatastores");

            return clientHandler.execute(new ClientExecutionParams<ListFhirDatastoresRequest, ListFhirDatastoresResponse>()
                    .withOperationName("ListFHIRDatastores").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listFhirDatastoresRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListFhirDatastoresRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all FHIR export jobs associated with an account and their statuses.
     * </p>
     *
     * @param listFhirExportJobsRequest
     * @return Result of the ListFHIRExportJobs operation returned by the service.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ResourceNotFoundException
     *         The requested data store was not found.
     * @throws AccessDeniedException
     *         Access is denied. Your account is not authorized to perform this operation.
     * @throws ThrottlingException
     *         The user has exceeded their maximum number of allowed calls to the given API.
     * @throws InternalServerException
     *         An unknown internal error occurred in the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.ListFHIRExportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/ListFHIRExportJobs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListFhirExportJobsResponse listFHIRExportJobs(ListFhirExportJobsRequest listFhirExportJobsRequest)
            throws ValidationException, ResourceNotFoundException, AccessDeniedException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListFhirExportJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListFhirExportJobsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listFhirExportJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFhirExportJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFHIRExportJobs");

            return clientHandler.execute(new ClientExecutionParams<ListFhirExportJobsRequest, ListFhirExportJobsResponse>()
                    .withOperationName("ListFHIRExportJobs").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listFhirExportJobsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListFhirExportJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List all FHIR import jobs associated with an account and their statuses.
     * </p>
     *
     * @param listFhirImportJobsRequest
     * @return Result of the ListFHIRImportJobs operation returned by the service.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ResourceNotFoundException
     *         The requested data store was not found.
     * @throws AccessDeniedException
     *         Access is denied. Your account is not authorized to perform this operation.
     * @throws ThrottlingException
     *         The user has exceeded their maximum number of allowed calls to the given API.
     * @throws InternalServerException
     *         An unknown internal error occurred in the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.ListFHIRImportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/ListFHIRImportJobs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListFhirImportJobsResponse listFHIRImportJobs(ListFhirImportJobsRequest listFhirImportJobsRequest)
            throws ValidationException, ResourceNotFoundException, AccessDeniedException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListFhirImportJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListFhirImportJobsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listFhirImportJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFhirImportJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFHIRImportJobs");

            return clientHandler.execute(new ClientExecutionParams<ListFhirImportJobsRequest, ListFhirImportJobsResponse>()
                    .withOperationName("ListFHIRImportJobs").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listFhirImportJobsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListFhirImportJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of all existing tags associated with a data store.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ResourceNotFoundException
     *         The requested data store was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Start a FHIR export job.
     * </p>
     *
     * @param startFhirExportJobRequest
     * @return Result of the StartFHIRExportJob operation returned by the service.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ThrottlingException
     *         The user has exceeded their maximum number of allowed calls to the given API.
     * @throws AccessDeniedException
     *         Access is denied. Your account is not authorized to perform this operation.
     * @throws ResourceNotFoundException
     *         The requested data store was not found.
     * @throws InternalServerException
     *         An unknown internal error occurred in the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.StartFHIRExportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/StartFHIRExportJob" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StartFhirExportJobResponse startFHIRExportJob(StartFhirExportJobRequest startFhirExportJobRequest)
            throws ValidationException, ThrottlingException, AccessDeniedException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartFhirExportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartFhirExportJobResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startFhirExportJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startFhirExportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartFHIRExportJob");

            return clientHandler.execute(new ClientExecutionParams<StartFhirExportJobRequest, StartFhirExportJobResponse>()
                    .withOperationName("StartFHIRExportJob").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startFhirExportJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartFhirExportJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Start importing bulk FHIR data into an ACTIVE data store. The import job imports FHIR data found in the
     * <code>InputDataConfig</code> object and stores processing results in the <code>JobOutputDataConfig</code> object.
     * </p>
     *
     * @param startFhirImportJobRequest
     * @return Result of the StartFHIRImportJob operation returned by the service.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ThrottlingException
     *         The user has exceeded their maximum number of allowed calls to the given API.
     * @throws AccessDeniedException
     *         Access is denied. Your account is not authorized to perform this operation.
     * @throws ResourceNotFoundException
     *         The requested data store was not found.
     * @throws InternalServerException
     *         An unknown internal error occurred in the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.StartFHIRImportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/StartFHIRImportJob" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StartFhirImportJobResponse startFHIRImportJob(StartFhirImportJobRequest startFhirImportJobRequest)
            throws ValidationException, ThrottlingException, AccessDeniedException, ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartFhirImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartFhirImportJobResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startFhirImportJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startFhirImportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartFHIRImportJob");

            return clientHandler.execute(new ClientExecutionParams<StartFhirImportJobRequest, StartFhirImportJobResponse>()
                    .withOperationName("StartFHIRImportJob").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startFhirImportJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartFhirImportJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Add a user-specifed key and value tag to a data store.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ResourceNotFoundException
     *         The requested data store was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Remove a user-specifed key and value tag from a data store.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         The user input parameter was invalid.
     * @throws ResourceNotFoundException
     *         The requested data store was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HealthLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HealthLakeClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/healthlake-2017-07-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, HealthLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "HealthLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Create an instance of {@link HealthLakeWaiter} using this client.
     * <p>
     * Waiters created via this method are managed by the SDK and resources will be released when the service client is
     * closed.
     *
     * @return an instance of {@link HealthLakeWaiter}
     */
    @Override
    public HealthLakeWaiter waiter() {
        return HealthLakeWaiter.builder().client(this).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        HealthLakeServiceClientConfigurationBuilder serviceConfigBuilder = new HealthLakeServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(HealthLakeException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON).protocolVersion("1.0");
    }

    @Override
    public final HealthLakeServiceClientConfiguration serviceClientConfiguration() {
        return new HealthLakeServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
