/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.healthlake.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.healthlake.HealthLakeClient;
import software.amazon.awssdk.services.healthlake.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirDatastoreRequest;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirDatastoreResponse;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirExportJobRequest;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirExportJobResponse;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirImportJobRequest;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirImportJobResponse;
import software.amazon.awssdk.services.healthlake.model.HealthLakeRequest;
import software.amazon.awssdk.services.healthlake.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultHealthLakeWaiter implements HealthLakeWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final HealthLakeClient client;

    private final AttributeMap managedResources;

    private final Waiter<DescribeFhirDatastoreResponse> fHIRDatastoreActiveWaiter;

    private final Waiter<DescribeFhirDatastoreResponse> fHIRDatastoreDeletedWaiter;

    private final Waiter<DescribeFhirExportJobResponse> fHIRExportJobCompletedWaiter;

    private final Waiter<DescribeFhirImportJobResponse> fHIRImportJobCompletedWaiter;

    private DefaultHealthLakeWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = HealthLakeClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.fHIRDatastoreActiveWaiter = Waiter.builder(DescribeFhirDatastoreResponse.class)
                .acceptors(fHIRDatastoreActiveWaiterAcceptors())
                .overrideConfiguration(fHIRDatastoreActiveWaiterConfig(builder.overrideConfiguration)).build();
        this.fHIRDatastoreDeletedWaiter = Waiter.builder(DescribeFhirDatastoreResponse.class)
                .acceptors(fHIRDatastoreDeletedWaiterAcceptors())
                .overrideConfiguration(fHIRDatastoreDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.fHIRExportJobCompletedWaiter = Waiter.builder(DescribeFhirExportJobResponse.class)
                .acceptors(fHIRExportJobCompletedWaiterAcceptors())
                .overrideConfiguration(fHIRExportJobCompletedWaiterConfig(builder.overrideConfiguration)).build();
        this.fHIRImportJobCompletedWaiter = Waiter.builder(DescribeFhirImportJobResponse.class)
                .acceptors(fHIRImportJobCompletedWaiterAcceptors())
                .overrideConfiguration(fHIRImportJobCompletedWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeFhirDatastoreResponse> waitUntilFHIRDatastoreActive(
            DescribeFhirDatastoreRequest describeFhirDatastoreRequest) {
        return fHIRDatastoreActiveWaiter.run(() -> client
                .describeFHIRDatastore(applyWaitersUserAgent(describeFhirDatastoreRequest)));
    }

    @Override
    public WaiterResponse<DescribeFhirDatastoreResponse> waitUntilFHIRDatastoreActive(
            DescribeFhirDatastoreRequest describeFhirDatastoreRequest, WaiterOverrideConfiguration overrideConfig) {
        return fHIRDatastoreActiveWaiter.run(
                () -> client.describeFHIRDatastore(applyWaitersUserAgent(describeFhirDatastoreRequest)),
                fHIRDatastoreActiveWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeFhirDatastoreResponse> waitUntilFHIRDatastoreDeleted(
            DescribeFhirDatastoreRequest describeFhirDatastoreRequest) {
        return fHIRDatastoreDeletedWaiter.run(() -> client
                .describeFHIRDatastore(applyWaitersUserAgent(describeFhirDatastoreRequest)));
    }

    @Override
    public WaiterResponse<DescribeFhirDatastoreResponse> waitUntilFHIRDatastoreDeleted(
            DescribeFhirDatastoreRequest describeFhirDatastoreRequest, WaiterOverrideConfiguration overrideConfig) {
        return fHIRDatastoreDeletedWaiter.run(
                () -> client.describeFHIRDatastore(applyWaitersUserAgent(describeFhirDatastoreRequest)),
                fHIRDatastoreDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeFhirExportJobResponse> waitUntilFHIRExportJobCompleted(
            DescribeFhirExportJobRequest describeFhirExportJobRequest) {
        return fHIRExportJobCompletedWaiter.run(() -> client
                .describeFHIRExportJob(applyWaitersUserAgent(describeFhirExportJobRequest)));
    }

    @Override
    public WaiterResponse<DescribeFhirExportJobResponse> waitUntilFHIRExportJobCompleted(
            DescribeFhirExportJobRequest describeFhirExportJobRequest, WaiterOverrideConfiguration overrideConfig) {
        return fHIRExportJobCompletedWaiter.run(
                () -> client.describeFHIRExportJob(applyWaitersUserAgent(describeFhirExportJobRequest)),
                fHIRExportJobCompletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeFhirImportJobResponse> waitUntilFHIRImportJobCompleted(
            DescribeFhirImportJobRequest describeFhirImportJobRequest) {
        return fHIRImportJobCompletedWaiter.run(() -> client
                .describeFHIRImportJob(applyWaitersUserAgent(describeFhirImportJobRequest)));
    }

    @Override
    public WaiterResponse<DescribeFhirImportJobResponse> waitUntilFHIRImportJobCompleted(
            DescribeFhirImportJobRequest describeFhirImportJobRequest, WaiterOverrideConfiguration overrideConfig) {
        return fHIRImportJobCompletedWaiter.run(
                () -> client.describeFHIRImportJob(applyWaitersUserAgent(describeFhirImportJobRequest)),
                fHIRImportJobCompletedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeFhirDatastoreResponse>> fHIRDatastoreActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeFhirDatastoreResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("DatastoreProperties").field("DatastoreStatus").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("DatastoreProperties").field("DatastoreStatus").value(), "CREATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (DatastoreProperties.DatastoreStatus=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("DatastoreProperties").field("DatastoreStatus").value(), "DELETED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (DatastoreProperties.DatastoreStatus=DELETED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeFhirDatastoreResponse>> fHIRDatastoreDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeFhirDatastoreResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("DatastoreProperties").field("DatastoreStatus").value(), "DELETED");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeFhirExportJobResponse>> fHIRExportJobCompletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeFhirExportJobResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("ExportJobProperties").field("JobStatus").value(), "COMPLETED");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("ExportJobProperties").field("JobStatus").value(), "COMPLETED_WITH_ERRORS");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("ExportJobProperties").field("JobStatus").value(), "CANCEL_COMPLETED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (ExportJobProperties.JobStatus=CANCEL_COMPLETED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("ExportJobProperties").field("JobStatus").value(), "FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (ExportJobProperties.JobStatus=FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("ExportJobProperties").field("JobStatus").value(), "CANCEL_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (ExportJobProperties.JobStatus=CANCEL_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeFhirImportJobResponse>> fHIRImportJobCompletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeFhirImportJobResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("ImportJobProperties").field("JobStatus").value(), "COMPLETED");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("ImportJobProperties").field("JobStatus").value(), "COMPLETED_WITH_ERRORS");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("ImportJobProperties").field("JobStatus").value(), "FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (ImportJobProperties.JobStatus=FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration fHIRDatastoreActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(360);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration fHIRDatastoreDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(360);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(120)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration fHIRExportJobCompletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(360);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(120)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration fHIRImportJobCompletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(720);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(120)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static HealthLakeWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends HealthLakeRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements HealthLakeWaiter.Builder {
        private HealthLakeClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public HealthLakeWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public HealthLakeWaiter.Builder client(HealthLakeClient client) {
            this.client = client;
            return this;
        }

        public HealthLakeWaiter build() {
            return new DefaultHealthLakeWaiter(this);
        }
    }
}
