/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.healthlake.waiters;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Immutable;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.healthlake.HealthLakeClient;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirDatastoreRequest;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirDatastoreResponse;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirExportJobRequest;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirExportJobResponse;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirImportJobRequest;
import software.amazon.awssdk.services.healthlake.model.DescribeFhirImportJobResponse;
import software.amazon.awssdk.utils.SdkAutoCloseable;

/**
 * Waiter utility class that polls a resource until a desired state is reached or until it is determined that the
 * resource will never enter into the desired state. This can be created using the static {@link #builder()} method
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
@Immutable
public interface HealthLakeWaiter extends SdkAutoCloseable {
    /**
     * Polls {@link HealthLakeClient#describeFHIRDatastore} API until the desired condition {@code FHIRDatastoreActive}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param describeFhirDatastoreRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeFhirDatastoreResponse> waitUntilFHIRDatastoreActive(
            DescribeFhirDatastoreRequest describeFhirDatastoreRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link HealthLakeClient#describeFHIRDatastore} API until the desired condition {@code FHIRDatastoreActive}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeFhirDatastoreRequest#builder()}
     *
     * @param describeFhirDatastoreRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeFhirDatastoreResponse> waitUntilFHIRDatastoreActive(
            Consumer<DescribeFhirDatastoreRequest.Builder> describeFhirDatastoreRequest) {
        return waitUntilFHIRDatastoreActive(DescribeFhirDatastoreRequest.builder().applyMutation(describeFhirDatastoreRequest)
                .build());
    }

    /**
     * Polls {@link HealthLakeClient#describeFHIRDatastore} API until the desired condition {@code FHIRDatastoreActive}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param describeFhirDatastoreRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeFhirDatastoreResponse> waitUntilFHIRDatastoreActive(
            DescribeFhirDatastoreRequest describeFhirDatastoreRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link HealthLakeClient#describeFHIRDatastore} API until the desired condition {@code FHIRDatastoreActive}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeFhirDatastoreRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeFhirDatastoreResponse> waitUntilFHIRDatastoreActive(
            Consumer<DescribeFhirDatastoreRequest.Builder> describeFhirDatastoreRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilFHIRDatastoreActive(DescribeFhirDatastoreRequest.builder().applyMutation(describeFhirDatastoreRequest)
                .build(), WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link HealthLakeClient#describeFHIRDatastore} API until the desired condition {@code FHIRDatastoreDeleted}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param describeFhirDatastoreRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeFhirDatastoreResponse> waitUntilFHIRDatastoreDeleted(
            DescribeFhirDatastoreRequest describeFhirDatastoreRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link HealthLakeClient#describeFHIRDatastore} API until the desired condition {@code FHIRDatastoreDeleted}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeFhirDatastoreRequest#builder()}
     *
     * @param describeFhirDatastoreRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeFhirDatastoreResponse> waitUntilFHIRDatastoreDeleted(
            Consumer<DescribeFhirDatastoreRequest.Builder> describeFhirDatastoreRequest) {
        return waitUntilFHIRDatastoreDeleted(DescribeFhirDatastoreRequest.builder().applyMutation(describeFhirDatastoreRequest)
                .build());
    }

    /**
     * Polls {@link HealthLakeClient#describeFHIRDatastore} API until the desired condition {@code FHIRDatastoreDeleted}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param describeFhirDatastoreRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeFhirDatastoreResponse> waitUntilFHIRDatastoreDeleted(
            DescribeFhirDatastoreRequest describeFhirDatastoreRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link HealthLakeClient#describeFHIRDatastore} API until the desired condition {@code FHIRDatastoreDeleted}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeFhirDatastoreRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeFhirDatastoreResponse> waitUntilFHIRDatastoreDeleted(
            Consumer<DescribeFhirDatastoreRequest.Builder> describeFhirDatastoreRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilFHIRDatastoreDeleted(DescribeFhirDatastoreRequest.builder().applyMutation(describeFhirDatastoreRequest)
                .build(), WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link HealthLakeClient#describeFHIRExportJob} API until the desired condition
     * {@code FHIRExportJobCompleted} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeFhirExportJobRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeFhirExportJobResponse> waitUntilFHIRExportJobCompleted(
            DescribeFhirExportJobRequest describeFhirExportJobRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link HealthLakeClient#describeFHIRExportJob} API until the desired condition
     * {@code FHIRExportJobCompleted} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeFhirExportJobRequest#builder()}
     *
     * @param describeFhirExportJobRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeFhirExportJobResponse> waitUntilFHIRExportJobCompleted(
            Consumer<DescribeFhirExportJobRequest.Builder> describeFhirExportJobRequest) {
        return waitUntilFHIRExportJobCompleted(DescribeFhirExportJobRequest.builder().applyMutation(describeFhirExportJobRequest)
                .build());
    }

    /**
     * Polls {@link HealthLakeClient#describeFHIRExportJob} API until the desired condition
     * {@code FHIRExportJobCompleted} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeFhirExportJobRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeFhirExportJobResponse> waitUntilFHIRExportJobCompleted(
            DescribeFhirExportJobRequest describeFhirExportJobRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link HealthLakeClient#describeFHIRExportJob} API until the desired condition
     * {@code FHIRExportJobCompleted} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeFhirExportJobRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeFhirExportJobResponse> waitUntilFHIRExportJobCompleted(
            Consumer<DescribeFhirExportJobRequest.Builder> describeFhirExportJobRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilFHIRExportJobCompleted(DescribeFhirExportJobRequest.builder().applyMutation(describeFhirExportJobRequest)
                .build(), WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link HealthLakeClient#describeFHIRImportJob} API until the desired condition
     * {@code FHIRImportJobCompleted} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeFhirImportJobRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeFhirImportJobResponse> waitUntilFHIRImportJobCompleted(
            DescribeFhirImportJobRequest describeFhirImportJobRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link HealthLakeClient#describeFHIRImportJob} API until the desired condition
     * {@code FHIRImportJobCompleted} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeFhirImportJobRequest#builder()}
     *
     * @param describeFhirImportJobRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeFhirImportJobResponse> waitUntilFHIRImportJobCompleted(
            Consumer<DescribeFhirImportJobRequest.Builder> describeFhirImportJobRequest) {
        return waitUntilFHIRImportJobCompleted(DescribeFhirImportJobRequest.builder().applyMutation(describeFhirImportJobRequest)
                .build());
    }

    /**
     * Polls {@link HealthLakeClient#describeFHIRImportJob} API until the desired condition
     * {@code FHIRImportJobCompleted} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeFhirImportJobRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeFhirImportJobResponse> waitUntilFHIRImportJobCompleted(
            DescribeFhirImportJobRequest describeFhirImportJobRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link HealthLakeClient#describeFHIRImportJob} API until the desired condition
     * {@code FHIRImportJobCompleted} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeFhirImportJobRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeFhirImportJobResponse> waitUntilFHIRImportJobCompleted(
            Consumer<DescribeFhirImportJobRequest.Builder> describeFhirImportJobRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilFHIRImportJobCompleted(DescribeFhirImportJobRequest.builder().applyMutation(describeFhirImportJobRequest)
                .build(), WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Create a builder that can be used to configure and create a {@link HealthLakeWaiter}.
     *
     * @return a builder
     */
    static Builder builder() {
        return DefaultHealthLakeWaiter.builder();
    }

    /**
     * Create an instance of {@link HealthLakeWaiter} with the default configuration.
     * <p>
     * <b>A default {@link HealthLakeClient} will be created to poll resources. It is recommended to share a single
     * instance of the waiter created via this method. If it is not desirable to share a waiter instance, invoke
     * {@link #close()} to release the resources once the waiter is not needed.</b>
     *
     * @return an instance of {@link HealthLakeWaiter}
     */
    static HealthLakeWaiter create() {
        return DefaultHealthLakeWaiter.builder().build();
    }

    interface Builder {
        /**
         * Defines overrides to the default SDK waiter configuration that should be used for waiters created from this
         * builder
         *
         * @param overrideConfiguration
         *        the override configuration to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration);

        /**
         * This is a convenient method to pass the override configuration without the need to create an instance
         * manually via {@link WaiterOverrideConfiguration#builder()}
         *
         * @param overrideConfiguration
         *        The consumer that will configure the overrideConfiguration
         * @return a reference to this object so that method calls can be chained together.
         * @see #overrideConfiguration(WaiterOverrideConfiguration)
         */
        default Builder overrideConfiguration(Consumer<WaiterOverrideConfiguration.Builder> overrideConfiguration) {
            WaiterOverrideConfiguration.Builder builder = WaiterOverrideConfiguration.builder();
            overrideConfiguration.accept(builder);
            return overrideConfiguration(builder.build());
        }

        /**
         * Sets a custom {@link HealthLakeClient} that will be used to poll the resource
         * <p>
         * This SDK client must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * client when the waiter is closed
         *
         * @param client
         *        the client to send the request
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder client(HealthLakeClient client);

        /**
         * Builds an instance of {@link HealthLakeWaiter} based on the configurations supplied to this builder
         *
         * @return An initialized {@link HealthLakeWaiter}
         */
        HealthLakeWaiter build();
    }
}
