/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.honeycode;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.honeycode.model.AccessDeniedException;
import software.amazon.awssdk.services.honeycode.model.AutomationExecutionException;
import software.amazon.awssdk.services.honeycode.model.AutomationExecutionTimeoutException;
import software.amazon.awssdk.services.honeycode.model.GetScreenDataRequest;
import software.amazon.awssdk.services.honeycode.model.GetScreenDataResponse;
import software.amazon.awssdk.services.honeycode.model.HoneycodeException;
import software.amazon.awssdk.services.honeycode.model.InternalServerException;
import software.amazon.awssdk.services.honeycode.model.InvokeScreenAutomationRequest;
import software.amazon.awssdk.services.honeycode.model.InvokeScreenAutomationResponse;
import software.amazon.awssdk.services.honeycode.model.RequestTimeoutException;
import software.amazon.awssdk.services.honeycode.model.ResourceNotFoundException;
import software.amazon.awssdk.services.honeycode.model.ServiceUnavailableException;
import software.amazon.awssdk.services.honeycode.model.ThrottlingException;
import software.amazon.awssdk.services.honeycode.model.ValidationException;
import software.amazon.awssdk.services.honeycode.transform.GetScreenDataRequestMarshaller;
import software.amazon.awssdk.services.honeycode.transform.InvokeScreenAutomationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link HoneycodeAsyncClient}.
 *
 * @see HoneycodeAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultHoneycodeAsyncClient implements HoneycodeAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultHoneycodeAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultHoneycodeAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * The GetScreenData API allows retrieval of data from a screen in a Honeycode app. The API allows setting local
     * variables in the screen to filter, sort or otherwise affect what will be displayed on the screen.
     * </p>
     *
     * @param getScreenDataRequest
     * @return A Java Future containing the result of the GetScreenData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Check that the
     *         workbook is owned by you and your IAM policy allows access to the screen/automation in the request.</li>
     *         <li>InternalServerException There were unexpected errors from the server.</li>
     *         <li>RequestTimeoutException The request timed out.</li>
     *         <li>ResourceNotFoundException A Workbook, App, Screen or Screen Automation was not found with the given
     *         ID.</li>
     *         <li>ServiceUnavailableException Remote service is unreachable.</li>
     *         <li>ThrottlingException Tps(transactions per second) rate reached.</li>
     *         <li>ValidationException Request is invalid. The message in the response contains details on why the
     *         request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HoneycodeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HoneycodeAsyncClient.GetScreenData
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/honeycode-2020-03-01/GetScreenData" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetScreenDataResponse> getScreenData(GetScreenDataRequest getScreenDataRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getScreenDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Honeycode");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetScreenData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetScreenDataResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetScreenDataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetScreenDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetScreenDataRequest, GetScreenDataResponse>()
                            .withOperationName("GetScreenData")
                            .withMarshaller(new GetScreenDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getScreenDataRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getScreenDataRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetScreenDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The InvokeScreenAutomation API allows invoking an action defined in a screen in a Honeycode app. The API allows
     * setting local variables, which can then be used in the automation being invoked. This allows automating the
     * Honeycode app interactions to write, update or delete data in the workbook.
     * </p>
     *
     * @param invokeScreenAutomationRequest
     * @return A Java Future containing the result of the InvokeScreenAutomation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Check that the
     *         workbook is owned by you and your IAM policy allows access to the screen/automation in the request.</li>
     *         <li>InternalServerException There were unexpected errors from the server.</li>
     *         <li>ResourceNotFoundException A Workbook, App, Screen or Screen Automation was not found with the given
     *         ID.</li>
     *         <li>ValidationException Request is invalid. The message in the response contains details on why the
     *         request is invalid.</li>
     *         <li>ThrottlingException Tps(transactions per second) rate reached.</li>
     *         <li>ServiceUnavailableException Remote service is unreachable.</li>
     *         <li>AutomationExecutionException The automation execution did not end successfully.</li>
     *         <li>AutomationExecutionTimeoutException The automation execution timed out.</li>
     *         <li>RequestTimeoutException The request timed out.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HoneycodeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HoneycodeAsyncClient.InvokeScreenAutomation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/honeycode-2020-03-01/InvokeScreenAutomation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<InvokeScreenAutomationResponse> invokeScreenAutomation(
            InvokeScreenAutomationRequest invokeScreenAutomationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, invokeScreenAutomationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Honeycode");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InvokeScreenAutomation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<InvokeScreenAutomationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, InvokeScreenAutomationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<InvokeScreenAutomationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<InvokeScreenAutomationRequest, InvokeScreenAutomationResponse>()
                            .withOperationName("InvokeScreenAutomation")
                            .withMarshaller(new InvokeScreenAutomationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(invokeScreenAutomationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = invokeScreenAutomationRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<InvokeScreenAutomationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(HoneycodeException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AutomationExecutionException")
                                .exceptionBuilderSupplier(AutomationExecutionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RequestTimeoutException")
                                .exceptionBuilderSupplier(RequestTimeoutException::builder).httpStatusCode(504).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AutomationExecutionTimeoutException")
                                .exceptionBuilderSupplier(AutomationExecutionTimeoutException::builder).httpStatusCode(504)
                                .build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
