/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.honeycode;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.honeycode.model.AccessDeniedException;
import software.amazon.awssdk.services.honeycode.model.AutomationExecutionException;
import software.amazon.awssdk.services.honeycode.model.AutomationExecutionTimeoutException;
import software.amazon.awssdk.services.honeycode.model.BatchCreateTableRowsRequest;
import software.amazon.awssdk.services.honeycode.model.BatchCreateTableRowsResponse;
import software.amazon.awssdk.services.honeycode.model.BatchDeleteTableRowsRequest;
import software.amazon.awssdk.services.honeycode.model.BatchDeleteTableRowsResponse;
import software.amazon.awssdk.services.honeycode.model.BatchUpdateTableRowsRequest;
import software.amazon.awssdk.services.honeycode.model.BatchUpdateTableRowsResponse;
import software.amazon.awssdk.services.honeycode.model.BatchUpsertTableRowsRequest;
import software.amazon.awssdk.services.honeycode.model.BatchUpsertTableRowsResponse;
import software.amazon.awssdk.services.honeycode.model.DescribeTableDataImportJobRequest;
import software.amazon.awssdk.services.honeycode.model.DescribeTableDataImportJobResponse;
import software.amazon.awssdk.services.honeycode.model.GetScreenDataRequest;
import software.amazon.awssdk.services.honeycode.model.GetScreenDataResponse;
import software.amazon.awssdk.services.honeycode.model.HoneycodeException;
import software.amazon.awssdk.services.honeycode.model.HoneycodeRequest;
import software.amazon.awssdk.services.honeycode.model.InternalServerException;
import software.amazon.awssdk.services.honeycode.model.InvokeScreenAutomationRequest;
import software.amazon.awssdk.services.honeycode.model.InvokeScreenAutomationResponse;
import software.amazon.awssdk.services.honeycode.model.ListTableColumnsRequest;
import software.amazon.awssdk.services.honeycode.model.ListTableColumnsResponse;
import software.amazon.awssdk.services.honeycode.model.ListTableRowsRequest;
import software.amazon.awssdk.services.honeycode.model.ListTableRowsResponse;
import software.amazon.awssdk.services.honeycode.model.ListTablesRequest;
import software.amazon.awssdk.services.honeycode.model.ListTablesResponse;
import software.amazon.awssdk.services.honeycode.model.QueryTableRowsRequest;
import software.amazon.awssdk.services.honeycode.model.QueryTableRowsResponse;
import software.amazon.awssdk.services.honeycode.model.RequestTimeoutException;
import software.amazon.awssdk.services.honeycode.model.ResourceNotFoundException;
import software.amazon.awssdk.services.honeycode.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.honeycode.model.ServiceUnavailableException;
import software.amazon.awssdk.services.honeycode.model.StartTableDataImportJobRequest;
import software.amazon.awssdk.services.honeycode.model.StartTableDataImportJobResponse;
import software.amazon.awssdk.services.honeycode.model.ThrottlingException;
import software.amazon.awssdk.services.honeycode.model.ValidationException;
import software.amazon.awssdk.services.honeycode.paginators.ListTableColumnsIterable;
import software.amazon.awssdk.services.honeycode.paginators.ListTableRowsIterable;
import software.amazon.awssdk.services.honeycode.paginators.ListTablesIterable;
import software.amazon.awssdk.services.honeycode.paginators.QueryTableRowsIterable;
import software.amazon.awssdk.services.honeycode.transform.BatchCreateTableRowsRequestMarshaller;
import software.amazon.awssdk.services.honeycode.transform.BatchDeleteTableRowsRequestMarshaller;
import software.amazon.awssdk.services.honeycode.transform.BatchUpdateTableRowsRequestMarshaller;
import software.amazon.awssdk.services.honeycode.transform.BatchUpsertTableRowsRequestMarshaller;
import software.amazon.awssdk.services.honeycode.transform.DescribeTableDataImportJobRequestMarshaller;
import software.amazon.awssdk.services.honeycode.transform.GetScreenDataRequestMarshaller;
import software.amazon.awssdk.services.honeycode.transform.InvokeScreenAutomationRequestMarshaller;
import software.amazon.awssdk.services.honeycode.transform.ListTableColumnsRequestMarshaller;
import software.amazon.awssdk.services.honeycode.transform.ListTableRowsRequestMarshaller;
import software.amazon.awssdk.services.honeycode.transform.ListTablesRequestMarshaller;
import software.amazon.awssdk.services.honeycode.transform.QueryTableRowsRequestMarshaller;
import software.amazon.awssdk.services.honeycode.transform.StartTableDataImportJobRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link HoneycodeClient}.
 *
 * @see HoneycodeClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultHoneycodeClient implements HoneycodeClient {
    private static final Logger log = Logger.loggerFor(DefaultHoneycodeClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultHoneycodeClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * The BatchCreateTableRows API allows you to create one or more rows at the end of a table in a workbook. The API
     * allows you to specify the values to set in some or all of the columns in the new rows.
     * </p>
     * <p>
     * If a column is not explicitly set in a specific row, then the column level formula specified in the table will be
     * applied to the new row. If there is no column level formula but the last row of the table has a formula, then
     * that formula will be copied down to the new row. If there is no column level formula and no formula in the last
     * row of the table, then that column will be left blank for the new rows.
     * </p>
     *
     * @param batchCreateTableRowsRequest
     * @return Result of the BatchCreateTableRows operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check that the workbook is owned by you and
     *         your IAM policy allows access to the resource in the request.
     * @throws InternalServerException
     *         There were unexpected errors from the server.
     * @throws RequestTimeoutException
     *         The request timed out.
     * @throws ResourceNotFoundException
     *         A Workbook, Table, App, Screen or Screen Automation was not found with the given ID.
     * @throws ServiceQuotaExceededException
     *         The request caused service quota to be breached.
     * @throws ServiceUnavailableException
     *         Remote service is unreachable.
     * @throws ThrottlingException
     *         Tps(transactions per second) rate reached.
     * @throws ValidationException
     *         Request is invalid. The message in the response contains details on why the request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HoneycodeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HoneycodeClient.BatchCreateTableRows
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/honeycode-2020-03-01/BatchCreateTableRows" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public BatchCreateTableRowsResponse batchCreateTableRows(BatchCreateTableRowsRequest batchCreateTableRowsRequest)
            throws AccessDeniedException, InternalServerException, RequestTimeoutException, ResourceNotFoundException,
            ServiceQuotaExceededException, ServiceUnavailableException, ThrottlingException, ValidationException,
            AwsServiceException, SdkClientException, HoneycodeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchCreateTableRowsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchCreateTableRowsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchCreateTableRowsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Honeycode");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchCreateTableRows");

            return clientHandler.execute(new ClientExecutionParams<BatchCreateTableRowsRequest, BatchCreateTableRowsResponse>()
                    .withOperationName("BatchCreateTableRows").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(batchCreateTableRowsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchCreateTableRowsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The BatchDeleteTableRows API allows you to delete one or more rows from a table in a workbook. You need to
     * specify the ids of the rows that you want to delete from the table.
     * </p>
     *
     * @param batchDeleteTableRowsRequest
     * @return Result of the BatchDeleteTableRows operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check that the workbook is owned by you and
     *         your IAM policy allows access to the resource in the request.
     * @throws InternalServerException
     *         There were unexpected errors from the server.
     * @throws ResourceNotFoundException
     *         A Workbook, Table, App, Screen or Screen Automation was not found with the given ID.
     * @throws ServiceUnavailableException
     *         Remote service is unreachable.
     * @throws ValidationException
     *         Request is invalid. The message in the response contains details on why the request is invalid.
     * @throws RequestTimeoutException
     *         The request timed out.
     * @throws ThrottlingException
     *         Tps(transactions per second) rate reached.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HoneycodeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HoneycodeClient.BatchDeleteTableRows
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/honeycode-2020-03-01/BatchDeleteTableRows" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public BatchDeleteTableRowsResponse batchDeleteTableRows(BatchDeleteTableRowsRequest batchDeleteTableRowsRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ServiceUnavailableException,
            ValidationException, RequestTimeoutException, ThrottlingException, AwsServiceException, SdkClientException,
            HoneycodeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchDeleteTableRowsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchDeleteTableRowsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchDeleteTableRowsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Honeycode");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDeleteTableRows");

            return clientHandler.execute(new ClientExecutionParams<BatchDeleteTableRowsRequest, BatchDeleteTableRowsResponse>()
                    .withOperationName("BatchDeleteTableRows").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(batchDeleteTableRowsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchDeleteTableRowsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The BatchUpdateTableRows API allows you to update one or more rows in a table in a workbook.
     * </p>
     * <p>
     * You can specify the values to set in some or all of the columns in the table for the specified rows. If a column
     * is not explicitly specified in a particular row, then that column will not be updated for that row. To clear out
     * the data in a specific cell, you need to set the value as an empty string ("").
     * </p>
     *
     * @param batchUpdateTableRowsRequest
     * @return Result of the BatchUpdateTableRows operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check that the workbook is owned by you and
     *         your IAM policy allows access to the resource in the request.
     * @throws InternalServerException
     *         There were unexpected errors from the server.
     * @throws ResourceNotFoundException
     *         A Workbook, Table, App, Screen or Screen Automation was not found with the given ID.
     * @throws ServiceUnavailableException
     *         Remote service is unreachable.
     * @throws ValidationException
     *         Request is invalid. The message in the response contains details on why the request is invalid.
     * @throws RequestTimeoutException
     *         The request timed out.
     * @throws ThrottlingException
     *         Tps(transactions per second) rate reached.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HoneycodeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HoneycodeClient.BatchUpdateTableRows
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/honeycode-2020-03-01/BatchUpdateTableRows" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public BatchUpdateTableRowsResponse batchUpdateTableRows(BatchUpdateTableRowsRequest batchUpdateTableRowsRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ServiceUnavailableException,
            ValidationException, RequestTimeoutException, ThrottlingException, AwsServiceException, SdkClientException,
            HoneycodeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchUpdateTableRowsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchUpdateTableRowsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchUpdateTableRowsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Honeycode");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchUpdateTableRows");

            return clientHandler.execute(new ClientExecutionParams<BatchUpdateTableRowsRequest, BatchUpdateTableRowsResponse>()
                    .withOperationName("BatchUpdateTableRows").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(batchUpdateTableRowsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchUpdateTableRowsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The BatchUpsertTableRows API allows you to upsert one or more rows in a table. The upsert operation takes a
     * filter expression as input and evaluates it to find matching rows on the destination table. If matching rows are
     * found, it will update the cells in the matching rows to new values specified in the request. If no matching rows
     * are found, a new row is added at the end of the table and the cells in that row are set to the new values
     * specified in the request.
     * </p>
     * <p>
     * You can specify the values to set in some or all of the columns in the table for the matching or newly appended
     * rows. If a column is not explicitly specified for a particular row, then that column will not be updated for that
     * row. To clear out the data in a specific cell, you need to set the value as an empty string ("").
     * </p>
     *
     * @param batchUpsertTableRowsRequest
     * @return Result of the BatchUpsertTableRows operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check that the workbook is owned by you and
     *         your IAM policy allows access to the resource in the request.
     * @throws InternalServerException
     *         There were unexpected errors from the server.
     * @throws RequestTimeoutException
     *         The request timed out.
     * @throws ResourceNotFoundException
     *         A Workbook, Table, App, Screen or Screen Automation was not found with the given ID.
     * @throws ServiceQuotaExceededException
     *         The request caused service quota to be breached.
     * @throws ServiceUnavailableException
     *         Remote service is unreachable.
     * @throws ThrottlingException
     *         Tps(transactions per second) rate reached.
     * @throws ValidationException
     *         Request is invalid. The message in the response contains details on why the request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HoneycodeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HoneycodeClient.BatchUpsertTableRows
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/honeycode-2020-03-01/BatchUpsertTableRows" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public BatchUpsertTableRowsResponse batchUpsertTableRows(BatchUpsertTableRowsRequest batchUpsertTableRowsRequest)
            throws AccessDeniedException, InternalServerException, RequestTimeoutException, ResourceNotFoundException,
            ServiceQuotaExceededException, ServiceUnavailableException, ThrottlingException, ValidationException,
            AwsServiceException, SdkClientException, HoneycodeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchUpsertTableRowsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchUpsertTableRowsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchUpsertTableRowsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Honeycode");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchUpsertTableRows");

            return clientHandler.execute(new ClientExecutionParams<BatchUpsertTableRowsRequest, BatchUpsertTableRowsResponse>()
                    .withOperationName("BatchUpsertTableRows").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(batchUpsertTableRowsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchUpsertTableRowsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The DescribeTableDataImportJob API allows you to retrieve the status and details of a table data import job.
     * </p>
     *
     * @param describeTableDataImportJobRequest
     * @return Result of the DescribeTableDataImportJob operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check that the workbook is owned by you and
     *         your IAM policy allows access to the resource in the request.
     * @throws InternalServerException
     *         There were unexpected errors from the server.
     * @throws ResourceNotFoundException
     *         A Workbook, Table, App, Screen or Screen Automation was not found with the given ID.
     * @throws ServiceUnavailableException
     *         Remote service is unreachable.
     * @throws ThrottlingException
     *         Tps(transactions per second) rate reached.
     * @throws ValidationException
     *         Request is invalid. The message in the response contains details on why the request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HoneycodeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HoneycodeClient.DescribeTableDataImportJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/honeycode-2020-03-01/DescribeTableDataImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeTableDataImportJobResponse describeTableDataImportJob(
            DescribeTableDataImportJobRequest describeTableDataImportJobRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ServiceUnavailableException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, HoneycodeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTableDataImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeTableDataImportJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTableDataImportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Honeycode");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTableDataImportJob");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeTableDataImportJobRequest, DescribeTableDataImportJobResponse>()
                            .withOperationName("DescribeTableDataImportJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeTableDataImportJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeTableDataImportJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The GetScreenData API allows retrieval of data from a screen in a Honeycode app. The API allows setting local
     * variables in the screen to filter, sort or otherwise affect what will be displayed on the screen.
     * </p>
     *
     * @param getScreenDataRequest
     * @return Result of the GetScreenData operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check that the workbook is owned by you and
     *         your IAM policy allows access to the resource in the request.
     * @throws InternalServerException
     *         There were unexpected errors from the server.
     * @throws RequestTimeoutException
     *         The request timed out.
     * @throws ResourceNotFoundException
     *         A Workbook, Table, App, Screen or Screen Automation was not found with the given ID.
     * @throws ServiceUnavailableException
     *         Remote service is unreachable.
     * @throws ThrottlingException
     *         Tps(transactions per second) rate reached.
     * @throws ValidationException
     *         Request is invalid. The message in the response contains details on why the request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HoneycodeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HoneycodeClient.GetScreenData
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/honeycode-2020-03-01/GetScreenData" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetScreenDataResponse getScreenData(GetScreenDataRequest getScreenDataRequest) throws AccessDeniedException,
            InternalServerException, RequestTimeoutException, ResourceNotFoundException, ServiceUnavailableException,
            ThrottlingException, ValidationException, AwsServiceException, SdkClientException, HoneycodeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetScreenDataResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetScreenDataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getScreenDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Honeycode");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetScreenData");

            return clientHandler.execute(new ClientExecutionParams<GetScreenDataRequest, GetScreenDataResponse>()
                    .withOperationName("GetScreenData").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getScreenDataRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetScreenDataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The InvokeScreenAutomation API allows invoking an action defined in a screen in a Honeycode app. The API allows
     * setting local variables, which can then be used in the automation being invoked. This allows automating the
     * Honeycode app interactions to write, update or delete data in the workbook.
     * </p>
     *
     * @param invokeScreenAutomationRequest
     * @return Result of the InvokeScreenAutomation operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check that the workbook is owned by you and
     *         your IAM policy allows access to the resource in the request.
     * @throws InternalServerException
     *         There were unexpected errors from the server.
     * @throws ResourceNotFoundException
     *         A Workbook, Table, App, Screen or Screen Automation was not found with the given ID.
     * @throws ValidationException
     *         Request is invalid. The message in the response contains details on why the request is invalid.
     * @throws ThrottlingException
     *         Tps(transactions per second) rate reached.
     * @throws ServiceUnavailableException
     *         Remote service is unreachable.
     * @throws AutomationExecutionException
     *         The automation execution did not end successfully.
     * @throws AutomationExecutionTimeoutException
     *         The automation execution timed out.
     * @throws RequestTimeoutException
     *         The request timed out.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HoneycodeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HoneycodeClient.InvokeScreenAutomation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/honeycode-2020-03-01/InvokeScreenAutomation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public InvokeScreenAutomationResponse invokeScreenAutomation(InvokeScreenAutomationRequest invokeScreenAutomationRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ValidationException,
            ThrottlingException, ServiceUnavailableException, AutomationExecutionException, AutomationExecutionTimeoutException,
            RequestTimeoutException, AwsServiceException, SdkClientException, HoneycodeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<InvokeScreenAutomationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, InvokeScreenAutomationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, invokeScreenAutomationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Honeycode");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InvokeScreenAutomation");

            return clientHandler
                    .execute(new ClientExecutionParams<InvokeScreenAutomationRequest, InvokeScreenAutomationResponse>()
                            .withOperationName("InvokeScreenAutomation").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(invokeScreenAutomationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new InvokeScreenAutomationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The ListTableColumns API allows you to retrieve a list of all the columns in a table in a workbook.
     * </p>
     *
     * @param listTableColumnsRequest
     * @return Result of the ListTableColumns operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check that the workbook is owned by you and
     *         your IAM policy allows access to the resource in the request.
     * @throws InternalServerException
     *         There were unexpected errors from the server.
     * @throws RequestTimeoutException
     *         The request timed out.
     * @throws ResourceNotFoundException
     *         A Workbook, Table, App, Screen or Screen Automation was not found with the given ID.
     * @throws ServiceUnavailableException
     *         Remote service is unreachable.
     * @throws ThrottlingException
     *         Tps(transactions per second) rate reached.
     * @throws ValidationException
     *         Request is invalid. The message in the response contains details on why the request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HoneycodeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HoneycodeClient.ListTableColumns
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/honeycode-2020-03-01/ListTableColumns" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTableColumnsResponse listTableColumns(ListTableColumnsRequest listTableColumnsRequest)
            throws AccessDeniedException, InternalServerException, RequestTimeoutException, ResourceNotFoundException,
            ServiceUnavailableException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            HoneycodeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTableColumnsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTableColumnsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTableColumnsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Honeycode");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTableColumns");

            return clientHandler.execute(new ClientExecutionParams<ListTableColumnsRequest, ListTableColumnsResponse>()
                    .withOperationName("ListTableColumns").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTableColumnsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTableColumnsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The ListTableColumns API allows you to retrieve a list of all the columns in a table in a workbook.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTableColumns(software.amazon.awssdk.services.honeycode.model.ListTableColumnsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.honeycode.paginators.ListTableColumnsIterable responses = client.listTableColumnsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.honeycode.paginators.ListTableColumnsIterable responses = client
     *             .listTableColumnsPaginator(request);
     *     for (software.amazon.awssdk.services.honeycode.model.ListTableColumnsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.honeycode.paginators.ListTableColumnsIterable responses = client.listTableColumnsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTableColumns(software.amazon.awssdk.services.honeycode.model.ListTableColumnsRequest)} operation.</b>
     * </p>
     *
     * @param listTableColumnsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check that the workbook is owned by you and
     *         your IAM policy allows access to the resource in the request.
     * @throws InternalServerException
     *         There were unexpected errors from the server.
     * @throws RequestTimeoutException
     *         The request timed out.
     * @throws ResourceNotFoundException
     *         A Workbook, Table, App, Screen or Screen Automation was not found with the given ID.
     * @throws ServiceUnavailableException
     *         Remote service is unreachable.
     * @throws ThrottlingException
     *         Tps(transactions per second) rate reached.
     * @throws ValidationException
     *         Request is invalid. The message in the response contains details on why the request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HoneycodeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HoneycodeClient.ListTableColumns
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/honeycode-2020-03-01/ListTableColumns" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTableColumnsIterable listTableColumnsPaginator(ListTableColumnsRequest listTableColumnsRequest)
            throws AccessDeniedException, InternalServerException, RequestTimeoutException, ResourceNotFoundException,
            ServiceUnavailableException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            HoneycodeException {
        return new ListTableColumnsIterable(this, applyPaginatorUserAgent(listTableColumnsRequest));
    }

    /**
     * <p>
     * The ListTableRows API allows you to retrieve a list of all the rows in a table in a workbook.
     * </p>
     *
     * @param listTableRowsRequest
     * @return Result of the ListTableRows operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check that the workbook is owned by you and
     *         your IAM policy allows access to the resource in the request.
     * @throws InternalServerException
     *         There were unexpected errors from the server.
     * @throws ResourceNotFoundException
     *         A Workbook, Table, App, Screen or Screen Automation was not found with the given ID.
     * @throws ServiceUnavailableException
     *         Remote service is unreachable.
     * @throws ValidationException
     *         Request is invalid. The message in the response contains details on why the request is invalid.
     * @throws RequestTimeoutException
     *         The request timed out.
     * @throws ThrottlingException
     *         Tps(transactions per second) rate reached.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HoneycodeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HoneycodeClient.ListTableRows
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/honeycode-2020-03-01/ListTableRows" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTableRowsResponse listTableRows(ListTableRowsRequest listTableRowsRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ServiceUnavailableException, ValidationException,
            RequestTimeoutException, ThrottlingException, AwsServiceException, SdkClientException, HoneycodeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTableRowsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTableRowsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTableRowsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Honeycode");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTableRows");

            return clientHandler.execute(new ClientExecutionParams<ListTableRowsRequest, ListTableRowsResponse>()
                    .withOperationName("ListTableRows").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTableRowsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTableRowsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The ListTableRows API allows you to retrieve a list of all the rows in a table in a workbook.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listTableRows(software.amazon.awssdk.services.honeycode.model.ListTableRowsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.honeycode.paginators.ListTableRowsIterable responses = client.listTableRowsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.honeycode.paginators.ListTableRowsIterable responses = client.listTableRowsPaginator(request);
     *     for (software.amazon.awssdk.services.honeycode.model.ListTableRowsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.honeycode.paginators.ListTableRowsIterable responses = client.listTableRowsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTableRows(software.amazon.awssdk.services.honeycode.model.ListTableRowsRequest)} operation.</b>
     * </p>
     *
     * @param listTableRowsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check that the workbook is owned by you and
     *         your IAM policy allows access to the resource in the request.
     * @throws InternalServerException
     *         There were unexpected errors from the server.
     * @throws ResourceNotFoundException
     *         A Workbook, Table, App, Screen or Screen Automation was not found with the given ID.
     * @throws ServiceUnavailableException
     *         Remote service is unreachable.
     * @throws ValidationException
     *         Request is invalid. The message in the response contains details on why the request is invalid.
     * @throws RequestTimeoutException
     *         The request timed out.
     * @throws ThrottlingException
     *         Tps(transactions per second) rate reached.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HoneycodeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HoneycodeClient.ListTableRows
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/honeycode-2020-03-01/ListTableRows" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTableRowsIterable listTableRowsPaginator(ListTableRowsRequest listTableRowsRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ServiceUnavailableException, ValidationException,
            RequestTimeoutException, ThrottlingException, AwsServiceException, SdkClientException, HoneycodeException {
        return new ListTableRowsIterable(this, applyPaginatorUserAgent(listTableRowsRequest));
    }

    /**
     * <p>
     * The ListTables API allows you to retrieve a list of all the tables in a workbook.
     * </p>
     *
     * @param listTablesRequest
     * @return Result of the ListTables operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check that the workbook is owned by you and
     *         your IAM policy allows access to the resource in the request.
     * @throws InternalServerException
     *         There were unexpected errors from the server.
     * @throws RequestTimeoutException
     *         The request timed out.
     * @throws ResourceNotFoundException
     *         A Workbook, Table, App, Screen or Screen Automation was not found with the given ID.
     * @throws ServiceUnavailableException
     *         Remote service is unreachable.
     * @throws ThrottlingException
     *         Tps(transactions per second) rate reached.
     * @throws ValidationException
     *         Request is invalid. The message in the response contains details on why the request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HoneycodeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HoneycodeClient.ListTables
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/honeycode-2020-03-01/ListTables" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTablesResponse listTables(ListTablesRequest listTablesRequest) throws AccessDeniedException,
            InternalServerException, RequestTimeoutException, ResourceNotFoundException, ServiceUnavailableException,
            ThrottlingException, ValidationException, AwsServiceException, SdkClientException, HoneycodeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTablesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTablesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTablesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Honeycode");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTables");

            return clientHandler
                    .execute(new ClientExecutionParams<ListTablesRequest, ListTablesResponse>().withOperationName("ListTables")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listTablesRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListTablesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The ListTables API allows you to retrieve a list of all the tables in a workbook.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listTables(software.amazon.awssdk.services.honeycode.model.ListTablesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.honeycode.paginators.ListTablesIterable responses = client.listTablesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.honeycode.paginators.ListTablesIterable responses = client.listTablesPaginator(request);
     *     for (software.amazon.awssdk.services.honeycode.model.ListTablesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.honeycode.paginators.ListTablesIterable responses = client.listTablesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTables(software.amazon.awssdk.services.honeycode.model.ListTablesRequest)} operation.</b>
     * </p>
     *
     * @param listTablesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check that the workbook is owned by you and
     *         your IAM policy allows access to the resource in the request.
     * @throws InternalServerException
     *         There were unexpected errors from the server.
     * @throws RequestTimeoutException
     *         The request timed out.
     * @throws ResourceNotFoundException
     *         A Workbook, Table, App, Screen or Screen Automation was not found with the given ID.
     * @throws ServiceUnavailableException
     *         Remote service is unreachable.
     * @throws ThrottlingException
     *         Tps(transactions per second) rate reached.
     * @throws ValidationException
     *         Request is invalid. The message in the response contains details on why the request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HoneycodeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HoneycodeClient.ListTables
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/honeycode-2020-03-01/ListTables" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTablesIterable listTablesPaginator(ListTablesRequest listTablesRequest) throws AccessDeniedException,
            InternalServerException, RequestTimeoutException, ResourceNotFoundException, ServiceUnavailableException,
            ThrottlingException, ValidationException, AwsServiceException, SdkClientException, HoneycodeException {
        return new ListTablesIterable(this, applyPaginatorUserAgent(listTablesRequest));
    }

    /**
     * <p>
     * The QueryTableRows API allows you to use a filter formula to query for specific rows in a table.
     * </p>
     *
     * @param queryTableRowsRequest
     * @return Result of the QueryTableRows operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check that the workbook is owned by you and
     *         your IAM policy allows access to the resource in the request.
     * @throws InternalServerException
     *         There were unexpected errors from the server.
     * @throws RequestTimeoutException
     *         The request timed out.
     * @throws ResourceNotFoundException
     *         A Workbook, Table, App, Screen or Screen Automation was not found with the given ID.
     * @throws ServiceUnavailableException
     *         Remote service is unreachable.
     * @throws ThrottlingException
     *         Tps(transactions per second) rate reached.
     * @throws ValidationException
     *         Request is invalid. The message in the response contains details on why the request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HoneycodeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HoneycodeClient.QueryTableRows
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/honeycode-2020-03-01/QueryTableRows" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public QueryTableRowsResponse queryTableRows(QueryTableRowsRequest queryTableRowsRequest) throws AccessDeniedException,
            InternalServerException, RequestTimeoutException, ResourceNotFoundException, ServiceUnavailableException,
            ThrottlingException, ValidationException, AwsServiceException, SdkClientException, HoneycodeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<QueryTableRowsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                QueryTableRowsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, queryTableRowsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Honeycode");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "QueryTableRows");

            return clientHandler.execute(new ClientExecutionParams<QueryTableRowsRequest, QueryTableRowsResponse>()
                    .withOperationName("QueryTableRows").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(queryTableRowsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new QueryTableRowsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The QueryTableRows API allows you to use a filter formula to query for specific rows in a table.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #queryTableRows(software.amazon.awssdk.services.honeycode.model.QueryTableRowsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.honeycode.paginators.QueryTableRowsIterable responses = client.queryTableRowsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.honeycode.paginators.QueryTableRowsIterable responses = client
     *             .queryTableRowsPaginator(request);
     *     for (software.amazon.awssdk.services.honeycode.model.QueryTableRowsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.honeycode.paginators.QueryTableRowsIterable responses = client.queryTableRowsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #queryTableRows(software.amazon.awssdk.services.honeycode.model.QueryTableRowsRequest)} operation.</b>
     * </p>
     *
     * @param queryTableRowsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check that the workbook is owned by you and
     *         your IAM policy allows access to the resource in the request.
     * @throws InternalServerException
     *         There were unexpected errors from the server.
     * @throws RequestTimeoutException
     *         The request timed out.
     * @throws ResourceNotFoundException
     *         A Workbook, Table, App, Screen or Screen Automation was not found with the given ID.
     * @throws ServiceUnavailableException
     *         Remote service is unreachable.
     * @throws ThrottlingException
     *         Tps(transactions per second) rate reached.
     * @throws ValidationException
     *         Request is invalid. The message in the response contains details on why the request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HoneycodeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HoneycodeClient.QueryTableRows
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/honeycode-2020-03-01/QueryTableRows" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public QueryTableRowsIterable queryTableRowsPaginator(QueryTableRowsRequest queryTableRowsRequest)
            throws AccessDeniedException, InternalServerException, RequestTimeoutException, ResourceNotFoundException,
            ServiceUnavailableException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            HoneycodeException {
        return new QueryTableRowsIterable(this, applyPaginatorUserAgent(queryTableRowsRequest));
    }

    /**
     * <p>
     * The StartTableDataImportJob API allows you to start an import job on a table. This API will only return the id of
     * the job that was started. To find out the status of the import request, you need to call the
     * DescribeTableDataImportJob API.
     * </p>
     *
     * @param startTableDataImportJobRequest
     * @return Result of the StartTableDataImportJob operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Check that the workbook is owned by you and
     *         your IAM policy allows access to the resource in the request.
     * @throws InternalServerException
     *         There were unexpected errors from the server.
     * @throws ResourceNotFoundException
     *         A Workbook, Table, App, Screen or Screen Automation was not found with the given ID.
     * @throws ServiceUnavailableException
     *         Remote service is unreachable.
     * @throws ThrottlingException
     *         Tps(transactions per second) rate reached.
     * @throws ValidationException
     *         Request is invalid. The message in the response contains details on why the request is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws HoneycodeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample HoneycodeClient.StartTableDataImportJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/honeycode-2020-03-01/StartTableDataImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartTableDataImportJobResponse startTableDataImportJob(StartTableDataImportJobRequest startTableDataImportJobRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ServiceUnavailableException,
            ThrottlingException, ValidationException, AwsServiceException, SdkClientException, HoneycodeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartTableDataImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartTableDataImportJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startTableDataImportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Honeycode");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartTableDataImportJob");

            return clientHandler
                    .execute(new ClientExecutionParams<StartTableDataImportJobRequest, StartTableDataImportJobResponse>()
                            .withOperationName("StartTableDataImportJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(startTableDataImportJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartTableDataImportJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(HoneycodeException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AutomationExecutionException")
                                .exceptionBuilderSupplier(AutomationExecutionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RequestTimeoutException")
                                .exceptionBuilderSupplier(RequestTimeoutException::builder).httpStatusCode(504).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AutomationExecutionTimeoutException")
                                .exceptionBuilderSupplier(AutomationExecutionTimeoutException::builder).httpStatusCode(504)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends HoneycodeRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
