/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.honeycode.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that represents a single cell in a table.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Cell implements SdkPojo, Serializable, ToCopyableBuilder<Cell.Builder, Cell> {
    private static final SdkField<String> FORMULA_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("formula")
            .getter(getter(Cell::formula)).setter(setter(Builder::formula))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("formula").build()).build();

    private static final SdkField<String> FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("format")
            .getter(getter(Cell::formatAsString)).setter(setter(Builder::format))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("format").build()).build();

    private static final SdkField<String> RAW_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("rawValue").getter(getter(Cell::rawValue)).setter(setter(Builder::rawValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("rawValue").build()).build();

    private static final SdkField<String> FORMATTED_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("formattedValue").getter(getter(Cell::formattedValue)).setter(setter(Builder::formattedValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("formattedValue").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FORMULA_FIELD, FORMAT_FIELD,
            RAW_VALUE_FIELD, FORMATTED_VALUE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String formula;

    private final String format;

    private final String rawValue;

    private final String formattedValue;

    private Cell(BuilderImpl builder) {
        this.formula = builder.formula;
        this.format = builder.format;
        this.rawValue = builder.rawValue;
        this.formattedValue = builder.formattedValue;
    }

    /**
     * <p>
     * The formula contained in the cell. This field is empty if a cell does not have a formula.
     * </p>
     * 
     * @return The formula contained in the cell. This field is empty if a cell does not have a formula.
     */
    public String formula() {
        return formula;
    }

    /**
     * <p>
     * The format of the cell. If this field is empty, then the format is either not specified in the workbook or the
     * format is set to AUTO.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link Format#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return The format of the cell. If this field is empty, then the format is either not specified in the workbook
     *         or the format is set to AUTO.
     * @see Format
     */
    public Format format() {
        return Format.fromValue(format);
    }

    /**
     * <p>
     * The format of the cell. If this field is empty, then the format is either not specified in the workbook or the
     * format is set to AUTO.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link Format#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return The format of the cell. If this field is empty, then the format is either not specified in the workbook
     *         or the format is set to AUTO.
     * @see Format
     */
    public String formatAsString() {
        return format;
    }

    /**
     * <p>
     * The raw value of the data contained in the cell. The raw value depends on the format of the data in the cell.
     * However the attribute in the API return value is always a string containing the raw value.
     * </p>
     * <p>
     * Cells with format DATE, DATE_TIME or TIME have the raw value as a floating point number where the whole number
     * represents the number of days since 1/1/1900 and the fractional part represents the fraction of the day since
     * midnight. For example, a cell with date 11/3/2020 has the raw value "44138". A cell with the time 9:00 AM has the
     * raw value "0.375" and a cell with date/time value of 11/3/2020 9:00 AM has the raw value "44138.375". Notice that
     * even though the raw value is a number in all three cases, it is still represented as a string.
     * </p>
     * <p>
     * Cells with format NUMBER, CURRENCY, PERCENTAGE and ACCOUNTING have the raw value of the data as the number
     * representing the data being displayed. For example, the number 1.325 with two decimal places in the format will
     * have it's raw value as "1.325" and formatted value as "1.33". A currency value for $10 will have the raw value as
     * "10" and formatted value as "$10.00". A value representing 20% with two decimal places in the format will have
     * its raw value as "0.2" and the formatted value as "20.00%". An accounting value of -$25 will have "-25" as the
     * raw value and "$ (25.00)" as the formatted value.
     * </p>
     * <p>
     * Cells with format TEXT will have the raw text as the raw value. For example, a cell with text "John Smith" will
     * have "John Smith" as both the raw value and the formatted value.
     * </p>
     * <p>
     * Cells with format CONTACT will have the name of the contact as a formatted value and the email address of the
     * contact as the raw value. For example, a contact for John Smith will have "John Smith" as the formatted value and
     * "john.smith@example.com" as the raw value.
     * </p>
     * <p>
     * Cells with format ROWLINK (aka picklist) will have the first column of the linked row as the formatted value and
     * the row id of the linked row as the raw value. For example, a cell containing a picklist to a table that displays
     * task status might have "Completed" as the formatted value and
     * "row:dfcefaee-5b37-4355-8f28-40c3e4ff5dd4/ca432b2f-b8eb-431d-9fb5-cbe0342f9f03" as the raw value.
     * </p>
     * <p>
     * Cells with format AUTO or cells without any format that are auto-detected as one of the formats above will
     * contain the raw and formatted values as mentioned above, based on the auto-detected formats. If there is no
     * auto-detected format, the raw and formatted values will be the same as the data in the cell.
     * </p>
     * 
     * @return The raw value of the data contained in the cell. The raw value depends on the format of the data in the
     *         cell. However the attribute in the API return value is always a string containing the raw value. </p>
     *         <p>
     *         Cells with format DATE, DATE_TIME or TIME have the raw value as a floating point number where the whole
     *         number represents the number of days since 1/1/1900 and the fractional part represents the fraction of
     *         the day since midnight. For example, a cell with date 11/3/2020 has the raw value "44138". A cell with
     *         the time 9:00 AM has the raw value "0.375" and a cell with date/time value of 11/3/2020 9:00 AM has the
     *         raw value "44138.375". Notice that even though the raw value is a number in all three cases, it is still
     *         represented as a string.
     *         </p>
     *         <p>
     *         Cells with format NUMBER, CURRENCY, PERCENTAGE and ACCOUNTING have the raw value of the data as the
     *         number representing the data being displayed. For example, the number 1.325 with two decimal places in
     *         the format will have it's raw value as "1.325" and formatted value as "1.33". A currency value for $10
     *         will have the raw value as "10" and formatted value as "$10.00". A value representing 20% with two
     *         decimal places in the format will have its raw value as "0.2" and the formatted value as "20.00%". An
     *         accounting value of -$25 will have "-25" as the raw value and "$ (25.00)" as the formatted value.
     *         </p>
     *         <p>
     *         Cells with format TEXT will have the raw text as the raw value. For example, a cell with text
     *         "John Smith" will have "John Smith" as both the raw value and the formatted value.
     *         </p>
     *         <p>
     *         Cells with format CONTACT will have the name of the contact as a formatted value and the email address of
     *         the contact as the raw value. For example, a contact for John Smith will have "John Smith" as the
     *         formatted value and "john.smith@example.com" as the raw value.
     *         </p>
     *         <p>
     *         Cells with format ROWLINK (aka picklist) will have the first column of the linked row as the formatted
     *         value and the row id of the linked row as the raw value. For example, a cell containing a picklist to a
     *         table that displays task status might have "Completed" as the formatted value and
     *         "row:dfcefaee-5b37-4355-8f28-40c3e4ff5dd4/ca432b2f-b8eb-431d-9fb5-cbe0342f9f03" as the raw value.
     *         </p>
     *         <p>
     *         Cells with format AUTO or cells without any format that are auto-detected as one of the formats above
     *         will contain the raw and formatted values as mentioned above, based on the auto-detected formats. If
     *         there is no auto-detected format, the raw and formatted values will be the same as the data in the cell.
     */
    public String rawValue() {
        return rawValue;
    }

    /**
     * <p>
     * The formatted value of the cell. This is the value that you see displayed in the cell in the UI.
     * </p>
     * <p>
     * Note that the formatted value of a cell is always represented as a string irrespective of the data that is stored
     * in the cell. For example, if a cell contains a date, the formatted value of the cell is the string representation
     * of the formatted date being shown in the cell in the UI. See details in the rawValue field below for how cells of
     * different formats will have different raw and formatted values.
     * </p>
     * 
     * @return The formatted value of the cell. This is the value that you see displayed in the cell in the UI. </p>
     *         <p>
     *         Note that the formatted value of a cell is always represented as a string irrespective of the data that
     *         is stored in the cell. For example, if a cell contains a date, the formatted value of the cell is the
     *         string representation of the formatted date being shown in the cell in the UI. See details in the
     *         rawValue field below for how cells of different formats will have different raw and formatted values.
     */
    public String formattedValue() {
        return formattedValue;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(formula());
        hashCode = 31 * hashCode + Objects.hashCode(formatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(rawValue());
        hashCode = 31 * hashCode + Objects.hashCode(formattedValue());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Cell)) {
            return false;
        }
        Cell other = (Cell) obj;
        return Objects.equals(formula(), other.formula()) && Objects.equals(formatAsString(), other.formatAsString())
                && Objects.equals(rawValue(), other.rawValue()) && Objects.equals(formattedValue(), other.formattedValue());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Cell").add("Formula", formula() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Format", formatAsString()).add("RawValue", rawValue()).add("FormattedValue", formattedValue()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "formula":
            return Optional.ofNullable(clazz.cast(formula()));
        case "format":
            return Optional.ofNullable(clazz.cast(formatAsString()));
        case "rawValue":
            return Optional.ofNullable(clazz.cast(rawValue()));
        case "formattedValue":
            return Optional.ofNullable(clazz.cast(formattedValue()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Cell, T> g) {
        return obj -> g.apply((Cell) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Cell> {
        /**
         * <p>
         * The formula contained in the cell. This field is empty if a cell does not have a formula.
         * </p>
         * 
         * @param formula
         *        The formula contained in the cell. This field is empty if a cell does not have a formula.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder formula(String formula);

        /**
         * <p>
         * The format of the cell. If this field is empty, then the format is either not specified in the workbook or
         * the format is set to AUTO.
         * </p>
         * 
         * @param format
         *        The format of the cell. If this field is empty, then the format is either not specified in the
         *        workbook or the format is set to AUTO.
         * @see Format
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Format
         */
        Builder format(String format);

        /**
         * <p>
         * The format of the cell. If this field is empty, then the format is either not specified in the workbook or
         * the format is set to AUTO.
         * </p>
         * 
         * @param format
         *        The format of the cell. If this field is empty, then the format is either not specified in the
         *        workbook or the format is set to AUTO.
         * @see Format
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Format
         */
        Builder format(Format format);

        /**
         * <p>
         * The raw value of the data contained in the cell. The raw value depends on the format of the data in the cell.
         * However the attribute in the API return value is always a string containing the raw value.
         * </p>
         * <p>
         * Cells with format DATE, DATE_TIME or TIME have the raw value as a floating point number where the whole
         * number represents the number of days since 1/1/1900 and the fractional part represents the fraction of the
         * day since midnight. For example, a cell with date 11/3/2020 has the raw value "44138". A cell with the time
         * 9:00 AM has the raw value "0.375" and a cell with date/time value of 11/3/2020 9:00 AM has the raw value
         * "44138.375". Notice that even though the raw value is a number in all three cases, it is still represented as
         * a string.
         * </p>
         * <p>
         * Cells with format NUMBER, CURRENCY, PERCENTAGE and ACCOUNTING have the raw value of the data as the number
         * representing the data being displayed. For example, the number 1.325 with two decimal places in the format
         * will have it's raw value as "1.325" and formatted value as "1.33". A currency value for $10 will have the raw
         * value as "10" and formatted value as "$10.00". A value representing 20% with two decimal places in the format
         * will have its raw value as "0.2" and the formatted value as "20.00%". An accounting value of -$25 will have
         * "-25" as the raw value and "$ (25.00)" as the formatted value.
         * </p>
         * <p>
         * Cells with format TEXT will have the raw text as the raw value. For example, a cell with text "John Smith"
         * will have "John Smith" as both the raw value and the formatted value.
         * </p>
         * <p>
         * Cells with format CONTACT will have the name of the contact as a formatted value and the email address of the
         * contact as the raw value. For example, a contact for John Smith will have "John Smith" as the formatted value
         * and "john.smith@example.com" as the raw value.
         * </p>
         * <p>
         * Cells with format ROWLINK (aka picklist) will have the first column of the linked row as the formatted value
         * and the row id of the linked row as the raw value. For example, a cell containing a picklist to a table that
         * displays task status might have "Completed" as the formatted value and
         * "row:dfcefaee-5b37-4355-8f28-40c3e4ff5dd4/ca432b2f-b8eb-431d-9fb5-cbe0342f9f03" as the raw value.
         * </p>
         * <p>
         * Cells with format AUTO or cells without any format that are auto-detected as one of the formats above will
         * contain the raw and formatted values as mentioned above, based on the auto-detected formats. If there is no
         * auto-detected format, the raw and formatted values will be the same as the data in the cell.
         * </p>
         * 
         * @param rawValue
         *        The raw value of the data contained in the cell. The raw value depends on the format of the data in
         *        the cell. However the attribute in the API return value is always a string containing the raw value.
         *        </p>
         *        <p>
         *        Cells with format DATE, DATE_TIME or TIME have the raw value as a floating point number where the
         *        whole number represents the number of days since 1/1/1900 and the fractional part represents the
         *        fraction of the day since midnight. For example, a cell with date 11/3/2020 has the raw value "44138".
         *        A cell with the time 9:00 AM has the raw value "0.375" and a cell with date/time value of 11/3/2020
         *        9:00 AM has the raw value "44138.375". Notice that even though the raw value is a number in all three
         *        cases, it is still represented as a string.
         *        </p>
         *        <p>
         *        Cells with format NUMBER, CURRENCY, PERCENTAGE and ACCOUNTING have the raw value of the data as the
         *        number representing the data being displayed. For example, the number 1.325 with two decimal places in
         *        the format will have it's raw value as "1.325" and formatted value as "1.33". A currency value for $10
         *        will have the raw value as "10" and formatted value as "$10.00". A value representing 20% with two
         *        decimal places in the format will have its raw value as "0.2" and the formatted value as "20.00%". An
         *        accounting value of -$25 will have "-25" as the raw value and "$ (25.00)" as the formatted value.
         *        </p>
         *        <p>
         *        Cells with format TEXT will have the raw text as the raw value. For example, a cell with text
         *        "John Smith" will have "John Smith" as both the raw value and the formatted value.
         *        </p>
         *        <p>
         *        Cells with format CONTACT will have the name of the contact as a formatted value and the email address
         *        of the contact as the raw value. For example, a contact for John Smith will have "John Smith" as the
         *        formatted value and "john.smith@example.com" as the raw value.
         *        </p>
         *        <p>
         *        Cells with format ROWLINK (aka picklist) will have the first column of the linked row as the formatted
         *        value and the row id of the linked row as the raw value. For example, a cell containing a picklist to
         *        a table that displays task status might have "Completed" as the formatted value and
         *        "row:dfcefaee-5b37-4355-8f28-40c3e4ff5dd4/ca432b2f-b8eb-431d-9fb5-cbe0342f9f03" as the raw value.
         *        </p>
         *        <p>
         *        Cells with format AUTO or cells without any format that are auto-detected as one of the formats above
         *        will contain the raw and formatted values as mentioned above, based on the auto-detected formats. If
         *        there is no auto-detected format, the raw and formatted values will be the same as the data in the
         *        cell.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rawValue(String rawValue);

        /**
         * <p>
         * The formatted value of the cell. This is the value that you see displayed in the cell in the UI.
         * </p>
         * <p>
         * Note that the formatted value of a cell is always represented as a string irrespective of the data that is
         * stored in the cell. For example, if a cell contains a date, the formatted value of the cell is the string
         * representation of the formatted date being shown in the cell in the UI. See details in the rawValue field
         * below for how cells of different formats will have different raw and formatted values.
         * </p>
         * 
         * @param formattedValue
         *        The formatted value of the cell. This is the value that you see displayed in the cell in the UI. </p>
         *        <p>
         *        Note that the formatted value of a cell is always represented as a string irrespective of the data
         *        that is stored in the cell. For example, if a cell contains a date, the formatted value of the cell is
         *        the string representation of the formatted date being shown in the cell in the UI. See details in the
         *        rawValue field below for how cells of different formats will have different raw and formatted values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder formattedValue(String formattedValue);
    }

    static final class BuilderImpl implements Builder {
        private String formula;

        private String format;

        private String rawValue;

        private String formattedValue;

        private BuilderImpl() {
        }

        private BuilderImpl(Cell model) {
            formula(model.formula);
            format(model.format);
            rawValue(model.rawValue);
            formattedValue(model.formattedValue);
        }

        public final String getFormula() {
            return formula;
        }

        @Override
        public final Builder formula(String formula) {
            this.formula = formula;
            return this;
        }

        public final void setFormula(String formula) {
            this.formula = formula;
        }

        public final String getFormat() {
            return format;
        }

        @Override
        public final Builder format(String format) {
            this.format = format;
            return this;
        }

        @Override
        public final Builder format(Format format) {
            this.format(format == null ? null : format.toString());
            return this;
        }

        public final void setFormat(String format) {
            this.format = format;
        }

        public final String getRawValue() {
            return rawValue;
        }

        @Override
        public final Builder rawValue(String rawValue) {
            this.rawValue = rawValue;
            return this;
        }

        public final void setRawValue(String rawValue) {
            this.rawValue = rawValue;
        }

        public final String getFormattedValue() {
            return formattedValue;
        }

        @Override
        public final Builder formattedValue(String formattedValue) {
            this.formattedValue = formattedValue;
            return this;
        }

        public final void setFormattedValue(String formattedValue) {
            this.formattedValue = formattedValue;
        }

        @Override
        public Cell build() {
            return new Cell(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
