/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iam.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that contains details about how a service-linked role is used, if that information is returned by the
 * service.
 * </p>
 * <p>
 * This data type is used as a response element in the <a>GetServiceLinkedRoleDeletionStatus</a> operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RoleUsageType implements SdkPojo, Serializable, ToCopyableBuilder<RoleUsageType.Builder, RoleUsageType> {
    private static final SdkField<String> REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(RoleUsageType::region)).setter(setter(Builder::region))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Region").build()).build();

    private static final SdkField<List<String>> RESOURCES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(RoleUsageType::resources))
            .setter(setter(Builder::resources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Resources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections
            .unmodifiableList(Arrays.asList(REGION_FIELD, RESOURCES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String region;

    private final List<String> resources;

    private RoleUsageType(BuilderImpl builder) {
        this.region = builder.region;
        this.resources = builder.resources;
    }

    /**
     * <p>
     * The name of the Region where the service-linked role is being used.
     * </p>
     * 
     * @return The name of the Region where the service-linked role is being used.
     */
    public String region() {
        return region;
    }

    /**
     * Returns true if the Resources property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasResources() {
        return resources != null && !(resources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The name of the resource that is using the service-linked role.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasResources()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The name of the resource that is using the service-linked role.
     */
    public List<String> resources() {
        return resources;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(region());
        hashCode = 31 * hashCode + Objects.hashCode(resources());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RoleUsageType)) {
            return false;
        }
        RoleUsageType other = (RoleUsageType) obj;
        return Objects.equals(region(), other.region()) && Objects.equals(resources(), other.resources());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("RoleUsageType").add("Region", region()).add("Resources", resources()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Region":
            return Optional.ofNullable(clazz.cast(region()));
        case "Resources":
            return Optional.ofNullable(clazz.cast(resources()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RoleUsageType, T> g) {
        return obj -> g.apply((RoleUsageType) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RoleUsageType> {
        /**
         * <p>
         * The name of the Region where the service-linked role is being used.
         * </p>
         * 
         * @param region
         *        The name of the Region where the service-linked role is being used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder region(String region);

        /**
         * <p>
         * The name of the resource that is using the service-linked role.
         * </p>
         * 
         * @param resources
         *        The name of the resource that is using the service-linked role.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(Collection<String> resources);

        /**
         * <p>
         * The name of the resource that is using the service-linked role.
         * </p>
         * 
         * @param resources
         *        The name of the resource that is using the service-linked role.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(String... resources);
    }

    static final class BuilderImpl implements Builder {
        private String region;

        private List<String> resources = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(RoleUsageType model) {
            region(model.region);
            resources(model.resources);
        }

        public final String getRegion() {
            return region;
        }

        @Override
        public final Builder region(String region) {
            this.region = region;
            return this;
        }

        public final void setRegion(String region) {
            this.region = region;
        }

        public final Collection<String> getResources() {
            return resources;
        }

        @Override
        public final Builder resources(Collection<String> resources) {
            this.resources = ArnListTypeCopier.copy(resources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resources(String... resources) {
            resources(Arrays.asList(resources));
            return this;
        }

        public final void setResources(Collection<String> resources) {
            this.resources = ArnListTypeCopier.copy(resources);
        }

        @Override
        public RoleUsageType build() {
            return new RoleUsageType(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
