/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iam.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about a managed policy.
 * </p>
 * <p>
 * This data type is used as a response element in the <a>CreatePolicy</a>, <a>GetPolicy</a>, and <a>ListPolicies</a>
 * operations.
 * </p>
 * <p>
 * For more information about managed policies, refer to <a
 * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/policies-managed-vs-inline.html">Managed policies and inline
 * policies</a> in the <i>IAM User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Policy implements SdkPojo, Serializable, ToCopyableBuilder<Policy.Builder, Policy> {
    private static final SdkField<String> POLICY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PolicyName").getter(getter(Policy::policyName)).setter(setter(Builder::policyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyName").build()).build();

    private static final SdkField<String> POLICY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PolicyId").getter(getter(Policy::policyId)).setter(setter(Builder::policyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyId").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(Policy::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Path")
            .getter(getter(Policy::path)).setter(setter(Builder::path))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Path").build()).build();

    private static final SdkField<String> DEFAULT_VERSION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DefaultVersionId").getter(getter(Policy::defaultVersionId)).setter(setter(Builder::defaultVersionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultVersionId").build()).build();

    private static final SdkField<Integer> ATTACHMENT_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("AttachmentCount").getter(getter(Policy::attachmentCount)).setter(setter(Builder::attachmentCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttachmentCount").build()).build();

    private static final SdkField<Integer> PERMISSIONS_BOUNDARY_USAGE_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("PermissionsBoundaryUsageCount")
            .getter(getter(Policy::permissionsBoundaryUsageCount))
            .setter(setter(Builder::permissionsBoundaryUsageCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PermissionsBoundaryUsageCount")
                    .build()).build();

    private static final SdkField<Boolean> IS_ATTACHABLE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IsAttachable").getter(getter(Policy::isAttachable)).setter(setter(Builder::isAttachable))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsAttachable").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(Policy::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<Instant> CREATE_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreateDate").getter(getter(Policy::createDate)).setter(setter(Builder::createDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreateDate").build()).build();

    private static final SdkField<Instant> UPDATE_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("UpdateDate").getter(getter(Policy::updateDate)).setter(setter(Builder::updateDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdateDate").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(Policy::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(POLICY_NAME_FIELD,
            POLICY_ID_FIELD, ARN_FIELD, PATH_FIELD, DEFAULT_VERSION_ID_FIELD, ATTACHMENT_COUNT_FIELD,
            PERMISSIONS_BOUNDARY_USAGE_COUNT_FIELD, IS_ATTACHABLE_FIELD, DESCRIPTION_FIELD, CREATE_DATE_FIELD, UPDATE_DATE_FIELD,
            TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String policyName;

    private final String policyId;

    private final String arn;

    private final String path;

    private final String defaultVersionId;

    private final Integer attachmentCount;

    private final Integer permissionsBoundaryUsageCount;

    private final Boolean isAttachable;

    private final String description;

    private final Instant createDate;

    private final Instant updateDate;

    private final List<Tag> tags;

    private Policy(BuilderImpl builder) {
        this.policyName = builder.policyName;
        this.policyId = builder.policyId;
        this.arn = builder.arn;
        this.path = builder.path;
        this.defaultVersionId = builder.defaultVersionId;
        this.attachmentCount = builder.attachmentCount;
        this.permissionsBoundaryUsageCount = builder.permissionsBoundaryUsageCount;
        this.isAttachable = builder.isAttachable;
        this.description = builder.description;
        this.createDate = builder.createDate;
        this.updateDate = builder.updateDate;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The friendly name (not ARN) identifying the policy.
     * </p>
     * 
     * @return The friendly name (not ARN) identifying the policy.
     */
    public final String policyName() {
        return policyName;
    }

    /**
     * <p>
     * The stable and unique string identifying the policy.
     * </p>
     * <p>
     * For more information about IDs, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">IAM identifiers</a> in the <i>IAM
     * User Guide</i>.
     * </p>
     * 
     * @return The stable and unique string identifying the policy.</p>
     *         <p>
     *         For more information about IDs, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">IAM identifiers</a> in the
     *         <i>IAM User Guide</i>.
     */
    public final String policyId() {
        return policyId;
    }

    /**
     * Returns the value of the Arn property for this object.
     * 
     * @return The value of the Arn property for this object.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The path to the policy.
     * </p>
     * <p>
     * For more information about paths, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">IAM identifiers</a> in the <i>IAM
     * User Guide</i>.
     * </p>
     * 
     * @return The path to the policy.</p>
     *         <p>
     *         For more information about paths, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">IAM identifiers</a> in the
     *         <i>IAM User Guide</i>.
     */
    public final String path() {
        return path;
    }

    /**
     * <p>
     * The identifier for the version of the policy that is set as the default version.
     * </p>
     * 
     * @return The identifier for the version of the policy that is set as the default version.
     */
    public final String defaultVersionId() {
        return defaultVersionId;
    }

    /**
     * <p>
     * The number of entities (users, groups, and roles) that the policy is attached to.
     * </p>
     * 
     * @return The number of entities (users, groups, and roles) that the policy is attached to.
     */
    public final Integer attachmentCount() {
        return attachmentCount;
    }

    /**
     * <p>
     * The number of entities (users and roles) for which the policy is used to set the permissions boundary.
     * </p>
     * <p>
     * For more information about permissions boundaries, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_boundaries.html">Permissions boundaries
     * for IAM identities </a> in the <i>IAM User Guide</i>.
     * </p>
     * 
     * @return The number of entities (users and roles) for which the policy is used to set the permissions boundary.
     *         </p>
     *         <p>
     *         For more information about permissions boundaries, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_boundaries.html">Permissions
     *         boundaries for IAM identities </a> in the <i>IAM User Guide</i>.
     */
    public final Integer permissionsBoundaryUsageCount() {
        return permissionsBoundaryUsageCount;
    }

    /**
     * <p>
     * Specifies whether the policy can be attached to an IAM user, group, or role.
     * </p>
     * 
     * @return Specifies whether the policy can be attached to an IAM user, group, or role.
     */
    public final Boolean isAttachable() {
        return isAttachable;
    }

    /**
     * <p>
     * A friendly description of the policy.
     * </p>
     * <p>
     * This element is included in the response to the <a>GetPolicy</a> operation. It is not included in the response to
     * the <a>ListPolicies</a> operation.
     * </p>
     * 
     * @return A friendly description of the policy.</p>
     *         <p>
     *         This element is included in the response to the <a>GetPolicy</a> operation. It is not included in the
     *         response to the <a>ListPolicies</a> operation.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The date and time, in <a href="http://www.iso.org/iso/iso8601">ISO 8601 date-time format</a>, when the policy was
     * created.
     * </p>
     * 
     * @return The date and time, in <a href="http://www.iso.org/iso/iso8601">ISO 8601 date-time format</a>, when the
     *         policy was created.
     */
    public final Instant createDate() {
        return createDate;
    }

    /**
     * <p>
     * The date and time, in <a href="http://www.iso.org/iso/iso8601">ISO 8601 date-time format</a>, when the policy was
     * last updated.
     * </p>
     * <p>
     * When a policy has only one version, this field contains the date and time when the policy was created. When a
     * policy has more than one version, this field contains the date and time when the most recent policy version was
     * created.
     * </p>
     * 
     * @return The date and time, in <a href="http://www.iso.org/iso/iso8601">ISO 8601 date-time format</a>, when the
     *         policy was last updated.</p>
     *         <p>
     *         When a policy has only one version, this field contains the date and time when the policy was created.
     *         When a policy has more than one version, this field contains the date and time when the most recent
     *         policy version was created.
     */
    public final Instant updateDate() {
        return updateDate;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of tags that are attached to the instance profile. For more information about tagging, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_tags.html">Tagging IAM resources</a> in the <i>IAM User
     * Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A list of tags that are attached to the instance profile. For more information about tagging, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_tags.html">Tagging IAM resources</a> in the
     *         <i>IAM User Guide</i>.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(policyName());
        hashCode = 31 * hashCode + Objects.hashCode(policyId());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(path());
        hashCode = 31 * hashCode + Objects.hashCode(defaultVersionId());
        hashCode = 31 * hashCode + Objects.hashCode(attachmentCount());
        hashCode = 31 * hashCode + Objects.hashCode(permissionsBoundaryUsageCount());
        hashCode = 31 * hashCode + Objects.hashCode(isAttachable());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(createDate());
        hashCode = 31 * hashCode + Objects.hashCode(updateDate());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Policy)) {
            return false;
        }
        Policy other = (Policy) obj;
        return Objects.equals(policyName(), other.policyName()) && Objects.equals(policyId(), other.policyId())
                && Objects.equals(arn(), other.arn()) && Objects.equals(path(), other.path())
                && Objects.equals(defaultVersionId(), other.defaultVersionId())
                && Objects.equals(attachmentCount(), other.attachmentCount())
                && Objects.equals(permissionsBoundaryUsageCount(), other.permissionsBoundaryUsageCount())
                && Objects.equals(isAttachable(), other.isAttachable()) && Objects.equals(description(), other.description())
                && Objects.equals(createDate(), other.createDate()) && Objects.equals(updateDate(), other.updateDate())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Policy").add("PolicyName", policyName()).add("PolicyId", policyId()).add("Arn", arn())
                .add("Path", path()).add("DefaultVersionId", defaultVersionId()).add("AttachmentCount", attachmentCount())
                .add("PermissionsBoundaryUsageCount", permissionsBoundaryUsageCount()).add("IsAttachable", isAttachable())
                .add("Description", description()).add("CreateDate", createDate()).add("UpdateDate", updateDate())
                .add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "PolicyName":
            return Optional.ofNullable(clazz.cast(policyName()));
        case "PolicyId":
            return Optional.ofNullable(clazz.cast(policyId()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Path":
            return Optional.ofNullable(clazz.cast(path()));
        case "DefaultVersionId":
            return Optional.ofNullable(clazz.cast(defaultVersionId()));
        case "AttachmentCount":
            return Optional.ofNullable(clazz.cast(attachmentCount()));
        case "PermissionsBoundaryUsageCount":
            return Optional.ofNullable(clazz.cast(permissionsBoundaryUsageCount()));
        case "IsAttachable":
            return Optional.ofNullable(clazz.cast(isAttachable()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "CreateDate":
            return Optional.ofNullable(clazz.cast(createDate()));
        case "UpdateDate":
            return Optional.ofNullable(clazz.cast(updateDate()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Policy, T> g) {
        return obj -> g.apply((Policy) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Policy> {
        /**
         * <p>
         * The friendly name (not ARN) identifying the policy.
         * </p>
         * 
         * @param policyName
         *        The friendly name (not ARN) identifying the policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyName(String policyName);

        /**
         * <p>
         * The stable and unique string identifying the policy.
         * </p>
         * <p>
         * For more information about IDs, see <a
         * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">IAM identifiers</a> in the
         * <i>IAM User Guide</i>.
         * </p>
         * 
         * @param policyId
         *        The stable and unique string identifying the policy.</p>
         *        <p>
         *        For more information about IDs, see <a
         *        href="https://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">IAM identifiers</a> in
         *        the <i>IAM User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyId(String policyId);

        /**
         * Sets the value of the Arn property for this object.
         *
         * @param arn
         *        The new value for the Arn property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The path to the policy.
         * </p>
         * <p>
         * For more information about paths, see <a
         * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">IAM identifiers</a> in the
         * <i>IAM User Guide</i>.
         * </p>
         * 
         * @param path
         *        The path to the policy.</p>
         *        <p>
         *        For more information about paths, see <a
         *        href="https://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">IAM identifiers</a> in
         *        the <i>IAM User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder path(String path);

        /**
         * <p>
         * The identifier for the version of the policy that is set as the default version.
         * </p>
         * 
         * @param defaultVersionId
         *        The identifier for the version of the policy that is set as the default version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultVersionId(String defaultVersionId);

        /**
         * <p>
         * The number of entities (users, groups, and roles) that the policy is attached to.
         * </p>
         * 
         * @param attachmentCount
         *        The number of entities (users, groups, and roles) that the policy is attached to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachmentCount(Integer attachmentCount);

        /**
         * <p>
         * The number of entities (users and roles) for which the policy is used to set the permissions boundary.
         * </p>
         * <p>
         * For more information about permissions boundaries, see <a
         * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_boundaries.html">Permissions
         * boundaries for IAM identities </a> in the <i>IAM User Guide</i>.
         * </p>
         * 
         * @param permissionsBoundaryUsageCount
         *        The number of entities (users and roles) for which the policy is used to set the permissions boundary.
         *        </p>
         *        <p>
         *        For more information about permissions boundaries, see <a
         *        href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_boundaries.html">Permissions
         *        boundaries for IAM identities </a> in the <i>IAM User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder permissionsBoundaryUsageCount(Integer permissionsBoundaryUsageCount);

        /**
         * <p>
         * Specifies whether the policy can be attached to an IAM user, group, or role.
         * </p>
         * 
         * @param isAttachable
         *        Specifies whether the policy can be attached to an IAM user, group, or role.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isAttachable(Boolean isAttachable);

        /**
         * <p>
         * A friendly description of the policy.
         * </p>
         * <p>
         * This element is included in the response to the <a>GetPolicy</a> operation. It is not included in the
         * response to the <a>ListPolicies</a> operation.
         * </p>
         * 
         * @param description
         *        A friendly description of the policy.</p>
         *        <p>
         *        This element is included in the response to the <a>GetPolicy</a> operation. It is not included in the
         *        response to the <a>ListPolicies</a> operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The date and time, in <a href="http://www.iso.org/iso/iso8601">ISO 8601 date-time format</a>, when the policy
         * was created.
         * </p>
         * 
         * @param createDate
         *        The date and time, in <a href="http://www.iso.org/iso/iso8601">ISO 8601 date-time format</a>, when the
         *        policy was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createDate(Instant createDate);

        /**
         * <p>
         * The date and time, in <a href="http://www.iso.org/iso/iso8601">ISO 8601 date-time format</a>, when the policy
         * was last updated.
         * </p>
         * <p>
         * When a policy has only one version, this field contains the date and time when the policy was created. When a
         * policy has more than one version, this field contains the date and time when the most recent policy version
         * was created.
         * </p>
         * 
         * @param updateDate
         *        The date and time, in <a href="http://www.iso.org/iso/iso8601">ISO 8601 date-time format</a>, when the
         *        policy was last updated.</p>
         *        <p>
         *        When a policy has only one version, this field contains the date and time when the policy was created.
         *        When a policy has more than one version, this field contains the date and time when the most recent
         *        policy version was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updateDate(Instant updateDate);

        /**
         * <p>
         * A list of tags that are attached to the instance profile. For more information about tagging, see <a
         * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_tags.html">Tagging IAM resources</a> in the <i>IAM
         * User Guide</i>.
         * </p>
         * 
         * @param tags
         *        A list of tags that are attached to the instance profile. For more information about tagging, see <a
         *        href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_tags.html">Tagging IAM resources</a> in the
         *        <i>IAM User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of tags that are attached to the instance profile. For more information about tagging, see <a
         * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_tags.html">Tagging IAM resources</a> in the <i>IAM
         * User Guide</i>.
         * </p>
         * 
         * @param tags
         *        A list of tags that are attached to the instance profile. For more information about tagging, see <a
         *        href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_tags.html">Tagging IAM resources</a> in the
         *        <i>IAM User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A list of tags that are attached to the instance profile. For more information about tagging, see <a
         * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_tags.html">Tagging IAM resources</a> in the <i>IAM
         * User Guide</i>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String policyName;

        private String policyId;

        private String arn;

        private String path;

        private String defaultVersionId;

        private Integer attachmentCount;

        private Integer permissionsBoundaryUsageCount;

        private Boolean isAttachable;

        private String description;

        private Instant createDate;

        private Instant updateDate;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Policy model) {
            policyName(model.policyName);
            policyId(model.policyId);
            arn(model.arn);
            path(model.path);
            defaultVersionId(model.defaultVersionId);
            attachmentCount(model.attachmentCount);
            permissionsBoundaryUsageCount(model.permissionsBoundaryUsageCount);
            isAttachable(model.isAttachable);
            description(model.description);
            createDate(model.createDate);
            updateDate(model.updateDate);
            tags(model.tags);
        }

        public final String getPolicyName() {
            return policyName;
        }

        public final void setPolicyName(String policyName) {
            this.policyName = policyName;
        }

        @Override
        @Transient
        public final Builder policyName(String policyName) {
            this.policyName = policyName;
            return this;
        }

        public final String getPolicyId() {
            return policyId;
        }

        public final void setPolicyId(String policyId) {
            this.policyId = policyId;
        }

        @Override
        @Transient
        public final Builder policyId(String policyId) {
            this.policyId = policyId;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        @Transient
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getPath() {
            return path;
        }

        public final void setPath(String path) {
            this.path = path;
        }

        @Override
        @Transient
        public final Builder path(String path) {
            this.path = path;
            return this;
        }

        public final String getDefaultVersionId() {
            return defaultVersionId;
        }

        public final void setDefaultVersionId(String defaultVersionId) {
            this.defaultVersionId = defaultVersionId;
        }

        @Override
        @Transient
        public final Builder defaultVersionId(String defaultVersionId) {
            this.defaultVersionId = defaultVersionId;
            return this;
        }

        public final Integer getAttachmentCount() {
            return attachmentCount;
        }

        public final void setAttachmentCount(Integer attachmentCount) {
            this.attachmentCount = attachmentCount;
        }

        @Override
        @Transient
        public final Builder attachmentCount(Integer attachmentCount) {
            this.attachmentCount = attachmentCount;
            return this;
        }

        public final Integer getPermissionsBoundaryUsageCount() {
            return permissionsBoundaryUsageCount;
        }

        public final void setPermissionsBoundaryUsageCount(Integer permissionsBoundaryUsageCount) {
            this.permissionsBoundaryUsageCount = permissionsBoundaryUsageCount;
        }

        @Override
        @Transient
        public final Builder permissionsBoundaryUsageCount(Integer permissionsBoundaryUsageCount) {
            this.permissionsBoundaryUsageCount = permissionsBoundaryUsageCount;
            return this;
        }

        public final Boolean getIsAttachable() {
            return isAttachable;
        }

        public final void setIsAttachable(Boolean isAttachable) {
            this.isAttachable = isAttachable;
        }

        @Override
        @Transient
        public final Builder isAttachable(Boolean isAttachable) {
            this.isAttachable = isAttachable;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        @Transient
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Instant getCreateDate() {
            return createDate;
        }

        public final void setCreateDate(Instant createDate) {
            this.createDate = createDate;
        }

        @Override
        @Transient
        public final Builder createDate(Instant createDate) {
            this.createDate = createDate;
            return this;
        }

        public final Instant getUpdateDate() {
            return updateDate;
        }

        public final void setUpdateDate(Instant updateDate) {
            this.updateDate = updateDate;
        }

        @Override
        @Transient
        public final Builder updateDate(Instant updateDate) {
            this.updateDate = updateDate;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = _tagListTypeCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = _tagListTypeCopier.copyFromBuilder(tags);
        }

        @Override
        @Transient
        public final Builder tags(Collection<Tag> tags) {
            this.tags = _tagListTypeCopier.copy(tags);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Policy build() {
            return new Policy(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
