/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iam.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.core.waiters.WaiterState;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.iam.IamClient;
import software.amazon.awssdk.services.iam.model.GetInstanceProfileRequest;
import software.amazon.awssdk.services.iam.model.GetInstanceProfileResponse;
import software.amazon.awssdk.services.iam.model.GetPolicyRequest;
import software.amazon.awssdk.services.iam.model.GetPolicyResponse;
import software.amazon.awssdk.services.iam.model.GetRoleRequest;
import software.amazon.awssdk.services.iam.model.GetRoleResponse;
import software.amazon.awssdk.services.iam.model.GetUserRequest;
import software.amazon.awssdk.services.iam.model.GetUserResponse;
import software.amazon.awssdk.services.iam.model.IamRequest;
import software.amazon.awssdk.services.iam.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultIamWaiter implements IamWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final IamClient client;

    private final AttributeMap managedResources;

    private final Waiter<GetInstanceProfileResponse> instanceProfileExistsWaiter;

    private final Waiter<GetUserResponse> userExistsWaiter;

    private final Waiter<GetRoleResponse> roleExistsWaiter;

    private final Waiter<GetPolicyResponse> policyExistsWaiter;

    private DefaultIamWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = IamClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.instanceProfileExistsWaiter = Waiter.builder(GetInstanceProfileResponse.class)
                .acceptors(instanceProfileExistsWaiterAcceptors())
                .overrideConfiguration(instanceProfileExistsWaiterConfig(builder.overrideConfiguration)).build();
        this.userExistsWaiter = Waiter.builder(GetUserResponse.class).acceptors(userExistsWaiterAcceptors())
                .overrideConfiguration(userExistsWaiterConfig(builder.overrideConfiguration)).build();
        this.roleExistsWaiter = Waiter.builder(GetRoleResponse.class).acceptors(roleExistsWaiterAcceptors())
                .overrideConfiguration(roleExistsWaiterConfig(builder.overrideConfiguration)).build();
        this.policyExistsWaiter = Waiter.builder(GetPolicyResponse.class).acceptors(policyExistsWaiterAcceptors())
                .overrideConfiguration(policyExistsWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<GetInstanceProfileResponse> waitUntilInstanceProfileExists(
            GetInstanceProfileRequest getInstanceProfileRequest) {
        return instanceProfileExistsWaiter.run(() -> client.getInstanceProfile(applyWaitersUserAgent(getInstanceProfileRequest)));
    }

    @Override
    public WaiterResponse<GetInstanceProfileResponse> waitUntilInstanceProfileExists(
            GetInstanceProfileRequest getInstanceProfileRequest, WaiterOverrideConfiguration overrideConfig) {
        return instanceProfileExistsWaiter.run(() -> client.getInstanceProfile(applyWaitersUserAgent(getInstanceProfileRequest)),
                instanceProfileExistsWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetPolicyResponse> waitUntilPolicyExists(GetPolicyRequest getPolicyRequest) {
        return policyExistsWaiter.run(() -> client.getPolicy(applyWaitersUserAgent(getPolicyRequest)));
    }

    @Override
    public WaiterResponse<GetPolicyResponse> waitUntilPolicyExists(GetPolicyRequest getPolicyRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return policyExistsWaiter.run(() -> client.getPolicy(applyWaitersUserAgent(getPolicyRequest)),
                policyExistsWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetRoleResponse> waitUntilRoleExists(GetRoleRequest getRoleRequest) {
        return roleExistsWaiter.run(() -> client.getRole(applyWaitersUserAgent(getRoleRequest)));
    }

    @Override
    public WaiterResponse<GetRoleResponse> waitUntilRoleExists(GetRoleRequest getRoleRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return roleExistsWaiter.run(() -> client.getRole(applyWaitersUserAgent(getRoleRequest)),
                roleExistsWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetUserResponse> waitUntilUserExists(GetUserRequest getUserRequest) {
        return userExistsWaiter.run(() -> client.getUser(applyWaitersUserAgent(getUserRequest)));
    }

    @Override
    public WaiterResponse<GetUserResponse> waitUntilUserExists(GetUserRequest getUserRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return userExistsWaiter.run(() -> client.getUser(applyWaitersUserAgent(getUserRequest)),
                userExistsWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super GetInstanceProfileResponse>> instanceProfileExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super GetInstanceProfileResponse>> result = new ArrayList<>();
        result.add(new WaitersRuntime.ResponseStatusAcceptor(200, WaiterState.SUCCESS));
        result.add(new WaitersRuntime.ResponseStatusAcceptor(404, WaiterState.RETRY));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetUserResponse>> userExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super GetUserResponse>> result = new ArrayList<>();
        result.add(new WaitersRuntime.ResponseStatusAcceptor(200, WaiterState.SUCCESS));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "NoSuchEntity")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetRoleResponse>> roleExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super GetRoleResponse>> result = new ArrayList<>();
        result.add(new WaitersRuntime.ResponseStatusAcceptor(200, WaiterState.SUCCESS));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "NoSuchEntity")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetPolicyResponse>> policyExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super GetPolicyResponse>> result = new ArrayList<>();
        result.add(new WaitersRuntime.ResponseStatusAcceptor(200, WaiterState.SUCCESS));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "NoSuchEntity")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration instanceProfileExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(1)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration userExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(1)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration roleExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(1)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration policyExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(1)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static IamWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends IamRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements IamWaiter.Builder {
        private IamClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public IamWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public IamWaiter.Builder client(IamClient client) {
            this.client = client;
            return this;
        }

        public IamWaiter build() {
            return new DefaultIamWaiter(this);
        }
    }
}
