/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iam.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetMfaDeviceResponse extends IamResponse implements
        ToCopyableBuilder<GetMfaDeviceResponse.Builder, GetMfaDeviceResponse> {
    private static final SdkField<String> USER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UserName").getter(getter(GetMfaDeviceResponse::userName)).setter(setter(Builder::userName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserName").build()).build();

    private static final SdkField<String> SERIAL_NUMBER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SerialNumber").getter(getter(GetMfaDeviceResponse::serialNumber)).setter(setter(Builder::serialNumber))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SerialNumber").build()).build();

    private static final SdkField<Instant> ENABLE_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EnableDate").getter(getter(GetMfaDeviceResponse::enableDate)).setter(setter(Builder::enableDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnableDate").build()).build();

    private static final SdkField<Map<String, String>> CERTIFICATIONS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Certifications")
            .getter(getter(GetMfaDeviceResponse::certifications))
            .setter(setter(Builder::certifications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Certifications").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(USER_NAME_FIELD,
            SERIAL_NUMBER_FIELD, ENABLE_DATE_FIELD, CERTIFICATIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("UserName", USER_NAME_FIELD);
                    put("SerialNumber", SERIAL_NUMBER_FIELD);
                    put("EnableDate", ENABLE_DATE_FIELD);
                    put("Certifications", CERTIFICATIONS_FIELD);
                }
            });

    private final String userName;

    private final String serialNumber;

    private final Instant enableDate;

    private final Map<String, String> certifications;

    private GetMfaDeviceResponse(BuilderImpl builder) {
        super(builder);
        this.userName = builder.userName;
        this.serialNumber = builder.serialNumber;
        this.enableDate = builder.enableDate;
        this.certifications = builder.certifications;
    }

    /**
     * <p>
     * The friendly name identifying the user.
     * </p>
     * 
     * @return The friendly name identifying the user.
     */
    public final String userName() {
        return userName;
    }

    /**
     * <p>
     * Serial number that uniquely identifies the MFA device. For this API, we only accept FIDO security key <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference-arns.html">ARNs</a>.
     * </p>
     * 
     * @return Serial number that uniquely identifies the MFA device. For this API, we only accept FIDO security key <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference-arns.html">ARNs</a>.
     */
    public final String serialNumber() {
        return serialNumber;
    }

    /**
     * <p>
     * The date that a specified user's MFA device was first enabled.
     * </p>
     * 
     * @return The date that a specified user's MFA device was first enabled.
     */
    public final Instant enableDate() {
        return enableDate;
    }

    /**
     * For responses, this returns true if the service returned a value for the Certifications property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCertifications() {
        return certifications != null && !(certifications instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The certifications of a specified user's MFA device. We currently provide FIPS-140-2, FIPS-140-3, and FIDO
     * certification levels obtained from <a href="https://fidoalliance.org/metadata/"> FIDO Alliance Metadata Service
     * (MDS)</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCertifications} method.
     * </p>
     * 
     * @return The certifications of a specified user's MFA device. We currently provide FIPS-140-2, FIPS-140-3, and
     *         FIDO certification levels obtained from <a href="https://fidoalliance.org/metadata/"> FIDO Alliance
     *         Metadata Service (MDS)</a>.
     */
    public final Map<String, String> certifications() {
        return certifications;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(userName());
        hashCode = 31 * hashCode + Objects.hashCode(serialNumber());
        hashCode = 31 * hashCode + Objects.hashCode(enableDate());
        hashCode = 31 * hashCode + Objects.hashCode(hasCertifications() ? certifications() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetMfaDeviceResponse)) {
            return false;
        }
        GetMfaDeviceResponse other = (GetMfaDeviceResponse) obj;
        return Objects.equals(userName(), other.userName()) && Objects.equals(serialNumber(), other.serialNumber())
                && Objects.equals(enableDate(), other.enableDate()) && hasCertifications() == other.hasCertifications()
                && Objects.equals(certifications(), other.certifications());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GetMfaDeviceResponse").add("UserName", userName()).add("SerialNumber", serialNumber())
                .add("EnableDate", enableDate()).add("Certifications", hasCertifications() ? certifications() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "UserName":
            return Optional.ofNullable(clazz.cast(userName()));
        case "SerialNumber":
            return Optional.ofNullable(clazz.cast(serialNumber()));
        case "EnableDate":
            return Optional.ofNullable(clazz.cast(enableDate()));
        case "Certifications":
            return Optional.ofNullable(clazz.cast(certifications()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<GetMfaDeviceResponse, T> g) {
        return obj -> g.apply((GetMfaDeviceResponse) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends IamResponse.Builder, SdkPojo, CopyableBuilder<Builder, GetMfaDeviceResponse> {
        /**
         * <p>
         * The friendly name identifying the user.
         * </p>
         * 
         * @param userName
         *        The friendly name identifying the user.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userName(String userName);

        /**
         * <p>
         * Serial number that uniquely identifies the MFA device. For this API, we only accept FIDO security key <a
         * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference-arns.html">ARNs</a>.
         * </p>
         * 
         * @param serialNumber
         *        Serial number that uniquely identifies the MFA device. For this API, we only accept FIDO security key
         *        <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference-arns.html">ARNs</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serialNumber(String serialNumber);

        /**
         * <p>
         * The date that a specified user's MFA device was first enabled.
         * </p>
         * 
         * @param enableDate
         *        The date that a specified user's MFA device was first enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableDate(Instant enableDate);

        /**
         * <p>
         * The certifications of a specified user's MFA device. We currently provide FIPS-140-2, FIPS-140-3, and FIDO
         * certification levels obtained from <a href="https://fidoalliance.org/metadata/"> FIDO Alliance Metadata
         * Service (MDS)</a>.
         * </p>
         * 
         * @param certifications
         *        The certifications of a specified user's MFA device. We currently provide FIPS-140-2, FIPS-140-3, and
         *        FIDO certification levels obtained from <a href="https://fidoalliance.org/metadata/"> FIDO Alliance
         *        Metadata Service (MDS)</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certifications(Map<String, String> certifications);
    }

    static final class BuilderImpl extends IamResponse.BuilderImpl implements Builder {
        private String userName;

        private String serialNumber;

        private Instant enableDate;

        private Map<String, String> certifications = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(GetMfaDeviceResponse model) {
            super(model);
            userName(model.userName);
            serialNumber(model.serialNumber);
            enableDate(model.enableDate);
            certifications(model.certifications);
        }

        public final String getUserName() {
            return userName;
        }

        public final void setUserName(String userName) {
            this.userName = userName;
        }

        @Override
        public final Builder userName(String userName) {
            this.userName = userName;
            return this;
        }

        public final String getSerialNumber() {
            return serialNumber;
        }

        public final void setSerialNumber(String serialNumber) {
            this.serialNumber = serialNumber;
        }

        @Override
        public final Builder serialNumber(String serialNumber) {
            this.serialNumber = serialNumber;
            return this;
        }

        public final Instant getEnableDate() {
            return enableDate;
        }

        public final void setEnableDate(Instant enableDate) {
            this.enableDate = enableDate;
        }

        @Override
        public final Builder enableDate(Instant enableDate) {
            this.enableDate = enableDate;
            return this;
        }

        public final Map<String, String> getCertifications() {
            if (certifications instanceof SdkAutoConstructMap) {
                return null;
            }
            return certifications;
        }

        public final void setCertifications(Map<String, String> certifications) {
            this.certifications = CertificationMapTypeCopier.copy(certifications);
        }

        @Override
        public final Builder certifications(Map<String, String> certifications) {
            this.certifications = CertificationMapTypeCopier.copy(certifications);
            return this;
        }

        @Override
        public GetMfaDeviceResponse build() {
            return new GetMfaDeviceResponse(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
