/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iam.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about an SSH public key, without the key's body or fingerprint.
 * </p>
 * <p>
 * This data type is used as a response element in the <a>ListSSHPublicKeys</a> operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SSHPublicKeyMetadata implements SdkPojo, Serializable,
        ToCopyableBuilder<SSHPublicKeyMetadata.Builder, SSHPublicKeyMetadata> {
    private static final SdkField<String> USER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UserName").getter(getter(SSHPublicKeyMetadata::userName)).setter(setter(Builder::userName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserName").build()).build();

    private static final SdkField<String> SSH_PUBLIC_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SSHPublicKeyId").getter(getter(SSHPublicKeyMetadata::sshPublicKeyId))
            .setter(setter(Builder::sshPublicKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SSHPublicKeyId").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(SSHPublicKeyMetadata::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Instant> UPLOAD_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("UploadDate").getter(getter(SSHPublicKeyMetadata::uploadDate)).setter(setter(Builder::uploadDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UploadDate").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(USER_NAME_FIELD,
            SSH_PUBLIC_KEY_ID_FIELD, STATUS_FIELD, UPLOAD_DATE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("UserName", USER_NAME_FIELD);
                    put("SSHPublicKeyId", SSH_PUBLIC_KEY_ID_FIELD);
                    put("Status", STATUS_FIELD);
                    put("UploadDate", UPLOAD_DATE_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String userName;

    private final String sshPublicKeyId;

    private final String status;

    private final Instant uploadDate;

    private SSHPublicKeyMetadata(BuilderImpl builder) {
        this.userName = builder.userName;
        this.sshPublicKeyId = builder.sshPublicKeyId;
        this.status = builder.status;
        this.uploadDate = builder.uploadDate;
    }

    /**
     * <p>
     * The name of the IAM user associated with the SSH public key.
     * </p>
     * 
     * @return The name of the IAM user associated with the SSH public key.
     */
    public final String userName() {
        return userName;
    }

    /**
     * <p>
     * The unique identifier for the SSH public key.
     * </p>
     * 
     * @return The unique identifier for the SSH public key.
     */
    public final String sshPublicKeyId() {
        return sshPublicKeyId;
    }

    /**
     * <p>
     * The status of the SSH public key. <code>Active</code> means that the key can be used for authentication with an
     * CodeCommit repository. <code>Inactive</code> means that the key cannot be used.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link StatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the SSH public key. <code>Active</code> means that the key can be used for authentication
     *         with an CodeCommit repository. <code>Inactive</code> means that the key cannot be used.
     * @see StatusType
     */
    public final StatusType status() {
        return StatusType.fromValue(status);
    }

    /**
     * <p>
     * The status of the SSH public key. <code>Active</code> means that the key can be used for authentication with an
     * CodeCommit repository. <code>Inactive</code> means that the key cannot be used.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link StatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the SSH public key. <code>Active</code> means that the key can be used for authentication
     *         with an CodeCommit repository. <code>Inactive</code> means that the key cannot be used.
     * @see StatusType
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The date and time, in <a href="http://www.iso.org/iso/iso8601">ISO 8601 date-time format</a>, when the SSH public
     * key was uploaded.
     * </p>
     * 
     * @return The date and time, in <a href="http://www.iso.org/iso/iso8601">ISO 8601 date-time format</a>, when the
     *         SSH public key was uploaded.
     */
    public final Instant uploadDate() {
        return uploadDate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(userName());
        hashCode = 31 * hashCode + Objects.hashCode(sshPublicKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(uploadDate());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SSHPublicKeyMetadata)) {
            return false;
        }
        SSHPublicKeyMetadata other = (SSHPublicKeyMetadata) obj;
        return Objects.equals(userName(), other.userName()) && Objects.equals(sshPublicKeyId(), other.sshPublicKeyId())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(uploadDate(), other.uploadDate());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SSHPublicKeyMetadata").add("UserName", userName()).add("SSHPublicKeyId", sshPublicKeyId())
                .add("Status", statusAsString()).add("UploadDate", uploadDate()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "UserName":
            return Optional.ofNullable(clazz.cast(userName()));
        case "SSHPublicKeyId":
            return Optional.ofNullable(clazz.cast(sshPublicKeyId()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "UploadDate":
            return Optional.ofNullable(clazz.cast(uploadDate()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<SSHPublicKeyMetadata, T> g) {
        return obj -> g.apply((SSHPublicKeyMetadata) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SSHPublicKeyMetadata> {
        /**
         * <p>
         * The name of the IAM user associated with the SSH public key.
         * </p>
         * 
         * @param userName
         *        The name of the IAM user associated with the SSH public key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userName(String userName);

        /**
         * <p>
         * The unique identifier for the SSH public key.
         * </p>
         * 
         * @param sshPublicKeyId
         *        The unique identifier for the SSH public key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sshPublicKeyId(String sshPublicKeyId);

        /**
         * <p>
         * The status of the SSH public key. <code>Active</code> means that the key can be used for authentication with
         * an CodeCommit repository. <code>Inactive</code> means that the key cannot be used.
         * </p>
         * 
         * @param status
         *        The status of the SSH public key. <code>Active</code> means that the key can be used for
         *        authentication with an CodeCommit repository. <code>Inactive</code> means that the key cannot be used.
         * @see StatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StatusType
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the SSH public key. <code>Active</code> means that the key can be used for authentication with
         * an CodeCommit repository. <code>Inactive</code> means that the key cannot be used.
         * </p>
         * 
         * @param status
         *        The status of the SSH public key. <code>Active</code> means that the key can be used for
         *        authentication with an CodeCommit repository. <code>Inactive</code> means that the key cannot be used.
         * @see StatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StatusType
         */
        Builder status(StatusType status);

        /**
         * <p>
         * The date and time, in <a href="http://www.iso.org/iso/iso8601">ISO 8601 date-time format</a>, when the SSH
         * public key was uploaded.
         * </p>
         * 
         * @param uploadDate
         *        The date and time, in <a href="http://www.iso.org/iso/iso8601">ISO 8601 date-time format</a>, when the
         *        SSH public key was uploaded.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder uploadDate(Instant uploadDate);
    }

    static final class BuilderImpl implements Builder {
        private String userName;

        private String sshPublicKeyId;

        private String status;

        private Instant uploadDate;

        private BuilderImpl() {
        }

        private BuilderImpl(SSHPublicKeyMetadata model) {
            userName(model.userName);
            sshPublicKeyId(model.sshPublicKeyId);
            status(model.status);
            uploadDate(model.uploadDate);
        }

        public final String getUserName() {
            return userName;
        }

        public final void setUserName(String userName) {
            this.userName = userName;
        }

        @Override
        public final Builder userName(String userName) {
            this.userName = userName;
            return this;
        }

        public final String getSshPublicKeyId() {
            return sshPublicKeyId;
        }

        public final void setSshPublicKeyId(String sshPublicKeyId) {
            this.sshPublicKeyId = sshPublicKeyId;
        }

        @Override
        public final Builder sshPublicKeyId(String sshPublicKeyId) {
            this.sshPublicKeyId = sshPublicKeyId;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(StatusType status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getUploadDate() {
            return uploadDate;
        }

        public final void setUploadDate(Instant uploadDate) {
            this.uploadDate = uploadDate;
        }

        @Override
        public final Builder uploadDate(Instant uploadDate) {
            this.uploadDate = uploadDate;
            return this;
        }

        @Override
        public SSHPublicKeyMetadata build() {
            return new SSHPublicKeyMetadata(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
