/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iam.waiters;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Immutable;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.iam.IamAsyncClient;
import software.amazon.awssdk.services.iam.model.GetInstanceProfileRequest;
import software.amazon.awssdk.services.iam.model.GetInstanceProfileResponse;
import software.amazon.awssdk.services.iam.model.GetPolicyRequest;
import software.amazon.awssdk.services.iam.model.GetPolicyResponse;
import software.amazon.awssdk.services.iam.model.GetRoleRequest;
import software.amazon.awssdk.services.iam.model.GetRoleResponse;
import software.amazon.awssdk.services.iam.model.GetUserRequest;
import software.amazon.awssdk.services.iam.model.GetUserResponse;
import software.amazon.awssdk.utils.SdkAutoCloseable;

/**
 * Waiter utility class that polls a resource until a desired state is reached or until it is determined that the
 * resource will never enter into the desired state. This can be created using the static {@link #builder()} method
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
@Immutable
public interface IamAsyncWaiter extends SdkAutoCloseable {
    /**
     * Polls {@link IamAsyncClient#getInstanceProfile} API until the desired condition {@code InstanceProfileExists} is
     * met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getInstanceProfileRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<GetInstanceProfileResponse>> waitUntilInstanceProfileExists(
            GetInstanceProfileRequest getInstanceProfileRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link IamAsyncClient#getInstanceProfile} API until the desired condition {@code InstanceProfileExists} is
     * met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetInstanceProfileRequest#builder()}
     *
     * @param getInstanceProfileRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<GetInstanceProfileResponse>> waitUntilInstanceProfileExists(
            Consumer<GetInstanceProfileRequest.Builder> getInstanceProfileRequest) {
        return waitUntilInstanceProfileExists(GetInstanceProfileRequest.builder().applyMutation(getInstanceProfileRequest)
                .build());
    }

    /**
     * Polls {@link IamAsyncClient#getInstanceProfile} API until the desired condition {@code InstanceProfileExists} is
     * met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getInstanceProfileRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetInstanceProfileResponse>> waitUntilInstanceProfileExists(
            GetInstanceProfileRequest getInstanceProfileRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link IamAsyncClient#getInstanceProfile} API until the desired condition {@code InstanceProfileExists} is
     * met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getInstanceProfileRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetInstanceProfileResponse>> waitUntilInstanceProfileExists(
            Consumer<GetInstanceProfileRequest.Builder> getInstanceProfileRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilInstanceProfileExists(GetInstanceProfileRequest.builder().applyMutation(getInstanceProfileRequest)
                .build(), WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link IamAsyncClient#getPolicy} API until the desired condition {@code PolicyExists} is met, or until it
     * is determined that the resource will never enter into the desired state
     *
     * @param getPolicyRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<GetPolicyResponse>> waitUntilPolicyExists(GetPolicyRequest getPolicyRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link IamAsyncClient#getPolicy} API until the desired condition {@code PolicyExists} is met, or until it
     * is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetPolicyRequest#builder()}
     *
     * @param getPolicyRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<GetPolicyResponse>> waitUntilPolicyExists(
            Consumer<GetPolicyRequest.Builder> getPolicyRequest) {
        return waitUntilPolicyExists(GetPolicyRequest.builder().applyMutation(getPolicyRequest).build());
    }

    /**
     * Polls {@link IamAsyncClient#getPolicy} API until the desired condition {@code PolicyExists} is met, or until it
     * is determined that the resource will never enter into the desired state
     *
     * @param getPolicyRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetPolicyResponse>> waitUntilPolicyExists(GetPolicyRequest getPolicyRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link IamAsyncClient#getPolicy} API until the desired condition {@code PolicyExists} is met, or until it
     * is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getPolicyRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetPolicyResponse>> waitUntilPolicyExists(
            Consumer<GetPolicyRequest.Builder> getPolicyRequest, Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilPolicyExists(GetPolicyRequest.builder().applyMutation(getPolicyRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link IamAsyncClient#getRole} API until the desired condition {@code RoleExists} is met, or until it is
     * determined that the resource will never enter into the desired state
     *
     * @param getRoleRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<GetRoleResponse>> waitUntilRoleExists(GetRoleRequest getRoleRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link IamAsyncClient#getRole} API until the desired condition {@code RoleExists} is met, or until it is
     * determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetRoleRequest#builder()}
     *
     * @param getRoleRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<GetRoleResponse>> waitUntilRoleExists(Consumer<GetRoleRequest.Builder> getRoleRequest) {
        return waitUntilRoleExists(GetRoleRequest.builder().applyMutation(getRoleRequest).build());
    }

    /**
     * Polls {@link IamAsyncClient#getRole} API until the desired condition {@code RoleExists} is met, or until it is
     * determined that the resource will never enter into the desired state
     *
     * @param getRoleRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetRoleResponse>> waitUntilRoleExists(GetRoleRequest getRoleRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link IamAsyncClient#getRole} API until the desired condition {@code RoleExists} is met, or until it is
     * determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getRoleRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetRoleResponse>> waitUntilRoleExists(
            Consumer<GetRoleRequest.Builder> getRoleRequest, Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilRoleExists(GetRoleRequest.builder().applyMutation(getRoleRequest).build(), WaiterOverrideConfiguration
                .builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link IamAsyncClient#getUser} API until the desired condition {@code UserExists} is met, or until it is
     * determined that the resource will never enter into the desired state
     *
     * @param getUserRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<GetUserResponse>> waitUntilUserExists(GetUserRequest getUserRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link IamAsyncClient#getUser} API until the desired condition {@code UserExists} is met, or until it is
     * determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetUserRequest#builder()}
     *
     * @param getUserRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<GetUserResponse>> waitUntilUserExists(Consumer<GetUserRequest.Builder> getUserRequest) {
        return waitUntilUserExists(GetUserRequest.builder().applyMutation(getUserRequest).build());
    }

    /**
     * Polls {@link IamAsyncClient#getUser} API until the desired condition {@code UserExists} is met, or until it is
     * determined that the resource will never enter into the desired state
     *
     * @param getUserRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetUserResponse>> waitUntilUserExists(GetUserRequest getUserRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link IamAsyncClient#getUser} API until the desired condition {@code UserExists} is met, or until it is
     * determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getUserRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetUserResponse>> waitUntilUserExists(
            Consumer<GetUserRequest.Builder> getUserRequest, Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilUserExists(GetUserRequest.builder().applyMutation(getUserRequest).build(), WaiterOverrideConfiguration
                .builder().applyMutation(overrideConfig).build());
    }

    /**
     * Create a builder that can be used to configure and create a {@link IamAsyncWaiter}.
     *
     * @return a builder
     */
    static Builder builder() {
        return DefaultIamAsyncWaiter.builder();
    }

    /**
     * Create an instance of {@link IamAsyncWaiter} with the default configuration.
     * <p>
     * <b>A default {@link IamAsyncClient} will be created to poll resources. It is recommended to share a single
     * instance of the waiter created via this method. If it is not desirable to share a waiter instance, invoke
     * {@link #close()} to release the resources once the waiter is not needed.</b>
     *
     * @return an instance of {@link IamAsyncWaiter}
     */
    static IamAsyncWaiter create() {
        return DefaultIamAsyncWaiter.builder().build();
    }

    interface Builder {
        /**
         * Sets a custom {@link ScheduledExecutorService} that will be used to schedule async polling attempts
         * <p>
         * This executorService must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * executorService when the waiter is closed
         *
         * @param executorService
         *        the executorService to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder scheduledExecutorService(ScheduledExecutorService executorService);

        /**
         * Defines overrides to the default SDK waiter configuration that should be used for waiters created from this
         * builder
         *
         * @param overrideConfiguration
         *        the override configuration to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration);

        /**
         * This is a convenient method to pass the override configuration without the need to create an instance
         * manually via {@link WaiterOverrideConfiguration#builder()}
         *
         * @param overrideConfiguration
         *        The consumer that will configure the overrideConfiguration
         * @return a reference to this object so that method calls can be chained together.
         * @see #overrideConfiguration(WaiterOverrideConfiguration)
         */
        default Builder overrideConfiguration(Consumer<WaiterOverrideConfiguration.Builder> overrideConfiguration) {
            WaiterOverrideConfiguration.Builder builder = WaiterOverrideConfiguration.builder();
            overrideConfiguration.accept(builder);
            return overrideConfiguration(builder.build());
        }

        /**
         * Sets a custom {@link IamAsyncClient} that will be used to poll the resource
         * <p>
         * This SDK client must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * client when the waiter is closed
         *
         * @param client
         *        the client to send the request
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder client(IamAsyncClient client);

        /**
         * Builds an instance of {@link IamAsyncWaiter} based on the configurations supplied to this builder
         *
         * @return An initialized {@link IamAsyncWaiter}
         */
        IamAsyncWaiter build();
    }
}
