/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.identitystore;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.identitystore.model.AccessDeniedException;
import software.amazon.awssdk.services.identitystore.model.DescribeGroupRequest;
import software.amazon.awssdk.services.identitystore.model.DescribeGroupResponse;
import software.amazon.awssdk.services.identitystore.model.DescribeUserRequest;
import software.amazon.awssdk.services.identitystore.model.DescribeUserResponse;
import software.amazon.awssdk.services.identitystore.model.IdentitystoreException;
import software.amazon.awssdk.services.identitystore.model.IdentitystoreRequest;
import software.amazon.awssdk.services.identitystore.model.InternalServerException;
import software.amazon.awssdk.services.identitystore.model.ListGroupsRequest;
import software.amazon.awssdk.services.identitystore.model.ListGroupsResponse;
import software.amazon.awssdk.services.identitystore.model.ListUsersRequest;
import software.amazon.awssdk.services.identitystore.model.ListUsersResponse;
import software.amazon.awssdk.services.identitystore.model.ResourceNotFoundException;
import software.amazon.awssdk.services.identitystore.model.ThrottlingException;
import software.amazon.awssdk.services.identitystore.model.ValidationException;
import software.amazon.awssdk.services.identitystore.paginators.ListGroupsIterable;
import software.amazon.awssdk.services.identitystore.paginators.ListUsersIterable;
import software.amazon.awssdk.services.identitystore.transform.DescribeGroupRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.DescribeUserRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.ListGroupsRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.ListUsersRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link IdentitystoreClient}.
 *
 * @see IdentitystoreClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIdentitystoreClient implements IdentitystoreClient {
    private static final Logger log = Logger.loggerFor(DefaultIdentitystoreClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultIdentitystoreClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Retrieves the group metadata and attributes from <code>GroupId</code> in an identity store.
     * </p>
     *
     * @param describeGroupRequest
     * @return Result of the DescribeGroup operation returned by the service.
     * @throws ResourceNotFoundException
     *         Indicates that a requested resource is not found.
     * @throws ValidationException
     *         The request failed because it contains a syntax error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Indicates that the principal has crossed the throttling limits of the API operations.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IdentitystoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IdentitystoreClient.DescribeGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/DescribeGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeGroupResponse describeGroup(DescribeGroupRequest describeGroupRequest) throws ResourceNotFoundException,
            ValidationException, AccessDeniedException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, IdentitystoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGroup");

            return clientHandler.execute(new ClientExecutionParams<DescribeGroupRequest, DescribeGroupResponse>()
                    .withOperationName("DescribeGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the user metadata and attributes from <code>UserId</code> in an identity store.
     * </p>
     *
     * @param describeUserRequest
     * @return Result of the DescribeUser operation returned by the service.
     * @throws ResourceNotFoundException
     *         Indicates that a requested resource is not found.
     * @throws ValidationException
     *         The request failed because it contains a syntax error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Indicates that the principal has crossed the throttling limits of the API operations.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IdentitystoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IdentitystoreClient.DescribeUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/DescribeUser" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeUserResponse describeUser(DescribeUserRequest describeUserRequest) throws ResourceNotFoundException,
            ValidationException, AccessDeniedException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, IdentitystoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeUser");

            return clientHandler.execute(new ClientExecutionParams<DescribeUserRequest, DescribeUserResponse>()
                    .withOperationName("DescribeUser").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeUserRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the attribute name and value of the group that you specified in the search. We only support
     * <code>DisplayName</code> as a valid filter attribute path currently, and filter is required. This API returns
     * minimum attributes, including <code>GroupId</code> and group <code>DisplayName</code> in the response.
     * </p>
     *
     * @param listGroupsRequest
     * @return Result of the ListGroups operation returned by the service.
     * @throws ValidationException
     *         The request failed because it contains a syntax error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         Indicates that a requested resource is not found.
     * @throws ThrottlingException
     *         Indicates that the principal has crossed the throttling limits of the API operations.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IdentitystoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IdentitystoreClient.ListGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/ListGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListGroupsResponse listGroups(ListGroupsRequest listGroupsRequest) throws ValidationException, AccessDeniedException,
            ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException, SdkClientException,
            IdentitystoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListGroupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGroups");

            return clientHandler
                    .execute(new ClientExecutionParams<ListGroupsRequest, ListGroupsResponse>().withOperationName("ListGroups")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listGroupsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the attribute name and value of the group that you specified in the search. We only support
     * <code>DisplayName</code> as a valid filter attribute path currently, and filter is required. This API returns
     * minimum attributes, including <code>GroupId</code> and group <code>DisplayName</code> in the response.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listGroups(software.amazon.awssdk.services.identitystore.model.ListGroupsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.identitystore.paginators.ListGroupsIterable responses = client.listGroupsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.identitystore.paginators.ListGroupsIterable responses = client.listGroupsPaginator(request);
     *     for (software.amazon.awssdk.services.identitystore.model.ListGroupsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.identitystore.paginators.ListGroupsIterable responses = client.listGroupsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGroups(software.amazon.awssdk.services.identitystore.model.ListGroupsRequest)} operation.</b>
     * </p>
     *
     * @param listGroupsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The request failed because it contains a syntax error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         Indicates that a requested resource is not found.
     * @throws ThrottlingException
     *         Indicates that the principal has crossed the throttling limits of the API operations.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IdentitystoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IdentitystoreClient.ListGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/ListGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListGroupsIterable listGroupsPaginator(ListGroupsRequest listGroupsRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, IdentitystoreException {
        return new ListGroupsIterable(this, applyPaginatorUserAgent(listGroupsRequest));
    }

    /**
     * <p>
     * Lists the attribute name and value of the user that you specified in the search. We only support
     * <code>UserName</code> as a valid filter attribute path currently, and filter is required. This API returns
     * minimum attributes, including <code>UserId</code> and <code>UserName</code> in the response.
     * </p>
     *
     * @param listUsersRequest
     * @return Result of the ListUsers operation returned by the service.
     * @throws ValidationException
     *         The request failed because it contains a syntax error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         Indicates that a requested resource is not found.
     * @throws ThrottlingException
     *         Indicates that the principal has crossed the throttling limits of the API operations.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IdentitystoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IdentitystoreClient.ListUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListUsersResponse listUsers(ListUsersRequest listUsersRequest) throws ValidationException, AccessDeniedException,
            ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException, SdkClientException,
            IdentitystoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListUsersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListUsersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listUsersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListUsers");

            return clientHandler.execute(new ClientExecutionParams<ListUsersRequest, ListUsersResponse>()
                    .withOperationName("ListUsers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listUsersRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ListUsersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the attribute name and value of the user that you specified in the search. We only support
     * <code>UserName</code> as a valid filter attribute path currently, and filter is required. This API returns
     * minimum attributes, including <code>UserId</code> and <code>UserName</code> in the response.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listUsers(software.amazon.awssdk.services.identitystore.model.ListUsersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.identitystore.paginators.ListUsersIterable responses = client.listUsersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.identitystore.paginators.ListUsersIterable responses = client.listUsersPaginator(request);
     *     for (software.amazon.awssdk.services.identitystore.model.ListUsersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.identitystore.paginators.ListUsersIterable responses = client.listUsersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listUsers(software.amazon.awssdk.services.identitystore.model.ListUsersRequest)} operation.</b>
     * </p>
     *
     * @param listUsersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The request failed because it contains a syntax error.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         Indicates that a requested resource is not found.
     * @throws ThrottlingException
     *         Indicates that the principal has crossed the throttling limits of the API operations.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IdentitystoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IdentitystoreClient.ListUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListUsersIterable listUsersPaginator(ListUsersRequest listUsersRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, IdentitystoreException {
        return new ListUsersIterable(this, applyPaginatorUserAgent(listUsersRequest));
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IdentitystoreException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends IdentitystoreRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
