/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.identitystore;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.identitystore.model.AccessDeniedException;
import software.amazon.awssdk.services.identitystore.model.DescribeGroupRequest;
import software.amazon.awssdk.services.identitystore.model.DescribeGroupResponse;
import software.amazon.awssdk.services.identitystore.model.DescribeUserRequest;
import software.amazon.awssdk.services.identitystore.model.DescribeUserResponse;
import software.amazon.awssdk.services.identitystore.model.IdentitystoreException;
import software.amazon.awssdk.services.identitystore.model.IdentitystoreRequest;
import software.amazon.awssdk.services.identitystore.model.InternalServerException;
import software.amazon.awssdk.services.identitystore.model.ListGroupsRequest;
import software.amazon.awssdk.services.identitystore.model.ListGroupsResponse;
import software.amazon.awssdk.services.identitystore.model.ListUsersRequest;
import software.amazon.awssdk.services.identitystore.model.ListUsersResponse;
import software.amazon.awssdk.services.identitystore.model.ResourceNotFoundException;
import software.amazon.awssdk.services.identitystore.model.ThrottlingException;
import software.amazon.awssdk.services.identitystore.model.ValidationException;
import software.amazon.awssdk.services.identitystore.paginators.ListGroupsPublisher;
import software.amazon.awssdk.services.identitystore.paginators.ListUsersPublisher;
import software.amazon.awssdk.services.identitystore.transform.DescribeGroupRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.DescribeUserRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.ListGroupsRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.ListUsersRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link IdentitystoreAsyncClient}.
 *
 * @see IdentitystoreAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIdentitystoreAsyncClient implements IdentitystoreAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultIdentitystoreAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultIdentitystoreAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Retrieves the group metadata and attributes from <code>GroupId</code> in an identity store.
     * </p>
     *
     * @param describeGroupRequest
     * @return A Java Future containing the result of the DescribeGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.DescribeGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/DescribeGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeGroupResponse> describeGroup(DescribeGroupRequest describeGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeGroupRequest, DescribeGroupResponse>()
                            .withOperationName("DescribeGroup")
                            .withMarshaller(new DescribeGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeGroupRequest));
            CompletableFuture<DescribeGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the user metadata and attributes from <code>UserId</code> in an identity store.
     * </p>
     *
     * @param describeUserRequest
     * @return A Java Future containing the result of the DescribeUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.DescribeUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/DescribeUser" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeUserResponse> describeUser(DescribeUserRequest describeUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeUserRequest, DescribeUserResponse>()
                            .withOperationName("DescribeUser").withMarshaller(new DescribeUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeUserRequest));
            CompletableFuture<DescribeUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the attribute name and value of the group that you specified in the search. We only support
     * <code>DisplayName</code> as a valid filter attribute path currently, and filter is required. This API returns
     * minimum attributes, including <code>GroupId</code> and group <code>DisplayName</code> in the response.
     * </p>
     *
     * @param listGroupsRequest
     * @return A Java Future containing the result of the ListGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.ListGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/ListGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListGroupsResponse> listGroups(ListGroupsRequest listGroupsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGroupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGroupsRequest, ListGroupsResponse>().withOperationName("ListGroups")
                            .withMarshaller(new ListGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listGroupsRequest));
            CompletableFuture<ListGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the attribute name and value of the group that you specified in the search. We only support
     * <code>DisplayName</code> as a valid filter attribute path currently, and filter is required. This API returns
     * minimum attributes, including <code>GroupId</code> and group <code>DisplayName</code> in the response.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listGroups(software.amazon.awssdk.services.identitystore.model.ListGroupsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.identitystore.paginators.ListGroupsPublisher publisher = client.listGroupsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.identitystore.paginators.ListGroupsPublisher publisher = client.listGroupsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.identitystore.model.ListGroupsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.identitystore.model.ListGroupsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGroups(software.amazon.awssdk.services.identitystore.model.ListGroupsRequest)} operation.</b>
     * </p>
     *
     * @param listGroupsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.ListGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/ListGroups" target="_top">AWS API
     *      Documentation</a>
     */
    public ListGroupsPublisher listGroupsPaginator(ListGroupsRequest listGroupsRequest) {
        return new ListGroupsPublisher(this, applyPaginatorUserAgent(listGroupsRequest));
    }

    /**
     * <p>
     * Lists the attribute name and value of the user that you specified in the search. We only support
     * <code>UserName</code> as a valid filter attribute path currently, and filter is required. This API returns
     * minimum attributes, including <code>UserId</code> and <code>UserName</code> in the response.
     * </p>
     *
     * @param listUsersRequest
     * @return A Java Future containing the result of the ListUsers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.ListUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListUsersResponse> listUsers(ListUsersRequest listUsersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listUsersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListUsers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListUsersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListUsersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListUsersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListUsersRequest, ListUsersResponse>().withOperationName("ListUsers")
                            .withMarshaller(new ListUsersRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(listUsersRequest));
            CompletableFuture<ListUsersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the attribute name and value of the user that you specified in the search. We only support
     * <code>UserName</code> as a valid filter attribute path currently, and filter is required. This API returns
     * minimum attributes, including <code>UserId</code> and <code>UserName</code> in the response.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listUsers(software.amazon.awssdk.services.identitystore.model.ListUsersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.identitystore.paginators.ListUsersPublisher publisher = client.listUsersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.identitystore.paginators.ListUsersPublisher publisher = client.listUsersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.identitystore.model.ListUsersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.identitystore.model.ListUsersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listUsers(software.amazon.awssdk.services.identitystore.model.ListUsersRequest)} operation.</b>
     * </p>
     *
     * @param listUsersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request failed because it contains a syntax error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException Indicates that a requested resource is not found.</li>
     *         <li>ThrottlingException Indicates that the principal has crossed the throttling limits of the API
     *         operations.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure with an internal server.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IdentitystoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IdentitystoreAsyncClient.ListUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    public ListUsersPublisher listUsersPaginator(ListUsersRequest listUsersRequest) {
        return new ListUsersPublisher(this, applyPaginatorUserAgent(listUsersRequest));
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IdentitystoreException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends IdentitystoreRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
