/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.identitystore;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.identitystore.internal.IdentitystoreServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.identitystore.model.AccessDeniedException;
import software.amazon.awssdk.services.identitystore.model.ConflictException;
import software.amazon.awssdk.services.identitystore.model.CreateGroupMembershipRequest;
import software.amazon.awssdk.services.identitystore.model.CreateGroupMembershipResponse;
import software.amazon.awssdk.services.identitystore.model.CreateGroupRequest;
import software.amazon.awssdk.services.identitystore.model.CreateGroupResponse;
import software.amazon.awssdk.services.identitystore.model.CreateUserRequest;
import software.amazon.awssdk.services.identitystore.model.CreateUserResponse;
import software.amazon.awssdk.services.identitystore.model.DeleteGroupMembershipRequest;
import software.amazon.awssdk.services.identitystore.model.DeleteGroupMembershipResponse;
import software.amazon.awssdk.services.identitystore.model.DeleteGroupRequest;
import software.amazon.awssdk.services.identitystore.model.DeleteGroupResponse;
import software.amazon.awssdk.services.identitystore.model.DeleteUserRequest;
import software.amazon.awssdk.services.identitystore.model.DeleteUserResponse;
import software.amazon.awssdk.services.identitystore.model.DescribeGroupMembershipRequest;
import software.amazon.awssdk.services.identitystore.model.DescribeGroupMembershipResponse;
import software.amazon.awssdk.services.identitystore.model.DescribeGroupRequest;
import software.amazon.awssdk.services.identitystore.model.DescribeGroupResponse;
import software.amazon.awssdk.services.identitystore.model.DescribeUserRequest;
import software.amazon.awssdk.services.identitystore.model.DescribeUserResponse;
import software.amazon.awssdk.services.identitystore.model.GetGroupIdRequest;
import software.amazon.awssdk.services.identitystore.model.GetGroupIdResponse;
import software.amazon.awssdk.services.identitystore.model.GetGroupMembershipIdRequest;
import software.amazon.awssdk.services.identitystore.model.GetGroupMembershipIdResponse;
import software.amazon.awssdk.services.identitystore.model.GetUserIdRequest;
import software.amazon.awssdk.services.identitystore.model.GetUserIdResponse;
import software.amazon.awssdk.services.identitystore.model.IdentitystoreException;
import software.amazon.awssdk.services.identitystore.model.InternalServerException;
import software.amazon.awssdk.services.identitystore.model.IsMemberInGroupsRequest;
import software.amazon.awssdk.services.identitystore.model.IsMemberInGroupsResponse;
import software.amazon.awssdk.services.identitystore.model.ListGroupMembershipsForMemberRequest;
import software.amazon.awssdk.services.identitystore.model.ListGroupMembershipsForMemberResponse;
import software.amazon.awssdk.services.identitystore.model.ListGroupMembershipsRequest;
import software.amazon.awssdk.services.identitystore.model.ListGroupMembershipsResponse;
import software.amazon.awssdk.services.identitystore.model.ListGroupsRequest;
import software.amazon.awssdk.services.identitystore.model.ListGroupsResponse;
import software.amazon.awssdk.services.identitystore.model.ListUsersRequest;
import software.amazon.awssdk.services.identitystore.model.ListUsersResponse;
import software.amazon.awssdk.services.identitystore.model.ResourceNotFoundException;
import software.amazon.awssdk.services.identitystore.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.identitystore.model.ThrottlingException;
import software.amazon.awssdk.services.identitystore.model.UpdateGroupRequest;
import software.amazon.awssdk.services.identitystore.model.UpdateGroupResponse;
import software.amazon.awssdk.services.identitystore.model.UpdateUserRequest;
import software.amazon.awssdk.services.identitystore.model.UpdateUserResponse;
import software.amazon.awssdk.services.identitystore.model.ValidationException;
import software.amazon.awssdk.services.identitystore.transform.CreateGroupMembershipRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.CreateGroupRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.CreateUserRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.DeleteGroupMembershipRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.DeleteGroupRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.DeleteUserRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.DescribeGroupMembershipRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.DescribeGroupRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.DescribeUserRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.GetGroupIdRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.GetGroupMembershipIdRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.GetUserIdRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.IsMemberInGroupsRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.ListGroupMembershipsForMemberRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.ListGroupMembershipsRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.ListGroupsRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.ListUsersRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.UpdateGroupRequestMarshaller;
import software.amazon.awssdk.services.identitystore.transform.UpdateUserRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link IdentitystoreClient}.
 *
 * @see IdentitystoreClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIdentitystoreClient implements IdentitystoreClient {
    private static final Logger log = Logger.loggerFor(DefaultIdentitystoreClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final IdentitystoreServiceClientConfiguration serviceClientConfiguration;

    protected DefaultIdentitystoreClient(IdentitystoreServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a group within the specified identity store.
     * </p>
     *
     * @param createGroupRequest
     * @return Result of the CreateGroup operation returned by the service.
     * @throws ResourceNotFoundException
     *         Indicates that a requested resource is not found.
     * @throws ThrottlingException
     *         Indicates that the principal has crossed the throttling limits of the API operations.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConflictException
     *         This request cannot be completed for one of the following reasons:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Performing the requested operation would violate an existing uniqueness claim in the identity store.
     *         Resolve the conflict before retrying this request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The requested resource was being concurrently modified by another request.
     *         </p>
     *         </li>
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         The request failed because it contains a syntax error.
     * @throws ServiceQuotaExceededException
     *         The request would cause the number of users or groups in the identity store to exceed the maximum
     *         allowed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IdentitystoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IdentitystoreClient.CreateGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/CreateGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateGroupResponse createGroup(CreateGroupRequest createGroupRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ConflictException, InternalServerException, ValidationException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, IdentitystoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createGroupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGroup");

            return clientHandler.execute(new ClientExecutionParams<CreateGroupRequest, CreateGroupResponse>()
                    .withOperationName("CreateGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createGroupRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a relationship between a member and a group. The following identifiers must be specified:
     * <code>GroupId</code>, <code>IdentityStoreId</code>, and <code>MemberId</code>.
     * </p>
     *
     * @param createGroupMembershipRequest
     * @return Result of the CreateGroupMembership operation returned by the service.
     * @throws ResourceNotFoundException
     *         Indicates that a requested resource is not found.
     * @throws ThrottlingException
     *         Indicates that the principal has crossed the throttling limits of the API operations.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConflictException
     *         This request cannot be completed for one of the following reasons:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Performing the requested operation would violate an existing uniqueness claim in the identity store.
     *         Resolve the conflict before retrying this request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The requested resource was being concurrently modified by another request.
     *         </p>
     *         </li>
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         The request failed because it contains a syntax error.
     * @throws ServiceQuotaExceededException
     *         The request would cause the number of users or groups in the identity store to exceed the maximum
     *         allowed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IdentitystoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IdentitystoreClient.CreateGroupMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/CreateGroupMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateGroupMembershipResponse createGroupMembership(CreateGroupMembershipRequest createGroupMembershipRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, ConflictException,
            InternalServerException, ValidationException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            IdentitystoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateGroupMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateGroupMembershipResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createGroupMembershipRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGroupMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGroupMembership");

            return clientHandler.execute(new ClientExecutionParams<CreateGroupMembershipRequest, CreateGroupMembershipResponse>()
                    .withOperationName("CreateGroupMembership").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createGroupMembershipRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateGroupMembershipRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a user within the specified identity store.
     * </p>
     *
     * @param createUserRequest
     * @return Result of the CreateUser operation returned by the service.
     * @throws ResourceNotFoundException
     *         Indicates that a requested resource is not found.
     * @throws ThrottlingException
     *         Indicates that the principal has crossed the throttling limits of the API operations.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConflictException
     *         This request cannot be completed for one of the following reasons:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Performing the requested operation would violate an existing uniqueness claim in the identity store.
     *         Resolve the conflict before retrying this request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The requested resource was being concurrently modified by another request.
     *         </p>
     *         </li>
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         The request failed because it contains a syntax error.
     * @throws ServiceQuotaExceededException
     *         The request would cause the number of users or groups in the identity store to exceed the maximum
     *         allowed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IdentitystoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IdentitystoreClient.CreateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/CreateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateUserResponse createUser(CreateUserRequest createUserRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ConflictException, InternalServerException, ValidationException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, IdentitystoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateUser");

            return clientHandler.execute(new ClientExecutionParams<CreateUserRequest, CreateUserResponse>()
                    .withOperationName("CreateUser").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createUserRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delete a group within an identity store given <code>GroupId</code>.
     * </p>
     *
     * @param deleteGroupRequest
     * @return Result of the DeleteGroup operation returned by the service.
     * @throws ResourceNotFoundException
     *         Indicates that a requested resource is not found.
     * @throws ThrottlingException
     *         Indicates that the principal has crossed the throttling limits of the API operations.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConflictException
     *         This request cannot be completed for one of the following reasons:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Performing the requested operation would violate an existing uniqueness claim in the identity store.
     *         Resolve the conflict before retrying this request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The requested resource was being concurrently modified by another request.
     *         </p>
     *         </li>
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         The request failed because it contains a syntax error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IdentitystoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IdentitystoreClient.DeleteGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/DeleteGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteGroupResponse deleteGroup(DeleteGroupRequest deleteGroupRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ConflictException, InternalServerException, ValidationException,
            AwsServiceException, SdkClientException, IdentitystoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteGroupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGroup");

            return clientHandler.execute(new ClientExecutionParams<DeleteGroupRequest, DeleteGroupResponse>()
                    .withOperationName("DeleteGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteGroupRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delete a membership within a group given <code>MembershipId</code>.
     * </p>
     *
     * @param deleteGroupMembershipRequest
     * @return Result of the DeleteGroupMembership operation returned by the service.
     * @throws ResourceNotFoundException
     *         Indicates that a requested resource is not found.
     * @throws ThrottlingException
     *         Indicates that the principal has crossed the throttling limits of the API operations.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConflictException
     *         This request cannot be completed for one of the following reasons:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Performing the requested operation would violate an existing uniqueness claim in the identity store.
     *         Resolve the conflict before retrying this request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The requested resource was being concurrently modified by another request.
     *         </p>
     *         </li>
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         The request failed because it contains a syntax error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IdentitystoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IdentitystoreClient.DeleteGroupMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/DeleteGroupMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteGroupMembershipResponse deleteGroupMembership(DeleteGroupMembershipRequest deleteGroupMembershipRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, ConflictException,
            InternalServerException, ValidationException, AwsServiceException, SdkClientException, IdentitystoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteGroupMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteGroupMembershipResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteGroupMembershipRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGroupMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGroupMembership");

            return clientHandler.execute(new ClientExecutionParams<DeleteGroupMembershipRequest, DeleteGroupMembershipResponse>()
                    .withOperationName("DeleteGroupMembership").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteGroupMembershipRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteGroupMembershipRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a user within an identity store given <code>UserId</code>.
     * </p>
     *
     * @param deleteUserRequest
     * @return Result of the DeleteUser operation returned by the service.
     * @throws ResourceNotFoundException
     *         Indicates that a requested resource is not found.
     * @throws ThrottlingException
     *         Indicates that the principal has crossed the throttling limits of the API operations.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConflictException
     *         This request cannot be completed for one of the following reasons:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Performing the requested operation would violate an existing uniqueness claim in the identity store.
     *         Resolve the conflict before retrying this request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The requested resource was being concurrently modified by another request.
     *         </p>
     *         </li>
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         The request failed because it contains a syntax error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IdentitystoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IdentitystoreClient.DeleteUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/DeleteUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteUserResponse deleteUser(DeleteUserRequest deleteUserRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ConflictException, InternalServerException, ValidationException,
            AwsServiceException, SdkClientException, IdentitystoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteUser");

            return clientHandler.execute(new ClientExecutionParams<DeleteUserRequest, DeleteUserResponse>()
                    .withOperationName("DeleteUser").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteUserRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the group metadata and attributes from <code>GroupId</code> in an identity store.
     * </p>
     * <note>
     * <p>
     * If you have administrator access to a member account, you can use this API from the member account. Read about <a
     * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_accounts_access.html">member
     * accounts</a> in the <i>Organizations User Guide</i>.
     * </p>
     * </note>
     *
     * @param describeGroupRequest
     * @return Result of the DescribeGroup operation returned by the service.
     * @throws ResourceNotFoundException
     *         Indicates that a requested resource is not found.
     * @throws ThrottlingException
     *         Indicates that the principal has crossed the throttling limits of the API operations.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         The request failed because it contains a syntax error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IdentitystoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IdentitystoreClient.DescribeGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/DescribeGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeGroupResponse describeGroup(DescribeGroupRequest describeGroupRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, IdentitystoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeGroupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGroup");

            return clientHandler.execute(new ClientExecutionParams<DescribeGroupRequest, DescribeGroupResponse>()
                    .withOperationName("DescribeGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(describeGroupRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves membership metadata and attributes from <code>MembershipId</code> in an identity store.
     * </p>
     * <note>
     * <p>
     * If you have administrator access to a member account, you can use this API from the member account. Read about <a
     * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_accounts_access.html">member
     * accounts</a> in the <i>Organizations User Guide</i>.
     * </p>
     * </note>
     *
     * @param describeGroupMembershipRequest
     * @return Result of the DescribeGroupMembership operation returned by the service.
     * @throws ResourceNotFoundException
     *         Indicates that a requested resource is not found.
     * @throws ThrottlingException
     *         Indicates that the principal has crossed the throttling limits of the API operations.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         The request failed because it contains a syntax error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IdentitystoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IdentitystoreClient.DescribeGroupMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/DescribeGroupMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeGroupMembershipResponse describeGroupMembership(DescribeGroupMembershipRequest describeGroupMembershipRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException,
            ValidationException, AwsServiceException, SdkClientException, IdentitystoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeGroupMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeGroupMembershipResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeGroupMembershipRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGroupMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGroupMembership");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeGroupMembershipRequest, DescribeGroupMembershipResponse>()
                            .withOperationName("DescribeGroupMembership").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(describeGroupMembershipRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeGroupMembershipRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the user metadata and attributes from the <code>UserId</code> in an identity store.
     * </p>
     * <note>
     * <p>
     * If you have administrator access to a member account, you can use this API from the member account. Read about <a
     * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_accounts_access.html">member
     * accounts</a> in the <i>Organizations User Guide</i>.
     * </p>
     * </note>
     *
     * @param describeUserRequest
     * @return Result of the DescribeUser operation returned by the service.
     * @throws ResourceNotFoundException
     *         Indicates that a requested resource is not found.
     * @throws ThrottlingException
     *         Indicates that the principal has crossed the throttling limits of the API operations.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         The request failed because it contains a syntax error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IdentitystoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IdentitystoreClient.DescribeUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/DescribeUser" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeUserResponse describeUser(DescribeUserRequest describeUserRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, IdentitystoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeUser");

            return clientHandler.execute(new ClientExecutionParams<DescribeUserRequest, DescribeUserResponse>()
                    .withOperationName("DescribeUser").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(describeUserRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves <code>GroupId</code> in an identity store.
     * </p>
     * <note>
     * <p>
     * If you have administrator access to a member account, you can use this API from the member account. Read about <a
     * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_accounts_access.html">member
     * accounts</a> in the <i>Organizations User Guide</i>.
     * </p>
     * </note>
     *
     * @param getGroupIdRequest
     * @return Result of the GetGroupId operation returned by the service.
     * @throws ResourceNotFoundException
     *         Indicates that a requested resource is not found.
     * @throws ThrottlingException
     *         Indicates that the principal has crossed the throttling limits of the API operations.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         The request failed because it contains a syntax error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IdentitystoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IdentitystoreClient.GetGroupId
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/GetGroupId" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetGroupIdResponse getGroupId(GetGroupIdRequest getGroupIdRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, IdentitystoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetGroupIdResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetGroupIdResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getGroupIdRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGroupIdRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGroupId");

            return clientHandler.execute(new ClientExecutionParams<GetGroupIdRequest, GetGroupIdResponse>()
                    .withOperationName("GetGroupId").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getGroupIdRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetGroupIdRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the <code>MembershipId</code> in an identity store.
     * </p>
     * <note>
     * <p>
     * If you have administrator access to a member account, you can use this API from the member account. Read about <a
     * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_accounts_access.html">member
     * accounts</a> in the <i>Organizations User Guide</i>.
     * </p>
     * </note>
     *
     * @param getGroupMembershipIdRequest
     * @return Result of the GetGroupMembershipId operation returned by the service.
     * @throws ResourceNotFoundException
     *         Indicates that a requested resource is not found.
     * @throws ThrottlingException
     *         Indicates that the principal has crossed the throttling limits of the API operations.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         The request failed because it contains a syntax error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IdentitystoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IdentitystoreClient.GetGroupMembershipId
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/GetGroupMembershipId"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetGroupMembershipIdResponse getGroupMembershipId(GetGroupMembershipIdRequest getGroupMembershipIdRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException,
            ValidationException, AwsServiceException, SdkClientException, IdentitystoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetGroupMembershipIdResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetGroupMembershipIdResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getGroupMembershipIdRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGroupMembershipIdRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGroupMembershipId");

            return clientHandler.execute(new ClientExecutionParams<GetGroupMembershipIdRequest, GetGroupMembershipIdResponse>()
                    .withOperationName("GetGroupMembershipId").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getGroupMembershipIdRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetGroupMembershipIdRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the <code>UserId</code> in an identity store.
     * </p>
     * <note>
     * <p>
     * If you have administrator access to a member account, you can use this API from the member account. Read about <a
     * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_accounts_access.html">member
     * accounts</a> in the <i>Organizations User Guide</i>.
     * </p>
     * </note>
     *
     * @param getUserIdRequest
     * @return Result of the GetUserId operation returned by the service.
     * @throws ResourceNotFoundException
     *         Indicates that a requested resource is not found.
     * @throws ThrottlingException
     *         Indicates that the principal has crossed the throttling limits of the API operations.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         The request failed because it contains a syntax error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IdentitystoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IdentitystoreClient.GetUserId
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/GetUserId" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetUserIdResponse getUserId(GetUserIdRequest getUserIdRequest) throws ResourceNotFoundException, ThrottlingException,
            AccessDeniedException, InternalServerException, ValidationException, AwsServiceException, SdkClientException,
            IdentitystoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetUserIdResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetUserIdResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getUserIdRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getUserIdRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetUserId");

            return clientHandler.execute(new ClientExecutionParams<GetUserIdRequest, GetUserIdResponse>()
                    .withOperationName("GetUserId").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getUserIdRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetUserIdRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Checks the user's membership in all requested groups and returns if the member exists in all queried groups.
     * </p>
     * <note>
     * <p>
     * If you have administrator access to a member account, you can use this API from the member account. Read about <a
     * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_accounts_access.html">member
     * accounts</a> in the <i>Organizations User Guide</i>.
     * </p>
     * </note>
     *
     * @param isMemberInGroupsRequest
     * @return Result of the IsMemberInGroups operation returned by the service.
     * @throws ResourceNotFoundException
     *         Indicates that a requested resource is not found.
     * @throws ThrottlingException
     *         Indicates that the principal has crossed the throttling limits of the API operations.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         The request failed because it contains a syntax error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IdentitystoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IdentitystoreClient.IsMemberInGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/IsMemberInGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public IsMemberInGroupsResponse isMemberInGroups(IsMemberInGroupsRequest isMemberInGroupsRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException,
            ValidationException, AwsServiceException, SdkClientException, IdentitystoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<IsMemberInGroupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                IsMemberInGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(isMemberInGroupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, isMemberInGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "IsMemberInGroups");

            return clientHandler.execute(new ClientExecutionParams<IsMemberInGroupsRequest, IsMemberInGroupsResponse>()
                    .withOperationName("IsMemberInGroups").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(isMemberInGroupsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new IsMemberInGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * For the specified group in the specified identity store, returns the list of all <code>GroupMembership</code>
     * objects and returns results in paginated form.
     * </p>
     * <note>
     * <p>
     * If you have administrator access to a member account, you can use this API from the member account. Read about <a
     * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_accounts_access.html">member
     * accounts</a> in the <i>Organizations User Guide</i>.
     * </p>
     * </note>
     *
     * @param listGroupMembershipsRequest
     * @return Result of the ListGroupMemberships operation returned by the service.
     * @throws ResourceNotFoundException
     *         Indicates that a requested resource is not found.
     * @throws ThrottlingException
     *         Indicates that the principal has crossed the throttling limits of the API operations.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         The request failed because it contains a syntax error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IdentitystoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IdentitystoreClient.ListGroupMemberships
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/ListGroupMemberships"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListGroupMembershipsResponse listGroupMemberships(ListGroupMembershipsRequest listGroupMembershipsRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException,
            ValidationException, AwsServiceException, SdkClientException, IdentitystoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListGroupMembershipsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListGroupMembershipsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listGroupMembershipsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGroupMembershipsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGroupMemberships");

            return clientHandler.execute(new ClientExecutionParams<ListGroupMembershipsRequest, ListGroupMembershipsResponse>()
                    .withOperationName("ListGroupMemberships").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listGroupMembershipsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListGroupMembershipsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * For the specified member in the specified identity store, returns the list of all <code>GroupMembership</code>
     * objects and returns results in paginated form.
     * </p>
     * <note>
     * <p>
     * If you have administrator access to a member account, you can use this API from the member account. Read about <a
     * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_accounts_access.html">member
     * accounts</a> in the <i>Organizations User Guide</i>.
     * </p>
     * </note>
     *
     * @param listGroupMembershipsForMemberRequest
     * @return Result of the ListGroupMembershipsForMember operation returned by the service.
     * @throws ResourceNotFoundException
     *         Indicates that a requested resource is not found.
     * @throws ThrottlingException
     *         Indicates that the principal has crossed the throttling limits of the API operations.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         The request failed because it contains a syntax error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IdentitystoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IdentitystoreClient.ListGroupMembershipsForMember
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/ListGroupMembershipsForMember"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListGroupMembershipsForMemberResponse listGroupMembershipsForMember(
            ListGroupMembershipsForMemberRequest listGroupMembershipsForMemberRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, IdentitystoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListGroupMembershipsForMemberResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListGroupMembershipsForMemberResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listGroupMembershipsForMemberRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listGroupMembershipsForMemberRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGroupMembershipsForMember");

            return clientHandler
                    .execute(new ClientExecutionParams<ListGroupMembershipsForMemberRequest, ListGroupMembershipsForMemberResponse>()
                            .withOperationName("ListGroupMembershipsForMember").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(listGroupMembershipsForMemberRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListGroupMembershipsForMemberRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all groups in the identity store. Returns a paginated list of complete <code>Group</code> objects.
     * Filtering for a <code>Group</code> by the <code>DisplayName</code> attribute is deprecated. Instead, use the
     * <code>GetGroupId</code> API action.
     * </p>
     * <note>
     * <p>
     * If you have administrator access to a member account, you can use this API from the member account. Read about <a
     * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_accounts_access.html">member
     * accounts</a> in the <i>Organizations User Guide</i>.
     * </p>
     * </note>
     *
     * @param listGroupsRequest
     * @return Result of the ListGroups operation returned by the service.
     * @throws ResourceNotFoundException
     *         Indicates that a requested resource is not found.
     * @throws ThrottlingException
     *         Indicates that the principal has crossed the throttling limits of the API operations.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         The request failed because it contains a syntax error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IdentitystoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IdentitystoreClient.ListGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/ListGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListGroupsResponse listGroups(ListGroupsRequest listGroupsRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, IdentitystoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListGroupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listGroupsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGroups");

            return clientHandler.execute(new ClientExecutionParams<ListGroupsRequest, ListGroupsResponse>()
                    .withOperationName("ListGroups").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listGroupsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all users in the identity store. Returns a paginated list of complete <code>User</code> objects. Filtering
     * for a <code>User</code> by the <code>UserName</code> attribute is deprecated. Instead, use the
     * <code>GetUserId</code> API action.
     * </p>
     * <note>
     * <p>
     * If you have administrator access to a member account, you can use this API from the member account. Read about <a
     * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_accounts_access.html">member
     * accounts</a> in the <i>Organizations User Guide</i>.
     * </p>
     * </note>
     *
     * @param listUsersRequest
     * @return Result of the ListUsers operation returned by the service.
     * @throws ResourceNotFoundException
     *         Indicates that a requested resource is not found.
     * @throws ThrottlingException
     *         Indicates that the principal has crossed the throttling limits of the API operations.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         The request failed because it contains a syntax error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IdentitystoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IdentitystoreClient.ListUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListUsersResponse listUsers(ListUsersRequest listUsersRequest) throws ResourceNotFoundException, ThrottlingException,
            AccessDeniedException, InternalServerException, ValidationException, AwsServiceException, SdkClientException,
            IdentitystoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListUsersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListUsersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listUsersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listUsersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListUsers");

            return clientHandler.execute(new ClientExecutionParams<ListUsersRequest, ListUsersResponse>()
                    .withOperationName("ListUsers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listUsersRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListUsersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * For the specified group in the specified identity store, updates the group metadata and attributes.
     * </p>
     *
     * @param updateGroupRequest
     * @return Result of the UpdateGroup operation returned by the service.
     * @throws ResourceNotFoundException
     *         Indicates that a requested resource is not found.
     * @throws ThrottlingException
     *         Indicates that the principal has crossed the throttling limits of the API operations.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConflictException
     *         This request cannot be completed for one of the following reasons:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Performing the requested operation would violate an existing uniqueness claim in the identity store.
     *         Resolve the conflict before retrying this request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The requested resource was being concurrently modified by another request.
     *         </p>
     *         </li>
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         The request failed because it contains a syntax error.
     * @throws ServiceQuotaExceededException
     *         The request would cause the number of users or groups in the identity store to exceed the maximum
     *         allowed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IdentitystoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IdentitystoreClient.UpdateGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/UpdateGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateGroupResponse updateGroup(UpdateGroupRequest updateGroupRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ConflictException, InternalServerException, ValidationException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, IdentitystoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateGroupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGroup");

            return clientHandler.execute(new ClientExecutionParams<UpdateGroupRequest, UpdateGroupResponse>()
                    .withOperationName("UpdateGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(updateGroupRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * For the specified user in the specified identity store, updates the user metadata and attributes.
     * </p>
     *
     * @param updateUserRequest
     * @return Result of the UpdateUser operation returned by the service.
     * @throws ResourceNotFoundException
     *         Indicates that a requested resource is not found.
     * @throws ThrottlingException
     *         Indicates that the principal has crossed the throttling limits of the API operations.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConflictException
     *         This request cannot be completed for one of the following reasons:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Performing the requested operation would violate an existing uniqueness claim in the identity store.
     *         Resolve the conflict before retrying this request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The requested resource was being concurrently modified by another request.
     *         </p>
     *         </li>
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure with an internal
     *         server.
     * @throws ValidationException
     *         The request failed because it contains a syntax error.
     * @throws ServiceQuotaExceededException
     *         The request would cause the number of users or groups in the identity store to exceed the maximum
     *         allowed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IdentitystoreException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IdentitystoreClient.UpdateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/identitystore-2020-06-15/UpdateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateUserResponse updateUser(UpdateUserRequest updateUserRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ConflictException, InternalServerException, ValidationException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, IdentitystoreException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "identitystore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateUser");

            return clientHandler.execute(new ClientExecutionParams<UpdateUserRequest, UpdateUserResponse>()
                    .withOperationName("UpdateUser").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(updateUserRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        IdentitystoreServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = IdentitystoreServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IdentitystoreException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public final IdentitystoreServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
