/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.imagebuilder.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An Image Builder image. You must specify exactly one recipe for the image – either a container recipe (
 * <code>containerRecipe</code>), which creates a container image, or an image recipe (<code>imageRecipe</code>), which
 * creates an AMI.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Image implements SdkPojo, Serializable, ToCopyableBuilder<Image.Builder, Image> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(Image::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(Image::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Image::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("version")
            .getter(getter(Image::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("version").build()).build();

    private static final SdkField<String> PLATFORM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("platform").getter(getter(Image::platformAsString)).setter(setter(Builder::platform))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("platform").build()).build();

    private static final SdkField<Boolean> ENHANCED_IMAGE_METADATA_ENABLED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("enhancedImageMetadataEnabled")
            .getter(getter(Image::enhancedImageMetadataEnabled))
            .setter(setter(Builder::enhancedImageMetadataEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("enhancedImageMetadataEnabled")
                    .build()).build();

    private static final SdkField<String> OS_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("osVersion").getter(getter(Image::osVersion)).setter(setter(Builder::osVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("osVersion").build()).build();

    private static final SdkField<ImageState> STATE_FIELD = SdkField.<ImageState> builder(MarshallingType.SDK_POJO)
            .memberName("state").getter(getter(Image::state)).setter(setter(Builder::state)).constructor(ImageState::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<ImageRecipe> IMAGE_RECIPE_FIELD = SdkField.<ImageRecipe> builder(MarshallingType.SDK_POJO)
            .memberName("imageRecipe").getter(getter(Image::imageRecipe)).setter(setter(Builder::imageRecipe))
            .constructor(ImageRecipe::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imageRecipe").build()).build();

    private static final SdkField<ContainerRecipe> CONTAINER_RECIPE_FIELD = SdkField
            .<ContainerRecipe> builder(MarshallingType.SDK_POJO).memberName("containerRecipe")
            .getter(getter(Image::containerRecipe)).setter(setter(Builder::containerRecipe))
            .constructor(ContainerRecipe::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerRecipe").build()).build();

    private static final SdkField<String> SOURCE_PIPELINE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sourcePipelineName").getter(getter(Image::sourcePipelineName))
            .setter(setter(Builder::sourcePipelineName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourcePipelineName").build())
            .build();

    private static final SdkField<String> SOURCE_PIPELINE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sourcePipelineArn").getter(getter(Image::sourcePipelineArn)).setter(setter(Builder::sourcePipelineArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourcePipelineArn").build()).build();

    private static final SdkField<InfrastructureConfiguration> INFRASTRUCTURE_CONFIGURATION_FIELD = SdkField
            .<InfrastructureConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("infrastructureConfiguration")
            .getter(getter(Image::infrastructureConfiguration))
            .setter(setter(Builder::infrastructureConfiguration))
            .constructor(InfrastructureConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("infrastructureConfiguration")
                    .build()).build();

    private static final SdkField<DistributionConfiguration> DISTRIBUTION_CONFIGURATION_FIELD = SdkField
            .<DistributionConfiguration> builder(MarshallingType.SDK_POJO).memberName("distributionConfiguration")
            .getter(getter(Image::distributionConfiguration)).setter(setter(Builder::distributionConfiguration))
            .constructor(DistributionConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("distributionConfiguration").build())
            .build();

    private static final SdkField<ImageTestsConfiguration> IMAGE_TESTS_CONFIGURATION_FIELD = SdkField
            .<ImageTestsConfiguration> builder(MarshallingType.SDK_POJO).memberName("imageTestsConfiguration")
            .getter(getter(Image::imageTestsConfiguration)).setter(setter(Builder::imageTestsConfiguration))
            .constructor(ImageTestsConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imageTestsConfiguration").build())
            .build();

    private static final SdkField<String> DATE_CREATED_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("dateCreated").getter(getter(Image::dateCreated)).setter(setter(Builder::dateCreated))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dateCreated").build()).build();

    private static final SdkField<OutputResources> OUTPUT_RESOURCES_FIELD = SdkField
            .<OutputResources> builder(MarshallingType.SDK_POJO).memberName("outputResources")
            .getter(getter(Image::outputResources)).setter(setter(Builder::outputResources))
            .constructor(OutputResources::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("outputResources").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(Image::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD, TYPE_FIELD,
            NAME_FIELD, VERSION_FIELD, PLATFORM_FIELD, ENHANCED_IMAGE_METADATA_ENABLED_FIELD, OS_VERSION_FIELD, STATE_FIELD,
            IMAGE_RECIPE_FIELD, CONTAINER_RECIPE_FIELD, SOURCE_PIPELINE_NAME_FIELD, SOURCE_PIPELINE_ARN_FIELD,
            INFRASTRUCTURE_CONFIGURATION_FIELD, DISTRIBUTION_CONFIGURATION_FIELD, IMAGE_TESTS_CONFIGURATION_FIELD,
            DATE_CREATED_FIELD, OUTPUT_RESOURCES_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final String type;

    private final String name;

    private final String version;

    private final String platform;

    private final Boolean enhancedImageMetadataEnabled;

    private final String osVersion;

    private final ImageState state;

    private final ImageRecipe imageRecipe;

    private final ContainerRecipe containerRecipe;

    private final String sourcePipelineName;

    private final String sourcePipelineArn;

    private final InfrastructureConfiguration infrastructureConfiguration;

    private final DistributionConfiguration distributionConfiguration;

    private final ImageTestsConfiguration imageTestsConfiguration;

    private final String dateCreated;

    private final OutputResources outputResources;

    private final Map<String, String> tags;

    private Image(BuilderImpl builder) {
        this.arn = builder.arn;
        this.type = builder.type;
        this.name = builder.name;
        this.version = builder.version;
        this.platform = builder.platform;
        this.enhancedImageMetadataEnabled = builder.enhancedImageMetadataEnabled;
        this.osVersion = builder.osVersion;
        this.state = builder.state;
        this.imageRecipe = builder.imageRecipe;
        this.containerRecipe = builder.containerRecipe;
        this.sourcePipelineName = builder.sourcePipelineName;
        this.sourcePipelineArn = builder.sourcePipelineArn;
        this.infrastructureConfiguration = builder.infrastructureConfiguration;
        this.distributionConfiguration = builder.distributionConfiguration;
        this.imageTestsConfiguration = builder.imageTestsConfiguration;
        this.dateCreated = builder.dateCreated;
        this.outputResources = builder.outputResources;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the image.
     * </p>
     * <note>
     * <p>
     * Semantic versioning is included in each object's Amazon Resource Name (ARN), at the level that applies to that
     * object as follows:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Versionless ARNs and Name ARNs do not include specific values in any of the nodes. The nodes are either left off
     * entirely, or they are specified as wildcards, for example: x.x.x.
     * </p>
     * </li>
     * <li>
     * <p>
     * Version ARNs have only the first three nodes: &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;
     * </p>
     * </li>
     * <li>
     * <p>
     * Build version ARNs have all four nodes, and point to a specific build for a specific version of an object.
     * </p>
     * </li>
     * </ol>
     * </note>
     * 
     * @return The Amazon Resource Name (ARN) of the image.</p> <note>
     *         <p>
     *         Semantic versioning is included in each object's Amazon Resource Name (ARN), at the level that applies to
     *         that object as follows:
     *         </p>
     *         <ol>
     *         <li>
     *         <p>
     *         Versionless ARNs and Name ARNs do not include specific values in any of the nodes. The nodes are either
     *         left off entirely, or they are specified as wildcards, for example: x.x.x.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Version ARNs have only the first three nodes: &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Build version ARNs have all four nodes, and point to a specific build for a specific version of an
     *         object.
     *         </p>
     *         </li>
     *         </ol>
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * Specifies whether this is an AMI or container image.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ImageType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Specifies whether this is an AMI or container image.
     * @see ImageType
     */
    public final ImageType type() {
        return ImageType.fromValue(type);
    }

    /**
     * <p>
     * Specifies whether this is an AMI or container image.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ImageType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Specifies whether this is an AMI or container image.
     * @see ImageType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The name of the image.
     * </p>
     * 
     * @return The name of the image.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The semantic version of the image.
     * </p>
     * <note>
     * <p>
     * The semantic version has four nodes: &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;/&lt;build&gt;. You can assign
     * values for the first three, and can filter on all of them.
     * </p>
     * <p>
     * <b>Assignment:</b> For the first three nodes you can assign any positive integer value, including zero, with an
     * upper limit of 2^30-1, or 1073741823 for each node. Image Builder automatically assigns the build number, and
     * that is not open for updates.
     * </p>
     * <p>
     * <b>Patterns:</b> You can use any numeric pattern that adheres to the assignment requirements for the nodes that
     * you can assign. For example, you might choose a software version pattern, such as 1.0.0, or a date, such as
     * 2021.01.01.
     * </p>
     * <p>
     * <b>Filtering:</b> When you retrieve or reference a resource with a semantic version, you can use wildcards (x) to
     * filter your results. When you use a wildcard in any node, all nodes to the right of the first wildcard must also
     * be wildcards. For example, specifying "1.2.x", or "1.x.x" works to filter list results, but neither "1.x.2", nor
     * "x.2.x" will work. You do not have to specify the build - Image Builder automatically uses a wildcard for that,
     * if applicable.
     * </p>
     * </note>
     * 
     * @return The semantic version of the image.</p> <note>
     *         <p>
     *         The semantic version has four nodes: &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;/&lt;build&gt;. You can
     *         assign values for the first three, and can filter on all of them.
     *         </p>
     *         <p>
     *         <b>Assignment:</b> For the first three nodes you can assign any positive integer value, including zero,
     *         with an upper limit of 2^30-1, or 1073741823 for each node. Image Builder automatically assigns the build
     *         number, and that is not open for updates.
     *         </p>
     *         <p>
     *         <b>Patterns:</b> You can use any numeric pattern that adheres to the assignment requirements for the
     *         nodes that you can assign. For example, you might choose a software version pattern, such as 1.0.0, or a
     *         date, such as 2021.01.01.
     *         </p>
     *         <p>
     *         <b>Filtering:</b> When you retrieve or reference a resource with a semantic version, you can use
     *         wildcards (x) to filter your results. When you use a wildcard in any node, all nodes to the right of the
     *         first wildcard must also be wildcards. For example, specifying "1.2.x", or "1.x.x" works to filter list
     *         results, but neither "1.x.2", nor "x.2.x" will work. You do not have to specify the build - Image Builder
     *         automatically uses a wildcard for that, if applicable.
     *         </p>
     */
    public final String version() {
        return version;
    }

    /**
     * <p>
     * The platform of the image.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #platform} will
     * return {@link Platform#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #platformAsString}.
     * </p>
     * 
     * @return The platform of the image.
     * @see Platform
     */
    public final Platform platform() {
        return Platform.fromValue(platform);
    }

    /**
     * <p>
     * The platform of the image.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #platform} will
     * return {@link Platform#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #platformAsString}.
     * </p>
     * 
     * @return The platform of the image.
     * @see Platform
     */
    public final String platformAsString() {
        return platform;
    }

    /**
     * <p>
     * Collects additional information about the image being created, including the operating system (OS) version and
     * package list. This information is used to enhance the overall experience of using EC2 Image Builder. Enabled by
     * default.
     * </p>
     * 
     * @return Collects additional information about the image being created, including the operating system (OS)
     *         version and package list. This information is used to enhance the overall experience of using EC2 Image
     *         Builder. Enabled by default.
     */
    public final Boolean enhancedImageMetadataEnabled() {
        return enhancedImageMetadataEnabled;
    }

    /**
     * <p>
     * The operating system version of the instance. For example, Amazon Linux 2, Ubuntu 18, or Microsoft Windows Server
     * 2019.
     * </p>
     * 
     * @return The operating system version of the instance. For example, Amazon Linux 2, Ubuntu 18, or Microsoft
     *         Windows Server 2019.
     */
    public final String osVersion() {
        return osVersion;
    }

    /**
     * <p>
     * The state of the image.
     * </p>
     * 
     * @return The state of the image.
     */
    public final ImageState state() {
        return state;
    }

    /**
     * <p>
     * The image recipe used when creating the image.
     * </p>
     * 
     * @return The image recipe used when creating the image.
     */
    public final ImageRecipe imageRecipe() {
        return imageRecipe;
    }

    /**
     * <p>
     * The recipe that is used to create an Image Builder container image.
     * </p>
     * 
     * @return The recipe that is used to create an Image Builder container image.
     */
    public final ContainerRecipe containerRecipe() {
        return containerRecipe;
    }

    /**
     * <p>
     * The name of the image pipeline that created this image.
     * </p>
     * 
     * @return The name of the image pipeline that created this image.
     */
    public final String sourcePipelineName() {
        return sourcePipelineName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the image pipeline that created this image.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the image pipeline that created this image.
     */
    public final String sourcePipelineArn() {
        return sourcePipelineArn;
    }

    /**
     * <p>
     * The infrastructure used when creating this image.
     * </p>
     * 
     * @return The infrastructure used when creating this image.
     */
    public final InfrastructureConfiguration infrastructureConfiguration() {
        return infrastructureConfiguration;
    }

    /**
     * <p>
     * The distribution configuration used when creating this image.
     * </p>
     * 
     * @return The distribution configuration used when creating this image.
     */
    public final DistributionConfiguration distributionConfiguration() {
        return distributionConfiguration;
    }

    /**
     * <p>
     * The image tests configuration used when creating this image.
     * </p>
     * 
     * @return The image tests configuration used when creating this image.
     */
    public final ImageTestsConfiguration imageTestsConfiguration() {
        return imageTestsConfiguration;
    }

    /**
     * <p>
     * The date on which this image was created.
     * </p>
     * 
     * @return The date on which this image was created.
     */
    public final String dateCreated() {
        return dateCreated;
    }

    /**
     * <p>
     * The output resources produced when creating this image.
     * </p>
     * 
     * @return The output resources produced when creating this image.
     */
    public final OutputResources outputResources() {
        return outputResources;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The tags of the image.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags of the image.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(platformAsString());
        hashCode = 31 * hashCode + Objects.hashCode(enhancedImageMetadataEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(osVersion());
        hashCode = 31 * hashCode + Objects.hashCode(state());
        hashCode = 31 * hashCode + Objects.hashCode(imageRecipe());
        hashCode = 31 * hashCode + Objects.hashCode(containerRecipe());
        hashCode = 31 * hashCode + Objects.hashCode(sourcePipelineName());
        hashCode = 31 * hashCode + Objects.hashCode(sourcePipelineArn());
        hashCode = 31 * hashCode + Objects.hashCode(infrastructureConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(distributionConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(imageTestsConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(dateCreated());
        hashCode = 31 * hashCode + Objects.hashCode(outputResources());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Image)) {
            return false;
        }
        Image other = (Image) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(name(), other.name()) && Objects.equals(version(), other.version())
                && Objects.equals(platformAsString(), other.platformAsString())
                && Objects.equals(enhancedImageMetadataEnabled(), other.enhancedImageMetadataEnabled())
                && Objects.equals(osVersion(), other.osVersion()) && Objects.equals(state(), other.state())
                && Objects.equals(imageRecipe(), other.imageRecipe())
                && Objects.equals(containerRecipe(), other.containerRecipe())
                && Objects.equals(sourcePipelineName(), other.sourcePipelineName())
                && Objects.equals(sourcePipelineArn(), other.sourcePipelineArn())
                && Objects.equals(infrastructureConfiguration(), other.infrastructureConfiguration())
                && Objects.equals(distributionConfiguration(), other.distributionConfiguration())
                && Objects.equals(imageTestsConfiguration(), other.imageTestsConfiguration())
                && Objects.equals(dateCreated(), other.dateCreated())
                && Objects.equals(outputResources(), other.outputResources()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Image").add("Arn", arn()).add("Type", typeAsString()).add("Name", name())
                .add("Version", version()).add("Platform", platformAsString())
                .add("EnhancedImageMetadataEnabled", enhancedImageMetadataEnabled()).add("OsVersion", osVersion())
                .add("State", state()).add("ImageRecipe", imageRecipe()).add("ContainerRecipe", containerRecipe())
                .add("SourcePipelineName", sourcePipelineName()).add("SourcePipelineArn", sourcePipelineArn())
                .add("InfrastructureConfiguration", infrastructureConfiguration())
                .add("DistributionConfiguration", distributionConfiguration())
                .add("ImageTestsConfiguration", imageTestsConfiguration()).add("DateCreated", dateCreated())
                .add("OutputResources", outputResources()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "version":
            return Optional.ofNullable(clazz.cast(version()));
        case "platform":
            return Optional.ofNullable(clazz.cast(platformAsString()));
        case "enhancedImageMetadataEnabled":
            return Optional.ofNullable(clazz.cast(enhancedImageMetadataEnabled()));
        case "osVersion":
            return Optional.ofNullable(clazz.cast(osVersion()));
        case "state":
            return Optional.ofNullable(clazz.cast(state()));
        case "imageRecipe":
            return Optional.ofNullable(clazz.cast(imageRecipe()));
        case "containerRecipe":
            return Optional.ofNullable(clazz.cast(containerRecipe()));
        case "sourcePipelineName":
            return Optional.ofNullable(clazz.cast(sourcePipelineName()));
        case "sourcePipelineArn":
            return Optional.ofNullable(clazz.cast(sourcePipelineArn()));
        case "infrastructureConfiguration":
            return Optional.ofNullable(clazz.cast(infrastructureConfiguration()));
        case "distributionConfiguration":
            return Optional.ofNullable(clazz.cast(distributionConfiguration()));
        case "imageTestsConfiguration":
            return Optional.ofNullable(clazz.cast(imageTestsConfiguration()));
        case "dateCreated":
            return Optional.ofNullable(clazz.cast(dateCreated()));
        case "outputResources":
            return Optional.ofNullable(clazz.cast(outputResources()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Image, T> g) {
        return obj -> g.apply((Image) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Image> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the image.
         * </p>
         * <note>
         * <p>
         * Semantic versioning is included in each object's Amazon Resource Name (ARN), at the level that applies to
         * that object as follows:
         * </p>
         * <ol>
         * <li>
         * <p>
         * Versionless ARNs and Name ARNs do not include specific values in any of the nodes. The nodes are either left
         * off entirely, or they are specified as wildcards, for example: x.x.x.
         * </p>
         * </li>
         * <li>
         * <p>
         * Version ARNs have only the first three nodes: &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;
         * </p>
         * </li>
         * <li>
         * <p>
         * Build version ARNs have all four nodes, and point to a specific build for a specific version of an object.
         * </p>
         * </li>
         * </ol>
         * </note>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the image.</p> <note>
         *        <p>
         *        Semantic versioning is included in each object's Amazon Resource Name (ARN), at the level that applies
         *        to that object as follows:
         *        </p>
         *        <ol>
         *        <li>
         *        <p>
         *        Versionless ARNs and Name ARNs do not include specific values in any of the nodes. The nodes are
         *        either left off entirely, or they are specified as wildcards, for example: x.x.x.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Version ARNs have only the first three nodes: &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Build version ARNs have all four nodes, and point to a specific build for a specific version of an
         *        object.
         *        </p>
         *        </li>
         *        </ol>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * Specifies whether this is an AMI or container image.
         * </p>
         * 
         * @param type
         *        Specifies whether this is an AMI or container image.
         * @see ImageType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImageType
         */
        Builder type(String type);

        /**
         * <p>
         * Specifies whether this is an AMI or container image.
         * </p>
         * 
         * @param type
         *        Specifies whether this is an AMI or container image.
         * @see ImageType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImageType
         */
        Builder type(ImageType type);

        /**
         * <p>
         * The name of the image.
         * </p>
         * 
         * @param name
         *        The name of the image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The semantic version of the image.
         * </p>
         * <note>
         * <p>
         * The semantic version has four nodes: &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;/&lt;build&gt;. You can assign
         * values for the first three, and can filter on all of them.
         * </p>
         * <p>
         * <b>Assignment:</b> For the first three nodes you can assign any positive integer value, including zero, with
         * an upper limit of 2^30-1, or 1073741823 for each node. Image Builder automatically assigns the build number,
         * and that is not open for updates.
         * </p>
         * <p>
         * <b>Patterns:</b> You can use any numeric pattern that adheres to the assignment requirements for the nodes
         * that you can assign. For example, you might choose a software version pattern, such as 1.0.0, or a date, such
         * as 2021.01.01.
         * </p>
         * <p>
         * <b>Filtering:</b> When you retrieve or reference a resource with a semantic version, you can use wildcards
         * (x) to filter your results. When you use a wildcard in any node, all nodes to the right of the first wildcard
         * must also be wildcards. For example, specifying "1.2.x", or "1.x.x" works to filter list results, but neither
         * "1.x.2", nor "x.2.x" will work. You do not have to specify the build - Image Builder automatically uses a
         * wildcard for that, if applicable.
         * </p>
         * </note>
         * 
         * @param version
         *        The semantic version of the image.</p> <note>
         *        <p>
         *        The semantic version has four nodes: &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;/&lt;build&gt;. You can
         *        assign values for the first three, and can filter on all of them.
         *        </p>
         *        <p>
         *        <b>Assignment:</b> For the first three nodes you can assign any positive integer value, including
         *        zero, with an upper limit of 2^30-1, or 1073741823 for each node. Image Builder automatically assigns
         *        the build number, and that is not open for updates.
         *        </p>
         *        <p>
         *        <b>Patterns:</b> You can use any numeric pattern that adheres to the assignment requirements for the
         *        nodes that you can assign. For example, you might choose a software version pattern, such as 1.0.0, or
         *        a date, such as 2021.01.01.
         *        </p>
         *        <p>
         *        <b>Filtering:</b> When you retrieve or reference a resource with a semantic version, you can use
         *        wildcards (x) to filter your results. When you use a wildcard in any node, all nodes to the right of
         *        the first wildcard must also be wildcards. For example, specifying "1.2.x", or "1.x.x" works to filter
         *        list results, but neither "1.x.2", nor "x.2.x" will work. You do not have to specify the build - Image
         *        Builder automatically uses a wildcard for that, if applicable.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * The platform of the image.
         * </p>
         * 
         * @param platform
         *        The platform of the image.
         * @see Platform
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Platform
         */
        Builder platform(String platform);

        /**
         * <p>
         * The platform of the image.
         * </p>
         * 
         * @param platform
         *        The platform of the image.
         * @see Platform
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Platform
         */
        Builder platform(Platform platform);

        /**
         * <p>
         * Collects additional information about the image being created, including the operating system (OS) version
         * and package list. This information is used to enhance the overall experience of using EC2 Image Builder.
         * Enabled by default.
         * </p>
         * 
         * @param enhancedImageMetadataEnabled
         *        Collects additional information about the image being created, including the operating system (OS)
         *        version and package list. This information is used to enhance the overall experience of using EC2
         *        Image Builder. Enabled by default.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enhancedImageMetadataEnabled(Boolean enhancedImageMetadataEnabled);

        /**
         * <p>
         * The operating system version of the instance. For example, Amazon Linux 2, Ubuntu 18, or Microsoft Windows
         * Server 2019.
         * </p>
         * 
         * @param osVersion
         *        The operating system version of the instance. For example, Amazon Linux 2, Ubuntu 18, or Microsoft
         *        Windows Server 2019.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder osVersion(String osVersion);

        /**
         * <p>
         * The state of the image.
         * </p>
         * 
         * @param state
         *        The state of the image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder state(ImageState state);

        /**
         * <p>
         * The state of the image.
         * </p>
         * This is a convenience that creates an instance of the {@link ImageState.Builder} avoiding the need to create
         * one manually via {@link ImageState#builder()}.
         *
         * When the {@link Consumer} completes, {@link ImageState.Builder#build()} is called immediately and its result
         * is passed to {@link #state(ImageState)}.
         * 
         * @param state
         *        a consumer that will call methods on {@link ImageState.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #state(ImageState)
         */
        default Builder state(Consumer<ImageState.Builder> state) {
            return state(ImageState.builder().applyMutation(state).build());
        }

        /**
         * <p>
         * The image recipe used when creating the image.
         * </p>
         * 
         * @param imageRecipe
         *        The image recipe used when creating the image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageRecipe(ImageRecipe imageRecipe);

        /**
         * <p>
         * The image recipe used when creating the image.
         * </p>
         * This is a convenience that creates an instance of the {@link ImageRecipe.Builder} avoiding the need to create
         * one manually via {@link ImageRecipe#builder()}.
         *
         * When the {@link Consumer} completes, {@link ImageRecipe.Builder#build()} is called immediately and its result
         * is passed to {@link #imageRecipe(ImageRecipe)}.
         * 
         * @param imageRecipe
         *        a consumer that will call methods on {@link ImageRecipe.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #imageRecipe(ImageRecipe)
         */
        default Builder imageRecipe(Consumer<ImageRecipe.Builder> imageRecipe) {
            return imageRecipe(ImageRecipe.builder().applyMutation(imageRecipe).build());
        }

        /**
         * <p>
         * The recipe that is used to create an Image Builder container image.
         * </p>
         * 
         * @param containerRecipe
         *        The recipe that is used to create an Image Builder container image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerRecipe(ContainerRecipe containerRecipe);

        /**
         * <p>
         * The recipe that is used to create an Image Builder container image.
         * </p>
         * This is a convenience that creates an instance of the {@link ContainerRecipe.Builder} avoiding the need to
         * create one manually via {@link ContainerRecipe#builder()}.
         *
         * When the {@link Consumer} completes, {@link ContainerRecipe.Builder#build()} is called immediately and its
         * result is passed to {@link #containerRecipe(ContainerRecipe)}.
         * 
         * @param containerRecipe
         *        a consumer that will call methods on {@link ContainerRecipe.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #containerRecipe(ContainerRecipe)
         */
        default Builder containerRecipe(Consumer<ContainerRecipe.Builder> containerRecipe) {
            return containerRecipe(ContainerRecipe.builder().applyMutation(containerRecipe).build());
        }

        /**
         * <p>
         * The name of the image pipeline that created this image.
         * </p>
         * 
         * @param sourcePipelineName
         *        The name of the image pipeline that created this image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourcePipelineName(String sourcePipelineName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the image pipeline that created this image.
         * </p>
         * 
         * @param sourcePipelineArn
         *        The Amazon Resource Name (ARN) of the image pipeline that created this image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourcePipelineArn(String sourcePipelineArn);

        /**
         * <p>
         * The infrastructure used when creating this image.
         * </p>
         * 
         * @param infrastructureConfiguration
         *        The infrastructure used when creating this image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder infrastructureConfiguration(InfrastructureConfiguration infrastructureConfiguration);

        /**
         * <p>
         * The infrastructure used when creating this image.
         * </p>
         * This is a convenience that creates an instance of the {@link InfrastructureConfiguration.Builder} avoiding
         * the need to create one manually via {@link InfrastructureConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link InfrastructureConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #infrastructureConfiguration(InfrastructureConfiguration)}.
         * 
         * @param infrastructureConfiguration
         *        a consumer that will call methods on {@link InfrastructureConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #infrastructureConfiguration(InfrastructureConfiguration)
         */
        default Builder infrastructureConfiguration(Consumer<InfrastructureConfiguration.Builder> infrastructureConfiguration) {
            return infrastructureConfiguration(InfrastructureConfiguration.builder().applyMutation(infrastructureConfiguration)
                    .build());
        }

        /**
         * <p>
         * The distribution configuration used when creating this image.
         * </p>
         * 
         * @param distributionConfiguration
         *        The distribution configuration used when creating this image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder distributionConfiguration(DistributionConfiguration distributionConfiguration);

        /**
         * <p>
         * The distribution configuration used when creating this image.
         * </p>
         * This is a convenience that creates an instance of the {@link DistributionConfiguration.Builder} avoiding the
         * need to create one manually via {@link DistributionConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link DistributionConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #distributionConfiguration(DistributionConfiguration)}.
         * 
         * @param distributionConfiguration
         *        a consumer that will call methods on {@link DistributionConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #distributionConfiguration(DistributionConfiguration)
         */
        default Builder distributionConfiguration(Consumer<DistributionConfiguration.Builder> distributionConfiguration) {
            return distributionConfiguration(DistributionConfiguration.builder().applyMutation(distributionConfiguration).build());
        }

        /**
         * <p>
         * The image tests configuration used when creating this image.
         * </p>
         * 
         * @param imageTestsConfiguration
         *        The image tests configuration used when creating this image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageTestsConfiguration(ImageTestsConfiguration imageTestsConfiguration);

        /**
         * <p>
         * The image tests configuration used when creating this image.
         * </p>
         * This is a convenience that creates an instance of the {@link ImageTestsConfiguration.Builder} avoiding the
         * need to create one manually via {@link ImageTestsConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link ImageTestsConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #imageTestsConfiguration(ImageTestsConfiguration)}.
         * 
         * @param imageTestsConfiguration
         *        a consumer that will call methods on {@link ImageTestsConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #imageTestsConfiguration(ImageTestsConfiguration)
         */
        default Builder imageTestsConfiguration(Consumer<ImageTestsConfiguration.Builder> imageTestsConfiguration) {
            return imageTestsConfiguration(ImageTestsConfiguration.builder().applyMutation(imageTestsConfiguration).build());
        }

        /**
         * <p>
         * The date on which this image was created.
         * </p>
         * 
         * @param dateCreated
         *        The date on which this image was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dateCreated(String dateCreated);

        /**
         * <p>
         * The output resources produced when creating this image.
         * </p>
         * 
         * @param outputResources
         *        The output resources produced when creating this image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputResources(OutputResources outputResources);

        /**
         * <p>
         * The output resources produced when creating this image.
         * </p>
         * This is a convenience that creates an instance of the {@link OutputResources.Builder} avoiding the need to
         * create one manually via {@link OutputResources#builder()}.
         *
         * When the {@link Consumer} completes, {@link OutputResources.Builder#build()} is called immediately and its
         * result is passed to {@link #outputResources(OutputResources)}.
         * 
         * @param outputResources
         *        a consumer that will call methods on {@link OutputResources.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #outputResources(OutputResources)
         */
        default Builder outputResources(Consumer<OutputResources.Builder> outputResources) {
            return outputResources(OutputResources.builder().applyMutation(outputResources).build());
        }

        /**
         * <p>
         * The tags of the image.
         * </p>
         * 
         * @param tags
         *        The tags of the image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private String type;

        private String name;

        private String version;

        private String platform;

        private Boolean enhancedImageMetadataEnabled;

        private String osVersion;

        private ImageState state;

        private ImageRecipe imageRecipe;

        private ContainerRecipe containerRecipe;

        private String sourcePipelineName;

        private String sourcePipelineArn;

        private InfrastructureConfiguration infrastructureConfiguration;

        private DistributionConfiguration distributionConfiguration;

        private ImageTestsConfiguration imageTestsConfiguration;

        private String dateCreated;

        private OutputResources outputResources;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Image model) {
            arn(model.arn);
            type(model.type);
            name(model.name);
            version(model.version);
            platform(model.platform);
            enhancedImageMetadataEnabled(model.enhancedImageMetadataEnabled);
            osVersion(model.osVersion);
            state(model.state);
            imageRecipe(model.imageRecipe);
            containerRecipe(model.containerRecipe);
            sourcePipelineName(model.sourcePipelineName);
            sourcePipelineArn(model.sourcePipelineArn);
            infrastructureConfiguration(model.infrastructureConfiguration);
            distributionConfiguration(model.distributionConfiguration);
            imageTestsConfiguration(model.imageTestsConfiguration);
            dateCreated(model.dateCreated);
            outputResources(model.outputResources);
            tags(model.tags);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        @Transient
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        @Transient
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        @Transient
        public final Builder type(ImageType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        @Transient
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final String getPlatform() {
            return platform;
        }

        public final void setPlatform(String platform) {
            this.platform = platform;
        }

        @Override
        @Transient
        public final Builder platform(String platform) {
            this.platform = platform;
            return this;
        }

        @Override
        @Transient
        public final Builder platform(Platform platform) {
            this.platform(platform == null ? null : platform.toString());
            return this;
        }

        public final Boolean getEnhancedImageMetadataEnabled() {
            return enhancedImageMetadataEnabled;
        }

        public final void setEnhancedImageMetadataEnabled(Boolean enhancedImageMetadataEnabled) {
            this.enhancedImageMetadataEnabled = enhancedImageMetadataEnabled;
        }

        @Override
        @Transient
        public final Builder enhancedImageMetadataEnabled(Boolean enhancedImageMetadataEnabled) {
            this.enhancedImageMetadataEnabled = enhancedImageMetadataEnabled;
            return this;
        }

        public final String getOsVersion() {
            return osVersion;
        }

        public final void setOsVersion(String osVersion) {
            this.osVersion = osVersion;
        }

        @Override
        @Transient
        public final Builder osVersion(String osVersion) {
            this.osVersion = osVersion;
            return this;
        }

        public final ImageState.Builder getState() {
            return state != null ? state.toBuilder() : null;
        }

        public final void setState(ImageState.BuilderImpl state) {
            this.state = state != null ? state.build() : null;
        }

        @Override
        @Transient
        public final Builder state(ImageState state) {
            this.state = state;
            return this;
        }

        public final ImageRecipe.Builder getImageRecipe() {
            return imageRecipe != null ? imageRecipe.toBuilder() : null;
        }

        public final void setImageRecipe(ImageRecipe.BuilderImpl imageRecipe) {
            this.imageRecipe = imageRecipe != null ? imageRecipe.build() : null;
        }

        @Override
        @Transient
        public final Builder imageRecipe(ImageRecipe imageRecipe) {
            this.imageRecipe = imageRecipe;
            return this;
        }

        public final ContainerRecipe.Builder getContainerRecipe() {
            return containerRecipe != null ? containerRecipe.toBuilder() : null;
        }

        public final void setContainerRecipe(ContainerRecipe.BuilderImpl containerRecipe) {
            this.containerRecipe = containerRecipe != null ? containerRecipe.build() : null;
        }

        @Override
        @Transient
        public final Builder containerRecipe(ContainerRecipe containerRecipe) {
            this.containerRecipe = containerRecipe;
            return this;
        }

        public final String getSourcePipelineName() {
            return sourcePipelineName;
        }

        public final void setSourcePipelineName(String sourcePipelineName) {
            this.sourcePipelineName = sourcePipelineName;
        }

        @Override
        @Transient
        public final Builder sourcePipelineName(String sourcePipelineName) {
            this.sourcePipelineName = sourcePipelineName;
            return this;
        }

        public final String getSourcePipelineArn() {
            return sourcePipelineArn;
        }

        public final void setSourcePipelineArn(String sourcePipelineArn) {
            this.sourcePipelineArn = sourcePipelineArn;
        }

        @Override
        @Transient
        public final Builder sourcePipelineArn(String sourcePipelineArn) {
            this.sourcePipelineArn = sourcePipelineArn;
            return this;
        }

        public final InfrastructureConfiguration.Builder getInfrastructureConfiguration() {
            return infrastructureConfiguration != null ? infrastructureConfiguration.toBuilder() : null;
        }

        public final void setInfrastructureConfiguration(InfrastructureConfiguration.BuilderImpl infrastructureConfiguration) {
            this.infrastructureConfiguration = infrastructureConfiguration != null ? infrastructureConfiguration.build() : null;
        }

        @Override
        @Transient
        public final Builder infrastructureConfiguration(InfrastructureConfiguration infrastructureConfiguration) {
            this.infrastructureConfiguration = infrastructureConfiguration;
            return this;
        }

        public final DistributionConfiguration.Builder getDistributionConfiguration() {
            return distributionConfiguration != null ? distributionConfiguration.toBuilder() : null;
        }

        public final void setDistributionConfiguration(DistributionConfiguration.BuilderImpl distributionConfiguration) {
            this.distributionConfiguration = distributionConfiguration != null ? distributionConfiguration.build() : null;
        }

        @Override
        @Transient
        public final Builder distributionConfiguration(DistributionConfiguration distributionConfiguration) {
            this.distributionConfiguration = distributionConfiguration;
            return this;
        }

        public final ImageTestsConfiguration.Builder getImageTestsConfiguration() {
            return imageTestsConfiguration != null ? imageTestsConfiguration.toBuilder() : null;
        }

        public final void setImageTestsConfiguration(ImageTestsConfiguration.BuilderImpl imageTestsConfiguration) {
            this.imageTestsConfiguration = imageTestsConfiguration != null ? imageTestsConfiguration.build() : null;
        }

        @Override
        @Transient
        public final Builder imageTestsConfiguration(ImageTestsConfiguration imageTestsConfiguration) {
            this.imageTestsConfiguration = imageTestsConfiguration;
            return this;
        }

        public final String getDateCreated() {
            return dateCreated;
        }

        public final void setDateCreated(String dateCreated) {
            this.dateCreated = dateCreated;
        }

        @Override
        @Transient
        public final Builder dateCreated(String dateCreated) {
            this.dateCreated = dateCreated;
            return this;
        }

        public final OutputResources.Builder getOutputResources() {
            return outputResources != null ? outputResources.toBuilder() : null;
        }

        public final void setOutputResources(OutputResources.BuilderImpl outputResources) {
            this.outputResources = outputResources != null ? outputResources.build() : null;
        }

        @Override
        @Transient
        public final Builder outputResources(OutputResources outputResources) {
            this.outputResources = outputResources;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        @Transient
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        @Override
        public Image build() {
            return new Image(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
