/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.imagebuilder.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An Image Builder image. You must specify exactly one recipe for the image – either a container recipe (
 * <code>containerRecipe</code>), which creates a container image, or an image recipe (<code>imageRecipe</code>), which
 * creates an AMI.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Image implements SdkPojo, Serializable, ToCopyableBuilder<Image.Builder, Image> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(Image::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(Image::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Image::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("version")
            .getter(getter(Image::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("version").build()).build();

    private static final SdkField<String> PLATFORM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("platform").getter(getter(Image::platformAsString)).setter(setter(Builder::platform))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("platform").build()).build();

    private static final SdkField<Boolean> ENHANCED_IMAGE_METADATA_ENABLED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("enhancedImageMetadataEnabled")
            .getter(getter(Image::enhancedImageMetadataEnabled))
            .setter(setter(Builder::enhancedImageMetadataEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("enhancedImageMetadataEnabled")
                    .build()).build();

    private static final SdkField<String> OS_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("osVersion").getter(getter(Image::osVersion)).setter(setter(Builder::osVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("osVersion").build()).build();

    private static final SdkField<ImageState> STATE_FIELD = SdkField.<ImageState> builder(MarshallingType.SDK_POJO)
            .memberName("state").getter(getter(Image::state)).setter(setter(Builder::state)).constructor(ImageState::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<ImageRecipe> IMAGE_RECIPE_FIELD = SdkField.<ImageRecipe> builder(MarshallingType.SDK_POJO)
            .memberName("imageRecipe").getter(getter(Image::imageRecipe)).setter(setter(Builder::imageRecipe))
            .constructor(ImageRecipe::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imageRecipe").build()).build();

    private static final SdkField<ContainerRecipe> CONTAINER_RECIPE_FIELD = SdkField
            .<ContainerRecipe> builder(MarshallingType.SDK_POJO).memberName("containerRecipe")
            .getter(getter(Image::containerRecipe)).setter(setter(Builder::containerRecipe))
            .constructor(ContainerRecipe::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerRecipe").build()).build();

    private static final SdkField<String> SOURCE_PIPELINE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sourcePipelineName").getter(getter(Image::sourcePipelineName))
            .setter(setter(Builder::sourcePipelineName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourcePipelineName").build())
            .build();

    private static final SdkField<String> SOURCE_PIPELINE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sourcePipelineArn").getter(getter(Image::sourcePipelineArn)).setter(setter(Builder::sourcePipelineArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourcePipelineArn").build()).build();

    private static final SdkField<InfrastructureConfiguration> INFRASTRUCTURE_CONFIGURATION_FIELD = SdkField
            .<InfrastructureConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("infrastructureConfiguration")
            .getter(getter(Image::infrastructureConfiguration))
            .setter(setter(Builder::infrastructureConfiguration))
            .constructor(InfrastructureConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("infrastructureConfiguration")
                    .build()).build();

    private static final SdkField<DistributionConfiguration> DISTRIBUTION_CONFIGURATION_FIELD = SdkField
            .<DistributionConfiguration> builder(MarshallingType.SDK_POJO).memberName("distributionConfiguration")
            .getter(getter(Image::distributionConfiguration)).setter(setter(Builder::distributionConfiguration))
            .constructor(DistributionConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("distributionConfiguration").build())
            .build();

    private static final SdkField<ImageTestsConfiguration> IMAGE_TESTS_CONFIGURATION_FIELD = SdkField
            .<ImageTestsConfiguration> builder(MarshallingType.SDK_POJO).memberName("imageTestsConfiguration")
            .getter(getter(Image::imageTestsConfiguration)).setter(setter(Builder::imageTestsConfiguration))
            .constructor(ImageTestsConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imageTestsConfiguration").build())
            .build();

    private static final SdkField<String> DATE_CREATED_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("dateCreated").getter(getter(Image::dateCreated)).setter(setter(Builder::dateCreated))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dateCreated").build()).build();

    private static final SdkField<OutputResources> OUTPUT_RESOURCES_FIELD = SdkField
            .<OutputResources> builder(MarshallingType.SDK_POJO).memberName("outputResources")
            .getter(getter(Image::outputResources)).setter(setter(Builder::outputResources))
            .constructor(OutputResources::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("outputResources").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(Image::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> BUILD_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("buildType").getter(getter(Image::buildTypeAsString)).setter(setter(Builder::buildType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("buildType").build()).build();

    private static final SdkField<String> IMAGE_SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("imageSource").getter(getter(Image::imageSourceAsString)).setter(setter(Builder::imageSource))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imageSource").build()).build();

    private static final SdkField<ImageScanState> SCAN_STATE_FIELD = SdkField.<ImageScanState> builder(MarshallingType.SDK_POJO)
            .memberName("scanState").getter(getter(Image::scanState)).setter(setter(Builder::scanState))
            .constructor(ImageScanState::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("scanState").build()).build();

    private static final SdkField<ImageScanningConfiguration> IMAGE_SCANNING_CONFIGURATION_FIELD = SdkField
            .<ImageScanningConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("imageScanningConfiguration")
            .getter(getter(Image::imageScanningConfiguration))
            .setter(setter(Builder::imageScanningConfiguration))
            .constructor(ImageScanningConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imageScanningConfiguration").build())
            .build();

    private static final SdkField<Instant> DEPRECATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("deprecationTime").getter(getter(Image::deprecationTime)).setter(setter(Builder::deprecationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deprecationTime").build()).build();

    private static final SdkField<String> LIFECYCLE_EXECUTION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("lifecycleExecutionId").getter(getter(Image::lifecycleExecutionId))
            .setter(setter(Builder::lifecycleExecutionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lifecycleExecutionId").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD, TYPE_FIELD,
            NAME_FIELD, VERSION_FIELD, PLATFORM_FIELD, ENHANCED_IMAGE_METADATA_ENABLED_FIELD, OS_VERSION_FIELD, STATE_FIELD,
            IMAGE_RECIPE_FIELD, CONTAINER_RECIPE_FIELD, SOURCE_PIPELINE_NAME_FIELD, SOURCE_PIPELINE_ARN_FIELD,
            INFRASTRUCTURE_CONFIGURATION_FIELD, DISTRIBUTION_CONFIGURATION_FIELD, IMAGE_TESTS_CONFIGURATION_FIELD,
            DATE_CREATED_FIELD, OUTPUT_RESOURCES_FIELD, TAGS_FIELD, BUILD_TYPE_FIELD, IMAGE_SOURCE_FIELD, SCAN_STATE_FIELD,
            IMAGE_SCANNING_CONFIGURATION_FIELD, DEPRECATION_TIME_FIELD, LIFECYCLE_EXECUTION_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final String type;

    private final String name;

    private final String version;

    private final String platform;

    private final Boolean enhancedImageMetadataEnabled;

    private final String osVersion;

    private final ImageState state;

    private final ImageRecipe imageRecipe;

    private final ContainerRecipe containerRecipe;

    private final String sourcePipelineName;

    private final String sourcePipelineArn;

    private final InfrastructureConfiguration infrastructureConfiguration;

    private final DistributionConfiguration distributionConfiguration;

    private final ImageTestsConfiguration imageTestsConfiguration;

    private final String dateCreated;

    private final OutputResources outputResources;

    private final Map<String, String> tags;

    private final String buildType;

    private final String imageSource;

    private final ImageScanState scanState;

    private final ImageScanningConfiguration imageScanningConfiguration;

    private final Instant deprecationTime;

    private final String lifecycleExecutionId;

    private Image(BuilderImpl builder) {
        this.arn = builder.arn;
        this.type = builder.type;
        this.name = builder.name;
        this.version = builder.version;
        this.platform = builder.platform;
        this.enhancedImageMetadataEnabled = builder.enhancedImageMetadataEnabled;
        this.osVersion = builder.osVersion;
        this.state = builder.state;
        this.imageRecipe = builder.imageRecipe;
        this.containerRecipe = builder.containerRecipe;
        this.sourcePipelineName = builder.sourcePipelineName;
        this.sourcePipelineArn = builder.sourcePipelineArn;
        this.infrastructureConfiguration = builder.infrastructureConfiguration;
        this.distributionConfiguration = builder.distributionConfiguration;
        this.imageTestsConfiguration = builder.imageTestsConfiguration;
        this.dateCreated = builder.dateCreated;
        this.outputResources = builder.outputResources;
        this.tags = builder.tags;
        this.buildType = builder.buildType;
        this.imageSource = builder.imageSource;
        this.scanState = builder.scanState;
        this.imageScanningConfiguration = builder.imageScanningConfiguration;
        this.deprecationTime = builder.deprecationTime;
        this.lifecycleExecutionId = builder.lifecycleExecutionId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the image.
     * </p>
     * <note>
     * <p>
     * Semantic versioning is included in each object's Amazon Resource Name (ARN), at the level that applies to that
     * object as follows:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Versionless ARNs and Name ARNs do not include specific values in any of the nodes. The nodes are either left off
     * entirely, or they are specified as wildcards, for example: x.x.x.
     * </p>
     * </li>
     * <li>
     * <p>
     * Version ARNs have only the first three nodes: &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;
     * </p>
     * </li>
     * <li>
     * <p>
     * Build version ARNs have all four nodes, and point to a specific build for a specific version of an object.
     * </p>
     * </li>
     * </ol>
     * </note>
     * 
     * @return The Amazon Resource Name (ARN) of the image.</p> <note>
     *         <p>
     *         Semantic versioning is included in each object's Amazon Resource Name (ARN), at the level that applies to
     *         that object as follows:
     *         </p>
     *         <ol>
     *         <li>
     *         <p>
     *         Versionless ARNs and Name ARNs do not include specific values in any of the nodes. The nodes are either
     *         left off entirely, or they are specified as wildcards, for example: x.x.x.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Version ARNs have only the first three nodes: &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Build version ARNs have all four nodes, and point to a specific build for a specific version of an
     *         object.
     *         </p>
     *         </li>
     *         </ol>
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * Specifies whether this image produces an AMI or a container image.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ImageType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Specifies whether this image produces an AMI or a container image.
     * @see ImageType
     */
    public final ImageType type() {
        return ImageType.fromValue(type);
    }

    /**
     * <p>
     * Specifies whether this image produces an AMI or a container image.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ImageType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Specifies whether this image produces an AMI or a container image.
     * @see ImageType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The name of the image.
     * </p>
     * 
     * @return The name of the image.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The semantic version of the image.
     * </p>
     * <note>
     * <p>
     * The semantic version has four nodes: &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;/&lt;build&gt;. You can assign
     * values for the first three, and can filter on all of them.
     * </p>
     * <p>
     * <b>Assignment:</b> For the first three nodes you can assign any positive integer value, including zero, with an
     * upper limit of 2^30-1, or 1073741823 for each node. Image Builder automatically assigns the build number to the
     * fourth node.
     * </p>
     * <p>
     * <b>Patterns:</b> You can use any numeric pattern that adheres to the assignment requirements for the nodes that
     * you can assign. For example, you might choose a software version pattern, such as 1.0.0, or a date, such as
     * 2021.01.01.
     * </p>
     * <p>
     * <b>Filtering:</b> With semantic versioning, you have the flexibility to use wildcards (x) to specify the most
     * recent versions or nodes when selecting the base image or components for your recipe. When you use a wildcard in
     * any node, all nodes to the right of the first wildcard must also be wildcards.
     * </p>
     * </note>
     * 
     * @return The semantic version of the image.</p> <note>
     *         <p>
     *         The semantic version has four nodes: &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;/&lt;build&gt;. You can
     *         assign values for the first three, and can filter on all of them.
     *         </p>
     *         <p>
     *         <b>Assignment:</b> For the first three nodes you can assign any positive integer value, including zero,
     *         with an upper limit of 2^30-1, or 1073741823 for each node. Image Builder automatically assigns the build
     *         number to the fourth node.
     *         </p>
     *         <p>
     *         <b>Patterns:</b> You can use any numeric pattern that adheres to the assignment requirements for the
     *         nodes that you can assign. For example, you might choose a software version pattern, such as 1.0.0, or a
     *         date, such as 2021.01.01.
     *         </p>
     *         <p>
     *         <b>Filtering:</b> With semantic versioning, you have the flexibility to use wildcards (x) to specify the
     *         most recent versions or nodes when selecting the base image or components for your recipe. When you use a
     *         wildcard in any node, all nodes to the right of the first wildcard must also be wildcards.
     *         </p>
     */
    public final String version() {
        return version;
    }

    /**
     * <p>
     * The image operating system platform, such as Linux or Windows.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #platform} will
     * return {@link Platform#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #platformAsString}.
     * </p>
     * 
     * @return The image operating system platform, such as Linux or Windows.
     * @see Platform
     */
    public final Platform platform() {
        return Platform.fromValue(platform);
    }

    /**
     * <p>
     * The image operating system platform, such as Linux or Windows.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #platform} will
     * return {@link Platform#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #platformAsString}.
     * </p>
     * 
     * @return The image operating system platform, such as Linux or Windows.
     * @see Platform
     */
    public final String platformAsString() {
        return platform;
    }

    /**
     * <p>
     * Indicates whether Image Builder collects additional information about the image, such as the operating system
     * (OS) version and package list.
     * </p>
     * 
     * @return Indicates whether Image Builder collects additional information about the image, such as the operating
     *         system (OS) version and package list.
     */
    public final Boolean enhancedImageMetadataEnabled() {
        return enhancedImageMetadataEnabled;
    }

    /**
     * <p>
     * The operating system version for instances that launch from this image. For example, Amazon Linux 2, Ubuntu 18,
     * or Microsoft Windows Server 2019.
     * </p>
     * 
     * @return The operating system version for instances that launch from this image. For example, Amazon Linux 2,
     *         Ubuntu 18, or Microsoft Windows Server 2019.
     */
    public final String osVersion() {
        return osVersion;
    }

    /**
     * <p>
     * The state of the image.
     * </p>
     * 
     * @return The state of the image.
     */
    public final ImageState state() {
        return state;
    }

    /**
     * <p>
     * For images that distribute an AMI, this is the image recipe that Image Builder used to create the image. For
     * container images, this is empty.
     * </p>
     * 
     * @return For images that distribute an AMI, this is the image recipe that Image Builder used to create the image.
     *         For container images, this is empty.
     */
    public final ImageRecipe imageRecipe() {
        return imageRecipe;
    }

    /**
     * <p>
     * For container images, this is the container recipe that Image Builder used to create the image. For images that
     * distribute an AMI, this is empty.
     * </p>
     * 
     * @return For container images, this is the container recipe that Image Builder used to create the image. For
     *         images that distribute an AMI, this is empty.
     */
    public final ContainerRecipe containerRecipe() {
        return containerRecipe;
    }

    /**
     * <p>
     * The name of the image pipeline that created this image.
     * </p>
     * 
     * @return The name of the image pipeline that created this image.
     */
    public final String sourcePipelineName() {
        return sourcePipelineName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the image pipeline that created this image.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the image pipeline that created this image.
     */
    public final String sourcePipelineArn() {
        return sourcePipelineArn;
    }

    /**
     * <p>
     * The infrastructure that Image Builder used to create this image.
     * </p>
     * 
     * @return The infrastructure that Image Builder used to create this image.
     */
    public final InfrastructureConfiguration infrastructureConfiguration() {
        return infrastructureConfiguration;
    }

    /**
     * <p>
     * The distribution configuration that Image Builder used to create this image.
     * </p>
     * 
     * @return The distribution configuration that Image Builder used to create this image.
     */
    public final DistributionConfiguration distributionConfiguration() {
        return distributionConfiguration;
    }

    /**
     * <p>
     * The image tests that ran when that Image Builder created this image.
     * </p>
     * 
     * @return The image tests that ran when that Image Builder created this image.
     */
    public final ImageTestsConfiguration imageTestsConfiguration() {
        return imageTestsConfiguration;
    }

    /**
     * <p>
     * The date on which Image Builder created this image.
     * </p>
     * 
     * @return The date on which Image Builder created this image.
     */
    public final String dateCreated() {
        return dateCreated;
    }

    /**
     * <p>
     * The output resources that Image Builder produces for this image.
     * </p>
     * 
     * @return The output resources that Image Builder produces for this image.
     */
    public final OutputResources outputResources() {
        return outputResources;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The tags that apply to this image.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags that apply to this image.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * Indicates the type of build that created this image. The build can be initiated in the following ways:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>USER_INITIATED</b> – A manual pipeline build request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>SCHEDULED</b> – A pipeline build initiated by a cron expression in the Image Builder pipeline, or from
     * EventBridge.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>IMPORT</b> – A VM import created the image to use as the base image for the recipe.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #buildType} will
     * return {@link BuildType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #buildTypeAsString}.
     * </p>
     * 
     * @return Indicates the type of build that created this image. The build can be initiated in the following
     *         ways:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>USER_INITIATED</b> – A manual pipeline build request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>SCHEDULED</b> – A pipeline build initiated by a cron expression in the Image Builder pipeline, or from
     *         EventBridge.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IMPORT</b> – A VM import created the image to use as the base image for the recipe.
     *         </p>
     *         </li>
     * @see BuildType
     */
    public final BuildType buildType() {
        return BuildType.fromValue(buildType);
    }

    /**
     * <p>
     * Indicates the type of build that created this image. The build can be initiated in the following ways:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>USER_INITIATED</b> – A manual pipeline build request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>SCHEDULED</b> – A pipeline build initiated by a cron expression in the Image Builder pipeline, or from
     * EventBridge.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>IMPORT</b> – A VM import created the image to use as the base image for the recipe.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #buildType} will
     * return {@link BuildType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #buildTypeAsString}.
     * </p>
     * 
     * @return Indicates the type of build that created this image. The build can be initiated in the following
     *         ways:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>USER_INITIATED</b> – A manual pipeline build request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>SCHEDULED</b> – A pipeline build initiated by a cron expression in the Image Builder pipeline, or from
     *         EventBridge.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IMPORT</b> – A VM import created the image to use as the base image for the recipe.
     *         </p>
     *         </li>
     * @see BuildType
     */
    public final String buildTypeAsString() {
        return buildType;
    }

    /**
     * <p>
     * The origin of the base image that Image Builder used to build this image.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #imageSource} will
     * return {@link ImageSource#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #imageSourceAsString}.
     * </p>
     * 
     * @return The origin of the base image that Image Builder used to build this image.
     * @see ImageSource
     */
    public final ImageSource imageSource() {
        return ImageSource.fromValue(imageSource);
    }

    /**
     * <p>
     * The origin of the base image that Image Builder used to build this image.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #imageSource} will
     * return {@link ImageSource#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #imageSourceAsString}.
     * </p>
     * 
     * @return The origin of the base image that Image Builder used to build this image.
     * @see ImageSource
     */
    public final String imageSourceAsString() {
        return imageSource;
    }

    /**
     * <p>
     * Contains information about the current state of scans for this image.
     * </p>
     * 
     * @return Contains information about the current state of scans for this image.
     */
    public final ImageScanState scanState() {
        return scanState;
    }

    /**
     * <p>
     * Contains settings for vulnerability scans.
     * </p>
     * 
     * @return Contains settings for vulnerability scans.
     */
    public final ImageScanningConfiguration imageScanningConfiguration() {
        return imageScanningConfiguration;
    }

    /**
     * <p>
     * The time when deprecation occurs for an image resource. This can be a past or future date.
     * </p>
     * 
     * @return The time when deprecation occurs for an image resource. This can be a past or future date.
     */
    public final Instant deprecationTime() {
        return deprecationTime;
    }

    /**
     * <p>
     * Identifies the last runtime instance of the lifecycle policy to take action on the image.
     * </p>
     * 
     * @return Identifies the last runtime instance of the lifecycle policy to take action on the image.
     */
    public final String lifecycleExecutionId() {
        return lifecycleExecutionId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(platformAsString());
        hashCode = 31 * hashCode + Objects.hashCode(enhancedImageMetadataEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(osVersion());
        hashCode = 31 * hashCode + Objects.hashCode(state());
        hashCode = 31 * hashCode + Objects.hashCode(imageRecipe());
        hashCode = 31 * hashCode + Objects.hashCode(containerRecipe());
        hashCode = 31 * hashCode + Objects.hashCode(sourcePipelineName());
        hashCode = 31 * hashCode + Objects.hashCode(sourcePipelineArn());
        hashCode = 31 * hashCode + Objects.hashCode(infrastructureConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(distributionConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(imageTestsConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(dateCreated());
        hashCode = 31 * hashCode + Objects.hashCode(outputResources());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(buildTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(imageSourceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(scanState());
        hashCode = 31 * hashCode + Objects.hashCode(imageScanningConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(deprecationTime());
        hashCode = 31 * hashCode + Objects.hashCode(lifecycleExecutionId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Image)) {
            return false;
        }
        Image other = (Image) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(name(), other.name()) && Objects.equals(version(), other.version())
                && Objects.equals(platformAsString(), other.platformAsString())
                && Objects.equals(enhancedImageMetadataEnabled(), other.enhancedImageMetadataEnabled())
                && Objects.equals(osVersion(), other.osVersion()) && Objects.equals(state(), other.state())
                && Objects.equals(imageRecipe(), other.imageRecipe())
                && Objects.equals(containerRecipe(), other.containerRecipe())
                && Objects.equals(sourcePipelineName(), other.sourcePipelineName())
                && Objects.equals(sourcePipelineArn(), other.sourcePipelineArn())
                && Objects.equals(infrastructureConfiguration(), other.infrastructureConfiguration())
                && Objects.equals(distributionConfiguration(), other.distributionConfiguration())
                && Objects.equals(imageTestsConfiguration(), other.imageTestsConfiguration())
                && Objects.equals(dateCreated(), other.dateCreated())
                && Objects.equals(outputResources(), other.outputResources()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(buildTypeAsString(), other.buildTypeAsString())
                && Objects.equals(imageSourceAsString(), other.imageSourceAsString())
                && Objects.equals(scanState(), other.scanState())
                && Objects.equals(imageScanningConfiguration(), other.imageScanningConfiguration())
                && Objects.equals(deprecationTime(), other.deprecationTime())
                && Objects.equals(lifecycleExecutionId(), other.lifecycleExecutionId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Image").add("Arn", arn()).add("Type", typeAsString()).add("Name", name())
                .add("Version", version()).add("Platform", platformAsString())
                .add("EnhancedImageMetadataEnabled", enhancedImageMetadataEnabled()).add("OsVersion", osVersion())
                .add("State", state()).add("ImageRecipe", imageRecipe()).add("ContainerRecipe", containerRecipe())
                .add("SourcePipelineName", sourcePipelineName()).add("SourcePipelineArn", sourcePipelineArn())
                .add("InfrastructureConfiguration", infrastructureConfiguration())
                .add("DistributionConfiguration", distributionConfiguration())
                .add("ImageTestsConfiguration", imageTestsConfiguration()).add("DateCreated", dateCreated())
                .add("OutputResources", outputResources()).add("Tags", hasTags() ? tags() : null)
                .add("BuildType", buildTypeAsString()).add("ImageSource", imageSourceAsString()).add("ScanState", scanState())
                .add("ImageScanningConfiguration", imageScanningConfiguration()).add("DeprecationTime", deprecationTime())
                .add("LifecycleExecutionId", lifecycleExecutionId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "version":
            return Optional.ofNullable(clazz.cast(version()));
        case "platform":
            return Optional.ofNullable(clazz.cast(platformAsString()));
        case "enhancedImageMetadataEnabled":
            return Optional.ofNullable(clazz.cast(enhancedImageMetadataEnabled()));
        case "osVersion":
            return Optional.ofNullable(clazz.cast(osVersion()));
        case "state":
            return Optional.ofNullable(clazz.cast(state()));
        case "imageRecipe":
            return Optional.ofNullable(clazz.cast(imageRecipe()));
        case "containerRecipe":
            return Optional.ofNullable(clazz.cast(containerRecipe()));
        case "sourcePipelineName":
            return Optional.ofNullable(clazz.cast(sourcePipelineName()));
        case "sourcePipelineArn":
            return Optional.ofNullable(clazz.cast(sourcePipelineArn()));
        case "infrastructureConfiguration":
            return Optional.ofNullable(clazz.cast(infrastructureConfiguration()));
        case "distributionConfiguration":
            return Optional.ofNullable(clazz.cast(distributionConfiguration()));
        case "imageTestsConfiguration":
            return Optional.ofNullable(clazz.cast(imageTestsConfiguration()));
        case "dateCreated":
            return Optional.ofNullable(clazz.cast(dateCreated()));
        case "outputResources":
            return Optional.ofNullable(clazz.cast(outputResources()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "buildType":
            return Optional.ofNullable(clazz.cast(buildTypeAsString()));
        case "imageSource":
            return Optional.ofNullable(clazz.cast(imageSourceAsString()));
        case "scanState":
            return Optional.ofNullable(clazz.cast(scanState()));
        case "imageScanningConfiguration":
            return Optional.ofNullable(clazz.cast(imageScanningConfiguration()));
        case "deprecationTime":
            return Optional.ofNullable(clazz.cast(deprecationTime()));
        case "lifecycleExecutionId":
            return Optional.ofNullable(clazz.cast(lifecycleExecutionId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Image, T> g) {
        return obj -> g.apply((Image) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Image> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the image.
         * </p>
         * <note>
         * <p>
         * Semantic versioning is included in each object's Amazon Resource Name (ARN), at the level that applies to
         * that object as follows:
         * </p>
         * <ol>
         * <li>
         * <p>
         * Versionless ARNs and Name ARNs do not include specific values in any of the nodes. The nodes are either left
         * off entirely, or they are specified as wildcards, for example: x.x.x.
         * </p>
         * </li>
         * <li>
         * <p>
         * Version ARNs have only the first three nodes: &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;
         * </p>
         * </li>
         * <li>
         * <p>
         * Build version ARNs have all four nodes, and point to a specific build for a specific version of an object.
         * </p>
         * </li>
         * </ol>
         * </note>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the image.</p> <note>
         *        <p>
         *        Semantic versioning is included in each object's Amazon Resource Name (ARN), at the level that applies
         *        to that object as follows:
         *        </p>
         *        <ol>
         *        <li>
         *        <p>
         *        Versionless ARNs and Name ARNs do not include specific values in any of the nodes. The nodes are
         *        either left off entirely, or they are specified as wildcards, for example: x.x.x.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Version ARNs have only the first three nodes: &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Build version ARNs have all four nodes, and point to a specific build for a specific version of an
         *        object.
         *        </p>
         *        </li>
         *        </ol>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * Specifies whether this image produces an AMI or a container image.
         * </p>
         * 
         * @param type
         *        Specifies whether this image produces an AMI or a container image.
         * @see ImageType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImageType
         */
        Builder type(String type);

        /**
         * <p>
         * Specifies whether this image produces an AMI or a container image.
         * </p>
         * 
         * @param type
         *        Specifies whether this image produces an AMI or a container image.
         * @see ImageType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImageType
         */
        Builder type(ImageType type);

        /**
         * <p>
         * The name of the image.
         * </p>
         * 
         * @param name
         *        The name of the image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The semantic version of the image.
         * </p>
         * <note>
         * <p>
         * The semantic version has four nodes: &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;/&lt;build&gt;. You can assign
         * values for the first three, and can filter on all of them.
         * </p>
         * <p>
         * <b>Assignment:</b> For the first three nodes you can assign any positive integer value, including zero, with
         * an upper limit of 2^30-1, or 1073741823 for each node. Image Builder automatically assigns the build number
         * to the fourth node.
         * </p>
         * <p>
         * <b>Patterns:</b> You can use any numeric pattern that adheres to the assignment requirements for the nodes
         * that you can assign. For example, you might choose a software version pattern, such as 1.0.0, or a date, such
         * as 2021.01.01.
         * </p>
         * <p>
         * <b>Filtering:</b> With semantic versioning, you have the flexibility to use wildcards (x) to specify the most
         * recent versions or nodes when selecting the base image or components for your recipe. When you use a wildcard
         * in any node, all nodes to the right of the first wildcard must also be wildcards.
         * </p>
         * </note>
         * 
         * @param version
         *        The semantic version of the image.</p> <note>
         *        <p>
         *        The semantic version has four nodes: &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;/&lt;build&gt;. You can
         *        assign values for the first three, and can filter on all of them.
         *        </p>
         *        <p>
         *        <b>Assignment:</b> For the first three nodes you can assign any positive integer value, including
         *        zero, with an upper limit of 2^30-1, or 1073741823 for each node. Image Builder automatically assigns
         *        the build number to the fourth node.
         *        </p>
         *        <p>
         *        <b>Patterns:</b> You can use any numeric pattern that adheres to the assignment requirements for the
         *        nodes that you can assign. For example, you might choose a software version pattern, such as 1.0.0, or
         *        a date, such as 2021.01.01.
         *        </p>
         *        <p>
         *        <b>Filtering:</b> With semantic versioning, you have the flexibility to use wildcards (x) to specify
         *        the most recent versions or nodes when selecting the base image or components for your recipe. When
         *        you use a wildcard in any node, all nodes to the right of the first wildcard must also be wildcards.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * The image operating system platform, such as Linux or Windows.
         * </p>
         * 
         * @param platform
         *        The image operating system platform, such as Linux or Windows.
         * @see Platform
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Platform
         */
        Builder platform(String platform);

        /**
         * <p>
         * The image operating system platform, such as Linux or Windows.
         * </p>
         * 
         * @param platform
         *        The image operating system platform, such as Linux or Windows.
         * @see Platform
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Platform
         */
        Builder platform(Platform platform);

        /**
         * <p>
         * Indicates whether Image Builder collects additional information about the image, such as the operating system
         * (OS) version and package list.
         * </p>
         * 
         * @param enhancedImageMetadataEnabled
         *        Indicates whether Image Builder collects additional information about the image, such as the operating
         *        system (OS) version and package list.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enhancedImageMetadataEnabled(Boolean enhancedImageMetadataEnabled);

        /**
         * <p>
         * The operating system version for instances that launch from this image. For example, Amazon Linux 2, Ubuntu
         * 18, or Microsoft Windows Server 2019.
         * </p>
         * 
         * @param osVersion
         *        The operating system version for instances that launch from this image. For example, Amazon Linux 2,
         *        Ubuntu 18, or Microsoft Windows Server 2019.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder osVersion(String osVersion);

        /**
         * <p>
         * The state of the image.
         * </p>
         * 
         * @param state
         *        The state of the image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder state(ImageState state);

        /**
         * <p>
         * The state of the image.
         * </p>
         * This is a convenience method that creates an instance of the {@link ImageState.Builder} avoiding the need to
         * create one manually via {@link ImageState#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ImageState.Builder#build()} is called immediately and its result
         * is passed to {@link #state(ImageState)}.
         * 
         * @param state
         *        a consumer that will call methods on {@link ImageState.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #state(ImageState)
         */
        default Builder state(Consumer<ImageState.Builder> state) {
            return state(ImageState.builder().applyMutation(state).build());
        }

        /**
         * <p>
         * For images that distribute an AMI, this is the image recipe that Image Builder used to create the image. For
         * container images, this is empty.
         * </p>
         * 
         * @param imageRecipe
         *        For images that distribute an AMI, this is the image recipe that Image Builder used to create the
         *        image. For container images, this is empty.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageRecipe(ImageRecipe imageRecipe);

        /**
         * <p>
         * For images that distribute an AMI, this is the image recipe that Image Builder used to create the image. For
         * container images, this is empty.
         * </p>
         * This is a convenience method that creates an instance of the {@link ImageRecipe.Builder} avoiding the need to
         * create one manually via {@link ImageRecipe#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ImageRecipe.Builder#build()} is called immediately and its result
         * is passed to {@link #imageRecipe(ImageRecipe)}.
         * 
         * @param imageRecipe
         *        a consumer that will call methods on {@link ImageRecipe.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #imageRecipe(ImageRecipe)
         */
        default Builder imageRecipe(Consumer<ImageRecipe.Builder> imageRecipe) {
            return imageRecipe(ImageRecipe.builder().applyMutation(imageRecipe).build());
        }

        /**
         * <p>
         * For container images, this is the container recipe that Image Builder used to create the image. For images
         * that distribute an AMI, this is empty.
         * </p>
         * 
         * @param containerRecipe
         *        For container images, this is the container recipe that Image Builder used to create the image. For
         *        images that distribute an AMI, this is empty.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerRecipe(ContainerRecipe containerRecipe);

        /**
         * <p>
         * For container images, this is the container recipe that Image Builder used to create the image. For images
         * that distribute an AMI, this is empty.
         * </p>
         * This is a convenience method that creates an instance of the {@link ContainerRecipe.Builder} avoiding the
         * need to create one manually via {@link ContainerRecipe#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ContainerRecipe.Builder#build()} is called immediately and its
         * result is passed to {@link #containerRecipe(ContainerRecipe)}.
         * 
         * @param containerRecipe
         *        a consumer that will call methods on {@link ContainerRecipe.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #containerRecipe(ContainerRecipe)
         */
        default Builder containerRecipe(Consumer<ContainerRecipe.Builder> containerRecipe) {
            return containerRecipe(ContainerRecipe.builder().applyMutation(containerRecipe).build());
        }

        /**
         * <p>
         * The name of the image pipeline that created this image.
         * </p>
         * 
         * @param sourcePipelineName
         *        The name of the image pipeline that created this image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourcePipelineName(String sourcePipelineName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the image pipeline that created this image.
         * </p>
         * 
         * @param sourcePipelineArn
         *        The Amazon Resource Name (ARN) of the image pipeline that created this image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourcePipelineArn(String sourcePipelineArn);

        /**
         * <p>
         * The infrastructure that Image Builder used to create this image.
         * </p>
         * 
         * @param infrastructureConfiguration
         *        The infrastructure that Image Builder used to create this image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder infrastructureConfiguration(InfrastructureConfiguration infrastructureConfiguration);

        /**
         * <p>
         * The infrastructure that Image Builder used to create this image.
         * </p>
         * This is a convenience method that creates an instance of the {@link InfrastructureConfiguration.Builder}
         * avoiding the need to create one manually via {@link InfrastructureConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InfrastructureConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #infrastructureConfiguration(InfrastructureConfiguration)}.
         * 
         * @param infrastructureConfiguration
         *        a consumer that will call methods on {@link InfrastructureConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #infrastructureConfiguration(InfrastructureConfiguration)
         */
        default Builder infrastructureConfiguration(Consumer<InfrastructureConfiguration.Builder> infrastructureConfiguration) {
            return infrastructureConfiguration(InfrastructureConfiguration.builder().applyMutation(infrastructureConfiguration)
                    .build());
        }

        /**
         * <p>
         * The distribution configuration that Image Builder used to create this image.
         * </p>
         * 
         * @param distributionConfiguration
         *        The distribution configuration that Image Builder used to create this image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder distributionConfiguration(DistributionConfiguration distributionConfiguration);

        /**
         * <p>
         * The distribution configuration that Image Builder used to create this image.
         * </p>
         * This is a convenience method that creates an instance of the {@link DistributionConfiguration.Builder}
         * avoiding the need to create one manually via {@link DistributionConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DistributionConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #distributionConfiguration(DistributionConfiguration)}.
         * 
         * @param distributionConfiguration
         *        a consumer that will call methods on {@link DistributionConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #distributionConfiguration(DistributionConfiguration)
         */
        default Builder distributionConfiguration(Consumer<DistributionConfiguration.Builder> distributionConfiguration) {
            return distributionConfiguration(DistributionConfiguration.builder().applyMutation(distributionConfiguration).build());
        }

        /**
         * <p>
         * The image tests that ran when that Image Builder created this image.
         * </p>
         * 
         * @param imageTestsConfiguration
         *        The image tests that ran when that Image Builder created this image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageTestsConfiguration(ImageTestsConfiguration imageTestsConfiguration);

        /**
         * <p>
         * The image tests that ran when that Image Builder created this image.
         * </p>
         * This is a convenience method that creates an instance of the {@link ImageTestsConfiguration.Builder} avoiding
         * the need to create one manually via {@link ImageTestsConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ImageTestsConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #imageTestsConfiguration(ImageTestsConfiguration)}.
         * 
         * @param imageTestsConfiguration
         *        a consumer that will call methods on {@link ImageTestsConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #imageTestsConfiguration(ImageTestsConfiguration)
         */
        default Builder imageTestsConfiguration(Consumer<ImageTestsConfiguration.Builder> imageTestsConfiguration) {
            return imageTestsConfiguration(ImageTestsConfiguration.builder().applyMutation(imageTestsConfiguration).build());
        }

        /**
         * <p>
         * The date on which Image Builder created this image.
         * </p>
         * 
         * @param dateCreated
         *        The date on which Image Builder created this image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dateCreated(String dateCreated);

        /**
         * <p>
         * The output resources that Image Builder produces for this image.
         * </p>
         * 
         * @param outputResources
         *        The output resources that Image Builder produces for this image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputResources(OutputResources outputResources);

        /**
         * <p>
         * The output resources that Image Builder produces for this image.
         * </p>
         * This is a convenience method that creates an instance of the {@link OutputResources.Builder} avoiding the
         * need to create one manually via {@link OutputResources#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OutputResources.Builder#build()} is called immediately and its
         * result is passed to {@link #outputResources(OutputResources)}.
         * 
         * @param outputResources
         *        a consumer that will call methods on {@link OutputResources.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #outputResources(OutputResources)
         */
        default Builder outputResources(Consumer<OutputResources.Builder> outputResources) {
            return outputResources(OutputResources.builder().applyMutation(outputResources).build());
        }

        /**
         * <p>
         * The tags that apply to this image.
         * </p>
         * 
         * @param tags
         *        The tags that apply to this image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * Indicates the type of build that created this image. The build can be initiated in the following ways:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>USER_INITIATED</b> – A manual pipeline build request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>SCHEDULED</b> – A pipeline build initiated by a cron expression in the Image Builder pipeline, or from
         * EventBridge.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>IMPORT</b> – A VM import created the image to use as the base image for the recipe.
         * </p>
         * </li>
         * </ul>
         * 
         * @param buildType
         *        Indicates the type of build that created this image. The build can be initiated in the following
         *        ways:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>USER_INITIATED</b> – A manual pipeline build request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>SCHEDULED</b> – A pipeline build initiated by a cron expression in the Image Builder pipeline, or
         *        from EventBridge.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>IMPORT</b> – A VM import created the image to use as the base image for the recipe.
         *        </p>
         *        </li>
         * @see BuildType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BuildType
         */
        Builder buildType(String buildType);

        /**
         * <p>
         * Indicates the type of build that created this image. The build can be initiated in the following ways:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>USER_INITIATED</b> – A manual pipeline build request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>SCHEDULED</b> – A pipeline build initiated by a cron expression in the Image Builder pipeline, or from
         * EventBridge.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>IMPORT</b> – A VM import created the image to use as the base image for the recipe.
         * </p>
         * </li>
         * </ul>
         * 
         * @param buildType
         *        Indicates the type of build that created this image. The build can be initiated in the following
         *        ways:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>USER_INITIATED</b> – A manual pipeline build request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>SCHEDULED</b> – A pipeline build initiated by a cron expression in the Image Builder pipeline, or
         *        from EventBridge.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>IMPORT</b> – A VM import created the image to use as the base image for the recipe.
         *        </p>
         *        </li>
         * @see BuildType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BuildType
         */
        Builder buildType(BuildType buildType);

        /**
         * <p>
         * The origin of the base image that Image Builder used to build this image.
         * </p>
         * 
         * @param imageSource
         *        The origin of the base image that Image Builder used to build this image.
         * @see ImageSource
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImageSource
         */
        Builder imageSource(String imageSource);

        /**
         * <p>
         * The origin of the base image that Image Builder used to build this image.
         * </p>
         * 
         * @param imageSource
         *        The origin of the base image that Image Builder used to build this image.
         * @see ImageSource
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImageSource
         */
        Builder imageSource(ImageSource imageSource);

        /**
         * <p>
         * Contains information about the current state of scans for this image.
         * </p>
         * 
         * @param scanState
         *        Contains information about the current state of scans for this image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scanState(ImageScanState scanState);

        /**
         * <p>
         * Contains information about the current state of scans for this image.
         * </p>
         * This is a convenience method that creates an instance of the {@link ImageScanState.Builder} avoiding the need
         * to create one manually via {@link ImageScanState#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ImageScanState.Builder#build()} is called immediately and its
         * result is passed to {@link #scanState(ImageScanState)}.
         * 
         * @param scanState
         *        a consumer that will call methods on {@link ImageScanState.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scanState(ImageScanState)
         */
        default Builder scanState(Consumer<ImageScanState.Builder> scanState) {
            return scanState(ImageScanState.builder().applyMutation(scanState).build());
        }

        /**
         * <p>
         * Contains settings for vulnerability scans.
         * </p>
         * 
         * @param imageScanningConfiguration
         *        Contains settings for vulnerability scans.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageScanningConfiguration(ImageScanningConfiguration imageScanningConfiguration);

        /**
         * <p>
         * Contains settings for vulnerability scans.
         * </p>
         * This is a convenience method that creates an instance of the {@link ImageScanningConfiguration.Builder}
         * avoiding the need to create one manually via {@link ImageScanningConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ImageScanningConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #imageScanningConfiguration(ImageScanningConfiguration)}.
         * 
         * @param imageScanningConfiguration
         *        a consumer that will call methods on {@link ImageScanningConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #imageScanningConfiguration(ImageScanningConfiguration)
         */
        default Builder imageScanningConfiguration(Consumer<ImageScanningConfiguration.Builder> imageScanningConfiguration) {
            return imageScanningConfiguration(ImageScanningConfiguration.builder().applyMutation(imageScanningConfiguration)
                    .build());
        }

        /**
         * <p>
         * The time when deprecation occurs for an image resource. This can be a past or future date.
         * </p>
         * 
         * @param deprecationTime
         *        The time when deprecation occurs for an image resource. This can be a past or future date.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deprecationTime(Instant deprecationTime);

        /**
         * <p>
         * Identifies the last runtime instance of the lifecycle policy to take action on the image.
         * </p>
         * 
         * @param lifecycleExecutionId
         *        Identifies the last runtime instance of the lifecycle policy to take action on the image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifecycleExecutionId(String lifecycleExecutionId);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private String type;

        private String name;

        private String version;

        private String platform;

        private Boolean enhancedImageMetadataEnabled;

        private String osVersion;

        private ImageState state;

        private ImageRecipe imageRecipe;

        private ContainerRecipe containerRecipe;

        private String sourcePipelineName;

        private String sourcePipelineArn;

        private InfrastructureConfiguration infrastructureConfiguration;

        private DistributionConfiguration distributionConfiguration;

        private ImageTestsConfiguration imageTestsConfiguration;

        private String dateCreated;

        private OutputResources outputResources;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private String buildType;

        private String imageSource;

        private ImageScanState scanState;

        private ImageScanningConfiguration imageScanningConfiguration;

        private Instant deprecationTime;

        private String lifecycleExecutionId;

        private BuilderImpl() {
        }

        private BuilderImpl(Image model) {
            arn(model.arn);
            type(model.type);
            name(model.name);
            version(model.version);
            platform(model.platform);
            enhancedImageMetadataEnabled(model.enhancedImageMetadataEnabled);
            osVersion(model.osVersion);
            state(model.state);
            imageRecipe(model.imageRecipe);
            containerRecipe(model.containerRecipe);
            sourcePipelineName(model.sourcePipelineName);
            sourcePipelineArn(model.sourcePipelineArn);
            infrastructureConfiguration(model.infrastructureConfiguration);
            distributionConfiguration(model.distributionConfiguration);
            imageTestsConfiguration(model.imageTestsConfiguration);
            dateCreated(model.dateCreated);
            outputResources(model.outputResources);
            tags(model.tags);
            buildType(model.buildType);
            imageSource(model.imageSource);
            scanState(model.scanState);
            imageScanningConfiguration(model.imageScanningConfiguration);
            deprecationTime(model.deprecationTime);
            lifecycleExecutionId(model.lifecycleExecutionId);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(ImageType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final String getPlatform() {
            return platform;
        }

        public final void setPlatform(String platform) {
            this.platform = platform;
        }

        @Override
        public final Builder platform(String platform) {
            this.platform = platform;
            return this;
        }

        @Override
        public final Builder platform(Platform platform) {
            this.platform(platform == null ? null : platform.toString());
            return this;
        }

        public final Boolean getEnhancedImageMetadataEnabled() {
            return enhancedImageMetadataEnabled;
        }

        public final void setEnhancedImageMetadataEnabled(Boolean enhancedImageMetadataEnabled) {
            this.enhancedImageMetadataEnabled = enhancedImageMetadataEnabled;
        }

        @Override
        public final Builder enhancedImageMetadataEnabled(Boolean enhancedImageMetadataEnabled) {
            this.enhancedImageMetadataEnabled = enhancedImageMetadataEnabled;
            return this;
        }

        public final String getOsVersion() {
            return osVersion;
        }

        public final void setOsVersion(String osVersion) {
            this.osVersion = osVersion;
        }

        @Override
        public final Builder osVersion(String osVersion) {
            this.osVersion = osVersion;
            return this;
        }

        public final ImageState.Builder getState() {
            return state != null ? state.toBuilder() : null;
        }

        public final void setState(ImageState.BuilderImpl state) {
            this.state = state != null ? state.build() : null;
        }

        @Override
        public final Builder state(ImageState state) {
            this.state = state;
            return this;
        }

        public final ImageRecipe.Builder getImageRecipe() {
            return imageRecipe != null ? imageRecipe.toBuilder() : null;
        }

        public final void setImageRecipe(ImageRecipe.BuilderImpl imageRecipe) {
            this.imageRecipe = imageRecipe != null ? imageRecipe.build() : null;
        }

        @Override
        public final Builder imageRecipe(ImageRecipe imageRecipe) {
            this.imageRecipe = imageRecipe;
            return this;
        }

        public final ContainerRecipe.Builder getContainerRecipe() {
            return containerRecipe != null ? containerRecipe.toBuilder() : null;
        }

        public final void setContainerRecipe(ContainerRecipe.BuilderImpl containerRecipe) {
            this.containerRecipe = containerRecipe != null ? containerRecipe.build() : null;
        }

        @Override
        public final Builder containerRecipe(ContainerRecipe containerRecipe) {
            this.containerRecipe = containerRecipe;
            return this;
        }

        public final String getSourcePipelineName() {
            return sourcePipelineName;
        }

        public final void setSourcePipelineName(String sourcePipelineName) {
            this.sourcePipelineName = sourcePipelineName;
        }

        @Override
        public final Builder sourcePipelineName(String sourcePipelineName) {
            this.sourcePipelineName = sourcePipelineName;
            return this;
        }

        public final String getSourcePipelineArn() {
            return sourcePipelineArn;
        }

        public final void setSourcePipelineArn(String sourcePipelineArn) {
            this.sourcePipelineArn = sourcePipelineArn;
        }

        @Override
        public final Builder sourcePipelineArn(String sourcePipelineArn) {
            this.sourcePipelineArn = sourcePipelineArn;
            return this;
        }

        public final InfrastructureConfiguration.Builder getInfrastructureConfiguration() {
            return infrastructureConfiguration != null ? infrastructureConfiguration.toBuilder() : null;
        }

        public final void setInfrastructureConfiguration(InfrastructureConfiguration.BuilderImpl infrastructureConfiguration) {
            this.infrastructureConfiguration = infrastructureConfiguration != null ? infrastructureConfiguration.build() : null;
        }

        @Override
        public final Builder infrastructureConfiguration(InfrastructureConfiguration infrastructureConfiguration) {
            this.infrastructureConfiguration = infrastructureConfiguration;
            return this;
        }

        public final DistributionConfiguration.Builder getDistributionConfiguration() {
            return distributionConfiguration != null ? distributionConfiguration.toBuilder() : null;
        }

        public final void setDistributionConfiguration(DistributionConfiguration.BuilderImpl distributionConfiguration) {
            this.distributionConfiguration = distributionConfiguration != null ? distributionConfiguration.build() : null;
        }

        @Override
        public final Builder distributionConfiguration(DistributionConfiguration distributionConfiguration) {
            this.distributionConfiguration = distributionConfiguration;
            return this;
        }

        public final ImageTestsConfiguration.Builder getImageTestsConfiguration() {
            return imageTestsConfiguration != null ? imageTestsConfiguration.toBuilder() : null;
        }

        public final void setImageTestsConfiguration(ImageTestsConfiguration.BuilderImpl imageTestsConfiguration) {
            this.imageTestsConfiguration = imageTestsConfiguration != null ? imageTestsConfiguration.build() : null;
        }

        @Override
        public final Builder imageTestsConfiguration(ImageTestsConfiguration imageTestsConfiguration) {
            this.imageTestsConfiguration = imageTestsConfiguration;
            return this;
        }

        public final String getDateCreated() {
            return dateCreated;
        }

        public final void setDateCreated(String dateCreated) {
            this.dateCreated = dateCreated;
        }

        @Override
        public final Builder dateCreated(String dateCreated) {
            this.dateCreated = dateCreated;
            return this;
        }

        public final OutputResources.Builder getOutputResources() {
            return outputResources != null ? outputResources.toBuilder() : null;
        }

        public final void setOutputResources(OutputResources.BuilderImpl outputResources) {
            this.outputResources = outputResources != null ? outputResources.build() : null;
        }

        @Override
        public final Builder outputResources(OutputResources outputResources) {
            this.outputResources = outputResources;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final String getBuildType() {
            return buildType;
        }

        public final void setBuildType(String buildType) {
            this.buildType = buildType;
        }

        @Override
        public final Builder buildType(String buildType) {
            this.buildType = buildType;
            return this;
        }

        @Override
        public final Builder buildType(BuildType buildType) {
            this.buildType(buildType == null ? null : buildType.toString());
            return this;
        }

        public final String getImageSource() {
            return imageSource;
        }

        public final void setImageSource(String imageSource) {
            this.imageSource = imageSource;
        }

        @Override
        public final Builder imageSource(String imageSource) {
            this.imageSource = imageSource;
            return this;
        }

        @Override
        public final Builder imageSource(ImageSource imageSource) {
            this.imageSource(imageSource == null ? null : imageSource.toString());
            return this;
        }

        public final ImageScanState.Builder getScanState() {
            return scanState != null ? scanState.toBuilder() : null;
        }

        public final void setScanState(ImageScanState.BuilderImpl scanState) {
            this.scanState = scanState != null ? scanState.build() : null;
        }

        @Override
        public final Builder scanState(ImageScanState scanState) {
            this.scanState = scanState;
            return this;
        }

        public final ImageScanningConfiguration.Builder getImageScanningConfiguration() {
            return imageScanningConfiguration != null ? imageScanningConfiguration.toBuilder() : null;
        }

        public final void setImageScanningConfiguration(ImageScanningConfiguration.BuilderImpl imageScanningConfiguration) {
            this.imageScanningConfiguration = imageScanningConfiguration != null ? imageScanningConfiguration.build() : null;
        }

        @Override
        public final Builder imageScanningConfiguration(ImageScanningConfiguration imageScanningConfiguration) {
            this.imageScanningConfiguration = imageScanningConfiguration;
            return this;
        }

        public final Instant getDeprecationTime() {
            return deprecationTime;
        }

        public final void setDeprecationTime(Instant deprecationTime) {
            this.deprecationTime = deprecationTime;
        }

        @Override
        public final Builder deprecationTime(Instant deprecationTime) {
            this.deprecationTime = deprecationTime;
            return this;
        }

        public final String getLifecycleExecutionId() {
            return lifecycleExecutionId;
        }

        public final void setLifecycleExecutionId(String lifecycleExecutionId) {
            this.lifecycleExecutionId = lifecycleExecutionId;
        }

        @Override
        public final Builder lifecycleExecutionId(String lifecycleExecutionId) {
            this.lifecycleExecutionId = lifecycleExecutionId;
            return this;
        }

        @Override
        public Image build() {
            return new Image(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
