/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.imagebuilder.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The defining characteristics of a specific version of an Amazon Web Services TOE component.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ComponentVersion implements SdkPojo, Serializable,
        ToCopyableBuilder<ComponentVersion.Builder, ComponentVersion> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(ComponentVersion::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(ComponentVersion::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("version")
            .getter(getter(ComponentVersion::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("version").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(ComponentVersion::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<String> PLATFORM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("platform").getter(getter(ComponentVersion::platformAsString)).setter(setter(Builder::platform))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("platform").build()).build();

    private static final SdkField<List<String>> SUPPORTED_OS_VERSIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("supportedOsVersions")
            .getter(getter(ComponentVersion::supportedOsVersions))
            .setter(setter(Builder::supportedOsVersions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("supportedOsVersions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(ComponentVersion::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> OWNER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("owner")
            .getter(getter(ComponentVersion::owner)).setter(setter(Builder::owner))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("owner").build()).build();

    private static final SdkField<String> DATE_CREATED_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("dateCreated").getter(getter(ComponentVersion::dateCreated)).setter(setter(Builder::dateCreated))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dateCreated").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(ComponentVersion::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<List<ProductCodeListItem>> PRODUCT_CODES_FIELD = SdkField
            .<List<ProductCodeListItem>> builder(MarshallingType.LIST)
            .memberName("productCodes")
            .getter(getter(ComponentVersion::productCodes))
            .setter(setter(Builder::productCodes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("productCodes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ProductCodeListItem> builder(MarshallingType.SDK_POJO)
                                            .constructor(ProductCodeListItem::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD, NAME_FIELD,
            VERSION_FIELD, DESCRIPTION_FIELD, PLATFORM_FIELD, SUPPORTED_OS_VERSIONS_FIELD, TYPE_FIELD, OWNER_FIELD,
            DATE_CREATED_FIELD, STATUS_FIELD, PRODUCT_CODES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final String name;

    private final String version;

    private final String description;

    private final String platform;

    private final List<String> supportedOsVersions;

    private final String type;

    private final String owner;

    private final String dateCreated;

    private final String status;

    private final List<ProductCodeListItem> productCodes;

    private ComponentVersion(BuilderImpl builder) {
        this.arn = builder.arn;
        this.name = builder.name;
        this.version = builder.version;
        this.description = builder.description;
        this.platform = builder.platform;
        this.supportedOsVersions = builder.supportedOsVersions;
        this.type = builder.type;
        this.owner = builder.owner;
        this.dateCreated = builder.dateCreated;
        this.status = builder.status;
        this.productCodes = builder.productCodes;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the component.
     * </p>
     * <note>
     * <p>
     * Semantic versioning is included in each object's Amazon Resource Name (ARN), at the level that applies to that
     * object as follows:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Versionless ARNs and Name ARNs do not include specific values in any of the nodes. The nodes are either left off
     * entirely, or they are specified as wildcards, for example: x.x.x.
     * </p>
     * </li>
     * <li>
     * <p>
     * Version ARNs have only the first three nodes: &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;
     * </p>
     * </li>
     * <li>
     * <p>
     * Build version ARNs have all four nodes, and point to a specific build for a specific version of an object.
     * </p>
     * </li>
     * </ol>
     * </note>
     * 
     * @return The Amazon Resource Name (ARN) of the component.</p> <note>
     *         <p>
     *         Semantic versioning is included in each object's Amazon Resource Name (ARN), at the level that applies to
     *         that object as follows:
     *         </p>
     *         <ol>
     *         <li>
     *         <p>
     *         Versionless ARNs and Name ARNs do not include specific values in any of the nodes. The nodes are either
     *         left off entirely, or they are specified as wildcards, for example: x.x.x.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Version ARNs have only the first three nodes: &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Build version ARNs have all four nodes, and point to a specific build for a specific version of an
     *         object.
     *         </p>
     *         </li>
     *         </ol>
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The name of the component.
     * </p>
     * 
     * @return The name of the component.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The semantic version of the component.
     * </p>
     * <note>
     * <p>
     * The semantic version has four nodes: &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;/&lt;build&gt;. You can assign
     * values for the first three, and can filter on all of them.
     * </p>
     * <p>
     * <b>Assignment:</b> For the first three nodes you can assign any positive integer value, including zero, with an
     * upper limit of 2^30-1, or 1073741823 for each node. Image Builder automatically assigns the build number to the
     * fourth node.
     * </p>
     * <p>
     * <b>Patterns:</b> You can use any numeric pattern that adheres to the assignment requirements for the nodes that
     * you can assign. For example, you might choose a software version pattern, such as 1.0.0, or a date, such as
     * 2021.01.01.
     * </p>
     * <p>
     * <b>Filtering:</b> With semantic versioning, you have the flexibility to use wildcards (x) to specify the most
     * recent versions or nodes when selecting the base image or components for your recipe. When you use a wildcard in
     * any node, all nodes to the right of the first wildcard must also be wildcards.
     * </p>
     * </note>
     * 
     * @return The semantic version of the component.</p> <note>
     *         <p>
     *         The semantic version has four nodes: &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;/&lt;build&gt;. You can
     *         assign values for the first three, and can filter on all of them.
     *         </p>
     *         <p>
     *         <b>Assignment:</b> For the first three nodes you can assign any positive integer value, including zero,
     *         with an upper limit of 2^30-1, or 1073741823 for each node. Image Builder automatically assigns the build
     *         number to the fourth node.
     *         </p>
     *         <p>
     *         <b>Patterns:</b> You can use any numeric pattern that adheres to the assignment requirements for the
     *         nodes that you can assign. For example, you might choose a software version pattern, such as 1.0.0, or a
     *         date, such as 2021.01.01.
     *         </p>
     *         <p>
     *         <b>Filtering:</b> With semantic versioning, you have the flexibility to use wildcards (x) to specify the
     *         most recent versions or nodes when selecting the base image or components for your recipe. When you use a
     *         wildcard in any node, all nodes to the right of the first wildcard must also be wildcards.
     *         </p>
     */
    public final String version() {
        return version;
    }

    /**
     * <p>
     * The description of the component.
     * </p>
     * 
     * @return The description of the component.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The platform of the component.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #platform} will
     * return {@link Platform#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #platformAsString}.
     * </p>
     * 
     * @return The platform of the component.
     * @see Platform
     */
    public final Platform platform() {
        return Platform.fromValue(platform);
    }

    /**
     * <p>
     * The platform of the component.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #platform} will
     * return {@link Platform#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #platformAsString}.
     * </p>
     * 
     * @return The platform of the component.
     * @see Platform
     */
    public final String platformAsString() {
        return platform;
    }

    /**
     * For responses, this returns true if the service returned a value for the SupportedOsVersions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSupportedOsVersions() {
        return supportedOsVersions != null && !(supportedOsVersions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * he operating system (OS) version supported by the component. If the OS information is available, a prefix match
     * is performed against the base image OS version during image recipe creation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportedOsVersions} method.
     * </p>
     * 
     * @return he operating system (OS) version supported by the component. If the OS information is available, a prefix
     *         match is performed against the base image OS version during image recipe creation.
     */
    public final List<String> supportedOsVersions() {
        return supportedOsVersions;
    }

    /**
     * <p>
     * The type of the component denotes whether the component is used to build the image or only to test it.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ComponentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the component denotes whether the component is used to build the image or only to test it.
     * @see ComponentType
     */
    public final ComponentType type() {
        return ComponentType.fromValue(type);
    }

    /**
     * <p>
     * The type of the component denotes whether the component is used to build the image or only to test it.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ComponentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the component denotes whether the component is used to build the image or only to test it.
     * @see ComponentType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The owner of the component.
     * </p>
     * 
     * @return The owner of the component.
     */
    public final String owner() {
        return owner;
    }

    /**
     * <p>
     * The date that the component was created.
     * </p>
     * 
     * @return The date that the component was created.
     */
    public final String dateCreated() {
        return dateCreated;
    }

    /**
     * <p>
     * Describes the current status of the component version.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ComponentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Describes the current status of the component version.
     * @see ComponentStatus
     */
    public final ComponentStatus status() {
        return ComponentStatus.fromValue(status);
    }

    /**
     * <p>
     * Describes the current status of the component version.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ComponentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Describes the current status of the component version.
     * @see ComponentStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * For responses, this returns true if the service returned a value for the ProductCodes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasProductCodes() {
        return productCodes != null && !(productCodes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Contains product codes that are used for billing purposes for Amazon Web Services Marketplace components.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProductCodes} method.
     * </p>
     * 
     * @return Contains product codes that are used for billing purposes for Amazon Web Services Marketplace components.
     */
    public final List<ProductCodeListItem> productCodes() {
        return productCodes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(platformAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasSupportedOsVersions() ? supportedOsVersions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(owner());
        hashCode = 31 * hashCode + Objects.hashCode(dateCreated());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasProductCodes() ? productCodes() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ComponentVersion)) {
            return false;
        }
        ComponentVersion other = (ComponentVersion) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(name(), other.name())
                && Objects.equals(version(), other.version()) && Objects.equals(description(), other.description())
                && Objects.equals(platformAsString(), other.platformAsString())
                && hasSupportedOsVersions() == other.hasSupportedOsVersions()
                && Objects.equals(supportedOsVersions(), other.supportedOsVersions())
                && Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(owner(), other.owner())
                && Objects.equals(dateCreated(), other.dateCreated()) && Objects.equals(statusAsString(), other.statusAsString())
                && hasProductCodes() == other.hasProductCodes() && Objects.equals(productCodes(), other.productCodes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ComponentVersion").add("Arn", arn()).add("Name", name()).add("Version", version())
                .add("Description", description()).add("Platform", platformAsString())
                .add("SupportedOsVersions", hasSupportedOsVersions() ? supportedOsVersions() : null).add("Type", typeAsString())
                .add("Owner", owner()).add("DateCreated", dateCreated()).add("Status", statusAsString())
                .add("ProductCodes", hasProductCodes() ? productCodes() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "version":
            return Optional.ofNullable(clazz.cast(version()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "platform":
            return Optional.ofNullable(clazz.cast(platformAsString()));
        case "supportedOsVersions":
            return Optional.ofNullable(clazz.cast(supportedOsVersions()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "owner":
            return Optional.ofNullable(clazz.cast(owner()));
        case "dateCreated":
            return Optional.ofNullable(clazz.cast(dateCreated()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "productCodes":
            return Optional.ofNullable(clazz.cast(productCodes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("arn", ARN_FIELD);
        map.put("name", NAME_FIELD);
        map.put("version", VERSION_FIELD);
        map.put("description", DESCRIPTION_FIELD);
        map.put("platform", PLATFORM_FIELD);
        map.put("supportedOsVersions", SUPPORTED_OS_VERSIONS_FIELD);
        map.put("type", TYPE_FIELD);
        map.put("owner", OWNER_FIELD);
        map.put("dateCreated", DATE_CREATED_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("productCodes", PRODUCT_CODES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ComponentVersion, T> g) {
        return obj -> g.apply((ComponentVersion) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ComponentVersion> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the component.
         * </p>
         * <note>
         * <p>
         * Semantic versioning is included in each object's Amazon Resource Name (ARN), at the level that applies to
         * that object as follows:
         * </p>
         * <ol>
         * <li>
         * <p>
         * Versionless ARNs and Name ARNs do not include specific values in any of the nodes. The nodes are either left
         * off entirely, or they are specified as wildcards, for example: x.x.x.
         * </p>
         * </li>
         * <li>
         * <p>
         * Version ARNs have only the first three nodes: &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;
         * </p>
         * </li>
         * <li>
         * <p>
         * Build version ARNs have all four nodes, and point to a specific build for a specific version of an object.
         * </p>
         * </li>
         * </ol>
         * </note>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the component.</p> <note>
         *        <p>
         *        Semantic versioning is included in each object's Amazon Resource Name (ARN), at the level that applies
         *        to that object as follows:
         *        </p>
         *        <ol>
         *        <li>
         *        <p>
         *        Versionless ARNs and Name ARNs do not include specific values in any of the nodes. The nodes are
         *        either left off entirely, or they are specified as wildcards, for example: x.x.x.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Version ARNs have only the first three nodes: &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Build version ARNs have all four nodes, and point to a specific build for a specific version of an
         *        object.
         *        </p>
         *        </li>
         *        </ol>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The name of the component.
         * </p>
         * 
         * @param name
         *        The name of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The semantic version of the component.
         * </p>
         * <note>
         * <p>
         * The semantic version has four nodes: &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;/&lt;build&gt;. You can assign
         * values for the first three, and can filter on all of them.
         * </p>
         * <p>
         * <b>Assignment:</b> For the first three nodes you can assign any positive integer value, including zero, with
         * an upper limit of 2^30-1, or 1073741823 for each node. Image Builder automatically assigns the build number
         * to the fourth node.
         * </p>
         * <p>
         * <b>Patterns:</b> You can use any numeric pattern that adheres to the assignment requirements for the nodes
         * that you can assign. For example, you might choose a software version pattern, such as 1.0.0, or a date, such
         * as 2021.01.01.
         * </p>
         * <p>
         * <b>Filtering:</b> With semantic versioning, you have the flexibility to use wildcards (x) to specify the most
         * recent versions or nodes when selecting the base image or components for your recipe. When you use a wildcard
         * in any node, all nodes to the right of the first wildcard must also be wildcards.
         * </p>
         * </note>
         * 
         * @param version
         *        The semantic version of the component.</p> <note>
         *        <p>
         *        The semantic version has four nodes: &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;/&lt;build&gt;. You can
         *        assign values for the first three, and can filter on all of them.
         *        </p>
         *        <p>
         *        <b>Assignment:</b> For the first three nodes you can assign any positive integer value, including
         *        zero, with an upper limit of 2^30-1, or 1073741823 for each node. Image Builder automatically assigns
         *        the build number to the fourth node.
         *        </p>
         *        <p>
         *        <b>Patterns:</b> You can use any numeric pattern that adheres to the assignment requirements for the
         *        nodes that you can assign. For example, you might choose a software version pattern, such as 1.0.0, or
         *        a date, such as 2021.01.01.
         *        </p>
         *        <p>
         *        <b>Filtering:</b> With semantic versioning, you have the flexibility to use wildcards (x) to specify
         *        the most recent versions or nodes when selecting the base image or components for your recipe. When
         *        you use a wildcard in any node, all nodes to the right of the first wildcard must also be wildcards.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * The description of the component.
         * </p>
         * 
         * @param description
         *        The description of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The platform of the component.
         * </p>
         * 
         * @param platform
         *        The platform of the component.
         * @see Platform
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Platform
         */
        Builder platform(String platform);

        /**
         * <p>
         * The platform of the component.
         * </p>
         * 
         * @param platform
         *        The platform of the component.
         * @see Platform
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Platform
         */
        Builder platform(Platform platform);

        /**
         * <p>
         * he operating system (OS) version supported by the component. If the OS information is available, a prefix
         * match is performed against the base image OS version during image recipe creation.
         * </p>
         * 
         * @param supportedOsVersions
         *        he operating system (OS) version supported by the component. If the OS information is available, a
         *        prefix match is performed against the base image OS version during image recipe creation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedOsVersions(Collection<String> supportedOsVersions);

        /**
         * <p>
         * he operating system (OS) version supported by the component. If the OS information is available, a prefix
         * match is performed against the base image OS version during image recipe creation.
         * </p>
         * 
         * @param supportedOsVersions
         *        he operating system (OS) version supported by the component. If the OS information is available, a
         *        prefix match is performed against the base image OS version during image recipe creation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedOsVersions(String... supportedOsVersions);

        /**
         * <p>
         * The type of the component denotes whether the component is used to build the image or only to test it.
         * </p>
         * 
         * @param type
         *        The type of the component denotes whether the component is used to build the image or only to test it.
         * @see ComponentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComponentType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of the component denotes whether the component is used to build the image or only to test it.
         * </p>
         * 
         * @param type
         *        The type of the component denotes whether the component is used to build the image or only to test it.
         * @see ComponentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComponentType
         */
        Builder type(ComponentType type);

        /**
         * <p>
         * The owner of the component.
         * </p>
         * 
         * @param owner
         *        The owner of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder owner(String owner);

        /**
         * <p>
         * The date that the component was created.
         * </p>
         * 
         * @param dateCreated
         *        The date that the component was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dateCreated(String dateCreated);

        /**
         * <p>
         * Describes the current status of the component version.
         * </p>
         * 
         * @param status
         *        Describes the current status of the component version.
         * @see ComponentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComponentStatus
         */
        Builder status(String status);

        /**
         * <p>
         * Describes the current status of the component version.
         * </p>
         * 
         * @param status
         *        Describes the current status of the component version.
         * @see ComponentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComponentStatus
         */
        Builder status(ComponentStatus status);

        /**
         * <p>
         * Contains product codes that are used for billing purposes for Amazon Web Services Marketplace components.
         * </p>
         * 
         * @param productCodes
         *        Contains product codes that are used for billing purposes for Amazon Web Services Marketplace
         *        components.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder productCodes(Collection<ProductCodeListItem> productCodes);

        /**
         * <p>
         * Contains product codes that are used for billing purposes for Amazon Web Services Marketplace components.
         * </p>
         * 
         * @param productCodes
         *        Contains product codes that are used for billing purposes for Amazon Web Services Marketplace
         *        components.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder productCodes(ProductCodeListItem... productCodes);

        /**
         * <p>
         * Contains product codes that are used for billing purposes for Amazon Web Services Marketplace components.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.imagebuilder.model.ProductCodeListItem.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.imagebuilder.model.ProductCodeListItem#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.imagebuilder.model.ProductCodeListItem.Builder#build()} is called
         * immediately and its result is passed to {@link #productCodes(List<ProductCodeListItem>)}.
         * 
         * @param productCodes
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.imagebuilder.model.ProductCodeListItem.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #productCodes(java.util.Collection<ProductCodeListItem>)
         */
        Builder productCodes(Consumer<ProductCodeListItem.Builder>... productCodes);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private String name;

        private String version;

        private String description;

        private String platform;

        private List<String> supportedOsVersions = DefaultSdkAutoConstructList.getInstance();

        private String type;

        private String owner;

        private String dateCreated;

        private String status;

        private List<ProductCodeListItem> productCodes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ComponentVersion model) {
            arn(model.arn);
            name(model.name);
            version(model.version);
            description(model.description);
            platform(model.platform);
            supportedOsVersions(model.supportedOsVersions);
            type(model.type);
            owner(model.owner);
            dateCreated(model.dateCreated);
            status(model.status);
            productCodes(model.productCodes);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getPlatform() {
            return platform;
        }

        public final void setPlatform(String platform) {
            this.platform = platform;
        }

        @Override
        public final Builder platform(String platform) {
            this.platform = platform;
            return this;
        }

        @Override
        public final Builder platform(Platform platform) {
            this.platform(platform == null ? null : platform.toString());
            return this;
        }

        public final Collection<String> getSupportedOsVersions() {
            if (supportedOsVersions instanceof SdkAutoConstructList) {
                return null;
            }
            return supportedOsVersions;
        }

        public final void setSupportedOsVersions(Collection<String> supportedOsVersions) {
            this.supportedOsVersions = OsVersionListCopier.copy(supportedOsVersions);
        }

        @Override
        public final Builder supportedOsVersions(Collection<String> supportedOsVersions) {
            this.supportedOsVersions = OsVersionListCopier.copy(supportedOsVersions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedOsVersions(String... supportedOsVersions) {
            supportedOsVersions(Arrays.asList(supportedOsVersions));
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(ComponentType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getOwner() {
            return owner;
        }

        public final void setOwner(String owner) {
            this.owner = owner;
        }

        @Override
        public final Builder owner(String owner) {
            this.owner = owner;
            return this;
        }

        public final String getDateCreated() {
            return dateCreated;
        }

        public final void setDateCreated(String dateCreated) {
            this.dateCreated = dateCreated;
        }

        @Override
        public final Builder dateCreated(String dateCreated) {
            this.dateCreated = dateCreated;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ComponentStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final List<ProductCodeListItem.Builder> getProductCodes() {
            List<ProductCodeListItem.Builder> result = ProductCodeListCopier.copyToBuilder(this.productCodes);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setProductCodes(Collection<ProductCodeListItem.BuilderImpl> productCodes) {
            this.productCodes = ProductCodeListCopier.copyFromBuilder(productCodes);
        }

        @Override
        public final Builder productCodes(Collection<ProductCodeListItem> productCodes) {
            this.productCodes = ProductCodeListCopier.copy(productCodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder productCodes(ProductCodeListItem... productCodes) {
            productCodes(Arrays.asList(productCodes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder productCodes(Consumer<ProductCodeListItem.Builder>... productCodes) {
            productCodes(Stream.of(productCodes).map(c -> ProductCodeListItem.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public ComponentVersion build() {
            return new ComponentVersion(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
