/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.imagebuilder.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Settings that Image Builder uses to configure the ECR repository and the output container images that Amazon
 * Inspector scans.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EcrConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<EcrConfiguration.Builder, EcrConfiguration> {
    private static final SdkField<String> REPOSITORY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("repositoryName").getter(getter(EcrConfiguration::repositoryName))
            .setter(setter(Builder::repositoryName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("repositoryName").build()).build();

    private static final SdkField<List<String>> CONTAINER_TAGS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("containerTags")
            .getter(getter(EcrConfiguration::containerTags))
            .setter(setter(Builder::containerTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerTags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(REPOSITORY_NAME_FIELD,
            CONTAINER_TAGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String repositoryName;

    private final List<String> containerTags;

    private EcrConfiguration(BuilderImpl builder) {
        this.repositoryName = builder.repositoryName;
        this.containerTags = builder.containerTags;
    }

    /**
     * <p>
     * The name of the container repository that Amazon Inspector scans to identify findings for your container images.
     * The name includes the path for the repository location. If you don’t provide this information, Image Builder
     * creates a repository in your account named <code>image-builder-image-scanning-repository</code> for vulnerability
     * scans of your output container images.
     * </p>
     * 
     * @return The name of the container repository that Amazon Inspector scans to identify findings for your container
     *         images. The name includes the path for the repository location. If you don’t provide this information,
     *         Image Builder creates a repository in your account named
     *         <code>image-builder-image-scanning-repository</code> for vulnerability scans of your output container
     *         images.
     */
    public final String repositoryName() {
        return repositoryName;
    }

    /**
     * For responses, this returns true if the service returned a value for the ContainerTags property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasContainerTags() {
        return containerTags != null && !(containerTags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Tags for Image Builder to apply to the output container image that Amazon Inspector scans. Tags can help you
     * identify and manage your scanned images.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasContainerTags} method.
     * </p>
     * 
     * @return Tags for Image Builder to apply to the output container image that Amazon Inspector scans. Tags can help
     *         you identify and manage your scanned images.
     */
    public final List<String> containerTags() {
        return containerTags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(repositoryName());
        hashCode = 31 * hashCode + Objects.hashCode(hasContainerTags() ? containerTags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EcrConfiguration)) {
            return false;
        }
        EcrConfiguration other = (EcrConfiguration) obj;
        return Objects.equals(repositoryName(), other.repositoryName()) && hasContainerTags() == other.hasContainerTags()
                && Objects.equals(containerTags(), other.containerTags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EcrConfiguration").add("RepositoryName", repositoryName())
                .add("ContainerTags", hasContainerTags() ? containerTags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "repositoryName":
            return Optional.ofNullable(clazz.cast(repositoryName()));
        case "containerTags":
            return Optional.ofNullable(clazz.cast(containerTags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("repositoryName", REPOSITORY_NAME_FIELD);
        map.put("containerTags", CONTAINER_TAGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<EcrConfiguration, T> g) {
        return obj -> g.apply((EcrConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EcrConfiguration> {
        /**
         * <p>
         * The name of the container repository that Amazon Inspector scans to identify findings for your container
         * images. The name includes the path for the repository location. If you don’t provide this information, Image
         * Builder creates a repository in your account named <code>image-builder-image-scanning-repository</code> for
         * vulnerability scans of your output container images.
         * </p>
         * 
         * @param repositoryName
         *        The name of the container repository that Amazon Inspector scans to identify findings for your
         *        container images. The name includes the path for the repository location. If you don’t provide this
         *        information, Image Builder creates a repository in your account named
         *        <code>image-builder-image-scanning-repository</code> for vulnerability scans of your output container
         *        images.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder repositoryName(String repositoryName);

        /**
         * <p>
         * Tags for Image Builder to apply to the output container image that Amazon Inspector scans. Tags can help you
         * identify and manage your scanned images.
         * </p>
         * 
         * @param containerTags
         *        Tags for Image Builder to apply to the output container image that Amazon Inspector scans. Tags can
         *        help you identify and manage your scanned images.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerTags(Collection<String> containerTags);

        /**
         * <p>
         * Tags for Image Builder to apply to the output container image that Amazon Inspector scans. Tags can help you
         * identify and manage your scanned images.
         * </p>
         * 
         * @param containerTags
         *        Tags for Image Builder to apply to the output container image that Amazon Inspector scans. Tags can
         *        help you identify and manage your scanned images.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerTags(String... containerTags);
    }

    static final class BuilderImpl implements Builder {
        private String repositoryName;

        private List<String> containerTags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(EcrConfiguration model) {
            repositoryName(model.repositoryName);
            containerTags(model.containerTags);
        }

        public final String getRepositoryName() {
            return repositoryName;
        }

        public final void setRepositoryName(String repositoryName) {
            this.repositoryName = repositoryName;
        }

        @Override
        public final Builder repositoryName(String repositoryName) {
            this.repositoryName = repositoryName;
            return this;
        }

        public final Collection<String> getContainerTags() {
            if (containerTags instanceof SdkAutoConstructList) {
                return null;
            }
            return containerTags;
        }

        public final void setContainerTags(Collection<String> containerTags) {
            this.containerTags = StringListCopier.copy(containerTags);
        }

        @Override
        public final Builder containerTags(Collection<String> containerTags) {
            this.containerTags = StringListCopier.copy(containerTags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder containerTags(String... containerTags) {
            containerTags(Arrays.asList(containerTags));
            return this;
        }

        @Override
        public EcrConfiguration build() {
            return new EcrConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
