/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.imagebuilder.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines a custom base AMI and block device mapping configurations of an instance used for building and testing
 * container images.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<InstanceConfiguration.Builder, InstanceConfiguration> {
    private static final SdkField<String> IMAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("image")
            .getter(getter(InstanceConfiguration::image)).setter(setter(Builder::image))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("image").build()).build();

    private static final SdkField<List<InstanceBlockDeviceMapping>> BLOCK_DEVICE_MAPPINGS_FIELD = SdkField
            .<List<InstanceBlockDeviceMapping>> builder(MarshallingType.LIST)
            .memberName("blockDeviceMappings")
            .getter(getter(InstanceConfiguration::blockDeviceMappings))
            .setter(setter(Builder::blockDeviceMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("blockDeviceMappings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<InstanceBlockDeviceMapping> builder(MarshallingType.SDK_POJO)
                                            .constructor(InstanceBlockDeviceMapping::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IMAGE_FIELD,
            BLOCK_DEVICE_MAPPINGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String image;

    private final List<InstanceBlockDeviceMapping> blockDeviceMappings;

    private InstanceConfiguration(BuilderImpl builder) {
        this.image = builder.image;
        this.blockDeviceMappings = builder.blockDeviceMappings;
    }

    /**
     * <p>
     * The AMI ID to use as the base image for a container build and test instance. If not specified, Image Builder will
     * use the appropriate ECS-optimized AMI as a base image.
     * </p>
     * 
     * @return The AMI ID to use as the base image for a container build and test instance. If not specified, Image
     *         Builder will use the appropriate ECS-optimized AMI as a base image.
     */
    public final String image() {
        return image;
    }

    /**
     * For responses, this returns true if the service returned a value for the BlockDeviceMappings property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasBlockDeviceMappings() {
        return blockDeviceMappings != null && !(blockDeviceMappings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Defines the block devices to attach for building an instance from this Image Builder AMI.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasBlockDeviceMappings} method.
     * </p>
     * 
     * @return Defines the block devices to attach for building an instance from this Image Builder AMI.
     */
    public final List<InstanceBlockDeviceMapping> blockDeviceMappings() {
        return blockDeviceMappings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(image());
        hashCode = 31 * hashCode + Objects.hashCode(hasBlockDeviceMappings() ? blockDeviceMappings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceConfiguration)) {
            return false;
        }
        InstanceConfiguration other = (InstanceConfiguration) obj;
        return Objects.equals(image(), other.image()) && hasBlockDeviceMappings() == other.hasBlockDeviceMappings()
                && Objects.equals(blockDeviceMappings(), other.blockDeviceMappings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InstanceConfiguration").add("Image", image())
                .add("BlockDeviceMappings", hasBlockDeviceMappings() ? blockDeviceMappings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "image":
            return Optional.ofNullable(clazz.cast(image()));
        case "blockDeviceMappings":
            return Optional.ofNullable(clazz.cast(blockDeviceMappings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("image", IMAGE_FIELD);
        map.put("blockDeviceMappings", BLOCK_DEVICE_MAPPINGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<InstanceConfiguration, T> g) {
        return obj -> g.apply((InstanceConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceConfiguration> {
        /**
         * <p>
         * The AMI ID to use as the base image for a container build and test instance. If not specified, Image Builder
         * will use the appropriate ECS-optimized AMI as a base image.
         * </p>
         * 
         * @param image
         *        The AMI ID to use as the base image for a container build and test instance. If not specified, Image
         *        Builder will use the appropriate ECS-optimized AMI as a base image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder image(String image);

        /**
         * <p>
         * Defines the block devices to attach for building an instance from this Image Builder AMI.
         * </p>
         * 
         * @param blockDeviceMappings
         *        Defines the block devices to attach for building an instance from this Image Builder AMI.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder blockDeviceMappings(Collection<InstanceBlockDeviceMapping> blockDeviceMappings);

        /**
         * <p>
         * Defines the block devices to attach for building an instance from this Image Builder AMI.
         * </p>
         * 
         * @param blockDeviceMappings
         *        Defines the block devices to attach for building an instance from this Image Builder AMI.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder blockDeviceMappings(InstanceBlockDeviceMapping... blockDeviceMappings);

        /**
         * <p>
         * Defines the block devices to attach for building an instance from this Image Builder AMI.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.imagebuilder.model.InstanceBlockDeviceMapping.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.imagebuilder.model.InstanceBlockDeviceMapping#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.imagebuilder.model.InstanceBlockDeviceMapping.Builder#build()} is
         * called immediately and its result is passed to {@link #blockDeviceMappings(List<InstanceBlockDeviceMapping>)}.
         * 
         * @param blockDeviceMappings
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.imagebuilder.model.InstanceBlockDeviceMapping.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #blockDeviceMappings(java.util.Collection<InstanceBlockDeviceMapping>)
         */
        Builder blockDeviceMappings(Consumer<InstanceBlockDeviceMapping.Builder>... blockDeviceMappings);
    }

    static final class BuilderImpl implements Builder {
        private String image;

        private List<InstanceBlockDeviceMapping> blockDeviceMappings = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceConfiguration model) {
            image(model.image);
            blockDeviceMappings(model.blockDeviceMappings);
        }

        public final String getImage() {
            return image;
        }

        public final void setImage(String image) {
            this.image = image;
        }

        @Override
        public final Builder image(String image) {
            this.image = image;
            return this;
        }

        public final List<InstanceBlockDeviceMapping.Builder> getBlockDeviceMappings() {
            List<InstanceBlockDeviceMapping.Builder> result = InstanceBlockDeviceMappingsCopier
                    .copyToBuilder(this.blockDeviceMappings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setBlockDeviceMappings(Collection<InstanceBlockDeviceMapping.BuilderImpl> blockDeviceMappings) {
            this.blockDeviceMappings = InstanceBlockDeviceMappingsCopier.copyFromBuilder(blockDeviceMappings);
        }

        @Override
        public final Builder blockDeviceMappings(Collection<InstanceBlockDeviceMapping> blockDeviceMappings) {
            this.blockDeviceMappings = InstanceBlockDeviceMappingsCopier.copy(blockDeviceMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder blockDeviceMappings(InstanceBlockDeviceMapping... blockDeviceMappings) {
            blockDeviceMappings(Arrays.asList(blockDeviceMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder blockDeviceMappings(Consumer<InstanceBlockDeviceMapping.Builder>... blockDeviceMappings) {
            blockDeviceMappings(Stream.of(blockDeviceMappings)
                    .map(c -> InstanceBlockDeviceMapping.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public InstanceConfiguration build() {
            return new InstanceConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
