/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.imagebuilder.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A detailed view of a component.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Component implements SdkPojo, Serializable, ToCopyableBuilder<Component.Builder, Component> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(Component::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Component::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("version")
            .getter(getter(Component::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("version").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(Component::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<String> CHANGE_DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("changeDescription").getter(getter(Component::changeDescription))
            .setter(setter(Builder::changeDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("changeDescription").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(Component::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> PLATFORM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("platform").getter(getter(Component::platformAsString)).setter(setter(Builder::platform))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("platform").build()).build();

    private static final SdkField<List<String>> SUPPORTED_OS_VERSIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("supportedOsVersions")
            .getter(getter(Component::supportedOsVersions))
            .setter(setter(Builder::supportedOsVersions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("supportedOsVersions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ComponentState> STATE_FIELD = SdkField.<ComponentState> builder(MarshallingType.SDK_POJO)
            .memberName("state").getter(getter(Component::state)).setter(setter(Builder::state))
            .constructor(ComponentState::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<List<ComponentParameterDetail>> PARAMETERS_FIELD = SdkField
            .<List<ComponentParameterDetail>> builder(MarshallingType.LIST)
            .memberName("parameters")
            .getter(getter(Component::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("parameters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ComponentParameterDetail> builder(MarshallingType.SDK_POJO)
                                            .constructor(ComponentParameterDetail::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> OWNER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("owner")
            .getter(getter(Component::owner)).setter(setter(Builder::owner))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("owner").build()).build();

    private static final SdkField<String> DATA_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("data")
            .getter(getter(Component::data)).setter(setter(Builder::data))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("data").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("kmsKeyId").getter(getter(Component::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("kmsKeyId").build()).build();

    private static final SdkField<Boolean> ENCRYPTED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("encrypted").getter(getter(Component::encrypted)).setter(setter(Builder::encrypted))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("encrypted").build()).build();

    private static final SdkField<String> DATE_CREATED_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("dateCreated").getter(getter(Component::dateCreated)).setter(setter(Builder::dateCreated))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dateCreated").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(Component::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> PUBLISHER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("publisher").getter(getter(Component::publisher)).setter(setter(Builder::publisher))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("publisher").build()).build();

    private static final SdkField<Boolean> OBFUSCATE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("obfuscate").getter(getter(Component::obfuscate)).setter(setter(Builder::obfuscate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("obfuscate").build()).build();

    private static final SdkField<List<ProductCodeListItem>> PRODUCT_CODES_FIELD = SdkField
            .<List<ProductCodeListItem>> builder(MarshallingType.LIST)
            .memberName("productCodes")
            .getter(getter(Component::productCodes))
            .setter(setter(Builder::productCodes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("productCodes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ProductCodeListItem> builder(MarshallingType.SDK_POJO)
                                            .constructor(ProductCodeListItem::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD, NAME_FIELD,
            VERSION_FIELD, DESCRIPTION_FIELD, CHANGE_DESCRIPTION_FIELD, TYPE_FIELD, PLATFORM_FIELD, SUPPORTED_OS_VERSIONS_FIELD,
            STATE_FIELD, PARAMETERS_FIELD, OWNER_FIELD, DATA_FIELD, KMS_KEY_ID_FIELD, ENCRYPTED_FIELD, DATE_CREATED_FIELD,
            TAGS_FIELD, PUBLISHER_FIELD, OBFUSCATE_FIELD, PRODUCT_CODES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final String name;

    private final String version;

    private final String description;

    private final String changeDescription;

    private final String type;

    private final String platform;

    private final List<String> supportedOsVersions;

    private final ComponentState state;

    private final List<ComponentParameterDetail> parameters;

    private final String owner;

    private final String data;

    private final String kmsKeyId;

    private final Boolean encrypted;

    private final String dateCreated;

    private final Map<String, String> tags;

    private final String publisher;

    private final Boolean obfuscate;

    private final List<ProductCodeListItem> productCodes;

    private Component(BuilderImpl builder) {
        this.arn = builder.arn;
        this.name = builder.name;
        this.version = builder.version;
        this.description = builder.description;
        this.changeDescription = builder.changeDescription;
        this.type = builder.type;
        this.platform = builder.platform;
        this.supportedOsVersions = builder.supportedOsVersions;
        this.state = builder.state;
        this.parameters = builder.parameters;
        this.owner = builder.owner;
        this.data = builder.data;
        this.kmsKeyId = builder.kmsKeyId;
        this.encrypted = builder.encrypted;
        this.dateCreated = builder.dateCreated;
        this.tags = builder.tags;
        this.publisher = builder.publisher;
        this.obfuscate = builder.obfuscate;
        this.productCodes = builder.productCodes;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the component.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the component.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The name of the component.
     * </p>
     * 
     * @return The name of the component.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The version of the component.
     * </p>
     * 
     * @return The version of the component.
     */
    public final String version() {
        return version;
    }

    /**
     * <p>
     * The description of the component.
     * </p>
     * 
     * @return The description of the component.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * Describes what change has been made in this version of the component, or what makes this version different from
     * other versions of the component.
     * </p>
     * 
     * @return Describes what change has been made in this version of the component, or what makes this version
     *         different from other versions of the component.
     */
    public final String changeDescription() {
        return changeDescription;
    }

    /**
     * <p>
     * The component type specifies whether Image Builder uses the component to build the image or only to test it.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ComponentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The component type specifies whether Image Builder uses the component to build the image or only to test
     *         it.
     * @see ComponentType
     */
    public final ComponentType type() {
        return ComponentType.fromValue(type);
    }

    /**
     * <p>
     * The component type specifies whether Image Builder uses the component to build the image or only to test it.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ComponentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The component type specifies whether Image Builder uses the component to build the image or only to test
     *         it.
     * @see ComponentType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The operating system platform of the component.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #platform} will
     * return {@link Platform#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #platformAsString}.
     * </p>
     * 
     * @return The operating system platform of the component.
     * @see Platform
     */
    public final Platform platform() {
        return Platform.fromValue(platform);
    }

    /**
     * <p>
     * The operating system platform of the component.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #platform} will
     * return {@link Platform#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #platformAsString}.
     * </p>
     * 
     * @return The operating system platform of the component.
     * @see Platform
     */
    public final String platformAsString() {
        return platform;
    }

    /**
     * For responses, this returns true if the service returned a value for the SupportedOsVersions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSupportedOsVersions() {
        return supportedOsVersions != null && !(supportedOsVersions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The operating system (OS) version supported by the component. If the OS information is available, Image Builder
     * performs a prefix match against the base image OS version during image recipe creation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportedOsVersions} method.
     * </p>
     * 
     * @return The operating system (OS) version supported by the component. If the OS information is available, Image
     *         Builder performs a prefix match against the base image OS version during image recipe creation.
     */
    public final List<String> supportedOsVersions() {
        return supportedOsVersions;
    }

    /**
     * <p>
     * Describes the current status of the component.
     * </p>
     * 
     * @return Describes the current status of the component.
     */
    public final ComponentState state() {
        return state;
    }

    /**
     * For responses, this returns true if the service returned a value for the Parameters property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Contains parameter details for each of the parameters that the component document defined for the component.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParameters} method.
     * </p>
     * 
     * @return Contains parameter details for each of the parameters that the component document defined for the
     *         component.
     */
    public final List<ComponentParameterDetail> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The owner of the component.
     * </p>
     * 
     * @return The owner of the component.
     */
    public final String owner() {
        return owner;
    }

    /**
     * <p>
     * Component data contains the YAML document content for the component.
     * </p>
     * 
     * @return Component data contains the YAML document content for the component.
     */
    public final String data() {
        return data;
    }

    /**
     * <p>
     * The KMS key identifier used to encrypt the component. This can be either the Key ARN or the Alias ARN. For more
     * information, see <a href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#key-id-key-ARN">Key
     * identifiers (KeyId)</a> in the <i>Key Management Service Developer Guide</i>.
     * </p>
     * 
     * @return The KMS key identifier used to encrypt the component. This can be either the Key ARN or the Alias ARN.
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#key-id-key-ARN">Key identifiers
     *         (KeyId)</a> in the <i>Key Management Service Developer Guide</i>.
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * The encryption status of the component.
     * </p>
     * 
     * @return The encryption status of the component.
     */
    public final Boolean encrypted() {
        return encrypted;
    }

    /**
     * <p>
     * The date that Image Builder created the component.
     * </p>
     * 
     * @return The date that Image Builder created the component.
     */
    public final String dateCreated() {
        return dateCreated;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The tags that apply to the component.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags that apply to the component.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * Contains the name of the publisher if this is a third-party component. Otherwise, this property is empty.
     * </p>
     * 
     * @return Contains the name of the publisher if this is a third-party component. Otherwise, this property is empty.
     */
    public final String publisher() {
        return publisher;
    }

    /**
     * <p>
     * Indicates whether component source is hidden from view in the console, and from component detail results for API,
     * CLI, or SDK operations.
     * </p>
     * 
     * @return Indicates whether component source is hidden from view in the console, and from component detail results
     *         for API, CLI, or SDK operations.
     */
    public final Boolean obfuscate() {
        return obfuscate;
    }

    /**
     * For responses, this returns true if the service returned a value for the ProductCodes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasProductCodes() {
        return productCodes != null && !(productCodes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Contains product codes that are used for billing purposes for Amazon Web Services Marketplace components.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProductCodes} method.
     * </p>
     * 
     * @return Contains product codes that are used for billing purposes for Amazon Web Services Marketplace components.
     */
    public final List<ProductCodeListItem> productCodes() {
        return productCodes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(changeDescription());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(platformAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasSupportedOsVersions() ? supportedOsVersions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(state());
        hashCode = 31 * hashCode + Objects.hashCode(hasParameters() ? parameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(owner());
        hashCode = 31 * hashCode + Objects.hashCode(data());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(encrypted());
        hashCode = 31 * hashCode + Objects.hashCode(dateCreated());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(publisher());
        hashCode = 31 * hashCode + Objects.hashCode(obfuscate());
        hashCode = 31 * hashCode + Objects.hashCode(hasProductCodes() ? productCodes() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Component)) {
            return false;
        }
        Component other = (Component) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(name(), other.name())
                && Objects.equals(version(), other.version()) && Objects.equals(description(), other.description())
                && Objects.equals(changeDescription(), other.changeDescription())
                && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(platformAsString(), other.platformAsString())
                && hasSupportedOsVersions() == other.hasSupportedOsVersions()
                && Objects.equals(supportedOsVersions(), other.supportedOsVersions()) && Objects.equals(state(), other.state())
                && hasParameters() == other.hasParameters() && Objects.equals(parameters(), other.parameters())
                && Objects.equals(owner(), other.owner()) && Objects.equals(data(), other.data())
                && Objects.equals(kmsKeyId(), other.kmsKeyId()) && Objects.equals(encrypted(), other.encrypted())
                && Objects.equals(dateCreated(), other.dateCreated()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(publisher(), other.publisher())
                && Objects.equals(obfuscate(), other.obfuscate()) && hasProductCodes() == other.hasProductCodes()
                && Objects.equals(productCodes(), other.productCodes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Component").add("Arn", arn()).add("Name", name()).add("Version", version())
                .add("Description", description()).add("ChangeDescription", changeDescription()).add("Type", typeAsString())
                .add("Platform", platformAsString())
                .add("SupportedOsVersions", hasSupportedOsVersions() ? supportedOsVersions() : null).add("State", state())
                .add("Parameters", hasParameters() ? parameters() : null).add("Owner", owner()).add("Data", data())
                .add("KmsKeyId", kmsKeyId()).add("Encrypted", encrypted()).add("DateCreated", dateCreated())
                .add("Tags", hasTags() ? tags() : null).add("Publisher", publisher()).add("Obfuscate", obfuscate())
                .add("ProductCodes", hasProductCodes() ? productCodes() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "version":
            return Optional.ofNullable(clazz.cast(version()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "changeDescription":
            return Optional.ofNullable(clazz.cast(changeDescription()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "platform":
            return Optional.ofNullable(clazz.cast(platformAsString()));
        case "supportedOsVersions":
            return Optional.ofNullable(clazz.cast(supportedOsVersions()));
        case "state":
            return Optional.ofNullable(clazz.cast(state()));
        case "parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "owner":
            return Optional.ofNullable(clazz.cast(owner()));
        case "data":
            return Optional.ofNullable(clazz.cast(data()));
        case "kmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "encrypted":
            return Optional.ofNullable(clazz.cast(encrypted()));
        case "dateCreated":
            return Optional.ofNullable(clazz.cast(dateCreated()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "publisher":
            return Optional.ofNullable(clazz.cast(publisher()));
        case "obfuscate":
            return Optional.ofNullable(clazz.cast(obfuscate()));
        case "productCodes":
            return Optional.ofNullable(clazz.cast(productCodes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("arn", ARN_FIELD);
        map.put("name", NAME_FIELD);
        map.put("version", VERSION_FIELD);
        map.put("description", DESCRIPTION_FIELD);
        map.put("changeDescription", CHANGE_DESCRIPTION_FIELD);
        map.put("type", TYPE_FIELD);
        map.put("platform", PLATFORM_FIELD);
        map.put("supportedOsVersions", SUPPORTED_OS_VERSIONS_FIELD);
        map.put("state", STATE_FIELD);
        map.put("parameters", PARAMETERS_FIELD);
        map.put("owner", OWNER_FIELD);
        map.put("data", DATA_FIELD);
        map.put("kmsKeyId", KMS_KEY_ID_FIELD);
        map.put("encrypted", ENCRYPTED_FIELD);
        map.put("dateCreated", DATE_CREATED_FIELD);
        map.put("tags", TAGS_FIELD);
        map.put("publisher", PUBLISHER_FIELD);
        map.put("obfuscate", OBFUSCATE_FIELD);
        map.put("productCodes", PRODUCT_CODES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Component, T> g) {
        return obj -> g.apply((Component) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Component> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the component.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The name of the component.
         * </p>
         * 
         * @param name
         *        The name of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The version of the component.
         * </p>
         * 
         * @param version
         *        The version of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * The description of the component.
         * </p>
         * 
         * @param description
         *        The description of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * Describes what change has been made in this version of the component, or what makes this version different
         * from other versions of the component.
         * </p>
         * 
         * @param changeDescription
         *        Describes what change has been made in this version of the component, or what makes this version
         *        different from other versions of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder changeDescription(String changeDescription);

        /**
         * <p>
         * The component type specifies whether Image Builder uses the component to build the image or only to test it.
         * </p>
         * 
         * @param type
         *        The component type specifies whether Image Builder uses the component to build the image or only to
         *        test it.
         * @see ComponentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComponentType
         */
        Builder type(String type);

        /**
         * <p>
         * The component type specifies whether Image Builder uses the component to build the image or only to test it.
         * </p>
         * 
         * @param type
         *        The component type specifies whether Image Builder uses the component to build the image or only to
         *        test it.
         * @see ComponentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComponentType
         */
        Builder type(ComponentType type);

        /**
         * <p>
         * The operating system platform of the component.
         * </p>
         * 
         * @param platform
         *        The operating system platform of the component.
         * @see Platform
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Platform
         */
        Builder platform(String platform);

        /**
         * <p>
         * The operating system platform of the component.
         * </p>
         * 
         * @param platform
         *        The operating system platform of the component.
         * @see Platform
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Platform
         */
        Builder platform(Platform platform);

        /**
         * <p>
         * The operating system (OS) version supported by the component. If the OS information is available, Image
         * Builder performs a prefix match against the base image OS version during image recipe creation.
         * </p>
         * 
         * @param supportedOsVersions
         *        The operating system (OS) version supported by the component. If the OS information is available,
         *        Image Builder performs a prefix match against the base image OS version during image recipe creation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedOsVersions(Collection<String> supportedOsVersions);

        /**
         * <p>
         * The operating system (OS) version supported by the component. If the OS information is available, Image
         * Builder performs a prefix match against the base image OS version during image recipe creation.
         * </p>
         * 
         * @param supportedOsVersions
         *        The operating system (OS) version supported by the component. If the OS information is available,
         *        Image Builder performs a prefix match against the base image OS version during image recipe creation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedOsVersions(String... supportedOsVersions);

        /**
         * <p>
         * Describes the current status of the component.
         * </p>
         * 
         * @param state
         *        Describes the current status of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder state(ComponentState state);

        /**
         * <p>
         * Describes the current status of the component.
         * </p>
         * This is a convenience method that creates an instance of the {@link ComponentState.Builder} avoiding the need
         * to create one manually via {@link ComponentState#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ComponentState.Builder#build()} is called immediately and its
         * result is passed to {@link #state(ComponentState)}.
         * 
         * @param state
         *        a consumer that will call methods on {@link ComponentState.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #state(ComponentState)
         */
        default Builder state(Consumer<ComponentState.Builder> state) {
            return state(ComponentState.builder().applyMutation(state).build());
        }

        /**
         * <p>
         * Contains parameter details for each of the parameters that the component document defined for the component.
         * </p>
         * 
         * @param parameters
         *        Contains parameter details for each of the parameters that the component document defined for the
         *        component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Collection<ComponentParameterDetail> parameters);

        /**
         * <p>
         * Contains parameter details for each of the parameters that the component document defined for the component.
         * </p>
         * 
         * @param parameters
         *        Contains parameter details for each of the parameters that the component document defined for the
         *        component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(ComponentParameterDetail... parameters);

        /**
         * <p>
         * Contains parameter details for each of the parameters that the component document defined for the component.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.imagebuilder.model.ComponentParameterDetail.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.imagebuilder.model.ComponentParameterDetail#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.imagebuilder.model.ComponentParameterDetail.Builder#build()} is called
         * immediately and its result is passed to {@link #parameters(List<ComponentParameterDetail>)}.
         * 
         * @param parameters
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.imagebuilder.model.ComponentParameterDetail.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parameters(java.util.Collection<ComponentParameterDetail>)
         */
        Builder parameters(Consumer<ComponentParameterDetail.Builder>... parameters);

        /**
         * <p>
         * The owner of the component.
         * </p>
         * 
         * @param owner
         *        The owner of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder owner(String owner);

        /**
         * <p>
         * Component data contains the YAML document content for the component.
         * </p>
         * 
         * @param data
         *        Component data contains the YAML document content for the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder data(String data);

        /**
         * <p>
         * The KMS key identifier used to encrypt the component. This can be either the Key ARN or the Alias ARN. For
         * more information, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#key-id-key-ARN">Key identifiers
         * (KeyId)</a> in the <i>Key Management Service Developer Guide</i>.
         * </p>
         * 
         * @param kmsKeyId
         *        The KMS key identifier used to encrypt the component. This can be either the Key ARN or the Alias ARN.
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#key-id-key-ARN">Key
         *        identifiers (KeyId)</a> in the <i>Key Management Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * The encryption status of the component.
         * </p>
         * 
         * @param encrypted
         *        The encryption status of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encrypted(Boolean encrypted);

        /**
         * <p>
         * The date that Image Builder created the component.
         * </p>
         * 
         * @param dateCreated
         *        The date that Image Builder created the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dateCreated(String dateCreated);

        /**
         * <p>
         * The tags that apply to the component.
         * </p>
         * 
         * @param tags
         *        The tags that apply to the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * Contains the name of the publisher if this is a third-party component. Otherwise, this property is empty.
         * </p>
         * 
         * @param publisher
         *        Contains the name of the publisher if this is a third-party component. Otherwise, this property is
         *        empty.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publisher(String publisher);

        /**
         * <p>
         * Indicates whether component source is hidden from view in the console, and from component detail results for
         * API, CLI, or SDK operations.
         * </p>
         * 
         * @param obfuscate
         *        Indicates whether component source is hidden from view in the console, and from component detail
         *        results for API, CLI, or SDK operations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder obfuscate(Boolean obfuscate);

        /**
         * <p>
         * Contains product codes that are used for billing purposes for Amazon Web Services Marketplace components.
         * </p>
         * 
         * @param productCodes
         *        Contains product codes that are used for billing purposes for Amazon Web Services Marketplace
         *        components.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder productCodes(Collection<ProductCodeListItem> productCodes);

        /**
         * <p>
         * Contains product codes that are used for billing purposes for Amazon Web Services Marketplace components.
         * </p>
         * 
         * @param productCodes
         *        Contains product codes that are used for billing purposes for Amazon Web Services Marketplace
         *        components.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder productCodes(ProductCodeListItem... productCodes);

        /**
         * <p>
         * Contains product codes that are used for billing purposes for Amazon Web Services Marketplace components.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.imagebuilder.model.ProductCodeListItem.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.imagebuilder.model.ProductCodeListItem#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.imagebuilder.model.ProductCodeListItem.Builder#build()} is called
         * immediately and its result is passed to {@link #productCodes(List<ProductCodeListItem>)}.
         * 
         * @param productCodes
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.imagebuilder.model.ProductCodeListItem.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #productCodes(java.util.Collection<ProductCodeListItem>)
         */
        Builder productCodes(Consumer<ProductCodeListItem.Builder>... productCodes);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private String name;

        private String version;

        private String description;

        private String changeDescription;

        private String type;

        private String platform;

        private List<String> supportedOsVersions = DefaultSdkAutoConstructList.getInstance();

        private ComponentState state;

        private List<ComponentParameterDetail> parameters = DefaultSdkAutoConstructList.getInstance();

        private String owner;

        private String data;

        private String kmsKeyId;

        private Boolean encrypted;

        private String dateCreated;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private String publisher;

        private Boolean obfuscate;

        private List<ProductCodeListItem> productCodes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Component model) {
            arn(model.arn);
            name(model.name);
            version(model.version);
            description(model.description);
            changeDescription(model.changeDescription);
            type(model.type);
            platform(model.platform);
            supportedOsVersions(model.supportedOsVersions);
            state(model.state);
            parameters(model.parameters);
            owner(model.owner);
            data(model.data);
            kmsKeyId(model.kmsKeyId);
            encrypted(model.encrypted);
            dateCreated(model.dateCreated);
            tags(model.tags);
            publisher(model.publisher);
            obfuscate(model.obfuscate);
            productCodes(model.productCodes);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getChangeDescription() {
            return changeDescription;
        }

        public final void setChangeDescription(String changeDescription) {
            this.changeDescription = changeDescription;
        }

        @Override
        public final Builder changeDescription(String changeDescription) {
            this.changeDescription = changeDescription;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(ComponentType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getPlatform() {
            return platform;
        }

        public final void setPlatform(String platform) {
            this.platform = platform;
        }

        @Override
        public final Builder platform(String platform) {
            this.platform = platform;
            return this;
        }

        @Override
        public final Builder platform(Platform platform) {
            this.platform(platform == null ? null : platform.toString());
            return this;
        }

        public final Collection<String> getSupportedOsVersions() {
            if (supportedOsVersions instanceof SdkAutoConstructList) {
                return null;
            }
            return supportedOsVersions;
        }

        public final void setSupportedOsVersions(Collection<String> supportedOsVersions) {
            this.supportedOsVersions = OsVersionListCopier.copy(supportedOsVersions);
        }

        @Override
        public final Builder supportedOsVersions(Collection<String> supportedOsVersions) {
            this.supportedOsVersions = OsVersionListCopier.copy(supportedOsVersions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedOsVersions(String... supportedOsVersions) {
            supportedOsVersions(Arrays.asList(supportedOsVersions));
            return this;
        }

        public final ComponentState.Builder getState() {
            return state != null ? state.toBuilder() : null;
        }

        public final void setState(ComponentState.BuilderImpl state) {
            this.state = state != null ? state.build() : null;
        }

        @Override
        public final Builder state(ComponentState state) {
            this.state = state;
            return this;
        }

        public final List<ComponentParameterDetail.Builder> getParameters() {
            List<ComponentParameterDetail.Builder> result = ComponentParameterDetailListCopier.copyToBuilder(this.parameters);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setParameters(Collection<ComponentParameterDetail.BuilderImpl> parameters) {
            this.parameters = ComponentParameterDetailListCopier.copyFromBuilder(parameters);
        }

        @Override
        public final Builder parameters(Collection<ComponentParameterDetail> parameters) {
            this.parameters = ComponentParameterDetailListCopier.copy(parameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(ComponentParameterDetail... parameters) {
            parameters(Arrays.asList(parameters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(Consumer<ComponentParameterDetail.Builder>... parameters) {
            parameters(Stream.of(parameters).map(c -> ComponentParameterDetail.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getOwner() {
            return owner;
        }

        public final void setOwner(String owner) {
            this.owner = owner;
        }

        @Override
        public final Builder owner(String owner) {
            this.owner = owner;
            return this;
        }

        public final String getData() {
            return data;
        }

        public final void setData(String data) {
            this.data = data;
        }

        @Override
        public final Builder data(String data) {
            this.data = data;
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final Boolean getEncrypted() {
            return encrypted;
        }

        public final void setEncrypted(Boolean encrypted) {
            this.encrypted = encrypted;
        }

        @Override
        public final Builder encrypted(Boolean encrypted) {
            this.encrypted = encrypted;
            return this;
        }

        public final String getDateCreated() {
            return dateCreated;
        }

        public final void setDateCreated(String dateCreated) {
            this.dateCreated = dateCreated;
        }

        @Override
        public final Builder dateCreated(String dateCreated) {
            this.dateCreated = dateCreated;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final String getPublisher() {
            return publisher;
        }

        public final void setPublisher(String publisher) {
            this.publisher = publisher;
        }

        @Override
        public final Builder publisher(String publisher) {
            this.publisher = publisher;
            return this;
        }

        public final Boolean getObfuscate() {
            return obfuscate;
        }

        public final void setObfuscate(Boolean obfuscate) {
            this.obfuscate = obfuscate;
        }

        @Override
        public final Builder obfuscate(Boolean obfuscate) {
            this.obfuscate = obfuscate;
            return this;
        }

        public final List<ProductCodeListItem.Builder> getProductCodes() {
            List<ProductCodeListItem.Builder> result = ProductCodeListCopier.copyToBuilder(this.productCodes);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setProductCodes(Collection<ProductCodeListItem.BuilderImpl> productCodes) {
            this.productCodes = ProductCodeListCopier.copyFromBuilder(productCodes);
        }

        @Override
        public final Builder productCodes(Collection<ProductCodeListItem> productCodes) {
            this.productCodes = ProductCodeListCopier.copy(productCodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder productCodes(ProductCodeListItem... productCodes) {
            productCodes(Arrays.asList(productCodes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder productCodes(Consumer<ProductCodeListItem.Builder>... productCodes) {
            productCodes(Stream.of(productCodes).map(c -> ProductCodeListItem.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public Component build() {
            return new Component(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
