/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.imagebuilder.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains details for a resource that the runtime instance of the lifecycle policy identified for action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LifecycleExecutionResource implements SdkPojo, Serializable,
        ToCopyableBuilder<LifecycleExecutionResource.Builder, LifecycleExecutionResource> {
    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("accountId").getter(getter(LifecycleExecutionResource::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accountId").build()).build();

    private static final SdkField<String> RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceId").getter(getter(LifecycleExecutionResource::resourceId)).setter(setter(Builder::resourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceId").build()).build();

    private static final SdkField<LifecycleExecutionResourceState> STATE_FIELD = SdkField
            .<LifecycleExecutionResourceState> builder(MarshallingType.SDK_POJO).memberName("state")
            .getter(getter(LifecycleExecutionResource::state)).setter(setter(Builder::state))
            .constructor(LifecycleExecutionResourceState::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<LifecycleExecutionResourceAction> ACTION_FIELD = SdkField
            .<LifecycleExecutionResourceAction> builder(MarshallingType.SDK_POJO).memberName("action")
            .getter(getter(LifecycleExecutionResource::action)).setter(setter(Builder::action))
            .constructor(LifecycleExecutionResourceAction::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("action").build()).build();

    private static final SdkField<String> REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("region")
            .getter(getter(LifecycleExecutionResource::region)).setter(setter(Builder::region))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("region").build()).build();

    private static final SdkField<List<LifecycleExecutionSnapshotResource>> SNAPSHOTS_FIELD = SdkField
            .<List<LifecycleExecutionSnapshotResource>> builder(MarshallingType.LIST)
            .memberName("snapshots")
            .getter(getter(LifecycleExecutionResource::snapshots))
            .setter(setter(Builder::snapshots))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("snapshots").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LifecycleExecutionSnapshotResource> builder(MarshallingType.SDK_POJO)
                                            .constructor(LifecycleExecutionSnapshotResource::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> IMAGE_URIS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("imageUris")
            .getter(getter(LifecycleExecutionResource::imageUris))
            .setter(setter(Builder::imageUris))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imageUris").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("startTime").getter(getter(LifecycleExecutionResource::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("endTime").getter(getter(LifecycleExecutionResource::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_ID_FIELD,
            RESOURCE_ID_FIELD, STATE_FIELD, ACTION_FIELD, REGION_FIELD, SNAPSHOTS_FIELD, IMAGE_URIS_FIELD, START_TIME_FIELD,
            END_TIME_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String accountId;

    private final String resourceId;

    private final LifecycleExecutionResourceState state;

    private final LifecycleExecutionResourceAction action;

    private final String region;

    private final List<LifecycleExecutionSnapshotResource> snapshots;

    private final List<String> imageUris;

    private final Instant startTime;

    private final Instant endTime;

    private LifecycleExecutionResource(BuilderImpl builder) {
        this.accountId = builder.accountId;
        this.resourceId = builder.resourceId;
        this.state = builder.state;
        this.action = builder.action;
        this.region = builder.region;
        this.snapshots = builder.snapshots;
        this.imageUris = builder.imageUris;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
    }

    /**
     * <p>
     * The account that owns the impacted resource.
     * </p>
     * 
     * @return The account that owns the impacted resource.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * Identifies the impacted resource. The resource ID depends on the type of resource, as follows.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Image Builder image resources: Amazon Resource Name (ARN)
     * </p>
     * </li>
     * <li>
     * <p>
     * Distributed AMIs: AMI ID
     * </p>
     * </li>
     * <li>
     * <p>
     * Container images distributed to an ECR repository: image URI or SHA Digest
     * </p>
     * </li>
     * </ul>
     * 
     * @return Identifies the impacted resource. The resource ID depends on the type of resource, as follows.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Image Builder image resources: Amazon Resource Name (ARN)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Distributed AMIs: AMI ID
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Container images distributed to an ECR repository: image URI or SHA Digest
     *         </p>
     *         </li>
     */
    public final String resourceId() {
        return resourceId;
    }

    /**
     * <p>
     * The runtime state for the lifecycle execution.
     * </p>
     * 
     * @return The runtime state for the lifecycle execution.
     */
    public final LifecycleExecutionResourceState state() {
        return state;
    }

    /**
     * <p>
     * The action to take for the identified resource.
     * </p>
     * 
     * @return The action to take for the identified resource.
     */
    public final LifecycleExecutionResourceAction action() {
        return action;
    }

    /**
     * <p>
     * The Amazon Web Services Region where the lifecycle execution resource is stored.
     * </p>
     * 
     * @return The Amazon Web Services Region where the lifecycle execution resource is stored.
     */
    public final String region() {
        return region;
    }

    /**
     * For responses, this returns true if the service returned a value for the Snapshots property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSnapshots() {
        return snapshots != null && !(snapshots instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of associated resource snapshots for the impacted resource if it’s an AMI.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSnapshots} method.
     * </p>
     * 
     * @return A list of associated resource snapshots for the impacted resource if it’s an AMI.
     */
    public final List<LifecycleExecutionSnapshotResource> snapshots() {
        return snapshots;
    }

    /**
     * For responses, this returns true if the service returned a value for the ImageUris property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasImageUris() {
        return imageUris != null && !(imageUris instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * For an impacted container image, this identifies a list of URIs for associated container images distributed to
     * ECR repositories.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasImageUris} method.
     * </p>
     * 
     * @return For an impacted container image, this identifies a list of URIs for associated container images
     *         distributed to ECR repositories.
     */
    public final List<String> imageUris() {
        return imageUris;
    }

    /**
     * <p>
     * The starting timestamp from the lifecycle action that was applied to the resource.
     * </p>
     * 
     * @return The starting timestamp from the lifecycle action that was applied to the resource.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The ending timestamp from the lifecycle action that was applied to the resource.
     * </p>
     * 
     * @return The ending timestamp from the lifecycle action that was applied to the resource.
     */
    public final Instant endTime() {
        return endTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceId());
        hashCode = 31 * hashCode + Objects.hashCode(state());
        hashCode = 31 * hashCode + Objects.hashCode(action());
        hashCode = 31 * hashCode + Objects.hashCode(region());
        hashCode = 31 * hashCode + Objects.hashCode(hasSnapshots() ? snapshots() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasImageUris() ? imageUris() : null);
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LifecycleExecutionResource)) {
            return false;
        }
        LifecycleExecutionResource other = (LifecycleExecutionResource) obj;
        return Objects.equals(accountId(), other.accountId()) && Objects.equals(resourceId(), other.resourceId())
                && Objects.equals(state(), other.state()) && Objects.equals(action(), other.action())
                && Objects.equals(region(), other.region()) && hasSnapshots() == other.hasSnapshots()
                && Objects.equals(snapshots(), other.snapshots()) && hasImageUris() == other.hasImageUris()
                && Objects.equals(imageUris(), other.imageUris()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(endTime(), other.endTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LifecycleExecutionResource").add("AccountId", accountId()).add("ResourceId", resourceId())
                .add("State", state()).add("Action", action()).add("Region", region())
                .add("Snapshots", hasSnapshots() ? snapshots() : null).add("ImageUris", hasImageUris() ? imageUris() : null)
                .add("StartTime", startTime()).add("EndTime", endTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "accountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "resourceId":
            return Optional.ofNullable(clazz.cast(resourceId()));
        case "state":
            return Optional.ofNullable(clazz.cast(state()));
        case "action":
            return Optional.ofNullable(clazz.cast(action()));
        case "region":
            return Optional.ofNullable(clazz.cast(region()));
        case "snapshots":
            return Optional.ofNullable(clazz.cast(snapshots()));
        case "imageUris":
            return Optional.ofNullable(clazz.cast(imageUris()));
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "endTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("accountId", ACCOUNT_ID_FIELD);
        map.put("resourceId", RESOURCE_ID_FIELD);
        map.put("state", STATE_FIELD);
        map.put("action", ACTION_FIELD);
        map.put("region", REGION_FIELD);
        map.put("snapshots", SNAPSHOTS_FIELD);
        map.put("imageUris", IMAGE_URIS_FIELD);
        map.put("startTime", START_TIME_FIELD);
        map.put("endTime", END_TIME_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<LifecycleExecutionResource, T> g) {
        return obj -> g.apply((LifecycleExecutionResource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LifecycleExecutionResource> {
        /**
         * <p>
         * The account that owns the impacted resource.
         * </p>
         * 
         * @param accountId
         *        The account that owns the impacted resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * Identifies the impacted resource. The resource ID depends on the type of resource, as follows.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Image Builder image resources: Amazon Resource Name (ARN)
         * </p>
         * </li>
         * <li>
         * <p>
         * Distributed AMIs: AMI ID
         * </p>
         * </li>
         * <li>
         * <p>
         * Container images distributed to an ECR repository: image URI or SHA Digest
         * </p>
         * </li>
         * </ul>
         * 
         * @param resourceId
         *        Identifies the impacted resource. The resource ID depends on the type of resource, as follows.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Image Builder image resources: Amazon Resource Name (ARN)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Distributed AMIs: AMI ID
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Container images distributed to an ECR repository: image URI or SHA Digest
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceId(String resourceId);

        /**
         * <p>
         * The runtime state for the lifecycle execution.
         * </p>
         * 
         * @param state
         *        The runtime state for the lifecycle execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder state(LifecycleExecutionResourceState state);

        /**
         * <p>
         * The runtime state for the lifecycle execution.
         * </p>
         * This is a convenience method that creates an instance of the {@link LifecycleExecutionResourceState.Builder}
         * avoiding the need to create one manually via {@link LifecycleExecutionResourceState#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LifecycleExecutionResourceState.Builder#build()} is called
         * immediately and its result is passed to {@link #state(LifecycleExecutionResourceState)}.
         * 
         * @param state
         *        a consumer that will call methods on {@link LifecycleExecutionResourceState.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #state(LifecycleExecutionResourceState)
         */
        default Builder state(Consumer<LifecycleExecutionResourceState.Builder> state) {
            return state(LifecycleExecutionResourceState.builder().applyMutation(state).build());
        }

        /**
         * <p>
         * The action to take for the identified resource.
         * </p>
         * 
         * @param action
         *        The action to take for the identified resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder action(LifecycleExecutionResourceAction action);

        /**
         * <p>
         * The action to take for the identified resource.
         * </p>
         * This is a convenience method that creates an instance of the {@link LifecycleExecutionResourceAction.Builder}
         * avoiding the need to create one manually via {@link LifecycleExecutionResourceAction#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LifecycleExecutionResourceAction.Builder#build()} is called
         * immediately and its result is passed to {@link #action(LifecycleExecutionResourceAction)}.
         * 
         * @param action
         *        a consumer that will call methods on {@link LifecycleExecutionResourceAction.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #action(LifecycleExecutionResourceAction)
         */
        default Builder action(Consumer<LifecycleExecutionResourceAction.Builder> action) {
            return action(LifecycleExecutionResourceAction.builder().applyMutation(action).build());
        }

        /**
         * <p>
         * The Amazon Web Services Region where the lifecycle execution resource is stored.
         * </p>
         * 
         * @param region
         *        The Amazon Web Services Region where the lifecycle execution resource is stored.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder region(String region);

        /**
         * <p>
         * A list of associated resource snapshots for the impacted resource if it’s an AMI.
         * </p>
         * 
         * @param snapshots
         *        A list of associated resource snapshots for the impacted resource if it’s an AMI.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshots(Collection<LifecycleExecutionSnapshotResource> snapshots);

        /**
         * <p>
         * A list of associated resource snapshots for the impacted resource if it’s an AMI.
         * </p>
         * 
         * @param snapshots
         *        A list of associated resource snapshots for the impacted resource if it’s an AMI.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshots(LifecycleExecutionSnapshotResource... snapshots);

        /**
         * <p>
         * A list of associated resource snapshots for the impacted resource if it’s an AMI.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.imagebuilder.model.LifecycleExecutionSnapshotResource.Builder}
         * avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.imagebuilder.model.LifecycleExecutionSnapshotResource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.imagebuilder.model.LifecycleExecutionSnapshotResource.Builder#build()}
         * is called immediately and its result is passed to {@link
         * #snapshots(List<LifecycleExecutionSnapshotResource>)}.
         * 
         * @param snapshots
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.imagebuilder.model.LifecycleExecutionSnapshotResource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #snapshots(java.util.Collection<LifecycleExecutionSnapshotResource>)
         */
        Builder snapshots(Consumer<LifecycleExecutionSnapshotResource.Builder>... snapshots);

        /**
         * <p>
         * For an impacted container image, this identifies a list of URIs for associated container images distributed
         * to ECR repositories.
         * </p>
         * 
         * @param imageUris
         *        For an impacted container image, this identifies a list of URIs for associated container images
         *        distributed to ECR repositories.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageUris(Collection<String> imageUris);

        /**
         * <p>
         * For an impacted container image, this identifies a list of URIs for associated container images distributed
         * to ECR repositories.
         * </p>
         * 
         * @param imageUris
         *        For an impacted container image, this identifies a list of URIs for associated container images
         *        distributed to ECR repositories.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageUris(String... imageUris);

        /**
         * <p>
         * The starting timestamp from the lifecycle action that was applied to the resource.
         * </p>
         * 
         * @param startTime
         *        The starting timestamp from the lifecycle action that was applied to the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The ending timestamp from the lifecycle action that was applied to the resource.
         * </p>
         * 
         * @param endTime
         *        The ending timestamp from the lifecycle action that was applied to the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);
    }

    static final class BuilderImpl implements Builder {
        private String accountId;

        private String resourceId;

        private LifecycleExecutionResourceState state;

        private LifecycleExecutionResourceAction action;

        private String region;

        private List<LifecycleExecutionSnapshotResource> snapshots = DefaultSdkAutoConstructList.getInstance();

        private List<String> imageUris = DefaultSdkAutoConstructList.getInstance();

        private Instant startTime;

        private Instant endTime;

        private BuilderImpl() {
        }

        private BuilderImpl(LifecycleExecutionResource model) {
            accountId(model.accountId);
            resourceId(model.resourceId);
            state(model.state);
            action(model.action);
            region(model.region);
            snapshots(model.snapshots);
            imageUris(model.imageUris);
            startTime(model.startTime);
            endTime(model.endTime);
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final String getResourceId() {
            return resourceId;
        }

        public final void setResourceId(String resourceId) {
            this.resourceId = resourceId;
        }

        @Override
        public final Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public final LifecycleExecutionResourceState.Builder getState() {
            return state != null ? state.toBuilder() : null;
        }

        public final void setState(LifecycleExecutionResourceState.BuilderImpl state) {
            this.state = state != null ? state.build() : null;
        }

        @Override
        public final Builder state(LifecycleExecutionResourceState state) {
            this.state = state;
            return this;
        }

        public final LifecycleExecutionResourceAction.Builder getAction() {
            return action != null ? action.toBuilder() : null;
        }

        public final void setAction(LifecycleExecutionResourceAction.BuilderImpl action) {
            this.action = action != null ? action.build() : null;
        }

        @Override
        public final Builder action(LifecycleExecutionResourceAction action) {
            this.action = action;
            return this;
        }

        public final String getRegion() {
            return region;
        }

        public final void setRegion(String region) {
            this.region = region;
        }

        @Override
        public final Builder region(String region) {
            this.region = region;
            return this;
        }

        public final List<LifecycleExecutionSnapshotResource.Builder> getSnapshots() {
            List<LifecycleExecutionSnapshotResource.Builder> result = LifecycleExecutionSnapshotResourceListCopier
                    .copyToBuilder(this.snapshots);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSnapshots(Collection<LifecycleExecutionSnapshotResource.BuilderImpl> snapshots) {
            this.snapshots = LifecycleExecutionSnapshotResourceListCopier.copyFromBuilder(snapshots);
        }

        @Override
        public final Builder snapshots(Collection<LifecycleExecutionSnapshotResource> snapshots) {
            this.snapshots = LifecycleExecutionSnapshotResourceListCopier.copy(snapshots);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder snapshots(LifecycleExecutionSnapshotResource... snapshots) {
            snapshots(Arrays.asList(snapshots));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder snapshots(Consumer<LifecycleExecutionSnapshotResource.Builder>... snapshots) {
            snapshots(Stream.of(snapshots).map(c -> LifecycleExecutionSnapshotResource.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getImageUris() {
            if (imageUris instanceof SdkAutoConstructList) {
                return null;
            }
            return imageUris;
        }

        public final void setImageUris(Collection<String> imageUris) {
            this.imageUris = StringListCopier.copy(imageUris);
        }

        @Override
        public final Builder imageUris(Collection<String> imageUris) {
            this.imageUris = StringListCopier.copy(imageUris);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder imageUris(String... imageUris) {
            imageUris(Arrays.asList(imageUris));
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        @Override
        public LifecycleExecutionResource build() {
            return new LifecycleExecutionResource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
