/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.inspector.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A snapshot of an Amazon Inspector assessment run that contains the findings of the assessment run .
 * </p>
 * <p>
 * Used as the response element in the <a>DescribeAssessmentRuns</a> action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AssessmentRun implements SdkPojo, Serializable, ToCopyableBuilder<AssessmentRun.Builder, AssessmentRun> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AssessmentRun::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AssessmentRun::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ASSESSMENT_TEMPLATE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AssessmentRun::assessmentTemplateArn)).setter(setter(Builder::assessmentTemplateArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("assessmentTemplateArn").build())
            .build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AssessmentRun::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<Integer> DURATION_IN_SECONDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(AssessmentRun::durationInSeconds)).setter(setter(Builder::durationInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("durationInSeconds").build()).build();

    private static final SdkField<List<String>> RULES_PACKAGE_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(AssessmentRun::rulesPackageArns))
            .setter(setter(Builder::rulesPackageArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("rulesPackageArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Attribute>> USER_ATTRIBUTES_FOR_FINDINGS_FIELD = SdkField
            .<List<Attribute>> builder(MarshallingType.LIST)
            .getter(getter(AssessmentRun::userAttributesForFindings))
            .setter(setter(Builder::userAttributesForFindings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("userAttributesForFindings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Attribute> builder(MarshallingType.SDK_POJO)
                                            .constructor(Attribute::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(AssessmentRun::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<Instant> STARTED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(AssessmentRun::startedAt)).setter(setter(Builder::startedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startedAt").build()).build();

    private static final SdkField<Instant> COMPLETED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(AssessmentRun::completedAt)).setter(setter(Builder::completedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("completedAt").build()).build();

    private static final SdkField<Instant> STATE_CHANGED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(AssessmentRun::stateChangedAt)).setter(setter(Builder::stateChangedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("stateChangedAt").build()).build();

    private static final SdkField<Boolean> DATA_COLLECTED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(AssessmentRun::dataCollected)).setter(setter(Builder::dataCollected))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dataCollected").build()).build();

    private static final SdkField<List<AssessmentRunStateChange>> STATE_CHANGES_FIELD = SdkField
            .<List<AssessmentRunStateChange>> builder(MarshallingType.LIST)
            .getter(getter(AssessmentRun::stateChanges))
            .setter(setter(Builder::stateChanges))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("stateChanges").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AssessmentRunStateChange> builder(MarshallingType.SDK_POJO)
                                            .constructor(AssessmentRunStateChange::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<AssessmentRunNotification>> NOTIFICATIONS_FIELD = SdkField
            .<List<AssessmentRunNotification>> builder(MarshallingType.LIST)
            .getter(getter(AssessmentRun::notifications))
            .setter(setter(Builder::notifications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("notifications").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AssessmentRunNotification> builder(MarshallingType.SDK_POJO)
                                            .constructor(AssessmentRunNotification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, Integer>> FINDING_COUNTS_FIELD = SdkField
            .<Map<String, Integer>> builder(MarshallingType.MAP)
            .getter(getter(AssessmentRun::findingCountsAsStrings))
            .setter(setter(Builder::findingCountsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("findingCounts").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<Integer> builder(MarshallingType.INTEGER)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD, NAME_FIELD,
            ASSESSMENT_TEMPLATE_ARN_FIELD, STATE_FIELD, DURATION_IN_SECONDS_FIELD, RULES_PACKAGE_ARNS_FIELD,
            USER_ATTRIBUTES_FOR_FINDINGS_FIELD, CREATED_AT_FIELD, STARTED_AT_FIELD, COMPLETED_AT_FIELD, STATE_CHANGED_AT_FIELD,
            DATA_COLLECTED_FIELD, STATE_CHANGES_FIELD, NOTIFICATIONS_FIELD, FINDING_COUNTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final String name;

    private final String assessmentTemplateArn;

    private final String state;

    private final Integer durationInSeconds;

    private final List<String> rulesPackageArns;

    private final List<Attribute> userAttributesForFindings;

    private final Instant createdAt;

    private final Instant startedAt;

    private final Instant completedAt;

    private final Instant stateChangedAt;

    private final Boolean dataCollected;

    private final List<AssessmentRunStateChange> stateChanges;

    private final List<AssessmentRunNotification> notifications;

    private final Map<String, Integer> findingCounts;

    private AssessmentRun(BuilderImpl builder) {
        this.arn = builder.arn;
        this.name = builder.name;
        this.assessmentTemplateArn = builder.assessmentTemplateArn;
        this.state = builder.state;
        this.durationInSeconds = builder.durationInSeconds;
        this.rulesPackageArns = builder.rulesPackageArns;
        this.userAttributesForFindings = builder.userAttributesForFindings;
        this.createdAt = builder.createdAt;
        this.startedAt = builder.startedAt;
        this.completedAt = builder.completedAt;
        this.stateChangedAt = builder.stateChangedAt;
        this.dataCollected = builder.dataCollected;
        this.stateChanges = builder.stateChanges;
        this.notifications = builder.notifications;
        this.findingCounts = builder.findingCounts;
    }

    /**
     * <p>
     * The ARN of the assessment run.
     * </p>
     * 
     * @return The ARN of the assessment run.
     */
    public String arn() {
        return arn;
    }

    /**
     * <p>
     * The auto-generated name for the assessment run.
     * </p>
     * 
     * @return The auto-generated name for the assessment run.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The ARN of the assessment template that is associated with the assessment run.
     * </p>
     * 
     * @return The ARN of the assessment template that is associated with the assessment run.
     */
    public String assessmentTemplateArn() {
        return assessmentTemplateArn;
    }

    /**
     * <p>
     * The state of the assessment run.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link AssessmentRunState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the assessment run.
     * @see AssessmentRunState
     */
    public AssessmentRunState state() {
        return AssessmentRunState.fromValue(state);
    }

    /**
     * <p>
     * The state of the assessment run.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link AssessmentRunState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the assessment run.
     * @see AssessmentRunState
     */
    public String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The duration of the assessment run.
     * </p>
     * 
     * @return The duration of the assessment run.
     */
    public Integer durationInSeconds() {
        return durationInSeconds;
    }

    /**
     * <p>
     * The rules packages selected for the assessment run.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The rules packages selected for the assessment run.
     */
    public List<String> rulesPackageArns() {
        return rulesPackageArns;
    }

    /**
     * <p>
     * The user-defined attributes that are assigned to every generated finding.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The user-defined attributes that are assigned to every generated finding.
     */
    public List<Attribute> userAttributesForFindings() {
        return userAttributesForFindings;
    }

    /**
     * <p>
     * The time when <a>StartAssessmentRun</a> was called.
     * </p>
     * 
     * @return The time when <a>StartAssessmentRun</a> was called.
     */
    public Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The time when <a>StartAssessmentRun</a> was called.
     * </p>
     * 
     * @return The time when <a>StartAssessmentRun</a> was called.
     */
    public Instant startedAt() {
        return startedAt;
    }

    /**
     * <p>
     * The assessment run completion time that corresponds to the rules packages evaluation completion time or failure.
     * </p>
     * 
     * @return The assessment run completion time that corresponds to the rules packages evaluation completion time or
     *         failure.
     */
    public Instant completedAt() {
        return completedAt;
    }

    /**
     * <p>
     * The last time when the assessment run's state changed.
     * </p>
     * 
     * @return The last time when the assessment run's state changed.
     */
    public Instant stateChangedAt() {
        return stateChangedAt;
    }

    /**
     * <p>
     * A Boolean value (true or false) that specifies whether the process of collecting data from the agents is
     * completed.
     * </p>
     * 
     * @return A Boolean value (true or false) that specifies whether the process of collecting data from the agents is
     *         completed.
     */
    public Boolean dataCollected() {
        return dataCollected;
    }

    /**
     * <p>
     * A list of the assessment run state changes.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of the assessment run state changes.
     */
    public List<AssessmentRunStateChange> stateChanges() {
        return stateChanges;
    }

    /**
     * <p>
     * A list of notifications for the event subscriptions. A notification about a particular generated finding is added
     * to this list only once.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of notifications for the event subscriptions. A notification about a particular generated finding
     *         is added to this list only once.
     */
    public List<AssessmentRunNotification> notifications() {
        return notifications;
    }

    /**
     * <p>
     * Provides a total count of generated findings per severity.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Provides a total count of generated findings per severity.
     */
    public Map<Severity, Integer> findingCounts() {
        return AssessmentRunFindingCountsCopier.copyStringToEnum(findingCounts);
    }

    /**
     * <p>
     * Provides a total count of generated findings per severity.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Provides a total count of generated findings per severity.
     */
    public Map<String, Integer> findingCountsAsStrings() {
        return findingCounts;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(assessmentTemplateArn());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(durationInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(rulesPackageArns());
        hashCode = 31 * hashCode + Objects.hashCode(userAttributesForFindings());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(startedAt());
        hashCode = 31 * hashCode + Objects.hashCode(completedAt());
        hashCode = 31 * hashCode + Objects.hashCode(stateChangedAt());
        hashCode = 31 * hashCode + Objects.hashCode(dataCollected());
        hashCode = 31 * hashCode + Objects.hashCode(stateChanges());
        hashCode = 31 * hashCode + Objects.hashCode(notifications());
        hashCode = 31 * hashCode + Objects.hashCode(findingCountsAsStrings());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AssessmentRun)) {
            return false;
        }
        AssessmentRun other = (AssessmentRun) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(name(), other.name())
                && Objects.equals(assessmentTemplateArn(), other.assessmentTemplateArn())
                && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(durationInSeconds(), other.durationInSeconds())
                && Objects.equals(rulesPackageArns(), other.rulesPackageArns())
                && Objects.equals(userAttributesForFindings(), other.userAttributesForFindings())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(startedAt(), other.startedAt())
                && Objects.equals(completedAt(), other.completedAt()) && Objects.equals(stateChangedAt(), other.stateChangedAt())
                && Objects.equals(dataCollected(), other.dataCollected()) && Objects.equals(stateChanges(), other.stateChanges())
                && Objects.equals(notifications(), other.notifications())
                && Objects.equals(findingCountsAsStrings(), other.findingCountsAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("AssessmentRun").add("Arn", arn()).add("Name", name())
                .add("AssessmentTemplateArn", assessmentTemplateArn()).add("State", stateAsString())
                .add("DurationInSeconds", durationInSeconds()).add("RulesPackageArns", rulesPackageArns())
                .add("UserAttributesForFindings", userAttributesForFindings()).add("CreatedAt", createdAt())
                .add("StartedAt", startedAt()).add("CompletedAt", completedAt()).add("StateChangedAt", stateChangedAt())
                .add("DataCollected", dataCollected()).add("StateChanges", stateChanges()).add("Notifications", notifications())
                .add("FindingCounts", findingCountsAsStrings()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "assessmentTemplateArn":
            return Optional.ofNullable(clazz.cast(assessmentTemplateArn()));
        case "state":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "durationInSeconds":
            return Optional.ofNullable(clazz.cast(durationInSeconds()));
        case "rulesPackageArns":
            return Optional.ofNullable(clazz.cast(rulesPackageArns()));
        case "userAttributesForFindings":
            return Optional.ofNullable(clazz.cast(userAttributesForFindings()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "startedAt":
            return Optional.ofNullable(clazz.cast(startedAt()));
        case "completedAt":
            return Optional.ofNullable(clazz.cast(completedAt()));
        case "stateChangedAt":
            return Optional.ofNullable(clazz.cast(stateChangedAt()));
        case "dataCollected":
            return Optional.ofNullable(clazz.cast(dataCollected()));
        case "stateChanges":
            return Optional.ofNullable(clazz.cast(stateChanges()));
        case "notifications":
            return Optional.ofNullable(clazz.cast(notifications()));
        case "findingCounts":
            return Optional.ofNullable(clazz.cast(findingCountsAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AssessmentRun, T> g) {
        return obj -> g.apply((AssessmentRun) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AssessmentRun> {
        /**
         * <p>
         * The ARN of the assessment run.
         * </p>
         * 
         * @param arn
         *        The ARN of the assessment run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The auto-generated name for the assessment run.
         * </p>
         * 
         * @param name
         *        The auto-generated name for the assessment run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ARN of the assessment template that is associated with the assessment run.
         * </p>
         * 
         * @param assessmentTemplateArn
         *        The ARN of the assessment template that is associated with the assessment run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder assessmentTemplateArn(String assessmentTemplateArn);

        /**
         * <p>
         * The state of the assessment run.
         * </p>
         * 
         * @param state
         *        The state of the assessment run.
         * @see AssessmentRunState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AssessmentRunState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the assessment run.
         * </p>
         * 
         * @param state
         *        The state of the assessment run.
         * @see AssessmentRunState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AssessmentRunState
         */
        Builder state(AssessmentRunState state);

        /**
         * <p>
         * The duration of the assessment run.
         * </p>
         * 
         * @param durationInSeconds
         *        The duration of the assessment run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder durationInSeconds(Integer durationInSeconds);

        /**
         * <p>
         * The rules packages selected for the assessment run.
         * </p>
         * 
         * @param rulesPackageArns
         *        The rules packages selected for the assessment run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rulesPackageArns(Collection<String> rulesPackageArns);

        /**
         * <p>
         * The rules packages selected for the assessment run.
         * </p>
         * 
         * @param rulesPackageArns
         *        The rules packages selected for the assessment run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rulesPackageArns(String... rulesPackageArns);

        /**
         * <p>
         * The user-defined attributes that are assigned to every generated finding.
         * </p>
         * 
         * @param userAttributesForFindings
         *        The user-defined attributes that are assigned to every generated finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userAttributesForFindings(Collection<Attribute> userAttributesForFindings);

        /**
         * <p>
         * The user-defined attributes that are assigned to every generated finding.
         * </p>
         * 
         * @param userAttributesForFindings
         *        The user-defined attributes that are assigned to every generated finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userAttributesForFindings(Attribute... userAttributesForFindings);

        /**
         * <p>
         * The user-defined attributes that are assigned to every generated finding.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Attribute>.Builder} avoiding the need to
         * create one manually via {@link List<Attribute>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Attribute>.Builder#build()} is called immediately and its
         * result is passed to {@link #userAttributesForFindings(List<Attribute>)}.
         * 
         * @param userAttributesForFindings
         *        a consumer that will call methods on {@link List<Attribute>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #userAttributesForFindings(List<Attribute>)
         */
        Builder userAttributesForFindings(Consumer<Attribute.Builder>... userAttributesForFindings);

        /**
         * <p>
         * The time when <a>StartAssessmentRun</a> was called.
         * </p>
         * 
         * @param createdAt
         *        The time when <a>StartAssessmentRun</a> was called.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The time when <a>StartAssessmentRun</a> was called.
         * </p>
         * 
         * @param startedAt
         *        The time when <a>StartAssessmentRun</a> was called.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startedAt(Instant startedAt);

        /**
         * <p>
         * The assessment run completion time that corresponds to the rules packages evaluation completion time or
         * failure.
         * </p>
         * 
         * @param completedAt
         *        The assessment run completion time that corresponds to the rules packages evaluation completion time
         *        or failure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completedAt(Instant completedAt);

        /**
         * <p>
         * The last time when the assessment run's state changed.
         * </p>
         * 
         * @param stateChangedAt
         *        The last time when the assessment run's state changed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateChangedAt(Instant stateChangedAt);

        /**
         * <p>
         * A Boolean value (true or false) that specifies whether the process of collecting data from the agents is
         * completed.
         * </p>
         * 
         * @param dataCollected
         *        A Boolean value (true or false) that specifies whether the process of collecting data from the agents
         *        is completed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataCollected(Boolean dataCollected);

        /**
         * <p>
         * A list of the assessment run state changes.
         * </p>
         * 
         * @param stateChanges
         *        A list of the assessment run state changes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateChanges(Collection<AssessmentRunStateChange> stateChanges);

        /**
         * <p>
         * A list of the assessment run state changes.
         * </p>
         * 
         * @param stateChanges
         *        A list of the assessment run state changes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateChanges(AssessmentRunStateChange... stateChanges);

        /**
         * <p>
         * A list of the assessment run state changes.
         * </p>
         * This is a convenience that creates an instance of the {@link List<AssessmentRunStateChange>.Builder} avoiding
         * the need to create one manually via {@link List<AssessmentRunStateChange>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AssessmentRunStateChange>.Builder#build()} is called
         * immediately and its result is passed to {@link #stateChanges(List<AssessmentRunStateChange>)}.
         * 
         * @param stateChanges
         *        a consumer that will call methods on {@link List<AssessmentRunStateChange>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #stateChanges(List<AssessmentRunStateChange>)
         */
        Builder stateChanges(Consumer<AssessmentRunStateChange.Builder>... stateChanges);

        /**
         * <p>
         * A list of notifications for the event subscriptions. A notification about a particular generated finding is
         * added to this list only once.
         * </p>
         * 
         * @param notifications
         *        A list of notifications for the event subscriptions. A notification about a particular generated
         *        finding is added to this list only once.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notifications(Collection<AssessmentRunNotification> notifications);

        /**
         * <p>
         * A list of notifications for the event subscriptions. A notification about a particular generated finding is
         * added to this list only once.
         * </p>
         * 
         * @param notifications
         *        A list of notifications for the event subscriptions. A notification about a particular generated
         *        finding is added to this list only once.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notifications(AssessmentRunNotification... notifications);

        /**
         * <p>
         * A list of notifications for the event subscriptions. A notification about a particular generated finding is
         * added to this list only once.
         * </p>
         * This is a convenience that creates an instance of the {@link List<AssessmentRunNotification>.Builder}
         * avoiding the need to create one manually via {@link List<AssessmentRunNotification>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AssessmentRunNotification>.Builder#build()} is called
         * immediately and its result is passed to {@link #notifications(List<AssessmentRunNotification>)}.
         * 
         * @param notifications
         *        a consumer that will call methods on {@link List<AssessmentRunNotification>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #notifications(List<AssessmentRunNotification>)
         */
        Builder notifications(Consumer<AssessmentRunNotification.Builder>... notifications);

        /**
         * <p>
         * Provides a total count of generated findings per severity.
         * </p>
         * 
         * @param findingCounts
         *        Provides a total count of generated findings per severity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingCountsWithStrings(Map<String, Integer> findingCounts);

        /**
         * <p>
         * Provides a total count of generated findings per severity.
         * </p>
         * 
         * @param findingCounts
         *        Provides a total count of generated findings per severity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingCounts(Map<Severity, Integer> findingCounts);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private String name;

        private String assessmentTemplateArn;

        private String state;

        private Integer durationInSeconds;

        private List<String> rulesPackageArns = DefaultSdkAutoConstructList.getInstance();

        private List<Attribute> userAttributesForFindings = DefaultSdkAutoConstructList.getInstance();

        private Instant createdAt;

        private Instant startedAt;

        private Instant completedAt;

        private Instant stateChangedAt;

        private Boolean dataCollected;

        private List<AssessmentRunStateChange> stateChanges = DefaultSdkAutoConstructList.getInstance();

        private List<AssessmentRunNotification> notifications = DefaultSdkAutoConstructList.getInstance();

        private Map<String, Integer> findingCounts = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AssessmentRun model) {
            arn(model.arn);
            name(model.name);
            assessmentTemplateArn(model.assessmentTemplateArn);
            state(model.state);
            durationInSeconds(model.durationInSeconds);
            rulesPackageArns(model.rulesPackageArns);
            userAttributesForFindings(model.userAttributesForFindings);
            createdAt(model.createdAt);
            startedAt(model.startedAt);
            completedAt(model.completedAt);
            stateChangedAt(model.stateChangedAt);
            dataCollected(model.dataCollected);
            stateChanges(model.stateChanges);
            notifications(model.notifications);
            findingCountsWithStrings(model.findingCounts);
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getAssessmentTemplateArn() {
            return assessmentTemplateArn;
        }

        @Override
        public final Builder assessmentTemplateArn(String assessmentTemplateArn) {
            this.assessmentTemplateArn = assessmentTemplateArn;
            return this;
        }

        public final void setAssessmentTemplateArn(String assessmentTemplateArn) {
            this.assessmentTemplateArn = assessmentTemplateArn;
        }

        public final String getStateAsString() {
            return state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(AssessmentRunState state) {
            this.state(state.toString());
            return this;
        }

        public final void setState(String state) {
            this.state = state;
        }

        public final Integer getDurationInSeconds() {
            return durationInSeconds;
        }

        @Override
        public final Builder durationInSeconds(Integer durationInSeconds) {
            this.durationInSeconds = durationInSeconds;
            return this;
        }

        public final void setDurationInSeconds(Integer durationInSeconds) {
            this.durationInSeconds = durationInSeconds;
        }

        public final Collection<String> getRulesPackageArns() {
            return rulesPackageArns;
        }

        @Override
        public final Builder rulesPackageArns(Collection<String> rulesPackageArns) {
            this.rulesPackageArns = AssessmentRulesPackageArnListCopier.copy(rulesPackageArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder rulesPackageArns(String... rulesPackageArns) {
            rulesPackageArns(Arrays.asList(rulesPackageArns));
            return this;
        }

        public final void setRulesPackageArns(Collection<String> rulesPackageArns) {
            this.rulesPackageArns = AssessmentRulesPackageArnListCopier.copy(rulesPackageArns);
        }

        public final Collection<Attribute.Builder> getUserAttributesForFindings() {
            return userAttributesForFindings != null ? userAttributesForFindings.stream().map(Attribute::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder userAttributesForFindings(Collection<Attribute> userAttributesForFindings) {
            this.userAttributesForFindings = UserAttributeListCopier.copy(userAttributesForFindings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder userAttributesForFindings(Attribute... userAttributesForFindings) {
            userAttributesForFindings(Arrays.asList(userAttributesForFindings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder userAttributesForFindings(Consumer<Attribute.Builder>... userAttributesForFindings) {
            userAttributesForFindings(Stream.of(userAttributesForFindings).map(c -> Attribute.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setUserAttributesForFindings(Collection<Attribute.BuilderImpl> userAttributesForFindings) {
            this.userAttributesForFindings = UserAttributeListCopier.copyFromBuilder(userAttributesForFindings);
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        public final Instant getStartedAt() {
            return startedAt;
        }

        @Override
        public final Builder startedAt(Instant startedAt) {
            this.startedAt = startedAt;
            return this;
        }

        public final void setStartedAt(Instant startedAt) {
            this.startedAt = startedAt;
        }

        public final Instant getCompletedAt() {
            return completedAt;
        }

        @Override
        public final Builder completedAt(Instant completedAt) {
            this.completedAt = completedAt;
            return this;
        }

        public final void setCompletedAt(Instant completedAt) {
            this.completedAt = completedAt;
        }

        public final Instant getStateChangedAt() {
            return stateChangedAt;
        }

        @Override
        public final Builder stateChangedAt(Instant stateChangedAt) {
            this.stateChangedAt = stateChangedAt;
            return this;
        }

        public final void setStateChangedAt(Instant stateChangedAt) {
            this.stateChangedAt = stateChangedAt;
        }

        public final Boolean getDataCollected() {
            return dataCollected;
        }

        @Override
        public final Builder dataCollected(Boolean dataCollected) {
            this.dataCollected = dataCollected;
            return this;
        }

        public final void setDataCollected(Boolean dataCollected) {
            this.dataCollected = dataCollected;
        }

        public final Collection<AssessmentRunStateChange.Builder> getStateChanges() {
            return stateChanges != null ? stateChanges.stream().map(AssessmentRunStateChange::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder stateChanges(Collection<AssessmentRunStateChange> stateChanges) {
            this.stateChanges = AssessmentRunStateChangeListCopier.copy(stateChanges);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stateChanges(AssessmentRunStateChange... stateChanges) {
            stateChanges(Arrays.asList(stateChanges));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stateChanges(Consumer<AssessmentRunStateChange.Builder>... stateChanges) {
            stateChanges(Stream.of(stateChanges).map(c -> AssessmentRunStateChange.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setStateChanges(Collection<AssessmentRunStateChange.BuilderImpl> stateChanges) {
            this.stateChanges = AssessmentRunStateChangeListCopier.copyFromBuilder(stateChanges);
        }

        public final Collection<AssessmentRunNotification.Builder> getNotifications() {
            return notifications != null ? notifications.stream().map(AssessmentRunNotification::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder notifications(Collection<AssessmentRunNotification> notifications) {
            this.notifications = AssessmentRunNotificationListCopier.copy(notifications);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notifications(AssessmentRunNotification... notifications) {
            notifications(Arrays.asList(notifications));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notifications(Consumer<AssessmentRunNotification.Builder>... notifications) {
            notifications(Stream.of(notifications).map(c -> AssessmentRunNotification.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setNotifications(Collection<AssessmentRunNotification.BuilderImpl> notifications) {
            this.notifications = AssessmentRunNotificationListCopier.copyFromBuilder(notifications);
        }

        public final Map<String, Integer> getFindingCountsAsStrings() {
            return findingCounts;
        }

        @Override
        public final Builder findingCountsWithStrings(Map<String, Integer> findingCounts) {
            this.findingCounts = AssessmentRunFindingCountsCopier.copy(findingCounts);
            return this;
        }

        @Override
        public final Builder findingCounts(Map<Severity, Integer> findingCounts) {
            this.findingCounts = AssessmentRunFindingCountsCopier.copyEnumToString(findingCounts);
            return this;
        }

        public final void setFindingCountsWithStrings(Map<String, Integer> findingCounts) {
            this.findingCounts = AssessmentRunFindingCountsCopier.copy(findingCounts);
        }

        @Override
        public AssessmentRun build() {
            return new AssessmentRun(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
