/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.inspector2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains details about a specific vulnerability Amazon Inspector can detect.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Vulnerability implements SdkPojo, Serializable, ToCopyableBuilder<Vulnerability.Builder, Vulnerability> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(Vulnerability::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<List<String>> CWES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("cwes")
            .getter(getter(Vulnerability::cwes))
            .setter(setter(Builder::cwes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cwes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<CisaData> CISA_DATA_FIELD = SdkField.<CisaData> builder(MarshallingType.SDK_POJO)
            .memberName("cisaData").getter(getter(Vulnerability::cisaData)).setter(setter(Builder::cisaData))
            .constructor(CisaData::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cisaData").build()).build();

    private static final SdkField<String> SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("source")
            .getter(getter(Vulnerability::sourceAsString)).setter(setter(Builder::source))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("source").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(Vulnerability::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<AtigData> ATIG_DATA_FIELD = SdkField.<AtigData> builder(MarshallingType.SDK_POJO)
            .memberName("atigData").getter(getter(Vulnerability::atigData)).setter(setter(Builder::atigData))
            .constructor(AtigData::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("atigData").build()).build();

    private static final SdkField<String> VENDOR_SEVERITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("vendorSeverity").getter(getter(Vulnerability::vendorSeverity)).setter(setter(Builder::vendorSeverity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("vendorSeverity").build()).build();

    private static final SdkField<Cvss4> CVSS4_FIELD = SdkField.<Cvss4> builder(MarshallingType.SDK_POJO).memberName("cvss4")
            .getter(getter(Vulnerability::cvss4)).setter(setter(Builder::cvss4)).constructor(Cvss4::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cvss4").build()).build();

    private static final SdkField<Cvss3> CVSS3_FIELD = SdkField.<Cvss3> builder(MarshallingType.SDK_POJO).memberName("cvss3")
            .getter(getter(Vulnerability::cvss3)).setter(setter(Builder::cvss3)).constructor(Cvss3::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cvss3").build()).build();

    private static final SdkField<List<String>> RELATED_VULNERABILITIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("relatedVulnerabilities")
            .getter(getter(Vulnerability::relatedVulnerabilities))
            .setter(setter(Builder::relatedVulnerabilities))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("relatedVulnerabilities").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Cvss2> CVSS2_FIELD = SdkField.<Cvss2> builder(MarshallingType.SDK_POJO).memberName("cvss2")
            .getter(getter(Vulnerability::cvss2)).setter(setter(Builder::cvss2)).constructor(Cvss2::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cvss2").build()).build();

    private static final SdkField<Instant> VENDOR_CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("vendorCreatedAt").getter(getter(Vulnerability::vendorCreatedAt))
            .setter(setter(Builder::vendorCreatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("vendorCreatedAt").build()).build();

    private static final SdkField<Instant> VENDOR_UPDATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("vendorUpdatedAt").getter(getter(Vulnerability::vendorUpdatedAt))
            .setter(setter(Builder::vendorUpdatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("vendorUpdatedAt").build()).build();

    private static final SdkField<String> SOURCE_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sourceUrl").getter(getter(Vulnerability::sourceUrl)).setter(setter(Builder::sourceUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceUrl").build()).build();

    private static final SdkField<List<String>> REFERENCE_URLS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("referenceUrls")
            .getter(getter(Vulnerability::referenceUrls))
            .setter(setter(Builder::referenceUrls))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("referenceUrls").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ExploitObserved> EXPLOIT_OBSERVED_FIELD = SdkField
            .<ExploitObserved> builder(MarshallingType.SDK_POJO).memberName("exploitObserved")
            .getter(getter(Vulnerability::exploitObserved)).setter(setter(Builder::exploitObserved))
            .constructor(ExploitObserved::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("exploitObserved").build()).build();

    private static final SdkField<List<String>> DETECTION_PLATFORMS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("detectionPlatforms")
            .getter(getter(Vulnerability::detectionPlatforms))
            .setter(setter(Builder::detectionPlatforms))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("detectionPlatforms").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Epss> EPSS_FIELD = SdkField.<Epss> builder(MarshallingType.SDK_POJO).memberName("epss")
            .getter(getter(Vulnerability::epss)).setter(setter(Builder::epss)).constructor(Epss::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("epss").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, CWES_FIELD,
            CISA_DATA_FIELD, SOURCE_FIELD, DESCRIPTION_FIELD, ATIG_DATA_FIELD, VENDOR_SEVERITY_FIELD, CVSS4_FIELD, CVSS3_FIELD,
            RELATED_VULNERABILITIES_FIELD, CVSS2_FIELD, VENDOR_CREATED_AT_FIELD, VENDOR_UPDATED_AT_FIELD, SOURCE_URL_FIELD,
            REFERENCE_URLS_FIELD, EXPLOIT_OBSERVED_FIELD, DETECTION_PLATFORMS_FIELD, EPSS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String id;

    private final List<String> cwes;

    private final CisaData cisaData;

    private final String source;

    private final String description;

    private final AtigData atigData;

    private final String vendorSeverity;

    private final Cvss4 cvss4;

    private final Cvss3 cvss3;

    private final List<String> relatedVulnerabilities;

    private final Cvss2 cvss2;

    private final Instant vendorCreatedAt;

    private final Instant vendorUpdatedAt;

    private final String sourceUrl;

    private final List<String> referenceUrls;

    private final ExploitObserved exploitObserved;

    private final List<String> detectionPlatforms;

    private final Epss epss;

    private Vulnerability(BuilderImpl builder) {
        this.id = builder.id;
        this.cwes = builder.cwes;
        this.cisaData = builder.cisaData;
        this.source = builder.source;
        this.description = builder.description;
        this.atigData = builder.atigData;
        this.vendorSeverity = builder.vendorSeverity;
        this.cvss4 = builder.cvss4;
        this.cvss3 = builder.cvss3;
        this.relatedVulnerabilities = builder.relatedVulnerabilities;
        this.cvss2 = builder.cvss2;
        this.vendorCreatedAt = builder.vendorCreatedAt;
        this.vendorUpdatedAt = builder.vendorUpdatedAt;
        this.sourceUrl = builder.sourceUrl;
        this.referenceUrls = builder.referenceUrls;
        this.exploitObserved = builder.exploitObserved;
        this.detectionPlatforms = builder.detectionPlatforms;
        this.epss = builder.epss;
    }

    /**
     * <p>
     * The ID for the specific vulnerability.
     * </p>
     * 
     * @return The ID for the specific vulnerability.
     */
    public final String id() {
        return id;
    }

    /**
     * For responses, this returns true if the service returned a value for the Cwes property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasCwes() {
        return cwes != null && !(cwes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Common Weakness Enumeration (CWE) associated with the vulnerability.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCwes} method.
     * </p>
     * 
     * @return The Common Weakness Enumeration (CWE) associated with the vulnerability.
     */
    public final List<String> cwes() {
        return cwes;
    }

    /**
     * <p>
     * An object that contains the Cybersecurity and Infrastructure Security Agency (CISA) details for the
     * vulnerability.
     * </p>
     * 
     * @return An object that contains the Cybersecurity and Infrastructure Security Agency (CISA) details for the
     *         vulnerability.
     */
    public final CisaData cisaData() {
        return cisaData;
    }

    /**
     * <p>
     * The source of the vulnerability information. Possible results are <code>RHEL</code>, <code>AMAZON_CVE</code>,
     * <code>DEBIAN</code> or <code>NVD</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #source} will
     * return {@link VulnerabilitySource#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #sourceAsString}.
     * </p>
     * 
     * @return The source of the vulnerability information. Possible results are <code>RHEL</code>,
     *         <code>AMAZON_CVE</code>, <code>DEBIAN</code> or <code>NVD</code>.
     * @see VulnerabilitySource
     */
    public final VulnerabilitySource source() {
        return VulnerabilitySource.fromValue(source);
    }

    /**
     * <p>
     * The source of the vulnerability information. Possible results are <code>RHEL</code>, <code>AMAZON_CVE</code>,
     * <code>DEBIAN</code> or <code>NVD</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #source} will
     * return {@link VulnerabilitySource#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #sourceAsString}.
     * </p>
     * 
     * @return The source of the vulnerability information. Possible results are <code>RHEL</code>,
     *         <code>AMAZON_CVE</code>, <code>DEBIAN</code> or <code>NVD</code>.
     * @see VulnerabilitySource
     */
    public final String sourceAsString() {
        return source;
    }

    /**
     * <p>
     * A description of the vulnerability.
     * </p>
     * 
     * @return A description of the vulnerability.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * An object that contains information about the Amazon Web Services Threat Intel Group (ATIG) details for the
     * vulnerability.
     * </p>
     * 
     * @return An object that contains information about the Amazon Web Services Threat Intel Group (ATIG) details for
     *         the vulnerability.
     */
    public final AtigData atigData() {
        return atigData;
    }

    /**
     * <p>
     * The severity assigned by the vendor.
     * </p>
     * 
     * @return The severity assigned by the vendor.
     */
    public final String vendorSeverity() {
        return vendorSeverity;
    }

    /**
     * <p>
     * An object that contains the Common Vulnerability Scoring System (CVSS) Version 4 details for the vulnerability.
     * </p>
     * 
     * @return An object that contains the Common Vulnerability Scoring System (CVSS) Version 4 details for the
     *         vulnerability.
     */
    public final Cvss4 cvss4() {
        return cvss4;
    }

    /**
     * <p>
     * An object that contains the Common Vulnerability Scoring System (CVSS) Version 3 details for the vulnerability.
     * </p>
     * 
     * @return An object that contains the Common Vulnerability Scoring System (CVSS) Version 3 details for the
     *         vulnerability.
     */
    public final Cvss3 cvss3() {
        return cvss3;
    }

    /**
     * For responses, this returns true if the service returned a value for the RelatedVulnerabilities property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasRelatedVulnerabilities() {
        return relatedVulnerabilities != null && !(relatedVulnerabilities instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of related vulnerabilities.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRelatedVulnerabilities} method.
     * </p>
     * 
     * @return A list of related vulnerabilities.
     */
    public final List<String> relatedVulnerabilities() {
        return relatedVulnerabilities;
    }

    /**
     * <p>
     * An object that contains the Common Vulnerability Scoring System (CVSS) Version 2 details for the vulnerability.
     * </p>
     * 
     * @return An object that contains the Common Vulnerability Scoring System (CVSS) Version 2 details for the
     *         vulnerability.
     */
    public final Cvss2 cvss2() {
        return cvss2;
    }

    /**
     * <p>
     * The date and time when the vendor created this vulnerability.
     * </p>
     * 
     * @return The date and time when the vendor created this vulnerability.
     */
    public final Instant vendorCreatedAt() {
        return vendorCreatedAt;
    }

    /**
     * <p>
     * The date and time when the vendor last updated this vulnerability.
     * </p>
     * 
     * @return The date and time when the vendor last updated this vulnerability.
     */
    public final Instant vendorUpdatedAt() {
        return vendorUpdatedAt;
    }

    /**
     * <p>
     * A link to the official source material for this vulnerability.
     * </p>
     * 
     * @return A link to the official source material for this vulnerability.
     */
    public final String sourceUrl() {
        return sourceUrl;
    }

    /**
     * For responses, this returns true if the service returned a value for the ReferenceUrls property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasReferenceUrls() {
        return referenceUrls != null && !(referenceUrls instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Links to various resources with more information on this vulnerability.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasReferenceUrls} method.
     * </p>
     * 
     * @return Links to various resources with more information on this vulnerability.
     */
    public final List<String> referenceUrls() {
        return referenceUrls;
    }

    /**
     * <p>
     * An object that contains details on when the exploit was observed.
     * </p>
     * 
     * @return An object that contains details on when the exploit was observed.
     */
    public final ExploitObserved exploitObserved() {
        return exploitObserved;
    }

    /**
     * For responses, this returns true if the service returned a value for the DetectionPlatforms property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDetectionPlatforms() {
        return detectionPlatforms != null && !(detectionPlatforms instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Platforms that the vulnerability can be detected on.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDetectionPlatforms} method.
     * </p>
     * 
     * @return Platforms that the vulnerability can be detected on.
     */
    public final List<String> detectionPlatforms() {
        return detectionPlatforms;
    }

    /**
     * <p>
     * An object that contains the Exploit Prediction Scoring System (EPSS) score for a vulnerability.
     * </p>
     * 
     * @return An object that contains the Exploit Prediction Scoring System (EPSS) score for a vulnerability.
     */
    public final Epss epss() {
        return epss;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(hasCwes() ? cwes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(cisaData());
        hashCode = 31 * hashCode + Objects.hashCode(sourceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(atigData());
        hashCode = 31 * hashCode + Objects.hashCode(vendorSeverity());
        hashCode = 31 * hashCode + Objects.hashCode(cvss4());
        hashCode = 31 * hashCode + Objects.hashCode(cvss3());
        hashCode = 31 * hashCode + Objects.hashCode(hasRelatedVulnerabilities() ? relatedVulnerabilities() : null);
        hashCode = 31 * hashCode + Objects.hashCode(cvss2());
        hashCode = 31 * hashCode + Objects.hashCode(vendorCreatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(vendorUpdatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(sourceUrl());
        hashCode = 31 * hashCode + Objects.hashCode(hasReferenceUrls() ? referenceUrls() : null);
        hashCode = 31 * hashCode + Objects.hashCode(exploitObserved());
        hashCode = 31 * hashCode + Objects.hashCode(hasDetectionPlatforms() ? detectionPlatforms() : null);
        hashCode = 31 * hashCode + Objects.hashCode(epss());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Vulnerability)) {
            return false;
        }
        Vulnerability other = (Vulnerability) obj;
        return Objects.equals(id(), other.id()) && hasCwes() == other.hasCwes() && Objects.equals(cwes(), other.cwes())
                && Objects.equals(cisaData(), other.cisaData()) && Objects.equals(sourceAsString(), other.sourceAsString())
                && Objects.equals(description(), other.description()) && Objects.equals(atigData(), other.atigData())
                && Objects.equals(vendorSeverity(), other.vendorSeverity()) && Objects.equals(cvss4(), other.cvss4())
                && Objects.equals(cvss3(), other.cvss3()) && hasRelatedVulnerabilities() == other.hasRelatedVulnerabilities()
                && Objects.equals(relatedVulnerabilities(), other.relatedVulnerabilities())
                && Objects.equals(cvss2(), other.cvss2()) && Objects.equals(vendorCreatedAt(), other.vendorCreatedAt())
                && Objects.equals(vendorUpdatedAt(), other.vendorUpdatedAt()) && Objects.equals(sourceUrl(), other.sourceUrl())
                && hasReferenceUrls() == other.hasReferenceUrls() && Objects.equals(referenceUrls(), other.referenceUrls())
                && Objects.equals(exploitObserved(), other.exploitObserved())
                && hasDetectionPlatforms() == other.hasDetectionPlatforms()
                && Objects.equals(detectionPlatforms(), other.detectionPlatforms()) && Objects.equals(epss(), other.epss());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Vulnerability").add("Id", id()).add("Cwes", hasCwes() ? cwes() : null)
                .add("CisaData", cisaData()).add("Source", sourceAsString()).add("Description", description())
                .add("AtigData", atigData()).add("VendorSeverity", vendorSeverity()).add("Cvss4", cvss4()).add("Cvss3", cvss3())
                .add("RelatedVulnerabilities", hasRelatedVulnerabilities() ? relatedVulnerabilities() : null)
                .add("Cvss2", cvss2()).add("VendorCreatedAt", vendorCreatedAt()).add("VendorUpdatedAt", vendorUpdatedAt())
                .add("SourceUrl", sourceUrl()).add("ReferenceUrls", hasReferenceUrls() ? referenceUrls() : null)
                .add("ExploitObserved", exploitObserved())
                .add("DetectionPlatforms", hasDetectionPlatforms() ? detectionPlatforms() : null).add("Epss", epss()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "cwes":
            return Optional.ofNullable(clazz.cast(cwes()));
        case "cisaData":
            return Optional.ofNullable(clazz.cast(cisaData()));
        case "source":
            return Optional.ofNullable(clazz.cast(sourceAsString()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "atigData":
            return Optional.ofNullable(clazz.cast(atigData()));
        case "vendorSeverity":
            return Optional.ofNullable(clazz.cast(vendorSeverity()));
        case "cvss4":
            return Optional.ofNullable(clazz.cast(cvss4()));
        case "cvss3":
            return Optional.ofNullable(clazz.cast(cvss3()));
        case "relatedVulnerabilities":
            return Optional.ofNullable(clazz.cast(relatedVulnerabilities()));
        case "cvss2":
            return Optional.ofNullable(clazz.cast(cvss2()));
        case "vendorCreatedAt":
            return Optional.ofNullable(clazz.cast(vendorCreatedAt()));
        case "vendorUpdatedAt":
            return Optional.ofNullable(clazz.cast(vendorUpdatedAt()));
        case "sourceUrl":
            return Optional.ofNullable(clazz.cast(sourceUrl()));
        case "referenceUrls":
            return Optional.ofNullable(clazz.cast(referenceUrls()));
        case "exploitObserved":
            return Optional.ofNullable(clazz.cast(exploitObserved()));
        case "detectionPlatforms":
            return Optional.ofNullable(clazz.cast(detectionPlatforms()));
        case "epss":
            return Optional.ofNullable(clazz.cast(epss()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("id", ID_FIELD);
        map.put("cwes", CWES_FIELD);
        map.put("cisaData", CISA_DATA_FIELD);
        map.put("source", SOURCE_FIELD);
        map.put("description", DESCRIPTION_FIELD);
        map.put("atigData", ATIG_DATA_FIELD);
        map.put("vendorSeverity", VENDOR_SEVERITY_FIELD);
        map.put("cvss4", CVSS4_FIELD);
        map.put("cvss3", CVSS3_FIELD);
        map.put("relatedVulnerabilities", RELATED_VULNERABILITIES_FIELD);
        map.put("cvss2", CVSS2_FIELD);
        map.put("vendorCreatedAt", VENDOR_CREATED_AT_FIELD);
        map.put("vendorUpdatedAt", VENDOR_UPDATED_AT_FIELD);
        map.put("sourceUrl", SOURCE_URL_FIELD);
        map.put("referenceUrls", REFERENCE_URLS_FIELD);
        map.put("exploitObserved", EXPLOIT_OBSERVED_FIELD);
        map.put("detectionPlatforms", DETECTION_PLATFORMS_FIELD);
        map.put("epss", EPSS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Vulnerability, T> g) {
        return obj -> g.apply((Vulnerability) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Vulnerability> {
        /**
         * <p>
         * The ID for the specific vulnerability.
         * </p>
         * 
         * @param id
         *        The ID for the specific vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The Common Weakness Enumeration (CWE) associated with the vulnerability.
         * </p>
         * 
         * @param cwes
         *        The Common Weakness Enumeration (CWE) associated with the vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cwes(Collection<String> cwes);

        /**
         * <p>
         * The Common Weakness Enumeration (CWE) associated with the vulnerability.
         * </p>
         * 
         * @param cwes
         *        The Common Weakness Enumeration (CWE) associated with the vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cwes(String... cwes);

        /**
         * <p>
         * An object that contains the Cybersecurity and Infrastructure Security Agency (CISA) details for the
         * vulnerability.
         * </p>
         * 
         * @param cisaData
         *        An object that contains the Cybersecurity and Infrastructure Security Agency (CISA) details for the
         *        vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cisaData(CisaData cisaData);

        /**
         * <p>
         * An object that contains the Cybersecurity and Infrastructure Security Agency (CISA) details for the
         * vulnerability.
         * </p>
         * This is a convenience method that creates an instance of the {@link CisaData.Builder} avoiding the need to
         * create one manually via {@link CisaData#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CisaData.Builder#build()} is called immediately and its result is
         * passed to {@link #cisaData(CisaData)}.
         * 
         * @param cisaData
         *        a consumer that will call methods on {@link CisaData.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cisaData(CisaData)
         */
        default Builder cisaData(Consumer<CisaData.Builder> cisaData) {
            return cisaData(CisaData.builder().applyMutation(cisaData).build());
        }

        /**
         * <p>
         * The source of the vulnerability information. Possible results are <code>RHEL</code>, <code>AMAZON_CVE</code>,
         * <code>DEBIAN</code> or <code>NVD</code>.
         * </p>
         * 
         * @param source
         *        The source of the vulnerability information. Possible results are <code>RHEL</code>,
         *        <code>AMAZON_CVE</code>, <code>DEBIAN</code> or <code>NVD</code>.
         * @see VulnerabilitySource
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VulnerabilitySource
         */
        Builder source(String source);

        /**
         * <p>
         * The source of the vulnerability information. Possible results are <code>RHEL</code>, <code>AMAZON_CVE</code>,
         * <code>DEBIAN</code> or <code>NVD</code>.
         * </p>
         * 
         * @param source
         *        The source of the vulnerability information. Possible results are <code>RHEL</code>,
         *        <code>AMAZON_CVE</code>, <code>DEBIAN</code> or <code>NVD</code>.
         * @see VulnerabilitySource
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VulnerabilitySource
         */
        Builder source(VulnerabilitySource source);

        /**
         * <p>
         * A description of the vulnerability.
         * </p>
         * 
         * @param description
         *        A description of the vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * An object that contains information about the Amazon Web Services Threat Intel Group (ATIG) details for the
         * vulnerability.
         * </p>
         * 
         * @param atigData
         *        An object that contains information about the Amazon Web Services Threat Intel Group (ATIG) details
         *        for the vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder atigData(AtigData atigData);

        /**
         * <p>
         * An object that contains information about the Amazon Web Services Threat Intel Group (ATIG) details for the
         * vulnerability.
         * </p>
         * This is a convenience method that creates an instance of the {@link AtigData.Builder} avoiding the need to
         * create one manually via {@link AtigData#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AtigData.Builder#build()} is called immediately and its result is
         * passed to {@link #atigData(AtigData)}.
         * 
         * @param atigData
         *        a consumer that will call methods on {@link AtigData.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #atigData(AtigData)
         */
        default Builder atigData(Consumer<AtigData.Builder> atigData) {
            return atigData(AtigData.builder().applyMutation(atigData).build());
        }

        /**
         * <p>
         * The severity assigned by the vendor.
         * </p>
         * 
         * @param vendorSeverity
         *        The severity assigned by the vendor.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vendorSeverity(String vendorSeverity);

        /**
         * <p>
         * An object that contains the Common Vulnerability Scoring System (CVSS) Version 4 details for the
         * vulnerability.
         * </p>
         * 
         * @param cvss4
         *        An object that contains the Common Vulnerability Scoring System (CVSS) Version 4 details for the
         *        vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cvss4(Cvss4 cvss4);

        /**
         * <p>
         * An object that contains the Common Vulnerability Scoring System (CVSS) Version 4 details for the
         * vulnerability.
         * </p>
         * This is a convenience method that creates an instance of the {@link Cvss4.Builder} avoiding the need to
         * create one manually via {@link Cvss4#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Cvss4.Builder#build()} is called immediately and its result is
         * passed to {@link #cvss4(Cvss4)}.
         * 
         * @param cvss4
         *        a consumer that will call methods on {@link Cvss4.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cvss4(Cvss4)
         */
        default Builder cvss4(Consumer<Cvss4.Builder> cvss4) {
            return cvss4(Cvss4.builder().applyMutation(cvss4).build());
        }

        /**
         * <p>
         * An object that contains the Common Vulnerability Scoring System (CVSS) Version 3 details for the
         * vulnerability.
         * </p>
         * 
         * @param cvss3
         *        An object that contains the Common Vulnerability Scoring System (CVSS) Version 3 details for the
         *        vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cvss3(Cvss3 cvss3);

        /**
         * <p>
         * An object that contains the Common Vulnerability Scoring System (CVSS) Version 3 details for the
         * vulnerability.
         * </p>
         * This is a convenience method that creates an instance of the {@link Cvss3.Builder} avoiding the need to
         * create one manually via {@link Cvss3#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Cvss3.Builder#build()} is called immediately and its result is
         * passed to {@link #cvss3(Cvss3)}.
         * 
         * @param cvss3
         *        a consumer that will call methods on {@link Cvss3.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cvss3(Cvss3)
         */
        default Builder cvss3(Consumer<Cvss3.Builder> cvss3) {
            return cvss3(Cvss3.builder().applyMutation(cvss3).build());
        }

        /**
         * <p>
         * A list of related vulnerabilities.
         * </p>
         * 
         * @param relatedVulnerabilities
         *        A list of related vulnerabilities.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedVulnerabilities(Collection<String> relatedVulnerabilities);

        /**
         * <p>
         * A list of related vulnerabilities.
         * </p>
         * 
         * @param relatedVulnerabilities
         *        A list of related vulnerabilities.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedVulnerabilities(String... relatedVulnerabilities);

        /**
         * <p>
         * An object that contains the Common Vulnerability Scoring System (CVSS) Version 2 details for the
         * vulnerability.
         * </p>
         * 
         * @param cvss2
         *        An object that contains the Common Vulnerability Scoring System (CVSS) Version 2 details for the
         *        vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cvss2(Cvss2 cvss2);

        /**
         * <p>
         * An object that contains the Common Vulnerability Scoring System (CVSS) Version 2 details for the
         * vulnerability.
         * </p>
         * This is a convenience method that creates an instance of the {@link Cvss2.Builder} avoiding the need to
         * create one manually via {@link Cvss2#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Cvss2.Builder#build()} is called immediately and its result is
         * passed to {@link #cvss2(Cvss2)}.
         * 
         * @param cvss2
         *        a consumer that will call methods on {@link Cvss2.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cvss2(Cvss2)
         */
        default Builder cvss2(Consumer<Cvss2.Builder> cvss2) {
            return cvss2(Cvss2.builder().applyMutation(cvss2).build());
        }

        /**
         * <p>
         * The date and time when the vendor created this vulnerability.
         * </p>
         * 
         * @param vendorCreatedAt
         *        The date and time when the vendor created this vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vendorCreatedAt(Instant vendorCreatedAt);

        /**
         * <p>
         * The date and time when the vendor last updated this vulnerability.
         * </p>
         * 
         * @param vendorUpdatedAt
         *        The date and time when the vendor last updated this vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vendorUpdatedAt(Instant vendorUpdatedAt);

        /**
         * <p>
         * A link to the official source material for this vulnerability.
         * </p>
         * 
         * @param sourceUrl
         *        A link to the official source material for this vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceUrl(String sourceUrl);

        /**
         * <p>
         * Links to various resources with more information on this vulnerability.
         * </p>
         * 
         * @param referenceUrls
         *        Links to various resources with more information on this vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder referenceUrls(Collection<String> referenceUrls);

        /**
         * <p>
         * Links to various resources with more information on this vulnerability.
         * </p>
         * 
         * @param referenceUrls
         *        Links to various resources with more information on this vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder referenceUrls(String... referenceUrls);

        /**
         * <p>
         * An object that contains details on when the exploit was observed.
         * </p>
         * 
         * @param exploitObserved
         *        An object that contains details on when the exploit was observed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exploitObserved(ExploitObserved exploitObserved);

        /**
         * <p>
         * An object that contains details on when the exploit was observed.
         * </p>
         * This is a convenience method that creates an instance of the {@link ExploitObserved.Builder} avoiding the
         * need to create one manually via {@link ExploitObserved#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ExploitObserved.Builder#build()} is called immediately and its
         * result is passed to {@link #exploitObserved(ExploitObserved)}.
         * 
         * @param exploitObserved
         *        a consumer that will call methods on {@link ExploitObserved.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #exploitObserved(ExploitObserved)
         */
        default Builder exploitObserved(Consumer<ExploitObserved.Builder> exploitObserved) {
            return exploitObserved(ExploitObserved.builder().applyMutation(exploitObserved).build());
        }

        /**
         * <p>
         * Platforms that the vulnerability can be detected on.
         * </p>
         * 
         * @param detectionPlatforms
         *        Platforms that the vulnerability can be detected on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder detectionPlatforms(Collection<String> detectionPlatforms);

        /**
         * <p>
         * Platforms that the vulnerability can be detected on.
         * </p>
         * 
         * @param detectionPlatforms
         *        Platforms that the vulnerability can be detected on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder detectionPlatforms(String... detectionPlatforms);

        /**
         * <p>
         * An object that contains the Exploit Prediction Scoring System (EPSS) score for a vulnerability.
         * </p>
         * 
         * @param epss
         *        An object that contains the Exploit Prediction Scoring System (EPSS) score for a vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder epss(Epss epss);

        /**
         * <p>
         * An object that contains the Exploit Prediction Scoring System (EPSS) score for a vulnerability.
         * </p>
         * This is a convenience method that creates an instance of the {@link Epss.Builder} avoiding the need to create
         * one manually via {@link Epss#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Epss.Builder#build()} is called immediately and its result is
         * passed to {@link #epss(Epss)}.
         * 
         * @param epss
         *        a consumer that will call methods on {@link Epss.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #epss(Epss)
         */
        default Builder epss(Consumer<Epss.Builder> epss) {
            return epss(Epss.builder().applyMutation(epss).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private List<String> cwes = DefaultSdkAutoConstructList.getInstance();

        private CisaData cisaData;

        private String source;

        private String description;

        private AtigData atigData;

        private String vendorSeverity;

        private Cvss4 cvss4;

        private Cvss3 cvss3;

        private List<String> relatedVulnerabilities = DefaultSdkAutoConstructList.getInstance();

        private Cvss2 cvss2;

        private Instant vendorCreatedAt;

        private Instant vendorUpdatedAt;

        private String sourceUrl;

        private List<String> referenceUrls = DefaultSdkAutoConstructList.getInstance();

        private ExploitObserved exploitObserved;

        private List<String> detectionPlatforms = DefaultSdkAutoConstructList.getInstance();

        private Epss epss;

        private BuilderImpl() {
        }

        private BuilderImpl(Vulnerability model) {
            id(model.id);
            cwes(model.cwes);
            cisaData(model.cisaData);
            source(model.source);
            description(model.description);
            atigData(model.atigData);
            vendorSeverity(model.vendorSeverity);
            cvss4(model.cvss4);
            cvss3(model.cvss3);
            relatedVulnerabilities(model.relatedVulnerabilities);
            cvss2(model.cvss2);
            vendorCreatedAt(model.vendorCreatedAt);
            vendorUpdatedAt(model.vendorUpdatedAt);
            sourceUrl(model.sourceUrl);
            referenceUrls(model.referenceUrls);
            exploitObserved(model.exploitObserved);
            detectionPlatforms(model.detectionPlatforms);
            epss(model.epss);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final Collection<String> getCwes() {
            if (cwes instanceof SdkAutoConstructList) {
                return null;
            }
            return cwes;
        }

        public final void setCwes(Collection<String> cwes) {
            this.cwes = CwesCopier.copy(cwes);
        }

        @Override
        public final Builder cwes(Collection<String> cwes) {
            this.cwes = CwesCopier.copy(cwes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder cwes(String... cwes) {
            cwes(Arrays.asList(cwes));
            return this;
        }

        public final CisaData.Builder getCisaData() {
            return cisaData != null ? cisaData.toBuilder() : null;
        }

        public final void setCisaData(CisaData.BuilderImpl cisaData) {
            this.cisaData = cisaData != null ? cisaData.build() : null;
        }

        @Override
        public final Builder cisaData(CisaData cisaData) {
            this.cisaData = cisaData;
            return this;
        }

        public final String getSource() {
            return source;
        }

        public final void setSource(String source) {
            this.source = source;
        }

        @Override
        public final Builder source(String source) {
            this.source = source;
            return this;
        }

        @Override
        public final Builder source(VulnerabilitySource source) {
            this.source(source == null ? null : source.toString());
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final AtigData.Builder getAtigData() {
            return atigData != null ? atigData.toBuilder() : null;
        }

        public final void setAtigData(AtigData.BuilderImpl atigData) {
            this.atigData = atigData != null ? atigData.build() : null;
        }

        @Override
        public final Builder atigData(AtigData atigData) {
            this.atigData = atigData;
            return this;
        }

        public final String getVendorSeverity() {
            return vendorSeverity;
        }

        public final void setVendorSeverity(String vendorSeverity) {
            this.vendorSeverity = vendorSeverity;
        }

        @Override
        public final Builder vendorSeverity(String vendorSeverity) {
            this.vendorSeverity = vendorSeverity;
            return this;
        }

        public final Cvss4.Builder getCvss4() {
            return cvss4 != null ? cvss4.toBuilder() : null;
        }

        public final void setCvss4(Cvss4.BuilderImpl cvss4) {
            this.cvss4 = cvss4 != null ? cvss4.build() : null;
        }

        @Override
        public final Builder cvss4(Cvss4 cvss4) {
            this.cvss4 = cvss4;
            return this;
        }

        public final Cvss3.Builder getCvss3() {
            return cvss3 != null ? cvss3.toBuilder() : null;
        }

        public final void setCvss3(Cvss3.BuilderImpl cvss3) {
            this.cvss3 = cvss3 != null ? cvss3.build() : null;
        }

        @Override
        public final Builder cvss3(Cvss3 cvss3) {
            this.cvss3 = cvss3;
            return this;
        }

        public final Collection<String> getRelatedVulnerabilities() {
            if (relatedVulnerabilities instanceof SdkAutoConstructList) {
                return null;
            }
            return relatedVulnerabilities;
        }

        public final void setRelatedVulnerabilities(Collection<String> relatedVulnerabilities) {
            this.relatedVulnerabilities = RelatedVulnerabilitiesCopier.copy(relatedVulnerabilities);
        }

        @Override
        public final Builder relatedVulnerabilities(Collection<String> relatedVulnerabilities) {
            this.relatedVulnerabilities = RelatedVulnerabilitiesCopier.copy(relatedVulnerabilities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relatedVulnerabilities(String... relatedVulnerabilities) {
            relatedVulnerabilities(Arrays.asList(relatedVulnerabilities));
            return this;
        }

        public final Cvss2.Builder getCvss2() {
            return cvss2 != null ? cvss2.toBuilder() : null;
        }

        public final void setCvss2(Cvss2.BuilderImpl cvss2) {
            this.cvss2 = cvss2 != null ? cvss2.build() : null;
        }

        @Override
        public final Builder cvss2(Cvss2 cvss2) {
            this.cvss2 = cvss2;
            return this;
        }

        public final Instant getVendorCreatedAt() {
            return vendorCreatedAt;
        }

        public final void setVendorCreatedAt(Instant vendorCreatedAt) {
            this.vendorCreatedAt = vendorCreatedAt;
        }

        @Override
        public final Builder vendorCreatedAt(Instant vendorCreatedAt) {
            this.vendorCreatedAt = vendorCreatedAt;
            return this;
        }

        public final Instant getVendorUpdatedAt() {
            return vendorUpdatedAt;
        }

        public final void setVendorUpdatedAt(Instant vendorUpdatedAt) {
            this.vendorUpdatedAt = vendorUpdatedAt;
        }

        @Override
        public final Builder vendorUpdatedAt(Instant vendorUpdatedAt) {
            this.vendorUpdatedAt = vendorUpdatedAt;
            return this;
        }

        public final String getSourceUrl() {
            return sourceUrl;
        }

        public final void setSourceUrl(String sourceUrl) {
            this.sourceUrl = sourceUrl;
        }

        @Override
        public final Builder sourceUrl(String sourceUrl) {
            this.sourceUrl = sourceUrl;
            return this;
        }

        public final Collection<String> getReferenceUrls() {
            if (referenceUrls instanceof SdkAutoConstructList) {
                return null;
            }
            return referenceUrls;
        }

        public final void setReferenceUrls(Collection<String> referenceUrls) {
            this.referenceUrls = VulnerabilityReferenceUrlsCopier.copy(referenceUrls);
        }

        @Override
        public final Builder referenceUrls(Collection<String> referenceUrls) {
            this.referenceUrls = VulnerabilityReferenceUrlsCopier.copy(referenceUrls);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder referenceUrls(String... referenceUrls) {
            referenceUrls(Arrays.asList(referenceUrls));
            return this;
        }

        public final ExploitObserved.Builder getExploitObserved() {
            return exploitObserved != null ? exploitObserved.toBuilder() : null;
        }

        public final void setExploitObserved(ExploitObserved.BuilderImpl exploitObserved) {
            this.exploitObserved = exploitObserved != null ? exploitObserved.build() : null;
        }

        @Override
        public final Builder exploitObserved(ExploitObserved exploitObserved) {
            this.exploitObserved = exploitObserved;
            return this;
        }

        public final Collection<String> getDetectionPlatforms() {
            if (detectionPlatforms instanceof SdkAutoConstructList) {
                return null;
            }
            return detectionPlatforms;
        }

        public final void setDetectionPlatforms(Collection<String> detectionPlatforms) {
            this.detectionPlatforms = DetectionPlatformsCopier.copy(detectionPlatforms);
        }

        @Override
        public final Builder detectionPlatforms(Collection<String> detectionPlatforms) {
            this.detectionPlatforms = DetectionPlatformsCopier.copy(detectionPlatforms);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder detectionPlatforms(String... detectionPlatforms) {
            detectionPlatforms(Arrays.asList(detectionPlatforms));
            return this;
        }

        public final Epss.Builder getEpss() {
            return epss != null ? epss.toBuilder() : null;
        }

        public final void setEpss(Epss.BuilderImpl epss) {
            this.epss = epss != null ? epss.build() : null;
        }

        @Override
        public final Builder epss(Epss epss) {
            this.epss = epss;
            return this;
        }

        @Override
        public Vulnerability build() {
            return new Vulnerability(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
