/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.internetmonitor;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.internetmonitor.model.AccessDeniedException;
import software.amazon.awssdk.services.internetmonitor.model.BadRequestException;
import software.amazon.awssdk.services.internetmonitor.model.ConflictException;
import software.amazon.awssdk.services.internetmonitor.model.CreateMonitorRequest;
import software.amazon.awssdk.services.internetmonitor.model.CreateMonitorResponse;
import software.amazon.awssdk.services.internetmonitor.model.DeleteMonitorRequest;
import software.amazon.awssdk.services.internetmonitor.model.DeleteMonitorResponse;
import software.amazon.awssdk.services.internetmonitor.model.GetHealthEventRequest;
import software.amazon.awssdk.services.internetmonitor.model.GetHealthEventResponse;
import software.amazon.awssdk.services.internetmonitor.model.GetMonitorRequest;
import software.amazon.awssdk.services.internetmonitor.model.GetMonitorResponse;
import software.amazon.awssdk.services.internetmonitor.model.InternalServerErrorException;
import software.amazon.awssdk.services.internetmonitor.model.InternalServerException;
import software.amazon.awssdk.services.internetmonitor.model.InternetMonitorException;
import software.amazon.awssdk.services.internetmonitor.model.LimitExceededException;
import software.amazon.awssdk.services.internetmonitor.model.ListHealthEventsRequest;
import software.amazon.awssdk.services.internetmonitor.model.ListHealthEventsResponse;
import software.amazon.awssdk.services.internetmonitor.model.ListMonitorsRequest;
import software.amazon.awssdk.services.internetmonitor.model.ListMonitorsResponse;
import software.amazon.awssdk.services.internetmonitor.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.internetmonitor.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.internetmonitor.model.NotFoundException;
import software.amazon.awssdk.services.internetmonitor.model.ResourceNotFoundException;
import software.amazon.awssdk.services.internetmonitor.model.TagResourceRequest;
import software.amazon.awssdk.services.internetmonitor.model.TagResourceResponse;
import software.amazon.awssdk.services.internetmonitor.model.ThrottlingException;
import software.amazon.awssdk.services.internetmonitor.model.TooManyRequestsException;
import software.amazon.awssdk.services.internetmonitor.model.UntagResourceRequest;
import software.amazon.awssdk.services.internetmonitor.model.UntagResourceResponse;
import software.amazon.awssdk.services.internetmonitor.model.UpdateMonitorRequest;
import software.amazon.awssdk.services.internetmonitor.model.UpdateMonitorResponse;
import software.amazon.awssdk.services.internetmonitor.model.ValidationException;
import software.amazon.awssdk.services.internetmonitor.paginators.ListHealthEventsIterable;
import software.amazon.awssdk.services.internetmonitor.paginators.ListMonitorsIterable;

/**
 * Service client for accessing Amazon CloudWatch Internet Monitor. This can be created using the static
 * {@link #builder()} method.
 *
 * <p>
 * Amazon CloudWatch Internet Monitor provides visibility into how internet issues impact the performance and
 * availability between your applications hosted on Amazon Web Services and your end users, reducing the time it takes
 * for you to diagnose these issues, from days to minutes. You can explore internet measurements for different time
 * frames and at different geographic granularities, and quickly visualize the impact of issues, and then take action to
 * improve your end users' experience, for example, by switching to other Amazon Web Services services or rerouting
 * traffic to your workload through differentAmazon Web Services Regions.
 * </p>
 * <p>
 * If the issue is caused by the Amazon Web Services network, you'll automatically receive an Amazon Web Services Health
 * Dashboard notification with the steps that Amazon Web Services is taking to mitigate the problem. To support
 * integrating health information for geographies and networks specific to your application, Internet Monitor delivers
 * measurements to CloudWatch Logs and CloudWatch Metrics. Internet Monitor also sends health events to Amazon
 * EventBridge, so you can set up notifications. Internet Monitor monitors internet connectivity for your application
 * through Amazon Virtual Private Clouds (VPCs), Amazon CloudFront distributions, and Amazon WorkSpaces directories.
 * </p>
 * <p>
 * To use Internet Monitor, you create a <i>monitor</i> and add resources to it, Virtual Private Clouds (VPCs), Amazon
 * CloudFront distributions, or WorkSpaces directories that show where your application's internet traffic is. Internet
 * Monitor then provides internet measurements from Amazon Web Services that are specific to the locations and networks
 * that communicate with your application. For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-InternetMonitor.html"> Using Amazon
 * CloudWatch Internet Monitor</a> in the Amazon CloudWatch User Guide.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface InternetMonitorClient extends AwsClient {
    String SERVICE_NAME = "internetmonitor";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "internetmonitor";

    /**
     * <p>
     * Creates a monitor in Amazon CloudWatch Internet Monitor. A monitor is built based on information from the
     * application resources that you add: Virtual Private Clouds (VPCs), Amazon CloudFront distributions, and
     * WorkSpaces directories.
     * </p>
     * <p>
     * After you create a monitor, you can view the internet performance for your application, scoped to a location, as
     * well as any health events that are impairing traffic. Internet Monitor can also diagnose whether the impairment
     * is on the Amazon Web Services network or is an issue with an internet service provider (ISP).
     * </p>
     *
     * @param createMonitorRequest
     * @return Result of the CreateMonitor operation returned by the service.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ConflictException
     *         The requested resource is in use.
     * @throws LimitExceededException
     *         The request exceeded a service quota.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.CreateMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/CreateMonitor" target="_top">AWS
     *      API Documentation</a>
     */
    default CreateMonitorResponse createMonitor(CreateMonitorRequest createMonitorRequest) throws InternalServerException,
            AccessDeniedException, ThrottlingException, ConflictException, LimitExceededException, ValidationException,
            AwsServiceException, SdkClientException, InternetMonitorException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a monitor in Amazon CloudWatch Internet Monitor. A monitor is built based on information from the
     * application resources that you add: Virtual Private Clouds (VPCs), Amazon CloudFront distributions, and
     * WorkSpaces directories.
     * </p>
     * <p>
     * After you create a monitor, you can view the internet performance for your application, scoped to a location, as
     * well as any health events that are impairing traffic. Internet Monitor can also diagnose whether the impairment
     * is on the Amazon Web Services network or is an issue with an internet service provider (ISP).
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateMonitorRequest.Builder} avoiding the need to
     * create one manually via {@link CreateMonitorRequest#builder()}
     * </p>
     *
     * @param createMonitorRequest
     *        A {@link Consumer} that will call methods on {@link CreateMonitorInput.Builder} to create a request.
     * @return Result of the CreateMonitor operation returned by the service.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ConflictException
     *         The requested resource is in use.
     * @throws LimitExceededException
     *         The request exceeded a service quota.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.CreateMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/CreateMonitor" target="_top">AWS
     *      API Documentation</a>
     */
    default CreateMonitorResponse createMonitor(Consumer<CreateMonitorRequest.Builder> createMonitorRequest)
            throws InternalServerException, AccessDeniedException, ThrottlingException, ConflictException,
            LimitExceededException, ValidationException, AwsServiceException, SdkClientException, InternetMonitorException {
        return createMonitor(CreateMonitorRequest.builder().applyMutation(createMonitorRequest).build());
    }

    /**
     * <p>
     * Deletes a monitor in Amazon CloudWatch Internet Monitor.
     * </p>
     *
     * @param deleteMonitorRequest
     * @return Result of the DeleteMonitor operation returned by the service.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.DeleteMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/DeleteMonitor" target="_top">AWS
     *      API Documentation</a>
     */
    default DeleteMonitorResponse deleteMonitor(DeleteMonitorRequest deleteMonitorRequest) throws InternalServerException,
            AccessDeniedException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            InternetMonitorException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a monitor in Amazon CloudWatch Internet Monitor.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteMonitorRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteMonitorRequest#builder()}
     * </p>
     *
     * @param deleteMonitorRequest
     *        A {@link Consumer} that will call methods on {@link DeleteMonitorInput.Builder} to create a request.
     * @return Result of the DeleteMonitor operation returned by the service.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.DeleteMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/DeleteMonitor" target="_top">AWS
     *      API Documentation</a>
     */
    default DeleteMonitorResponse deleteMonitor(Consumer<DeleteMonitorRequest.Builder> deleteMonitorRequest)
            throws InternalServerException, AccessDeniedException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, InternetMonitorException {
        return deleteMonitor(DeleteMonitorRequest.builder().applyMutation(deleteMonitorRequest).build());
    }

    /**
     * <p>
     * Gets information the Amazon CloudWatch Internet Monitor has created and stored about a health event for a
     * specified monitor. This information includes the impacted locations, and all of the information related to the
     * event by location.
     * </p>
     * <p>
     * The information returned includes the performance, availability, and round-trip time impact, information about
     * the network providers, the event type, and so on.
     * </p>
     * <p>
     * Information rolled up at the global traffic level is also returned, including the impact type and total traffic
     * impact.
     * </p>
     *
     * @param getHealthEventRequest
     * @return Result of the GetHealthEvent operation returned by the service.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.GetHealthEvent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/GetHealthEvent"
     *      target="_top">AWS API Documentation</a>
     */
    default GetHealthEventResponse getHealthEvent(GetHealthEventRequest getHealthEventRequest) throws InternalServerException,
            AccessDeniedException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            InternetMonitorException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets information the Amazon CloudWatch Internet Monitor has created and stored about a health event for a
     * specified monitor. This information includes the impacted locations, and all of the information related to the
     * event by location.
     * </p>
     * <p>
     * The information returned includes the performance, availability, and round-trip time impact, information about
     * the network providers, the event type, and so on.
     * </p>
     * <p>
     * Information rolled up at the global traffic level is also returned, including the impact type and total traffic
     * impact.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetHealthEventRequest.Builder} avoiding the need to
     * create one manually via {@link GetHealthEventRequest#builder()}
     * </p>
     *
     * @param getHealthEventRequest
     *        A {@link Consumer} that will call methods on {@link GetHealthEventInput.Builder} to create a request.
     * @return Result of the GetHealthEvent operation returned by the service.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.GetHealthEvent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/GetHealthEvent"
     *      target="_top">AWS API Documentation</a>
     */
    default GetHealthEventResponse getHealthEvent(Consumer<GetHealthEventRequest.Builder> getHealthEventRequest)
            throws InternalServerException, AccessDeniedException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, InternetMonitorException {
        return getHealthEvent(GetHealthEventRequest.builder().applyMutation(getHealthEventRequest).build());
    }

    /**
     * <p>
     * Gets information about a monitor in Amazon CloudWatch Internet Monitor based on a monitor name. The information
     * returned includes the Amazon Resource Name (ARN), create time, modified time, resources included in the monitor,
     * and status information.
     * </p>
     *
     * @param getMonitorRequest
     * @return Result of the GetMonitor operation returned by the service.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.GetMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/GetMonitor" target="_top">AWS
     *      API Documentation</a>
     */
    default GetMonitorResponse getMonitor(GetMonitorRequest getMonitorRequest) throws InternalServerException,
            AccessDeniedException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            InternetMonitorException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets information about a monitor in Amazon CloudWatch Internet Monitor based on a monitor name. The information
     * returned includes the Amazon Resource Name (ARN), create time, modified time, resources included in the monitor,
     * and status information.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetMonitorRequest.Builder} avoiding the need to
     * create one manually via {@link GetMonitorRequest#builder()}
     * </p>
     *
     * @param getMonitorRequest
     *        A {@link Consumer} that will call methods on {@link GetMonitorInput.Builder} to create a request.
     * @return Result of the GetMonitor operation returned by the service.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.GetMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/GetMonitor" target="_top">AWS
     *      API Documentation</a>
     */
    default GetMonitorResponse getMonitor(Consumer<GetMonitorRequest.Builder> getMonitorRequest) throws InternalServerException,
            AccessDeniedException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            InternetMonitorException {
        return getMonitor(GetMonitorRequest.builder().applyMutation(getMonitorRequest).build());
    }

    /**
     * <p>
     * Lists all health events for a monitor in Amazon CloudWatch Internet Monitor. Returns all information for health
     * events including the client location information the network cause and status, event start and end time,
     * percentage of total traffic impacted, and status.
     * </p>
     * <note>
     * <p>
     * Health events that have start times during the time frame that is requested are not included in the list of
     * health events.
     * </p>
     * </note>
     *
     * @param listHealthEventsRequest
     * @return Result of the ListHealthEvents operation returned by the service.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.ListHealthEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/ListHealthEvents"
     *      target="_top">AWS API Documentation</a>
     */
    default ListHealthEventsResponse listHealthEvents(ListHealthEventsRequest listHealthEventsRequest)
            throws InternalServerException, AccessDeniedException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, InternetMonitorException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all health events for a monitor in Amazon CloudWatch Internet Monitor. Returns all information for health
     * events including the client location information the network cause and status, event start and end time,
     * percentage of total traffic impacted, and status.
     * </p>
     * <note>
     * <p>
     * Health events that have start times during the time frame that is requested are not included in the list of
     * health events.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListHealthEventsRequest.Builder} avoiding the need
     * to create one manually via {@link ListHealthEventsRequest#builder()}
     * </p>
     *
     * @param listHealthEventsRequest
     *        A {@link Consumer} that will call methods on {@link ListHealthEventsInput.Builder} to create a request.
     * @return Result of the ListHealthEvents operation returned by the service.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.ListHealthEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/ListHealthEvents"
     *      target="_top">AWS API Documentation</a>
     */
    default ListHealthEventsResponse listHealthEvents(Consumer<ListHealthEventsRequest.Builder> listHealthEventsRequest)
            throws InternalServerException, AccessDeniedException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, InternetMonitorException {
        return listHealthEvents(ListHealthEventsRequest.builder().applyMutation(listHealthEventsRequest).build());
    }

    /**
     * <p>
     * Lists all health events for a monitor in Amazon CloudWatch Internet Monitor. Returns all information for health
     * events including the client location information the network cause and status, event start and end time,
     * percentage of total traffic impacted, and status.
     * </p>
     * <note>
     * <p>
     * Health events that have start times during the time frame that is requested are not included in the list of
     * health events.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #listHealthEvents(software.amazon.awssdk.services.internetmonitor.model.ListHealthEventsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.internetmonitor.paginators.ListHealthEventsIterable responses = client.listHealthEventsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.internetmonitor.paginators.ListHealthEventsIterable responses = client
     *             .listHealthEventsPaginator(request);
     *     for (software.amazon.awssdk.services.internetmonitor.model.ListHealthEventsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.internetmonitor.paginators.ListHealthEventsIterable responses = client.listHealthEventsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listHealthEvents(software.amazon.awssdk.services.internetmonitor.model.ListHealthEventsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listHealthEventsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.ListHealthEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/ListHealthEvents"
     *      target="_top">AWS API Documentation</a>
     */
    default ListHealthEventsIterable listHealthEventsPaginator(ListHealthEventsRequest listHealthEventsRequest)
            throws InternalServerException, AccessDeniedException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, InternetMonitorException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all health events for a monitor in Amazon CloudWatch Internet Monitor. Returns all information for health
     * events including the client location information the network cause and status, event start and end time,
     * percentage of total traffic impacted, and status.
     * </p>
     * <note>
     * <p>
     * Health events that have start times during the time frame that is requested are not included in the list of
     * health events.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #listHealthEvents(software.amazon.awssdk.services.internetmonitor.model.ListHealthEventsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.internetmonitor.paginators.ListHealthEventsIterable responses = client.listHealthEventsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.internetmonitor.paginators.ListHealthEventsIterable responses = client
     *             .listHealthEventsPaginator(request);
     *     for (software.amazon.awssdk.services.internetmonitor.model.ListHealthEventsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.internetmonitor.paginators.ListHealthEventsIterable responses = client.listHealthEventsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listHealthEvents(software.amazon.awssdk.services.internetmonitor.model.ListHealthEventsRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListHealthEventsRequest.Builder} avoiding the need
     * to create one manually via {@link ListHealthEventsRequest#builder()}
     * </p>
     *
     * @param listHealthEventsRequest
     *        A {@link Consumer} that will call methods on {@link ListHealthEventsInput.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.ListHealthEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/ListHealthEvents"
     *      target="_top">AWS API Documentation</a>
     */
    default ListHealthEventsIterable listHealthEventsPaginator(Consumer<ListHealthEventsRequest.Builder> listHealthEventsRequest)
            throws InternalServerException, AccessDeniedException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, InternetMonitorException {
        return listHealthEventsPaginator(ListHealthEventsRequest.builder().applyMutation(listHealthEventsRequest).build());
    }

    /**
     * <p>
     * Lists all of your monitors for Amazon CloudWatch Internet Monitor and their statuses, along with the Amazon
     * Resource Name (ARN) and name of each monitor.
     * </p>
     *
     * @param listMonitorsRequest
     * @return Result of the ListMonitors operation returned by the service.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.ListMonitors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/ListMonitors" target="_top">AWS
     *      API Documentation</a>
     */
    default ListMonitorsResponse listMonitors(ListMonitorsRequest listMonitorsRequest) throws InternalServerException,
            AccessDeniedException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            InternetMonitorException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all of your monitors for Amazon CloudWatch Internet Monitor and their statuses, along with the Amazon
     * Resource Name (ARN) and name of each monitor.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListMonitorsRequest.Builder} avoiding the need to
     * create one manually via {@link ListMonitorsRequest#builder()}
     * </p>
     *
     * @param listMonitorsRequest
     *        A {@link Consumer} that will call methods on {@link ListMonitorsInput.Builder} to create a request.
     * @return Result of the ListMonitors operation returned by the service.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.ListMonitors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/ListMonitors" target="_top">AWS
     *      API Documentation</a>
     */
    default ListMonitorsResponse listMonitors(Consumer<ListMonitorsRequest.Builder> listMonitorsRequest)
            throws InternalServerException, AccessDeniedException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, InternetMonitorException {
        return listMonitors(ListMonitorsRequest.builder().applyMutation(listMonitorsRequest).build());
    }

    /**
     * <p>
     * Lists all of your monitors for Amazon CloudWatch Internet Monitor and their statuses, along with the Amazon
     * Resource Name (ARN) and name of each monitor.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listMonitors(software.amazon.awssdk.services.internetmonitor.model.ListMonitorsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.internetmonitor.paginators.ListMonitorsIterable responses = client.listMonitorsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.internetmonitor.paginators.ListMonitorsIterable responses = client
     *             .listMonitorsPaginator(request);
     *     for (software.amazon.awssdk.services.internetmonitor.model.ListMonitorsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.internetmonitor.paginators.ListMonitorsIterable responses = client.listMonitorsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMonitors(software.amazon.awssdk.services.internetmonitor.model.ListMonitorsRequest)} operation.</b>
     * </p>
     *
     * @param listMonitorsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.ListMonitors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/ListMonitors" target="_top">AWS
     *      API Documentation</a>
     */
    default ListMonitorsIterable listMonitorsPaginator(ListMonitorsRequest listMonitorsRequest) throws InternalServerException,
            AccessDeniedException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            InternetMonitorException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all of your monitors for Amazon CloudWatch Internet Monitor and their statuses, along with the Amazon
     * Resource Name (ARN) and name of each monitor.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listMonitors(software.amazon.awssdk.services.internetmonitor.model.ListMonitorsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.internetmonitor.paginators.ListMonitorsIterable responses = client.listMonitorsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.internetmonitor.paginators.ListMonitorsIterable responses = client
     *             .listMonitorsPaginator(request);
     *     for (software.amazon.awssdk.services.internetmonitor.model.ListMonitorsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.internetmonitor.paginators.ListMonitorsIterable responses = client.listMonitorsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMonitors(software.amazon.awssdk.services.internetmonitor.model.ListMonitorsRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListMonitorsRequest.Builder} avoiding the need to
     * create one manually via {@link ListMonitorsRequest#builder()}
     * </p>
     *
     * @param listMonitorsRequest
     *        A {@link Consumer} that will call methods on {@link ListMonitorsInput.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.ListMonitors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/ListMonitors" target="_top">AWS
     *      API Documentation</a>
     */
    default ListMonitorsIterable listMonitorsPaginator(Consumer<ListMonitorsRequest.Builder> listMonitorsRequest)
            throws InternalServerException, AccessDeniedException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, InternetMonitorException {
        return listMonitorsPaginator(ListMonitorsRequest.builder().applyMutation(listMonitorsRequest).build());
    }

    /**
     * <p>
     * Lists the tags for a resource. Tags are supported only for monitors in Amazon CloudWatch Internet Monitor.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws TooManyRequestsException
     *         There were too many requests.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws NotFoundException
     *         The request specifies something that doesn't exist.
     * @throws BadRequestException
     *         A bad request was received.
     * @throws InternalServerErrorException
     *         There was an internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws TooManyRequestsException, AccessDeniedException, NotFoundException, BadRequestException,
            InternalServerErrorException, AwsServiceException, SdkClientException, InternetMonitorException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the tags for a resource. Tags are supported only for monitors in Amazon CloudWatch Internet Monitor.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on {@link ListTagsForResourceInput.Builder} to create a request.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws TooManyRequestsException
     *         There were too many requests.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws NotFoundException
     *         The request specifies something that doesn't exist.
     * @throws BadRequestException
     *         A bad request was received.
     * @throws InternalServerErrorException
     *         There was an internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) throws TooManyRequestsException,
            AccessDeniedException, NotFoundException, BadRequestException, InternalServerErrorException, AwsServiceException,
            SdkClientException, InternetMonitorException {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Adds a tag to a resource. Tags are supported only for monitors in Amazon CloudWatch Internet Monitor. You can add
     * a maximum of 50 tags in Internet Monitor.
     * </p>
     * <p>
     * A minimum of one tag is required for this call. It returns an error if you use the <code>TagResource</code>
     * request with 0 tags.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws TooManyRequestsException
     *         There were too many requests.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws NotFoundException
     *         The request specifies something that doesn't exist.
     * @throws BadRequestException
     *         A bad request was received.
     * @throws InternalServerErrorException
     *         There was an internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws TooManyRequestsException,
            AccessDeniedException, NotFoundException, BadRequestException, InternalServerErrorException, AwsServiceException,
            SdkClientException, InternetMonitorException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Adds a tag to a resource. Tags are supported only for monitors in Amazon CloudWatch Internet Monitor. You can add
     * a maximum of 50 tags in Internet Monitor.
     * </p>
     * <p>
     * A minimum of one tag is required for this call. It returns an error if you use the <code>TagResource</code>
     * request with 0 tags.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on {@link TagResourceInput.Builder} to create a request.
     * @return Result of the TagResource operation returned by the service.
     * @throws TooManyRequestsException
     *         There were too many requests.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws NotFoundException
     *         The request specifies something that doesn't exist.
     * @throws BadRequestException
     *         A bad request was received.
     * @throws InternalServerErrorException
     *         There was an internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default TagResourceResponse tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest)
            throws TooManyRequestsException, AccessDeniedException, NotFoundException, BadRequestException,
            InternalServerErrorException, AwsServiceException, SdkClientException, InternetMonitorException {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes a tag from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws TooManyRequestsException
     *         There were too many requests.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws NotFoundException
     *         The request specifies something that doesn't exist.
     * @throws BadRequestException
     *         A bad request was received.
     * @throws InternalServerErrorException
     *         There was an internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws TooManyRequestsException,
            AccessDeniedException, NotFoundException, BadRequestException, InternalServerErrorException, AwsServiceException,
            SdkClientException, InternetMonitorException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes a tag from a resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on {@link UntagResourceInput.Builder} to create a request.
     * @return Result of the UntagResource operation returned by the service.
     * @throws TooManyRequestsException
     *         There were too many requests.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws NotFoundException
     *         The request specifies something that doesn't exist.
     * @throws BadRequestException
     *         A bad request was received.
     * @throws InternalServerErrorException
     *         There was an internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default UntagResourceResponse untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest)
            throws TooManyRequestsException, AccessDeniedException, NotFoundException, BadRequestException,
            InternalServerErrorException, AwsServiceException, SdkClientException, InternetMonitorException {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Updates a monitor. You can update a monitor to add or remove resources, or to change the status of the monitor.
     * You can't change the name of a monitor.
     * </p>
     *
     * @param updateMonitorRequest
     * @return Result of the UpdateMonitor operation returned by the service.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws ResourceNotFoundException
     *         The request specifies a resource that doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws LimitExceededException
     *         The request exceeded a service quota.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.UpdateMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/UpdateMonitor" target="_top">AWS
     *      API Documentation</a>
     */
    default UpdateMonitorResponse updateMonitor(UpdateMonitorRequest updateMonitorRequest) throws InternalServerException,
            ResourceNotFoundException, AccessDeniedException, ThrottlingException, LimitExceededException, ValidationException,
            AwsServiceException, SdkClientException, InternetMonitorException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates a monitor. You can update a monitor to add or remove resources, or to change the status of the monitor.
     * You can't change the name of a monitor.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateMonitorRequest.Builder} avoiding the need to
     * create one manually via {@link UpdateMonitorRequest#builder()}
     * </p>
     *
     * @param updateMonitorRequest
     *        A {@link Consumer} that will call methods on {@link UpdateMonitorInput.Builder} to create a request.
     * @return Result of the UpdateMonitor operation returned by the service.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws ResourceNotFoundException
     *         The request specifies a resource that doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws LimitExceededException
     *         The request exceeded a service quota.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.UpdateMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/UpdateMonitor" target="_top">AWS
     *      API Documentation</a>
     */
    default UpdateMonitorResponse updateMonitor(Consumer<UpdateMonitorRequest.Builder> updateMonitorRequest)
            throws InternalServerException, ResourceNotFoundException, AccessDeniedException, ThrottlingException,
            LimitExceededException, ValidationException, AwsServiceException, SdkClientException, InternetMonitorException {
        return updateMonitor(UpdateMonitorRequest.builder().applyMutation(updateMonitorRequest).build());
    }

    /**
     * Create a {@link InternetMonitorClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static InternetMonitorClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link InternetMonitorClient}.
     */
    static InternetMonitorClientBuilder builder() {
        return new DefaultInternetMonitorClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    InternetMonitorServiceClientConfiguration serviceClientConfiguration();
}
