/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.internetmonitor.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateMonitorRequest extends InternetMonitorRequest implements
        ToCopyableBuilder<CreateMonitorRequest.Builder, CreateMonitorRequest> {
    private static final SdkField<String> MONITOR_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MonitorName").getter(getter(CreateMonitorRequest::monitorName)).setter(setter(Builder::monitorName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MonitorName").build()).build();

    private static final SdkField<List<String>> RESOURCES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Resources")
            .getter(getter(CreateMonitorRequest::resources))
            .setter(setter(Builder::resources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Resources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CLIENT_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientToken")
            .getter(getter(CreateMonitorRequest::clientToken))
            .setter(setter(Builder::clientToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientToken").build(),
                    DefaultValueTrait.idempotencyToken()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Tags")
            .getter(getter(CreateMonitorRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Integer> MAX_CITY_NETWORKS_TO_MONITOR_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("MaxCityNetworksToMonitor")
            .getter(getter(CreateMonitorRequest::maxCityNetworksToMonitor)).setter(setter(Builder::maxCityNetworksToMonitor))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxCityNetworksToMonitor").build())
            .build();

    private static final SdkField<InternetMeasurementsLogDelivery> INTERNET_MEASUREMENTS_LOG_DELIVERY_FIELD = SdkField
            .<InternetMeasurementsLogDelivery> builder(MarshallingType.SDK_POJO)
            .memberName("InternetMeasurementsLogDelivery")
            .getter(getter(CreateMonitorRequest::internetMeasurementsLogDelivery))
            .setter(setter(Builder::internetMeasurementsLogDelivery))
            .constructor(InternetMeasurementsLogDelivery::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InternetMeasurementsLogDelivery")
                    .build()).build();

    private static final SdkField<Integer> TRAFFIC_PERCENTAGE_TO_MONITOR_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("TrafficPercentageToMonitor")
            .getter(getter(CreateMonitorRequest::trafficPercentageToMonitor))
            .setter(setter(Builder::trafficPercentageToMonitor))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrafficPercentageToMonitor").build())
            .build();

    private static final SdkField<HealthEventsConfig> HEALTH_EVENTS_CONFIG_FIELD = SdkField
            .<HealthEventsConfig> builder(MarshallingType.SDK_POJO).memberName("HealthEventsConfig")
            .getter(getter(CreateMonitorRequest::healthEventsConfig)).setter(setter(Builder::healthEventsConfig))
            .constructor(HealthEventsConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HealthEventsConfig").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MONITOR_NAME_FIELD,
            RESOURCES_FIELD, CLIENT_TOKEN_FIELD, TAGS_FIELD, MAX_CITY_NETWORKS_TO_MONITOR_FIELD,
            INTERNET_MEASUREMENTS_LOG_DELIVERY_FIELD, TRAFFIC_PERCENTAGE_TO_MONITOR_FIELD, HEALTH_EVENTS_CONFIG_FIELD));

    private final String monitorName;

    private final List<String> resources;

    private final String clientToken;

    private final Map<String, String> tags;

    private final Integer maxCityNetworksToMonitor;

    private final InternetMeasurementsLogDelivery internetMeasurementsLogDelivery;

    private final Integer trafficPercentageToMonitor;

    private final HealthEventsConfig healthEventsConfig;

    private CreateMonitorRequest(BuilderImpl builder) {
        super(builder);
        this.monitorName = builder.monitorName;
        this.resources = builder.resources;
        this.clientToken = builder.clientToken;
        this.tags = builder.tags;
        this.maxCityNetworksToMonitor = builder.maxCityNetworksToMonitor;
        this.internetMeasurementsLogDelivery = builder.internetMeasurementsLogDelivery;
        this.trafficPercentageToMonitor = builder.trafficPercentageToMonitor;
        this.healthEventsConfig = builder.healthEventsConfig;
    }

    /**
     * <p>
     * The name of the monitor.
     * </p>
     * 
     * @return The name of the monitor.
     */
    public final String monitorName() {
        return monitorName;
    }

    /**
     * For responses, this returns true if the service returned a value for the Resources property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasResources() {
        return resources != null && !(resources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The resources to include in a monitor, which you provide as a set of Amazon Resource Names (ARNs). Resources can
     * be VPCs, NLBs, Amazon CloudFront distributions, or Amazon WorkSpaces directories.
     * </p>
     * <p>
     * You can add a combination of VPCs and CloudFront distributions, or you can add WorkSpaces directories, or you can
     * add NLBs. You can't add NLBs or WorkSpaces directories together with any other resources.
     * </p>
     * <note>
     * <p>
     * If you add only Amazon VPC resources, at least one VPC must have an Internet Gateway attached to it, to make sure
     * that it has internet connectivity.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResources} method.
     * </p>
     * 
     * @return The resources to include in a monitor, which you provide as a set of Amazon Resource Names (ARNs).
     *         Resources can be VPCs, NLBs, Amazon CloudFront distributions, or Amazon WorkSpaces directories.</p>
     *         <p>
     *         You can add a combination of VPCs and CloudFront distributions, or you can add WorkSpaces directories, or
     *         you can add NLBs. You can't add NLBs or WorkSpaces directories together with any other resources.
     *         </p>
     *         <note>
     *         <p>
     *         If you add only Amazon VPC resources, at least one VPC must have an Internet Gateway attached to it, to
     *         make sure that it has internet connectivity.
     *         </p>
     */
    public final List<String> resources() {
        return resources;
    }

    /**
     * <p>
     * A unique, case-sensitive string of up to 64 ASCII characters that you specify to make an idempotent API request.
     * Don't reuse the same client token for other API requests.
     * </p>
     * 
     * @return A unique, case-sensitive string of up to 64 ASCII characters that you specify to make an idempotent API
     *         request. Don't reuse the same client token for other API requests.
     */
    public final String clientToken() {
        return clientToken;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The tags for a monitor. You can add a maximum of 50 tags in Internet Monitor.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags for a monitor. You can add a maximum of 50 tags in Internet Monitor.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The maximum number of city-networks to monitor for your resources. A city-network is the location (city) where
     * clients access your application resources from and the ASN or network provider, such as an internet service
     * provider (ISP), that clients access the resources through. Setting this limit can help control billing costs.
     * </p>
     * <p>
     * To learn more, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/IMCityNetworksMaximum.html">Choosing a
     * city-network maximum value </a> in the Amazon CloudWatch Internet Monitor section of the <i>CloudWatch User
     * Guide</i>.
     * </p>
     * 
     * @return The maximum number of city-networks to monitor for your resources. A city-network is the location (city)
     *         where clients access your application resources from and the ASN or network provider, such as an internet
     *         service provider (ISP), that clients access the resources through. Setting this limit can help control
     *         billing costs.</p>
     *         <p>
     *         To learn more, see <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/IMCityNetworksMaximum.html">Choosing
     *         a city-network maximum value </a> in the Amazon CloudWatch Internet Monitor section of the <i>CloudWatch
     *         User Guide</i>.
     */
    public final Integer maxCityNetworksToMonitor() {
        return maxCityNetworksToMonitor;
    }

    /**
     * <p>
     * Publish internet measurements for Internet Monitor to an Amazon S3 bucket in addition to CloudWatch Logs.
     * </p>
     * 
     * @return Publish internet measurements for Internet Monitor to an Amazon S3 bucket in addition to CloudWatch Logs.
     */
    public final InternetMeasurementsLogDelivery internetMeasurementsLogDelivery() {
        return internetMeasurementsLogDelivery;
    }

    /**
     * <p>
     * The percentage of the internet-facing traffic for your application that you want to monitor with this monitor. If
     * you set a city-networks maximum, that limit overrides the traffic percentage that you set.
     * </p>
     * <p>
     * To learn more, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/IMTrafficPercentage.html">Choosing an
     * application traffic percentage to monitor </a> in the Amazon CloudWatch Internet Monitor section of the
     * <i>CloudWatch User Guide</i>.
     * </p>
     * 
     * @return The percentage of the internet-facing traffic for your application that you want to monitor with this
     *         monitor. If you set a city-networks maximum, that limit overrides the traffic percentage that you
     *         set.</p>
     *         <p>
     *         To learn more, see <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/IMTrafficPercentage.html">Choosing
     *         an application traffic percentage to monitor </a> in the Amazon CloudWatch Internet Monitor section of
     *         the <i>CloudWatch User Guide</i>.
     */
    public final Integer trafficPercentageToMonitor() {
        return trafficPercentageToMonitor;
    }

    /**
     * <p>
     * Defines the threshold percentages and other configuration information for when Amazon CloudWatch Internet Monitor
     * creates a health event. Internet Monitor creates a health event when an internet issue that affects your
     * application end users has a health score percentage that is at or below a specific threshold, and, sometimes,
     * when other criteria are met.
     * </p>
     * <p>
     * If you don't set a health event threshold, the default value is 95%.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-overview.html#IMUpdateThresholdFromOverview"
     * > Change health event thresholds</a> in the Internet Monitor section of the <i>CloudWatch User Guide</i>.
     * </p>
     * 
     * @return Defines the threshold percentages and other configuration information for when Amazon CloudWatch Internet
     *         Monitor creates a health event. Internet Monitor creates a health event when an internet issue that
     *         affects your application end users has a health score percentage that is at or below a specific
     *         threshold, and, sometimes, when other criteria are met.</p>
     *         <p>
     *         If you don't set a health event threshold, the default value is 95%.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-overview.html#IMUpdateThresholdFromOverview"
     *         > Change health event thresholds</a> in the Internet Monitor section of the <i>CloudWatch User Guide</i>.
     */
    public final HealthEventsConfig healthEventsConfig() {
        return healthEventsConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(monitorName());
        hashCode = 31 * hashCode + Objects.hashCode(hasResources() ? resources() : null);
        hashCode = 31 * hashCode + Objects.hashCode(clientToken());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(maxCityNetworksToMonitor());
        hashCode = 31 * hashCode + Objects.hashCode(internetMeasurementsLogDelivery());
        hashCode = 31 * hashCode + Objects.hashCode(trafficPercentageToMonitor());
        hashCode = 31 * hashCode + Objects.hashCode(healthEventsConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateMonitorRequest)) {
            return false;
        }
        CreateMonitorRequest other = (CreateMonitorRequest) obj;
        return Objects.equals(monitorName(), other.monitorName()) && hasResources() == other.hasResources()
                && Objects.equals(resources(), other.resources()) && Objects.equals(clientToken(), other.clientToken())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(maxCityNetworksToMonitor(), other.maxCityNetworksToMonitor())
                && Objects.equals(internetMeasurementsLogDelivery(), other.internetMeasurementsLogDelivery())
                && Objects.equals(trafficPercentageToMonitor(), other.trafficPercentageToMonitor())
                && Objects.equals(healthEventsConfig(), other.healthEventsConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateMonitorRequest").add("MonitorName", monitorName())
                .add("Resources", hasResources() ? resources() : null).add("ClientToken", clientToken())
                .add("Tags", hasTags() ? tags() : null).add("MaxCityNetworksToMonitor", maxCityNetworksToMonitor())
                .add("InternetMeasurementsLogDelivery", internetMeasurementsLogDelivery())
                .add("TrafficPercentageToMonitor", trafficPercentageToMonitor()).add("HealthEventsConfig", healthEventsConfig())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MonitorName":
            return Optional.ofNullable(clazz.cast(monitorName()));
        case "Resources":
            return Optional.ofNullable(clazz.cast(resources()));
        case "ClientToken":
            return Optional.ofNullable(clazz.cast(clientToken()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "MaxCityNetworksToMonitor":
            return Optional.ofNullable(clazz.cast(maxCityNetworksToMonitor()));
        case "InternetMeasurementsLogDelivery":
            return Optional.ofNullable(clazz.cast(internetMeasurementsLogDelivery()));
        case "TrafficPercentageToMonitor":
            return Optional.ofNullable(clazz.cast(trafficPercentageToMonitor()));
        case "HealthEventsConfig":
            return Optional.ofNullable(clazz.cast(healthEventsConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateMonitorRequest, T> g) {
        return obj -> g.apply((CreateMonitorRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends InternetMonitorRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateMonitorRequest> {
        /**
         * <p>
         * The name of the monitor.
         * </p>
         * 
         * @param monitorName
         *        The name of the monitor.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder monitorName(String monitorName);

        /**
         * <p>
         * The resources to include in a monitor, which you provide as a set of Amazon Resource Names (ARNs). Resources
         * can be VPCs, NLBs, Amazon CloudFront distributions, or Amazon WorkSpaces directories.
         * </p>
         * <p>
         * You can add a combination of VPCs and CloudFront distributions, or you can add WorkSpaces directories, or you
         * can add NLBs. You can't add NLBs or WorkSpaces directories together with any other resources.
         * </p>
         * <note>
         * <p>
         * If you add only Amazon VPC resources, at least one VPC must have an Internet Gateway attached to it, to make
         * sure that it has internet connectivity.
         * </p>
         * </note>
         * 
         * @param resources
         *        The resources to include in a monitor, which you provide as a set of Amazon Resource Names (ARNs).
         *        Resources can be VPCs, NLBs, Amazon CloudFront distributions, or Amazon WorkSpaces directories.</p>
         *        <p>
         *        You can add a combination of VPCs and CloudFront distributions, or you can add WorkSpaces directories,
         *        or you can add NLBs. You can't add NLBs or WorkSpaces directories together with any other resources.
         *        </p>
         *        <note>
         *        <p>
         *        If you add only Amazon VPC resources, at least one VPC must have an Internet Gateway attached to it,
         *        to make sure that it has internet connectivity.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(Collection<String> resources);

        /**
         * <p>
         * The resources to include in a monitor, which you provide as a set of Amazon Resource Names (ARNs). Resources
         * can be VPCs, NLBs, Amazon CloudFront distributions, or Amazon WorkSpaces directories.
         * </p>
         * <p>
         * You can add a combination of VPCs and CloudFront distributions, or you can add WorkSpaces directories, or you
         * can add NLBs. You can't add NLBs or WorkSpaces directories together with any other resources.
         * </p>
         * <note>
         * <p>
         * If you add only Amazon VPC resources, at least one VPC must have an Internet Gateway attached to it, to make
         * sure that it has internet connectivity.
         * </p>
         * </note>
         * 
         * @param resources
         *        The resources to include in a monitor, which you provide as a set of Amazon Resource Names (ARNs).
         *        Resources can be VPCs, NLBs, Amazon CloudFront distributions, or Amazon WorkSpaces directories.</p>
         *        <p>
         *        You can add a combination of VPCs and CloudFront distributions, or you can add WorkSpaces directories,
         *        or you can add NLBs. You can't add NLBs or WorkSpaces directories together with any other resources.
         *        </p>
         *        <note>
         *        <p>
         *        If you add only Amazon VPC resources, at least one VPC must have an Internet Gateway attached to it,
         *        to make sure that it has internet connectivity.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(String... resources);

        /**
         * <p>
         * A unique, case-sensitive string of up to 64 ASCII characters that you specify to make an idempotent API
         * request. Don't reuse the same client token for other API requests.
         * </p>
         * 
         * @param clientToken
         *        A unique, case-sensitive string of up to 64 ASCII characters that you specify to make an idempotent
         *        API request. Don't reuse the same client token for other API requests.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientToken(String clientToken);

        /**
         * <p>
         * The tags for a monitor. You can add a maximum of 50 tags in Internet Monitor.
         * </p>
         * 
         * @param tags
         *        The tags for a monitor. You can add a maximum of 50 tags in Internet Monitor.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The maximum number of city-networks to monitor for your resources. A city-network is the location (city)
         * where clients access your application resources from and the ASN or network provider, such as an internet
         * service provider (ISP), that clients access the resources through. Setting this limit can help control
         * billing costs.
         * </p>
         * <p>
         * To learn more, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/IMCityNetworksMaximum.html">Choosing a
         * city-network maximum value </a> in the Amazon CloudWatch Internet Monitor section of the <i>CloudWatch User
         * Guide</i>.
         * </p>
         * 
         * @param maxCityNetworksToMonitor
         *        The maximum number of city-networks to monitor for your resources. A city-network is the location
         *        (city) where clients access your application resources from and the ASN or network provider, such as
         *        an internet service provider (ISP), that clients access the resources through. Setting this limit can
         *        help control billing costs.</p>
         *        <p>
         *        To learn more, see <a
         *        href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/IMCityNetworksMaximum.html"
         *        >Choosing a city-network maximum value </a> in the Amazon CloudWatch Internet Monitor section of the
         *        <i>CloudWatch User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxCityNetworksToMonitor(Integer maxCityNetworksToMonitor);

        /**
         * <p>
         * Publish internet measurements for Internet Monitor to an Amazon S3 bucket in addition to CloudWatch Logs.
         * </p>
         * 
         * @param internetMeasurementsLogDelivery
         *        Publish internet measurements for Internet Monitor to an Amazon S3 bucket in addition to CloudWatch
         *        Logs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder internetMeasurementsLogDelivery(InternetMeasurementsLogDelivery internetMeasurementsLogDelivery);

        /**
         * <p>
         * Publish internet measurements for Internet Monitor to an Amazon S3 bucket in addition to CloudWatch Logs.
         * </p>
         * This is a convenience method that creates an instance of the {@link InternetMeasurementsLogDelivery.Builder}
         * avoiding the need to create one manually via {@link InternetMeasurementsLogDelivery#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InternetMeasurementsLogDelivery.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #internetMeasurementsLogDelivery(InternetMeasurementsLogDelivery)}.
         * 
         * @param internetMeasurementsLogDelivery
         *        a consumer that will call methods on {@link InternetMeasurementsLogDelivery.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #internetMeasurementsLogDelivery(InternetMeasurementsLogDelivery)
         */
        default Builder internetMeasurementsLogDelivery(
                Consumer<InternetMeasurementsLogDelivery.Builder> internetMeasurementsLogDelivery) {
            return internetMeasurementsLogDelivery(InternetMeasurementsLogDelivery.builder()
                    .applyMutation(internetMeasurementsLogDelivery).build());
        }

        /**
         * <p>
         * The percentage of the internet-facing traffic for your application that you want to monitor with this
         * monitor. If you set a city-networks maximum, that limit overrides the traffic percentage that you set.
         * </p>
         * <p>
         * To learn more, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/IMTrafficPercentage.html">Choosing an
         * application traffic percentage to monitor </a> in the Amazon CloudWatch Internet Monitor section of the
         * <i>CloudWatch User Guide</i>.
         * </p>
         * 
         * @param trafficPercentageToMonitor
         *        The percentage of the internet-facing traffic for your application that you want to monitor with this
         *        monitor. If you set a city-networks maximum, that limit overrides the traffic percentage that you
         *        set.</p>
         *        <p>
         *        To learn more, see <a
         *        href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/IMTrafficPercentage.html"
         *        >Choosing an application traffic percentage to monitor </a> in the Amazon CloudWatch Internet Monitor
         *        section of the <i>CloudWatch User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trafficPercentageToMonitor(Integer trafficPercentageToMonitor);

        /**
         * <p>
         * Defines the threshold percentages and other configuration information for when Amazon CloudWatch Internet
         * Monitor creates a health event. Internet Monitor creates a health event when an internet issue that affects
         * your application end users has a health score percentage that is at or below a specific threshold, and,
         * sometimes, when other criteria are met.
         * </p>
         * <p>
         * If you don't set a health event threshold, the default value is 95%.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-overview.html#IMUpdateThresholdFromOverview"
         * > Change health event thresholds</a> in the Internet Monitor section of the <i>CloudWatch User Guide</i>.
         * </p>
         * 
         * @param healthEventsConfig
         *        Defines the threshold percentages and other configuration information for when Amazon CloudWatch
         *        Internet Monitor creates a health event. Internet Monitor creates a health event when an internet
         *        issue that affects your application end users has a health score percentage that is at or below a
         *        specific threshold, and, sometimes, when other criteria are met.</p>
         *        <p>
         *        If you don't set a health event threshold, the default value is 95%.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-overview.html#IMUpdateThresholdFromOverview"
         *        > Change health event thresholds</a> in the Internet Monitor section of the <i>CloudWatch User
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder healthEventsConfig(HealthEventsConfig healthEventsConfig);

        /**
         * <p>
         * Defines the threshold percentages and other configuration information for when Amazon CloudWatch Internet
         * Monitor creates a health event. Internet Monitor creates a health event when an internet issue that affects
         * your application end users has a health score percentage that is at or below a specific threshold, and,
         * sometimes, when other criteria are met.
         * </p>
         * <p>
         * If you don't set a health event threshold, the default value is 95%.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-overview.html#IMUpdateThresholdFromOverview"
         * > Change health event thresholds</a> in the Internet Monitor section of the <i>CloudWatch User Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link HealthEventsConfig.Builder} avoiding the
         * need to create one manually via {@link HealthEventsConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link HealthEventsConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #healthEventsConfig(HealthEventsConfig)}.
         * 
         * @param healthEventsConfig
         *        a consumer that will call methods on {@link HealthEventsConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #healthEventsConfig(HealthEventsConfig)
         */
        default Builder healthEventsConfig(Consumer<HealthEventsConfig.Builder> healthEventsConfig) {
            return healthEventsConfig(HealthEventsConfig.builder().applyMutation(healthEventsConfig).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends InternetMonitorRequest.BuilderImpl implements Builder {
        private String monitorName;

        private List<String> resources = DefaultSdkAutoConstructList.getInstance();

        private String clientToken;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private Integer maxCityNetworksToMonitor;

        private InternetMeasurementsLogDelivery internetMeasurementsLogDelivery;

        private Integer trafficPercentageToMonitor;

        private HealthEventsConfig healthEventsConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateMonitorRequest model) {
            super(model);
            monitorName(model.monitorName);
            resources(model.resources);
            clientToken(model.clientToken);
            tags(model.tags);
            maxCityNetworksToMonitor(model.maxCityNetworksToMonitor);
            internetMeasurementsLogDelivery(model.internetMeasurementsLogDelivery);
            trafficPercentageToMonitor(model.trafficPercentageToMonitor);
            healthEventsConfig(model.healthEventsConfig);
        }

        public final String getMonitorName() {
            return monitorName;
        }

        public final void setMonitorName(String monitorName) {
            this.monitorName = monitorName;
        }

        @Override
        public final Builder monitorName(String monitorName) {
            this.monitorName = monitorName;
            return this;
        }

        public final Collection<String> getResources() {
            if (resources instanceof SdkAutoConstructList) {
                return null;
            }
            return resources;
        }

        public final void setResources(Collection<String> resources) {
            this.resources = SetOfARNsCopier.copy(resources);
        }

        @Override
        public final Builder resources(Collection<String> resources) {
            this.resources = SetOfARNsCopier.copy(resources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resources(String... resources) {
            resources(Arrays.asList(resources));
            return this;
        }

        public final String getClientToken() {
            return clientToken;
        }

        public final void setClientToken(String clientToken) {
            this.clientToken = clientToken;
        }

        @Override
        public final Builder clientToken(String clientToken) {
            this.clientToken = clientToken;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final Integer getMaxCityNetworksToMonitor() {
            return maxCityNetworksToMonitor;
        }

        public final void setMaxCityNetworksToMonitor(Integer maxCityNetworksToMonitor) {
            this.maxCityNetworksToMonitor = maxCityNetworksToMonitor;
        }

        @Override
        public final Builder maxCityNetworksToMonitor(Integer maxCityNetworksToMonitor) {
            this.maxCityNetworksToMonitor = maxCityNetworksToMonitor;
            return this;
        }

        public final InternetMeasurementsLogDelivery.Builder getInternetMeasurementsLogDelivery() {
            return internetMeasurementsLogDelivery != null ? internetMeasurementsLogDelivery.toBuilder() : null;
        }

        public final void setInternetMeasurementsLogDelivery(
                InternetMeasurementsLogDelivery.BuilderImpl internetMeasurementsLogDelivery) {
            this.internetMeasurementsLogDelivery = internetMeasurementsLogDelivery != null ? internetMeasurementsLogDelivery
                    .build() : null;
        }

        @Override
        public final Builder internetMeasurementsLogDelivery(InternetMeasurementsLogDelivery internetMeasurementsLogDelivery) {
            this.internetMeasurementsLogDelivery = internetMeasurementsLogDelivery;
            return this;
        }

        public final Integer getTrafficPercentageToMonitor() {
            return trafficPercentageToMonitor;
        }

        public final void setTrafficPercentageToMonitor(Integer trafficPercentageToMonitor) {
            this.trafficPercentageToMonitor = trafficPercentageToMonitor;
        }

        @Override
        public final Builder trafficPercentageToMonitor(Integer trafficPercentageToMonitor) {
            this.trafficPercentageToMonitor = trafficPercentageToMonitor;
            return this;
        }

        public final HealthEventsConfig.Builder getHealthEventsConfig() {
            return healthEventsConfig != null ? healthEventsConfig.toBuilder() : null;
        }

        public final void setHealthEventsConfig(HealthEventsConfig.BuilderImpl healthEventsConfig) {
            this.healthEventsConfig = healthEventsConfig != null ? healthEventsConfig.build() : null;
        }

        @Override
        public final Builder healthEventsConfig(HealthEventsConfig healthEventsConfig) {
            this.healthEventsConfig = healthEventsConfig;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateMonitorRequest build() {
            return new CreateMonitorRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
