/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.internetmonitor;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.internetmonitor.internal.InternetMonitorServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.internetmonitor.model.AccessDeniedException;
import software.amazon.awssdk.services.internetmonitor.model.BadRequestException;
import software.amazon.awssdk.services.internetmonitor.model.ConflictException;
import software.amazon.awssdk.services.internetmonitor.model.CreateMonitorRequest;
import software.amazon.awssdk.services.internetmonitor.model.CreateMonitorResponse;
import software.amazon.awssdk.services.internetmonitor.model.DeleteMonitorRequest;
import software.amazon.awssdk.services.internetmonitor.model.DeleteMonitorResponse;
import software.amazon.awssdk.services.internetmonitor.model.GetHealthEventRequest;
import software.amazon.awssdk.services.internetmonitor.model.GetHealthEventResponse;
import software.amazon.awssdk.services.internetmonitor.model.GetMonitorRequest;
import software.amazon.awssdk.services.internetmonitor.model.GetMonitorResponse;
import software.amazon.awssdk.services.internetmonitor.model.GetQueryResultsRequest;
import software.amazon.awssdk.services.internetmonitor.model.GetQueryResultsResponse;
import software.amazon.awssdk.services.internetmonitor.model.GetQueryStatusRequest;
import software.amazon.awssdk.services.internetmonitor.model.GetQueryStatusResponse;
import software.amazon.awssdk.services.internetmonitor.model.InternalServerErrorException;
import software.amazon.awssdk.services.internetmonitor.model.InternalServerException;
import software.amazon.awssdk.services.internetmonitor.model.InternetMonitorException;
import software.amazon.awssdk.services.internetmonitor.model.LimitExceededException;
import software.amazon.awssdk.services.internetmonitor.model.ListHealthEventsRequest;
import software.amazon.awssdk.services.internetmonitor.model.ListHealthEventsResponse;
import software.amazon.awssdk.services.internetmonitor.model.ListMonitorsRequest;
import software.amazon.awssdk.services.internetmonitor.model.ListMonitorsResponse;
import software.amazon.awssdk.services.internetmonitor.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.internetmonitor.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.internetmonitor.model.NotFoundException;
import software.amazon.awssdk.services.internetmonitor.model.ResourceNotFoundException;
import software.amazon.awssdk.services.internetmonitor.model.StartQueryRequest;
import software.amazon.awssdk.services.internetmonitor.model.StartQueryResponse;
import software.amazon.awssdk.services.internetmonitor.model.StopQueryRequest;
import software.amazon.awssdk.services.internetmonitor.model.StopQueryResponse;
import software.amazon.awssdk.services.internetmonitor.model.TagResourceRequest;
import software.amazon.awssdk.services.internetmonitor.model.TagResourceResponse;
import software.amazon.awssdk.services.internetmonitor.model.ThrottlingException;
import software.amazon.awssdk.services.internetmonitor.model.TooManyRequestsException;
import software.amazon.awssdk.services.internetmonitor.model.UntagResourceRequest;
import software.amazon.awssdk.services.internetmonitor.model.UntagResourceResponse;
import software.amazon.awssdk.services.internetmonitor.model.UpdateMonitorRequest;
import software.amazon.awssdk.services.internetmonitor.model.UpdateMonitorResponse;
import software.amazon.awssdk.services.internetmonitor.model.ValidationException;
import software.amazon.awssdk.services.internetmonitor.transform.CreateMonitorRequestMarshaller;
import software.amazon.awssdk.services.internetmonitor.transform.DeleteMonitorRequestMarshaller;
import software.amazon.awssdk.services.internetmonitor.transform.GetHealthEventRequestMarshaller;
import software.amazon.awssdk.services.internetmonitor.transform.GetMonitorRequestMarshaller;
import software.amazon.awssdk.services.internetmonitor.transform.GetQueryResultsRequestMarshaller;
import software.amazon.awssdk.services.internetmonitor.transform.GetQueryStatusRequestMarshaller;
import software.amazon.awssdk.services.internetmonitor.transform.ListHealthEventsRequestMarshaller;
import software.amazon.awssdk.services.internetmonitor.transform.ListMonitorsRequestMarshaller;
import software.amazon.awssdk.services.internetmonitor.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.internetmonitor.transform.StartQueryRequestMarshaller;
import software.amazon.awssdk.services.internetmonitor.transform.StopQueryRequestMarshaller;
import software.amazon.awssdk.services.internetmonitor.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.internetmonitor.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.internetmonitor.transform.UpdateMonitorRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link InternetMonitorClient}.
 *
 * @see InternetMonitorClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultInternetMonitorClient implements InternetMonitorClient {
    private static final Logger log = Logger.loggerFor(DefaultInternetMonitorClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultInternetMonitorClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a monitor in Amazon CloudWatch Internet Monitor. A monitor is built based on information from the
     * application resources that you add: VPCs, Network Load Balancers (NLBs), Amazon CloudFront distributions, and
     * Amazon WorkSpaces directories. Internet Monitor then publishes internet measurements from Amazon Web Services
     * that are specific to the <i>city-networks</i>. That is, the locations and ASNs (typically internet service
     * providers or ISPs), where clients access your application. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-InternetMonitor.html">Using
     * Amazon CloudWatch Internet Monitor</a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     * <p>
     * When you create a monitor, you choose the percentage of traffic that you want to monitor. You can also set a
     * maximum limit for the number of city-networks where client traffic is monitored, that caps the total traffic that
     * Internet Monitor monitors. A city-network maximum is the limit of city-networks, but you only pay for the number
     * of city-networks that are actually monitored. You can update your monitor at any time to change the percentage of
     * traffic to monitor or the city-networks maximum. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/IMCityNetworksMaximum.html">Choosing a
     * city-network maximum value</a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     *
     * @param createMonitorRequest
     * @return Result of the CreateMonitor operation returned by the service.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ConflictException
     *         The requested resource is in use.
     * @throws LimitExceededException
     *         The request exceeded a service quota.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.CreateMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/CreateMonitor" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateMonitorResponse createMonitor(CreateMonitorRequest createMonitorRequest) throws InternalServerException,
            AccessDeniedException, ThrottlingException, ConflictException, LimitExceededException, ValidationException,
            AwsServiceException, SdkClientException, InternetMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateMonitorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateMonitorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createMonitorRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "InternetMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMonitor");

            return clientHandler.execute(new ClientExecutionParams<CreateMonitorRequest, CreateMonitorResponse>()
                    .withOperationName("CreateMonitor").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createMonitorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateMonitorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a monitor in Amazon CloudWatch Internet Monitor.
     * </p>
     *
     * @param deleteMonitorRequest
     * @return Result of the DeleteMonitor operation returned by the service.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.DeleteMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/DeleteMonitor" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteMonitorResponse deleteMonitor(DeleteMonitorRequest deleteMonitorRequest) throws InternalServerException,
            AccessDeniedException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            InternetMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteMonitorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteMonitorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteMonitorRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "InternetMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMonitor");

            return clientHandler.execute(new ClientExecutionParams<DeleteMonitorRequest, DeleteMonitorResponse>()
                    .withOperationName("DeleteMonitor").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteMonitorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteMonitorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information the Amazon CloudWatch Internet Monitor has created and stored about a health event for a
     * specified monitor. This information includes the impacted locations, and all the information related to the
     * event, by location.
     * </p>
     * <p>
     * The information returned includes the impact on performance, availability, and round-trip time, information about
     * the network providers (ASNs), the event type, and so on.
     * </p>
     * <p>
     * Information rolled up at the global traffic level is also returned, including the impact type and total traffic
     * impact.
     * </p>
     *
     * @param getHealthEventRequest
     * @return Result of the GetHealthEvent operation returned by the service.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.GetHealthEvent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/GetHealthEvent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetHealthEventResponse getHealthEvent(GetHealthEventRequest getHealthEventRequest) throws InternalServerException,
            AccessDeniedException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            InternetMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetHealthEventResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetHealthEventResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getHealthEventRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getHealthEventRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "InternetMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetHealthEvent");

            return clientHandler.execute(new ClientExecutionParams<GetHealthEventRequest, GetHealthEventResponse>()
                    .withOperationName("GetHealthEvent").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getHealthEventRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetHealthEventRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about a monitor in Amazon CloudWatch Internet Monitor based on a monitor name. The information
     * returned includes the Amazon Resource Name (ARN), create time, modified time, resources included in the monitor,
     * and status information.
     * </p>
     *
     * @param getMonitorRequest
     * @return Result of the GetMonitor operation returned by the service.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.GetMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/GetMonitor" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetMonitorResponse getMonitor(GetMonitorRequest getMonitorRequest) throws InternalServerException,
            AccessDeniedException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            InternetMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMonitorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetMonitorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMonitorRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "InternetMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMonitor");

            return clientHandler.execute(new ClientExecutionParams<GetMonitorRequest, GetMonitorResponse>()
                    .withOperationName("GetMonitor").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getMonitorRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMonitorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Return the data for a query with the Amazon CloudWatch Internet Monitor query interface. Specify the query that
     * you want to return results for by providing a <code>QueryId</code> and a monitor name.
     * </p>
     * <p>
     * For more information about using the query interface, including examples, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-view-cw-tools-cwim-query.html"
     * >Using the Amazon CloudWatch Internet Monitor query interface</a> in the Amazon CloudWatch Internet Monitor User
     * Guide.
     * </p>
     *
     * @param getQueryResultsRequest
     * @return Result of the GetQueryResults operation returned by the service.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws LimitExceededException
     *         The request exceeded a service quota.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.GetQueryResults
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/GetQueryResults"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetQueryResultsResponse getQueryResults(GetQueryResultsRequest getQueryResultsRequest) throws InternalServerException,
            AccessDeniedException, ThrottlingException, LimitExceededException, ValidationException, AwsServiceException,
            SdkClientException, InternetMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetQueryResultsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetQueryResultsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getQueryResultsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getQueryResultsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "InternetMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQueryResults");

            return clientHandler.execute(new ClientExecutionParams<GetQueryResultsRequest, GetQueryResultsResponse>()
                    .withOperationName("GetQueryResults").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getQueryResultsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetQueryResultsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the current status of a query for the Amazon CloudWatch Internet Monitor query interface, for a specified
     * query ID and monitor. When you run a query, check the status to make sure that the query has
     * <code>SUCCEEDED</code> before you review the results.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>QUEUED</code>: The query is scheduled to run.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RUNNING</code>: The query is in progress but not complete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SUCCEEDED</code>: The query completed sucessfully.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code>: The query failed due to an error.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CANCELED</code>: The query was canceled.
     * </p>
     * </li>
     * </ul>
     *
     * @param getQueryStatusRequest
     * @return Result of the GetQueryStatus operation returned by the service.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws LimitExceededException
     *         The request exceeded a service quota.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.GetQueryStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/GetQueryStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetQueryStatusResponse getQueryStatus(GetQueryStatusRequest getQueryStatusRequest) throws InternalServerException,
            AccessDeniedException, ThrottlingException, LimitExceededException, ValidationException, AwsServiceException,
            SdkClientException, InternetMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetQueryStatusResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetQueryStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getQueryStatusRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getQueryStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "InternetMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQueryStatus");

            return clientHandler.execute(new ClientExecutionParams<GetQueryStatusRequest, GetQueryStatusResponse>()
                    .withOperationName("GetQueryStatus").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getQueryStatusRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetQueryStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all health events for a monitor in Amazon CloudWatch Internet Monitor. Returns information for health
     * events including the event start and end time and the status.
     * </p>
     * <note>
     * <p>
     * Health events that have start times during the time frame that is requested are not included in the list of
     * health events.
     * </p>
     * </note>
     *
     * @param listHealthEventsRequest
     * @return Result of the ListHealthEvents operation returned by the service.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.ListHealthEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/ListHealthEvents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListHealthEventsResponse listHealthEvents(ListHealthEventsRequest listHealthEventsRequest)
            throws InternalServerException, AccessDeniedException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, InternetMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListHealthEventsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListHealthEventsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listHealthEventsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listHealthEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "InternetMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListHealthEvents");

            return clientHandler.execute(new ClientExecutionParams<ListHealthEventsRequest, ListHealthEventsResponse>()
                    .withOperationName("ListHealthEvents").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listHealthEventsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListHealthEventsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all of your monitors for Amazon CloudWatch Internet Monitor and their statuses, along with the Amazon
     * Resource Name (ARN) and name of each monitor.
     * </p>
     *
     * @param listMonitorsRequest
     * @return Result of the ListMonitors operation returned by the service.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.ListMonitors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/ListMonitors" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListMonitorsResponse listMonitors(ListMonitorsRequest listMonitorsRequest) throws InternalServerException,
            AccessDeniedException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            InternetMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListMonitorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListMonitorsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMonitorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMonitorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "InternetMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMonitors");

            return clientHandler.execute(new ClientExecutionParams<ListMonitorsRequest, ListMonitorsResponse>()
                    .withOperationName("ListMonitors").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listMonitorsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListMonitorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags for a resource. Tags are supported only for monitors in Amazon CloudWatch Internet Monitor.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws TooManyRequestsException
     *         There were too many requests.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws NotFoundException
     *         The request specifies something that doesn't exist.
     * @throws BadRequestException
     *         A bad request was received.
     * @throws InternalServerErrorException
     *         There was an internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws TooManyRequestsException, AccessDeniedException, NotFoundException, BadRequestException,
            InternalServerErrorException, AwsServiceException, SdkClientException, InternetMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "InternetMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Start a query to return data for a specific query type for the Amazon CloudWatch Internet Monitor query
     * interface. Specify a time period for the data that you want returned by using <code>StartTime</code> and
     * <code>EndTime</code>. You filter the query results to return by providing parameters that you specify with
     * <code>FilterParameters</code>.
     * </p>
     * <p>
     * For more information about using the query interface, including examples, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-view-cw-tools-cwim-query.html"
     * >Using the Amazon CloudWatch Internet Monitor query interface</a> in the Amazon CloudWatch Internet Monitor User
     * Guide.
     * </p>
     *
     * @param startQueryRequest
     * @return Result of the StartQuery operation returned by the service.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws LimitExceededException
     *         The request exceeded a service quota.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.StartQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/StartQuery" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StartQueryResponse startQuery(StartQueryRequest startQueryRequest) throws InternalServerException,
            AccessDeniedException, ThrottlingException, LimitExceededException, ValidationException, AwsServiceException,
            SdkClientException, InternetMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startQueryRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "InternetMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartQuery");

            return clientHandler.execute(new ClientExecutionParams<StartQueryRequest, StartQueryResponse>()
                    .withOperationName("StartQuery").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(startQueryRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stop a query that is progress for a specific monitor.
     * </p>
     *
     * @param stopQueryRequest
     * @return Result of the StopQuery operation returned by the service.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws LimitExceededException
     *         The request exceeded a service quota.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.StopQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/StopQuery" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopQueryResponse stopQuery(StopQueryRequest stopQueryRequest) throws InternalServerException, AccessDeniedException,
            ThrottlingException, LimitExceededException, ValidationException, AwsServiceException, SdkClientException,
            InternetMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopQueryRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "InternetMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopQuery");

            return clientHandler.execute(new ClientExecutionParams<StopQueryRequest, StopQueryResponse>()
                    .withOperationName("StopQuery").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(stopQueryRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds a tag to a resource. Tags are supported only for monitors in Amazon CloudWatch Internet Monitor. You can add
     * a maximum of 50 tags in Internet Monitor.
     * </p>
     * <p>
     * A minimum of one tag is required for this call. It returns an error if you use the <code>TagResource</code>
     * request with 0 tags.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws TooManyRequestsException
     *         There were too many requests.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws NotFoundException
     *         The request specifies something that doesn't exist.
     * @throws BadRequestException
     *         A bad request was received.
     * @throws InternalServerErrorException
     *         There was an internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws TooManyRequestsException,
            AccessDeniedException, NotFoundException, BadRequestException, InternalServerErrorException, AwsServiceException,
            SdkClientException, InternetMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "InternetMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a tag from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws TooManyRequestsException
     *         There were too many requests.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws NotFoundException
     *         The request specifies something that doesn't exist.
     * @throws BadRequestException
     *         A bad request was received.
     * @throws InternalServerErrorException
     *         There was an internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws TooManyRequestsException,
            AccessDeniedException, NotFoundException, BadRequestException, InternalServerErrorException, AwsServiceException,
            SdkClientException, InternetMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "InternetMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a monitor. You can update a monitor to change the percentage of traffic to monitor or the maximum number
     * of city-networks (locations and ASNs), to add or remove resources, or to change the status of the monitor. Note
     * that you can't change the name of a monitor.
     * </p>
     * <p>
     * The city-network maximum that you choose is the limit, but you only pay for the number of city-networks that are
     * actually monitored. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/IMCityNetworksMaximum.html">Choosing a
     * city-network maximum value</a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     *
     * @param updateMonitorRequest
     * @return Result of the UpdateMonitor operation returned by the service.
     * @throws InternalServerException
     *         An internal error occurred.
     * @throws ResourceNotFoundException
     *         The request specifies a resource that doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient permission to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws LimitExceededException
     *         The request exceeded a service quota.
     * @throws ValidationException
     *         Invalid request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InternetMonitorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InternetMonitorClient.UpdateMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/internetmonitor-2021-06-03/UpdateMonitor" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateMonitorResponse updateMonitor(UpdateMonitorRequest updateMonitorRequest) throws InternalServerException,
            ResourceNotFoundException, AccessDeniedException, ThrottlingException, LimitExceededException, ValidationException,
            AwsServiceException, SdkClientException, InternetMonitorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateMonitorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateMonitorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateMonitorRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "InternetMonitor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMonitor");

            return clientHandler.execute(new ClientExecutionParams<UpdateMonitorRequest, UpdateMonitorResponse>()
                    .withOperationName("UpdateMonitor").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateMonitorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateMonitorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        InternetMonitorServiceClientConfigurationBuilder serviceConfigBuilder = new InternetMonitorServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(InternetMonitorException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(403).build());
    }

    @Override
    public final InternetMonitorServiceClientConfiguration serviceClientConfiguration() {
        return new InternetMonitorServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
