/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.internetmonitor.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class StartQueryRequest extends InternetMonitorRequest implements
        ToCopyableBuilder<StartQueryRequest.Builder, StartQueryRequest> {
    private static final SdkField<String> MONITOR_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MonitorName").getter(getter(StartQueryRequest::monitorName)).setter(setter(Builder::monitorName))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("MonitorName").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTime")
            .getter(getter(StartQueryRequest::startTime))
            .setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndTime")
            .getter(getter(StartQueryRequest::endTime))
            .setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> QUERY_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("QueryType").getter(getter(StartQueryRequest::queryTypeAsString)).setter(setter(Builder::queryType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QueryType").build()).build();

    private static final SdkField<List<FilterParameter>> FILTER_PARAMETERS_FIELD = SdkField
            .<List<FilterParameter>> builder(MarshallingType.LIST)
            .memberName("FilterParameters")
            .getter(getter(StartQueryRequest::filterParameters))
            .setter(setter(Builder::filterParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FilterParameters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<FilterParameter> builder(MarshallingType.SDK_POJO)
                                            .constructor(FilterParameter::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> LINKED_ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LinkedAccountId").getter(getter(StartQueryRequest::linkedAccountId))
            .setter(setter(Builder::linkedAccountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LinkedAccountId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MONITOR_NAME_FIELD,
            START_TIME_FIELD, END_TIME_FIELD, QUERY_TYPE_FIELD, FILTER_PARAMETERS_FIELD, LINKED_ACCOUNT_ID_FIELD));

    private final String monitorName;

    private final Instant startTime;

    private final Instant endTime;

    private final String queryType;

    private final List<FilterParameter> filterParameters;

    private final String linkedAccountId;

    private StartQueryRequest(BuilderImpl builder) {
        super(builder);
        this.monitorName = builder.monitorName;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.queryType = builder.queryType;
        this.filterParameters = builder.filterParameters;
        this.linkedAccountId = builder.linkedAccountId;
    }

    /**
     * <p>
     * The name of the monitor to query.
     * </p>
     * 
     * @return The name of the monitor to query.
     */
    public final String monitorName() {
        return monitorName;
    }

    /**
     * <p>
     * The timestamp that is the beginning of the period that you want to retrieve data for with your query.
     * </p>
     * 
     * @return The timestamp that is the beginning of the period that you want to retrieve data for with your query.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The timestamp that is the end of the period that you want to retrieve data for with your query.
     * </p>
     * 
     * @return The timestamp that is the end of the period that you want to retrieve data for with your query.
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * The type of query to run. The following are the three types of queries that you can run using the Internet
     * Monitor query interface:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>MEASUREMENTS</code>: Provides availability score, performance score, total traffic, and round-trip times,
     * at 5 minute intervals.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TOP_LOCATIONS</code>: Provides availability score, performance score, total traffic, and time to first byte
     * (TTFB) information, for the top location and ASN combinations that you're monitoring, by traffic volume.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TOP_LOCATION_DETAILS</code>: Provides TTFB for Amazon CloudFront, your current configuration, and the best
     * performing EC2 configuration, at 1 hour intervals.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OVERALL_TRAFFIC_SUGGESTIONS</code>: Provides TTFB, using a 30-day weighted average, for all traffic in each
     * Amazon Web Services location that is monitored.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OVERALL_TRAFFIC_SUGGESTIONS_DETAILS</code>: Provides TTFB, using a 30-day weighted average, for each top
     * location, for a proposed Amazon Web Services location. Must provide a Amazon Web Services location to search.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For lists of the fields returned with each query type and more information about how each type of query is
     * performed, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-view-cw-tools-cwim-query.html">
     * Using the Amazon CloudWatch Internet Monitor query interface</a> in the Amazon CloudWatch Internet Monitor User
     * Guide.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #queryType} will
     * return {@link QueryType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #queryTypeAsString}.
     * </p>
     * 
     * @return The type of query to run. The following are the three types of queries that you can run using the
     *         Internet Monitor query interface:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>MEASUREMENTS</code>: Provides availability score, performance score, total traffic, and round-trip
     *         times, at 5 minute intervals.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TOP_LOCATIONS</code>: Provides availability score, performance score, total traffic, and time to
     *         first byte (TTFB) information, for the top location and ASN combinations that you're monitoring, by
     *         traffic volume.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TOP_LOCATION_DETAILS</code>: Provides TTFB for Amazon CloudFront, your current configuration, and
     *         the best performing EC2 configuration, at 1 hour intervals.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OVERALL_TRAFFIC_SUGGESTIONS</code>: Provides TTFB, using a 30-day weighted average, for all traffic
     *         in each Amazon Web Services location that is monitored.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OVERALL_TRAFFIC_SUGGESTIONS_DETAILS</code>: Provides TTFB, using a 30-day weighted average, for
     *         each top location, for a proposed Amazon Web Services location. Must provide a Amazon Web Services
     *         location to search.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For lists of the fields returned with each query type and more information about how each type of query
     *         is performed, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-view-cw-tools-cwim-query.html"
     *         > Using the Amazon CloudWatch Internet Monitor query interface</a> in the Amazon CloudWatch Internet
     *         Monitor User Guide.
     * @see QueryType
     */
    public final QueryType queryType() {
        return QueryType.fromValue(queryType);
    }

    /**
     * <p>
     * The type of query to run. The following are the three types of queries that you can run using the Internet
     * Monitor query interface:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>MEASUREMENTS</code>: Provides availability score, performance score, total traffic, and round-trip times,
     * at 5 minute intervals.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TOP_LOCATIONS</code>: Provides availability score, performance score, total traffic, and time to first byte
     * (TTFB) information, for the top location and ASN combinations that you're monitoring, by traffic volume.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TOP_LOCATION_DETAILS</code>: Provides TTFB for Amazon CloudFront, your current configuration, and the best
     * performing EC2 configuration, at 1 hour intervals.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OVERALL_TRAFFIC_SUGGESTIONS</code>: Provides TTFB, using a 30-day weighted average, for all traffic in each
     * Amazon Web Services location that is monitored.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OVERALL_TRAFFIC_SUGGESTIONS_DETAILS</code>: Provides TTFB, using a 30-day weighted average, for each top
     * location, for a proposed Amazon Web Services location. Must provide a Amazon Web Services location to search.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For lists of the fields returned with each query type and more information about how each type of query is
     * performed, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-view-cw-tools-cwim-query.html">
     * Using the Amazon CloudWatch Internet Monitor query interface</a> in the Amazon CloudWatch Internet Monitor User
     * Guide.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #queryType} will
     * return {@link QueryType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #queryTypeAsString}.
     * </p>
     * 
     * @return The type of query to run. The following are the three types of queries that you can run using the
     *         Internet Monitor query interface:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>MEASUREMENTS</code>: Provides availability score, performance score, total traffic, and round-trip
     *         times, at 5 minute intervals.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TOP_LOCATIONS</code>: Provides availability score, performance score, total traffic, and time to
     *         first byte (TTFB) information, for the top location and ASN combinations that you're monitoring, by
     *         traffic volume.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TOP_LOCATION_DETAILS</code>: Provides TTFB for Amazon CloudFront, your current configuration, and
     *         the best performing EC2 configuration, at 1 hour intervals.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OVERALL_TRAFFIC_SUGGESTIONS</code>: Provides TTFB, using a 30-day weighted average, for all traffic
     *         in each Amazon Web Services location that is monitored.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OVERALL_TRAFFIC_SUGGESTIONS_DETAILS</code>: Provides TTFB, using a 30-day weighted average, for
     *         each top location, for a proposed Amazon Web Services location. Must provide a Amazon Web Services
     *         location to search.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For lists of the fields returned with each query type and more information about how each type of query
     *         is performed, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-view-cw-tools-cwim-query.html"
     *         > Using the Amazon CloudWatch Internet Monitor query interface</a> in the Amazon CloudWatch Internet
     *         Monitor User Guide.
     * @see QueryType
     */
    public final String queryTypeAsString() {
        return queryType;
    }

    /**
     * For responses, this returns true if the service returned a value for the FilterParameters property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFilterParameters() {
        return filterParameters != null && !(filterParameters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The <code>FilterParameters</code> field that you use with Amazon CloudWatch Internet Monitor queries is a string
     * the defines how you want a query to be filtered. The filter parameters that you can specify depend on the query
     * type, since each query type returns a different set of Internet Monitor data.
     * </p>
     * <p>
     * For more information about specifying filter parameters, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-view-cw-tools-cwim-query.html"
     * >Using the Amazon CloudWatch Internet Monitor query interface</a> in the Amazon CloudWatch Internet Monitor User
     * Guide.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFilterParameters} method.
     * </p>
     * 
     * @return The <code>FilterParameters</code> field that you use with Amazon CloudWatch Internet Monitor queries is a
     *         string the defines how you want a query to be filtered. The filter parameters that you can specify depend
     *         on the query type, since each query type returns a different set of Internet Monitor data.</p>
     *         <p>
     *         For more information about specifying filter parameters, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-view-cw-tools-cwim-query.html"
     *         >Using the Amazon CloudWatch Internet Monitor query interface</a> in the Amazon CloudWatch Internet
     *         Monitor User Guide.
     */
    public final List<FilterParameter> filterParameters() {
        return filterParameters;
    }

    /**
     * <p>
     * The account ID for an account that you've set up cross-account sharing for in Amazon CloudWatch Internet Monitor.
     * You configure cross-account sharing by using Amazon CloudWatch Observability Access Manager. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cwim-cross-account.html">Internet Monitor
     * cross-account observability</a> in the Amazon CloudWatch Internet Monitor User Guide.
     * </p>
     * 
     * @return The account ID for an account that you've set up cross-account sharing for in Amazon CloudWatch Internet
     *         Monitor. You configure cross-account sharing by using Amazon CloudWatch Observability Access Manager. For
     *         more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cwim-cross-account.html">Internet
     *         Monitor cross-account observability</a> in the Amazon CloudWatch Internet Monitor User Guide.
     */
    public final String linkedAccountId() {
        return linkedAccountId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(monitorName());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(queryTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasFilterParameters() ? filterParameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(linkedAccountId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StartQueryRequest)) {
            return false;
        }
        StartQueryRequest other = (StartQueryRequest) obj;
        return Objects.equals(monitorName(), other.monitorName()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(endTime(), other.endTime()) && Objects.equals(queryTypeAsString(), other.queryTypeAsString())
                && hasFilterParameters() == other.hasFilterParameters()
                && Objects.equals(filterParameters(), other.filterParameters())
                && Objects.equals(linkedAccountId(), other.linkedAccountId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StartQueryRequest").add("MonitorName", monitorName()).add("StartTime", startTime())
                .add("EndTime", endTime()).add("QueryType", queryTypeAsString())
                .add("FilterParameters", hasFilterParameters() ? filterParameters() : null)
                .add("LinkedAccountId", linkedAccountId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MonitorName":
            return Optional.ofNullable(clazz.cast(monitorName()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "EndTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "QueryType":
            return Optional.ofNullable(clazz.cast(queryTypeAsString()));
        case "FilterParameters":
            return Optional.ofNullable(clazz.cast(filterParameters()));
        case "LinkedAccountId":
            return Optional.ofNullable(clazz.cast(linkedAccountId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StartQueryRequest, T> g) {
        return obj -> g.apply((StartQueryRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends InternetMonitorRequest.Builder, SdkPojo, CopyableBuilder<Builder, StartQueryRequest> {
        /**
         * <p>
         * The name of the monitor to query.
         * </p>
         * 
         * @param monitorName
         *        The name of the monitor to query.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder monitorName(String monitorName);

        /**
         * <p>
         * The timestamp that is the beginning of the period that you want to retrieve data for with your query.
         * </p>
         * 
         * @param startTime
         *        The timestamp that is the beginning of the period that you want to retrieve data for with your query.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The timestamp that is the end of the period that you want to retrieve data for with your query.
         * </p>
         * 
         * @param endTime
         *        The timestamp that is the end of the period that you want to retrieve data for with your query.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * The type of query to run. The following are the three types of queries that you can run using the Internet
         * Monitor query interface:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>MEASUREMENTS</code>: Provides availability score, performance score, total traffic, and round-trip
         * times, at 5 minute intervals.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TOP_LOCATIONS</code>: Provides availability score, performance score, total traffic, and time to first
         * byte (TTFB) information, for the top location and ASN combinations that you're monitoring, by traffic volume.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TOP_LOCATION_DETAILS</code>: Provides TTFB for Amazon CloudFront, your current configuration, and the
         * best performing EC2 configuration, at 1 hour intervals.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OVERALL_TRAFFIC_SUGGESTIONS</code>: Provides TTFB, using a 30-day weighted average, for all traffic in
         * each Amazon Web Services location that is monitored.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OVERALL_TRAFFIC_SUGGESTIONS_DETAILS</code>: Provides TTFB, using a 30-day weighted average, for each
         * top location, for a proposed Amazon Web Services location. Must provide a Amazon Web Services location to
         * search.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For lists of the fields returned with each query type and more information about how each type of query is
         * performed, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-view-cw-tools-cwim-query.html">
         * Using the Amazon CloudWatch Internet Monitor query interface</a> in the Amazon CloudWatch Internet Monitor
         * User Guide.
         * </p>
         * 
         * @param queryType
         *        The type of query to run. The following are the three types of queries that you can run using the
         *        Internet Monitor query interface:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>MEASUREMENTS</code>: Provides availability score, performance score, total traffic, and
         *        round-trip times, at 5 minute intervals.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TOP_LOCATIONS</code>: Provides availability score, performance score, total traffic, and time to
         *        first byte (TTFB) information, for the top location and ASN combinations that you're monitoring, by
         *        traffic volume.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TOP_LOCATION_DETAILS</code>: Provides TTFB for Amazon CloudFront, your current configuration,
         *        and the best performing EC2 configuration, at 1 hour intervals.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OVERALL_TRAFFIC_SUGGESTIONS</code>: Provides TTFB, using a 30-day weighted average, for all
         *        traffic in each Amazon Web Services location that is monitored.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OVERALL_TRAFFIC_SUGGESTIONS_DETAILS</code>: Provides TTFB, using a 30-day weighted average, for
         *        each top location, for a proposed Amazon Web Services location. Must provide a Amazon Web Services
         *        location to search.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For lists of the fields returned with each query type and more information about how each type of
         *        query is performed, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-view-cw-tools-cwim-query.html"
         *        > Using the Amazon CloudWatch Internet Monitor query interface</a> in the Amazon CloudWatch Internet
         *        Monitor User Guide.
         * @see QueryType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see QueryType
         */
        Builder queryType(String queryType);

        /**
         * <p>
         * The type of query to run. The following are the three types of queries that you can run using the Internet
         * Monitor query interface:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>MEASUREMENTS</code>: Provides availability score, performance score, total traffic, and round-trip
         * times, at 5 minute intervals.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TOP_LOCATIONS</code>: Provides availability score, performance score, total traffic, and time to first
         * byte (TTFB) information, for the top location and ASN combinations that you're monitoring, by traffic volume.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TOP_LOCATION_DETAILS</code>: Provides TTFB for Amazon CloudFront, your current configuration, and the
         * best performing EC2 configuration, at 1 hour intervals.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OVERALL_TRAFFIC_SUGGESTIONS</code>: Provides TTFB, using a 30-day weighted average, for all traffic in
         * each Amazon Web Services location that is monitored.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OVERALL_TRAFFIC_SUGGESTIONS_DETAILS</code>: Provides TTFB, using a 30-day weighted average, for each
         * top location, for a proposed Amazon Web Services location. Must provide a Amazon Web Services location to
         * search.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For lists of the fields returned with each query type and more information about how each type of query is
         * performed, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-view-cw-tools-cwim-query.html">
         * Using the Amazon CloudWatch Internet Monitor query interface</a> in the Amazon CloudWatch Internet Monitor
         * User Guide.
         * </p>
         * 
         * @param queryType
         *        The type of query to run. The following are the three types of queries that you can run using the
         *        Internet Monitor query interface:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>MEASUREMENTS</code>: Provides availability score, performance score, total traffic, and
         *        round-trip times, at 5 minute intervals.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TOP_LOCATIONS</code>: Provides availability score, performance score, total traffic, and time to
         *        first byte (TTFB) information, for the top location and ASN combinations that you're monitoring, by
         *        traffic volume.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TOP_LOCATION_DETAILS</code>: Provides TTFB for Amazon CloudFront, your current configuration,
         *        and the best performing EC2 configuration, at 1 hour intervals.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OVERALL_TRAFFIC_SUGGESTIONS</code>: Provides TTFB, using a 30-day weighted average, for all
         *        traffic in each Amazon Web Services location that is monitored.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OVERALL_TRAFFIC_SUGGESTIONS_DETAILS</code>: Provides TTFB, using a 30-day weighted average, for
         *        each top location, for a proposed Amazon Web Services location. Must provide a Amazon Web Services
         *        location to search.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For lists of the fields returned with each query type and more information about how each type of
         *        query is performed, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-view-cw-tools-cwim-query.html"
         *        > Using the Amazon CloudWatch Internet Monitor query interface</a> in the Amazon CloudWatch Internet
         *        Monitor User Guide.
         * @see QueryType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see QueryType
         */
        Builder queryType(QueryType queryType);

        /**
         * <p>
         * The <code>FilterParameters</code> field that you use with Amazon CloudWatch Internet Monitor queries is a
         * string the defines how you want a query to be filtered. The filter parameters that you can specify depend on
         * the query type, since each query type returns a different set of Internet Monitor data.
         * </p>
         * <p>
         * For more information about specifying filter parameters, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-view-cw-tools-cwim-query.html"
         * >Using the Amazon CloudWatch Internet Monitor query interface</a> in the Amazon CloudWatch Internet Monitor
         * User Guide.
         * </p>
         * 
         * @param filterParameters
         *        The <code>FilterParameters</code> field that you use with Amazon CloudWatch Internet Monitor queries
         *        is a string the defines how you want a query to be filtered. The filter parameters that you can
         *        specify depend on the query type, since each query type returns a different set of Internet Monitor
         *        data.</p>
         *        <p>
         *        For more information about specifying filter parameters, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-view-cw-tools-cwim-query.html"
         *        >Using the Amazon CloudWatch Internet Monitor query interface</a> in the Amazon CloudWatch Internet
         *        Monitor User Guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filterParameters(Collection<FilterParameter> filterParameters);

        /**
         * <p>
         * The <code>FilterParameters</code> field that you use with Amazon CloudWatch Internet Monitor queries is a
         * string the defines how you want a query to be filtered. The filter parameters that you can specify depend on
         * the query type, since each query type returns a different set of Internet Monitor data.
         * </p>
         * <p>
         * For more information about specifying filter parameters, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-view-cw-tools-cwim-query.html"
         * >Using the Amazon CloudWatch Internet Monitor query interface</a> in the Amazon CloudWatch Internet Monitor
         * User Guide.
         * </p>
         * 
         * @param filterParameters
         *        The <code>FilterParameters</code> field that you use with Amazon CloudWatch Internet Monitor queries
         *        is a string the defines how you want a query to be filtered. The filter parameters that you can
         *        specify depend on the query type, since each query type returns a different set of Internet Monitor
         *        data.</p>
         *        <p>
         *        For more information about specifying filter parameters, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-view-cw-tools-cwim-query.html"
         *        >Using the Amazon CloudWatch Internet Monitor query interface</a> in the Amazon CloudWatch Internet
         *        Monitor User Guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filterParameters(FilterParameter... filterParameters);

        /**
         * <p>
         * The <code>FilterParameters</code> field that you use with Amazon CloudWatch Internet Monitor queries is a
         * string the defines how you want a query to be filtered. The filter parameters that you can specify depend on
         * the query type, since each query type returns a different set of Internet Monitor data.
         * </p>
         * <p>
         * For more information about specifying filter parameters, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-IM-view-cw-tools-cwim-query.html"
         * >Using the Amazon CloudWatch Internet Monitor query interface</a> in the Amazon CloudWatch Internet Monitor
         * User Guide.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.internetmonitor.model.FilterParameter.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.internetmonitor.model.FilterParameter#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.internetmonitor.model.FilterParameter.Builder#build()} is called
         * immediately and its result is passed to {@link #filterParameters(List<FilterParameter>)}.
         * 
         * @param filterParameters
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.internetmonitor.model.FilterParameter.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #filterParameters(java.util.Collection<FilterParameter>)
         */
        Builder filterParameters(Consumer<FilterParameter.Builder>... filterParameters);

        /**
         * <p>
         * The account ID for an account that you've set up cross-account sharing for in Amazon CloudWatch Internet
         * Monitor. You configure cross-account sharing by using Amazon CloudWatch Observability Access Manager. For
         * more information, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cwim-cross-account.html">Internet
         * Monitor cross-account observability</a> in the Amazon CloudWatch Internet Monitor User Guide.
         * </p>
         * 
         * @param linkedAccountId
         *        The account ID for an account that you've set up cross-account sharing for in Amazon CloudWatch
         *        Internet Monitor. You configure cross-account sharing by using Amazon CloudWatch Observability Access
         *        Manager. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cwim-cross-account.html">Internet
         *        Monitor cross-account observability</a> in the Amazon CloudWatch Internet Monitor User Guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder linkedAccountId(String linkedAccountId);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends InternetMonitorRequest.BuilderImpl implements Builder {
        private String monitorName;

        private Instant startTime;

        private Instant endTime;

        private String queryType;

        private List<FilterParameter> filterParameters = DefaultSdkAutoConstructList.getInstance();

        private String linkedAccountId;

        private BuilderImpl() {
        }

        private BuilderImpl(StartQueryRequest model) {
            super(model);
            monitorName(model.monitorName);
            startTime(model.startTime);
            endTime(model.endTime);
            queryType(model.queryType);
            filterParameters(model.filterParameters);
            linkedAccountId(model.linkedAccountId);
        }

        public final String getMonitorName() {
            return monitorName;
        }

        public final void setMonitorName(String monitorName) {
            this.monitorName = monitorName;
        }

        @Override
        public final Builder monitorName(String monitorName) {
            this.monitorName = monitorName;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final String getQueryType() {
            return queryType;
        }

        public final void setQueryType(String queryType) {
            this.queryType = queryType;
        }

        @Override
        public final Builder queryType(String queryType) {
            this.queryType = queryType;
            return this;
        }

        @Override
        public final Builder queryType(QueryType queryType) {
            this.queryType(queryType == null ? null : queryType.toString());
            return this;
        }

        public final List<FilterParameter.Builder> getFilterParameters() {
            List<FilterParameter.Builder> result = FilterParametersCopier.copyToBuilder(this.filterParameters);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setFilterParameters(Collection<FilterParameter.BuilderImpl> filterParameters) {
            this.filterParameters = FilterParametersCopier.copyFromBuilder(filterParameters);
        }

        @Override
        public final Builder filterParameters(Collection<FilterParameter> filterParameters) {
            this.filterParameters = FilterParametersCopier.copy(filterParameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filterParameters(FilterParameter... filterParameters) {
            filterParameters(Arrays.asList(filterParameters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filterParameters(Consumer<FilterParameter.Builder>... filterParameters) {
            filterParameters(Stream.of(filterParameters).map(c -> FilterParameter.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getLinkedAccountId() {
            return linkedAccountId;
        }

        public final void setLinkedAccountId(String linkedAccountId) {
            this.linkedAccountId = linkedAccountId;
        }

        @Override
        public final Builder linkedAccountId(String linkedAccountId) {
            this.linkedAccountId = linkedAccountId;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public StartQueryRequest build() {
            return new StartQueryRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
