/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.invoicing;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.invoicing.internal.InvoicingServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.invoicing.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.invoicing.model.AccessDeniedException;
import software.amazon.awssdk.services.invoicing.model.BatchGetInvoiceProfileRequest;
import software.amazon.awssdk.services.invoicing.model.BatchGetInvoiceProfileResponse;
import software.amazon.awssdk.services.invoicing.model.CreateInvoiceUnitRequest;
import software.amazon.awssdk.services.invoicing.model.CreateInvoiceUnitResponse;
import software.amazon.awssdk.services.invoicing.model.DeleteInvoiceUnitRequest;
import software.amazon.awssdk.services.invoicing.model.DeleteInvoiceUnitResponse;
import software.amazon.awssdk.services.invoicing.model.GetInvoiceUnitRequest;
import software.amazon.awssdk.services.invoicing.model.GetInvoiceUnitResponse;
import software.amazon.awssdk.services.invoicing.model.InternalServerException;
import software.amazon.awssdk.services.invoicing.model.InvoicingException;
import software.amazon.awssdk.services.invoicing.model.ListInvoiceSummariesRequest;
import software.amazon.awssdk.services.invoicing.model.ListInvoiceSummariesResponse;
import software.amazon.awssdk.services.invoicing.model.ListInvoiceUnitsRequest;
import software.amazon.awssdk.services.invoicing.model.ListInvoiceUnitsResponse;
import software.amazon.awssdk.services.invoicing.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.invoicing.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.invoicing.model.ResourceNotFoundException;
import software.amazon.awssdk.services.invoicing.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.invoicing.model.TagResourceRequest;
import software.amazon.awssdk.services.invoicing.model.TagResourceResponse;
import software.amazon.awssdk.services.invoicing.model.ThrottlingException;
import software.amazon.awssdk.services.invoicing.model.UntagResourceRequest;
import software.amazon.awssdk.services.invoicing.model.UntagResourceResponse;
import software.amazon.awssdk.services.invoicing.model.UpdateInvoiceUnitRequest;
import software.amazon.awssdk.services.invoicing.model.UpdateInvoiceUnitResponse;
import software.amazon.awssdk.services.invoicing.model.ValidationException;
import software.amazon.awssdk.services.invoicing.transform.BatchGetInvoiceProfileRequestMarshaller;
import software.amazon.awssdk.services.invoicing.transform.CreateInvoiceUnitRequestMarshaller;
import software.amazon.awssdk.services.invoicing.transform.DeleteInvoiceUnitRequestMarshaller;
import software.amazon.awssdk.services.invoicing.transform.GetInvoiceUnitRequestMarshaller;
import software.amazon.awssdk.services.invoicing.transform.ListInvoiceSummariesRequestMarshaller;
import software.amazon.awssdk.services.invoicing.transform.ListInvoiceUnitsRequestMarshaller;
import software.amazon.awssdk.services.invoicing.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.invoicing.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.invoicing.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.invoicing.transform.UpdateInvoiceUnitRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link InvoicingClient}.
 *
 * @see InvoicingClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultInvoicingClient implements InvoicingClient {
    private static final Logger log = Logger.loggerFor(DefaultInvoicingClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultInvoicingClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "Invoicing" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * This gets the invoice profile associated with a set of accounts. The accounts must be linked accounts under the
     * requester management account organization.
     * </p>
     *
     * @param batchGetInvoiceProfileRequest
     * @return Result of the BatchGetInvoiceProfile operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The processing request failed because of an unknown error, exception, or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InvoicingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InvoicingClient.BatchGetInvoiceProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/invoicing-2024-12-01/BatchGetInvoiceProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchGetInvoiceProfileResponse batchGetInvoiceProfile(BatchGetInvoiceProfileRequest batchGetInvoiceProfileRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, InvoicingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchGetInvoiceProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchGetInvoiceProfileResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchGetInvoiceProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetInvoiceProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Invoicing");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetInvoiceProfile");

            return clientHandler
                    .execute(new ClientExecutionParams<BatchGetInvoiceProfileRequest, BatchGetInvoiceProfileResponse>()
                            .withOperationName("BatchGetInvoiceProfile").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(batchGetInvoiceProfileRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new BatchGetInvoiceProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This creates a new invoice unit with the provided definition.
     * </p>
     *
     * @param createInvoiceUnitRequest
     * @return Result of the CreateInvoiceUnit operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The processing request failed because of an unknown error, exception, or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InvoicingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InvoicingClient.CreateInvoiceUnit
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/invoicing-2024-12-01/CreateInvoiceUnit" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateInvoiceUnitResponse createInvoiceUnit(CreateInvoiceUnitRequest createInvoiceUnitRequest)
            throws ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, InvoicingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateInvoiceUnitResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateInvoiceUnitResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createInvoiceUnitRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createInvoiceUnitRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Invoicing");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateInvoiceUnit");

            return clientHandler.execute(new ClientExecutionParams<CreateInvoiceUnitRequest, CreateInvoiceUnitResponse>()
                    .withOperationName("CreateInvoiceUnit").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createInvoiceUnitRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateInvoiceUnitRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This deletes an invoice unit with the provided invoice unit ARN.
     * </p>
     *
     * @param deleteInvoiceUnitRequest
     * @return Result of the DeleteInvoiceUnit operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The processing request failed because of an unknown error, exception, or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InvoicingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InvoicingClient.DeleteInvoiceUnit
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/invoicing-2024-12-01/DeleteInvoiceUnit" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteInvoiceUnitResponse deleteInvoiceUnit(DeleteInvoiceUnitRequest deleteInvoiceUnitRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, InvoicingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteInvoiceUnitResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteInvoiceUnitResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteInvoiceUnitRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteInvoiceUnitRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Invoicing");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteInvoiceUnit");

            return clientHandler.execute(new ClientExecutionParams<DeleteInvoiceUnitRequest, DeleteInvoiceUnitResponse>()
                    .withOperationName("DeleteInvoiceUnit").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteInvoiceUnitRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteInvoiceUnitRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This retrieves the invoice unit definition.
     * </p>
     *
     * @param getInvoiceUnitRequest
     * @return Result of the GetInvoiceUnit operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The processing request failed because of an unknown error, exception, or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InvoicingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InvoicingClient.GetInvoiceUnit
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/invoicing-2024-12-01/GetInvoiceUnit" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetInvoiceUnitResponse getInvoiceUnit(GetInvoiceUnitRequest getInvoiceUnitRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, InvoicingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetInvoiceUnitResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetInvoiceUnitResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getInvoiceUnitRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInvoiceUnitRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Invoicing");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInvoiceUnit");

            return clientHandler.execute(new ClientExecutionParams<GetInvoiceUnitRequest, GetInvoiceUnitResponse>()
                    .withOperationName("GetInvoiceUnit").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getInvoiceUnitRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetInvoiceUnitRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves your invoice details programmatically, without line item details.
     * </p>
     *
     * @param listInvoiceSummariesRequest
     * @return Result of the ListInvoiceSummaries operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The processing request failed because of an unknown error, exception, or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InvoicingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InvoicingClient.ListInvoiceSummaries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/invoicing-2024-12-01/ListInvoiceSummaries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListInvoiceSummariesResponse listInvoiceSummaries(ListInvoiceSummariesRequest listInvoiceSummariesRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, InvoicingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInvoiceSummariesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListInvoiceSummariesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listInvoiceSummariesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInvoiceSummariesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Invoicing");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInvoiceSummaries");

            return clientHandler.execute(new ClientExecutionParams<ListInvoiceSummariesRequest, ListInvoiceSummariesResponse>()
                    .withOperationName("ListInvoiceSummaries").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listInvoiceSummariesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListInvoiceSummariesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This fetches a list of all invoice unit definitions for a given account, as of the provided <code>AsOf</code>
     * date.
     * </p>
     *
     * @param listInvoiceUnitsRequest
     * @return Result of the ListInvoiceUnits operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The processing request failed because of an unknown error, exception, or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InvoicingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InvoicingClient.ListInvoiceUnits
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/invoicing-2024-12-01/ListInvoiceUnits" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListInvoiceUnitsResponse listInvoiceUnits(ListInvoiceUnitsRequest listInvoiceUnitsRequest) throws ValidationException,
            InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            InvoicingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInvoiceUnitsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListInvoiceUnitsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listInvoiceUnitsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInvoiceUnitsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Invoicing");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInvoiceUnits");

            return clientHandler.execute(new ClientExecutionParams<ListInvoiceUnitsRequest, ListInvoiceUnitsResponse>()
                    .withOperationName("ListInvoiceUnits").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listInvoiceUnitsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListInvoiceUnitsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags for a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The processing request failed because of an unknown error, exception, or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InvoicingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InvoicingClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/invoicing-2024-12-01/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, InvoicingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Invoicing");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds a tag to a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The processing request failed because of an unknown error, exception, or failure.
     * @throws ServiceQuotaExceededException
     *         The request was rejected because it attempted to create resources beyond the current Amazon Web Services
     *         account limits. The error message describes the limit exceeded.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InvoicingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InvoicingClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/invoicing-2024-12-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, ServiceQuotaExceededException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, InvoicingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Invoicing");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a tag from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The processing request failed because of an unknown error, exception, or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InvoicingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InvoicingClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/invoicing-2024-12-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, InvoicingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Invoicing");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * You can update the invoice unit configuration at any time, and Amazon Web Services will use the latest
     * configuration at the end of the month.
     * </p>
     *
     * @param updateInvoiceUnitRequest
     * @return Result of the UpdateInvoiceUnit operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The processing request failed because of an unknown error, exception, or failure.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws InvoicingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample InvoicingClient.UpdateInvoiceUnit
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/invoicing-2024-12-01/UpdateInvoiceUnit" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateInvoiceUnitResponse updateInvoiceUnit(UpdateInvoiceUnitRequest updateInvoiceUnitRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, InvoicingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateInvoiceUnitResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateInvoiceUnitResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateInvoiceUnitRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateInvoiceUnitRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Invoicing");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateInvoiceUnit");

            return clientHandler.execute(new ClientExecutionParams<UpdateInvoiceUnitRequest, UpdateInvoiceUnitResponse>()
                    .withOperationName("UpdateInvoiceUnit").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateInvoiceUnitRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateInvoiceUnitRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        InvoicingServiceClientConfigurationBuilder serviceConfigBuilder = new InvoicingServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(InvoicingException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON).protocolVersion("1.0");
    }

    @Override
    public final InvoicingServiceClientConfiguration serviceClientConfiguration() {
        return new InvoicingServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
