/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iot.model;

import java.time.Instant;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.services.iot.transform.AuthorizerDescriptionMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The authorizer description.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AuthorizerDescription implements StructuredPojo,
        ToCopyableBuilder<AuthorizerDescription.Builder, AuthorizerDescription> {
    private final String authorizerName;

    private final String authorizerArn;

    private final String authorizerFunctionArn;

    private final String tokenKeyName;

    private final Map<String, String> tokenSigningPublicKeys;

    private final String status;

    private final Instant creationDate;

    private final Instant lastModifiedDate;

    private AuthorizerDescription(BuilderImpl builder) {
        this.authorizerName = builder.authorizerName;
        this.authorizerArn = builder.authorizerArn;
        this.authorizerFunctionArn = builder.authorizerFunctionArn;
        this.tokenKeyName = builder.tokenKeyName;
        this.tokenSigningPublicKeys = builder.tokenSigningPublicKeys;
        this.status = builder.status;
        this.creationDate = builder.creationDate;
        this.lastModifiedDate = builder.lastModifiedDate;
    }

    /**
     * <p>
     * The authorizer name.
     * </p>
     * 
     * @return The authorizer name.
     */
    public String authorizerName() {
        return authorizerName;
    }

    /**
     * <p>
     * The authorizer ARN.
     * </p>
     * 
     * @return The authorizer ARN.
     */
    public String authorizerArn() {
        return authorizerArn;
    }

    /**
     * <p>
     * The authorizer's Lambda function ARN.
     * </p>
     * 
     * @return The authorizer's Lambda function ARN.
     */
    public String authorizerFunctionArn() {
        return authorizerFunctionArn;
    }

    /**
     * <p>
     * The key used to extract the token from the HTTP headers.
     * </p>
     * 
     * @return The key used to extract the token from the HTTP headers.
     */
    public String tokenKeyName() {
        return tokenKeyName;
    }

    /**
     * <p>
     * The public keys used to validate the token signature returned by your custom authentication service.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The public keys used to validate the token signature returned by your custom authentication service.
     */
    public Map<String, String> tokenSigningPublicKeys() {
        return tokenSigningPublicKeys;
    }

    /**
     * <p>
     * The status of the authorizer.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AuthorizerStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the authorizer.
     * @see AuthorizerStatus
     */
    public AuthorizerStatus status() {
        return AuthorizerStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the authorizer.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AuthorizerStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the authorizer.
     * @see AuthorizerStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The UNIX timestamp of when the authorizer was created.
     * </p>
     * 
     * @return The UNIX timestamp of when the authorizer was created.
     */
    public Instant creationDate() {
        return creationDate;
    }

    /**
     * <p>
     * The UNIX timestamp of when the authorizer was last updated.
     * </p>
     * 
     * @return The UNIX timestamp of when the authorizer was last updated.
     */
    public Instant lastModifiedDate() {
        return lastModifiedDate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(authorizerName());
        hashCode = 31 * hashCode + Objects.hashCode(authorizerArn());
        hashCode = 31 * hashCode + Objects.hashCode(authorizerFunctionArn());
        hashCode = 31 * hashCode + Objects.hashCode(tokenKeyName());
        hashCode = 31 * hashCode + Objects.hashCode(tokenSigningPublicKeys());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(creationDate());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedDate());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AuthorizerDescription)) {
            return false;
        }
        AuthorizerDescription other = (AuthorizerDescription) obj;
        return Objects.equals(authorizerName(), other.authorizerName()) && Objects.equals(authorizerArn(), other.authorizerArn())
                && Objects.equals(authorizerFunctionArn(), other.authorizerFunctionArn())
                && Objects.equals(tokenKeyName(), other.tokenKeyName())
                && Objects.equals(tokenSigningPublicKeys(), other.tokenSigningPublicKeys())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(creationDate(), other.creationDate())
                && Objects.equals(lastModifiedDate(), other.lastModifiedDate());
    }

    @Override
    public String toString() {
        return ToString.builder("AuthorizerDescription").add("AuthorizerName", authorizerName())
                .add("AuthorizerArn", authorizerArn()).add("AuthorizerFunctionArn", authorizerFunctionArn())
                .add("TokenKeyName", tokenKeyName()).add("TokenSigningPublicKeys", tokenSigningPublicKeys())
                .add("Status", statusAsString()).add("CreationDate", creationDate()).add("LastModifiedDate", lastModifiedDate())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "authorizerName":
            return Optional.ofNullable(clazz.cast(authorizerName()));
        case "authorizerArn":
            return Optional.ofNullable(clazz.cast(authorizerArn()));
        case "authorizerFunctionArn":
            return Optional.ofNullable(clazz.cast(authorizerFunctionArn()));
        case "tokenKeyName":
            return Optional.ofNullable(clazz.cast(tokenKeyName()));
        case "tokenSigningPublicKeys":
            return Optional.ofNullable(clazz.cast(tokenSigningPublicKeys()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "creationDate":
            return Optional.ofNullable(clazz.cast(creationDate()));
        case "lastModifiedDate":
            return Optional.ofNullable(clazz.cast(lastModifiedDate()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        AuthorizerDescriptionMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, AuthorizerDescription> {
        /**
         * <p>
         * The authorizer name.
         * </p>
         * 
         * @param authorizerName
         *        The authorizer name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorizerName(String authorizerName);

        /**
         * <p>
         * The authorizer ARN.
         * </p>
         * 
         * @param authorizerArn
         *        The authorizer ARN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorizerArn(String authorizerArn);

        /**
         * <p>
         * The authorizer's Lambda function ARN.
         * </p>
         * 
         * @param authorizerFunctionArn
         *        The authorizer's Lambda function ARN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorizerFunctionArn(String authorizerFunctionArn);

        /**
         * <p>
         * The key used to extract the token from the HTTP headers.
         * </p>
         * 
         * @param tokenKeyName
         *        The key used to extract the token from the HTTP headers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tokenKeyName(String tokenKeyName);

        /**
         * <p>
         * The public keys used to validate the token signature returned by your custom authentication service.
         * </p>
         * 
         * @param tokenSigningPublicKeys
         *        The public keys used to validate the token signature returned by your custom authentication service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tokenSigningPublicKeys(Map<String, String> tokenSigningPublicKeys);

        /**
         * <p>
         * The status of the authorizer.
         * </p>
         * 
         * @param status
         *        The status of the authorizer.
         * @see AuthorizerStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AuthorizerStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the authorizer.
         * </p>
         * 
         * @param status
         *        The status of the authorizer.
         * @see AuthorizerStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AuthorizerStatus
         */
        Builder status(AuthorizerStatus status);

        /**
         * <p>
         * The UNIX timestamp of when the authorizer was created.
         * </p>
         * 
         * @param creationDate
         *        The UNIX timestamp of when the authorizer was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDate(Instant creationDate);

        /**
         * <p>
         * The UNIX timestamp of when the authorizer was last updated.
         * </p>
         * 
         * @param lastModifiedDate
         *        The UNIX timestamp of when the authorizer was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedDate(Instant lastModifiedDate);
    }

    static final class BuilderImpl implements Builder {
        private String authorizerName;

        private String authorizerArn;

        private String authorizerFunctionArn;

        private String tokenKeyName;

        private Map<String, String> tokenSigningPublicKeys = DefaultSdkAutoConstructMap.getInstance();

        private String status;

        private Instant creationDate;

        private Instant lastModifiedDate;

        private BuilderImpl() {
        }

        private BuilderImpl(AuthorizerDescription model) {
            authorizerName(model.authorizerName);
            authorizerArn(model.authorizerArn);
            authorizerFunctionArn(model.authorizerFunctionArn);
            tokenKeyName(model.tokenKeyName);
            tokenSigningPublicKeys(model.tokenSigningPublicKeys);
            status(model.status);
            creationDate(model.creationDate);
            lastModifiedDate(model.lastModifiedDate);
        }

        public final String getAuthorizerName() {
            return authorizerName;
        }

        @Override
        public final Builder authorizerName(String authorizerName) {
            this.authorizerName = authorizerName;
            return this;
        }

        public final void setAuthorizerName(String authorizerName) {
            this.authorizerName = authorizerName;
        }

        public final String getAuthorizerArn() {
            return authorizerArn;
        }

        @Override
        public final Builder authorizerArn(String authorizerArn) {
            this.authorizerArn = authorizerArn;
            return this;
        }

        public final void setAuthorizerArn(String authorizerArn) {
            this.authorizerArn = authorizerArn;
        }

        public final String getAuthorizerFunctionArn() {
            return authorizerFunctionArn;
        }

        @Override
        public final Builder authorizerFunctionArn(String authorizerFunctionArn) {
            this.authorizerFunctionArn = authorizerFunctionArn;
            return this;
        }

        public final void setAuthorizerFunctionArn(String authorizerFunctionArn) {
            this.authorizerFunctionArn = authorizerFunctionArn;
        }

        public final String getTokenKeyName() {
            return tokenKeyName;
        }

        @Override
        public final Builder tokenKeyName(String tokenKeyName) {
            this.tokenKeyName = tokenKeyName;
            return this;
        }

        public final void setTokenKeyName(String tokenKeyName) {
            this.tokenKeyName = tokenKeyName;
        }

        public final Map<String, String> getTokenSigningPublicKeys() {
            return tokenSigningPublicKeys;
        }

        @Override
        public final Builder tokenSigningPublicKeys(Map<String, String> tokenSigningPublicKeys) {
            this.tokenSigningPublicKeys = PublicKeyMapCopier.copy(tokenSigningPublicKeys);
            return this;
        }

        public final void setTokenSigningPublicKeys(Map<String, String> tokenSigningPublicKeys) {
            this.tokenSigningPublicKeys = PublicKeyMapCopier.copy(tokenSigningPublicKeys);
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(AuthorizerStatus status) {
            this.status(status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final Instant getCreationDate() {
            return creationDate;
        }

        @Override
        public final Builder creationDate(Instant creationDate) {
            this.creationDate = creationDate;
            return this;
        }

        public final void setCreationDate(Instant creationDate) {
            this.creationDate = creationDate;
        }

        public final Instant getLastModifiedDate() {
            return lastModifiedDate;
        }

        @Override
        public final Builder lastModifiedDate(Instant lastModifiedDate) {
            this.lastModifiedDate = lastModifiedDate;
            return this;
        }

        public final void setLastModifiedDate(Instant lastModifiedDate) {
            this.lastModifiedDate = lastModifiedDate;
        }

        @Override
        public AuthorizerDescription build() {
            return new AuthorizerDescription(this);
        }
    }
}
