/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iot.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.services.iot.transform.ImplicitDenyMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information that implicitly denies authorization. When policy doesn't explicitly deny or allow an action on a
 * resource it is considered an implicit deny.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ImplicitDeny implements StructuredPojo, ToCopyableBuilder<ImplicitDeny.Builder, ImplicitDeny> {
    private final List<Policy> policies;

    private ImplicitDeny(BuilderImpl builder) {
        this.policies = builder.policies;
    }

    /**
     * <p>
     * Policies that don't contain a matching allow or deny statement for the specified action on the specified
     * resource.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Policies that don't contain a matching allow or deny statement for the specified action on the specified
     *         resource.
     */
    public List<Policy> policies() {
        return policies;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(policies());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ImplicitDeny)) {
            return false;
        }
        ImplicitDeny other = (ImplicitDeny) obj;
        return Objects.equals(policies(), other.policies());
    }

    @Override
    public String toString() {
        return ToString.builder("ImplicitDeny").add("Policies", policies()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "policies":
            return Optional.ofNullable(clazz.cast(policies()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        ImplicitDenyMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, ImplicitDeny> {
        /**
         * <p>
         * Policies that don't contain a matching allow or deny statement for the specified action on the specified
         * resource.
         * </p>
         * 
         * @param policies
         *        Policies that don't contain a matching allow or deny statement for the specified action on the
         *        specified resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policies(Collection<Policy> policies);

        /**
         * <p>
         * Policies that don't contain a matching allow or deny statement for the specified action on the specified
         * resource.
         * </p>
         * 
         * @param policies
         *        Policies that don't contain a matching allow or deny statement for the specified action on the
         *        specified resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policies(Policy... policies);

        /**
         * <p>
         * Policies that don't contain a matching allow or deny statement for the specified action on the specified
         * resource.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Policy>.Builder} avoiding the need to
         * create one manually via {@link List<Policy>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Policy>.Builder#build()} is called immediately and its
         * result is passed to {@link #policies(List<Policy>)}.
         * 
         * @param policies
         *        a consumer that will call methods on {@link List<Policy>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #policies(List<Policy>)
         */
        Builder policies(Consumer<Policy.Builder>... policies);
    }

    static final class BuilderImpl implements Builder {
        private List<Policy> policies = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ImplicitDeny model) {
            policies(model.policies);
        }

        public final Collection<Policy.Builder> getPolicies() {
            return policies != null ? policies.stream().map(Policy::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder policies(Collection<Policy> policies) {
            this.policies = PoliciesCopier.copy(policies);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder policies(Policy... policies) {
            policies(Arrays.asList(policies));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder policies(Consumer<Policy.Builder>... policies) {
            policies(Stream.of(policies).map(c -> Policy.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setPolicies(Collection<Policy.BuilderImpl> policies) {
            this.policies = PoliciesCopier.copyFromBuilder(policies);
        }

        @Override
        public ImplicitDeny build() {
            return new ImplicitDeny(this);
        }
    }
}
