/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iot.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.services.iot.transform.JobMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The <code>Job</code> object contains details about a job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Job implements StructuredPojo, ToCopyableBuilder<Job.Builder, Job> {
    private final String jobArn;

    private final String jobId;

    private final String targetSelection;

    private final String status;

    private final Boolean forceCanceled;

    private final String comment;

    private final List<String> targets;

    private final String description;

    private final PresignedUrlConfig presignedUrlConfig;

    private final JobExecutionsRolloutConfig jobExecutionsRolloutConfig;

    private final Instant createdAt;

    private final Instant lastUpdatedAt;

    private final Instant completedAt;

    private final JobProcessDetails jobProcessDetails;

    private Job(BuilderImpl builder) {
        this.jobArn = builder.jobArn;
        this.jobId = builder.jobId;
        this.targetSelection = builder.targetSelection;
        this.status = builder.status;
        this.forceCanceled = builder.forceCanceled;
        this.comment = builder.comment;
        this.targets = builder.targets;
        this.description = builder.description;
        this.presignedUrlConfig = builder.presignedUrlConfig;
        this.jobExecutionsRolloutConfig = builder.jobExecutionsRolloutConfig;
        this.createdAt = builder.createdAt;
        this.lastUpdatedAt = builder.lastUpdatedAt;
        this.completedAt = builder.completedAt;
        this.jobProcessDetails = builder.jobProcessDetails;
    }

    /**
     * <p>
     * An ARN identifying the job with format "arn:aws:iot:region:account:job/jobId".
     * </p>
     * 
     * @return An ARN identifying the job with format "arn:aws:iot:region:account:job/jobId".
     */
    public String jobArn() {
        return jobArn;
    }

    /**
     * <p>
     * The unique identifier you assigned to this job when it was created.
     * </p>
     * 
     * @return The unique identifier you assigned to this job when it was created.
     */
    public String jobId() {
        return jobId;
    }

    /**
     * <p>
     * Specifies whether the job will continue to run (CONTINUOUS), or will be complete after all those things specified
     * as targets have completed the job (SNAPSHOT). If continuous, the job may also be run on a thing when a change is
     * detected in a target. For example, a job will run on a device when the thing representing the device is added to
     * a target group, even after the job was completed by all things originally in the group.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #targetSelection}
     * will return {@link TargetSelection#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #targetSelectionAsString}.
     * </p>
     * 
     * @return Specifies whether the job will continue to run (CONTINUOUS), or will be complete after all those things
     *         specified as targets have completed the job (SNAPSHOT). If continuous, the job may also be run on a thing
     *         when a change is detected in a target. For example, a job will run on a device when the thing
     *         representing the device is added to a target group, even after the job was completed by all things
     *         originally in the group.
     * @see TargetSelection
     */
    public TargetSelection targetSelection() {
        return TargetSelection.fromValue(targetSelection);
    }

    /**
     * <p>
     * Specifies whether the job will continue to run (CONTINUOUS), or will be complete after all those things specified
     * as targets have completed the job (SNAPSHOT). If continuous, the job may also be run on a thing when a change is
     * detected in a target. For example, a job will run on a device when the thing representing the device is added to
     * a target group, even after the job was completed by all things originally in the group.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #targetSelection}
     * will return {@link TargetSelection#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #targetSelectionAsString}.
     * </p>
     * 
     * @return Specifies whether the job will continue to run (CONTINUOUS), or will be complete after all those things
     *         specified as targets have completed the job (SNAPSHOT). If continuous, the job may also be run on a thing
     *         when a change is detected in a target. For example, a job will run on a device when the thing
     *         representing the device is added to a target group, even after the job was completed by all things
     *         originally in the group.
     * @see TargetSelection
     */
    public String targetSelectionAsString() {
        return targetSelection;
    }

    /**
     * <p>
     * The status of the job, one of <code>IN_PROGRESS</code>, <code>CANCELED</code>, or <code>COMPLETED</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link JobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the job, one of <code>IN_PROGRESS</code>, <code>CANCELED</code>, or <code>COMPLETED</code>.
     * @see JobStatus
     */
    public JobStatus status() {
        return JobStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the job, one of <code>IN_PROGRESS</code>, <code>CANCELED</code>, or <code>COMPLETED</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link JobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the job, one of <code>IN_PROGRESS</code>, <code>CANCELED</code>, or <code>COMPLETED</code>.
     * @see JobStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Will be <code>true</code> if the job was canceled with the optional <code>force</code> parameter set to
     * <code>true</code>.
     * </p>
     * 
     * @return Will be <code>true</code> if the job was canceled with the optional <code>force</code> parameter set to
     *         <code>true</code>.
     */
    public Boolean forceCanceled() {
        return forceCanceled;
    }

    /**
     * <p>
     * If the job was updated, describes the reason for the update.
     * </p>
     * 
     * @return If the job was updated, describes the reason for the update.
     */
    public String comment() {
        return comment;
    }

    /**
     * <p>
     * A list of IoT things and thing groups to which the job should be sent.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of IoT things and thing groups to which the job should be sent.
     */
    public List<String> targets() {
        return targets;
    }

    /**
     * <p>
     * A short text description of the job.
     * </p>
     * 
     * @return A short text description of the job.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * Configuration for pre-signed S3 URLs.
     * </p>
     * 
     * @return Configuration for pre-signed S3 URLs.
     */
    public PresignedUrlConfig presignedUrlConfig() {
        return presignedUrlConfig;
    }

    /**
     * <p>
     * Allows you to create a staged rollout of a job.
     * </p>
     * 
     * @return Allows you to create a staged rollout of a job.
     */
    public JobExecutionsRolloutConfig jobExecutionsRolloutConfig() {
        return jobExecutionsRolloutConfig;
    }

    /**
     * <p>
     * The time, in milliseconds since the epoch, when the job was created.
     * </p>
     * 
     * @return The time, in milliseconds since the epoch, when the job was created.
     */
    public Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The time, in milliseconds since the epoch, when the job was last updated.
     * </p>
     * 
     * @return The time, in milliseconds since the epoch, when the job was last updated.
     */
    public Instant lastUpdatedAt() {
        return lastUpdatedAt;
    }

    /**
     * <p>
     * The time, in milliseconds since the epoch, when the job was completed.
     * </p>
     * 
     * @return The time, in milliseconds since the epoch, when the job was completed.
     */
    public Instant completedAt() {
        return completedAt;
    }

    /**
     * <p>
     * Details about the job process.
     * </p>
     * 
     * @return Details about the job process.
     */
    public JobProcessDetails jobProcessDetails() {
        return jobProcessDetails;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(jobArn());
        hashCode = 31 * hashCode + Objects.hashCode(jobId());
        hashCode = 31 * hashCode + Objects.hashCode(targetSelectionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(forceCanceled());
        hashCode = 31 * hashCode + Objects.hashCode(comment());
        hashCode = 31 * hashCode + Objects.hashCode(targets());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(presignedUrlConfig());
        hashCode = 31 * hashCode + Objects.hashCode(jobExecutionsRolloutConfig());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(completedAt());
        hashCode = 31 * hashCode + Objects.hashCode(jobProcessDetails());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Job)) {
            return false;
        }
        Job other = (Job) obj;
        return Objects.equals(jobArn(), other.jobArn()) && Objects.equals(jobId(), other.jobId())
                && Objects.equals(targetSelectionAsString(), other.targetSelectionAsString())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(forceCanceled(), other.forceCanceled()) && Objects.equals(comment(), other.comment())
                && Objects.equals(targets(), other.targets()) && Objects.equals(description(), other.description())
                && Objects.equals(presignedUrlConfig(), other.presignedUrlConfig())
                && Objects.equals(jobExecutionsRolloutConfig(), other.jobExecutionsRolloutConfig())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(lastUpdatedAt(), other.lastUpdatedAt())
                && Objects.equals(completedAt(), other.completedAt())
                && Objects.equals(jobProcessDetails(), other.jobProcessDetails());
    }

    @Override
    public String toString() {
        return ToString.builder("Job").add("JobArn", jobArn()).add("JobId", jobId())
                .add("TargetSelection", targetSelectionAsString()).add("Status", statusAsString())
                .add("ForceCanceled", forceCanceled()).add("Comment", comment()).add("Targets", targets())
                .add("Description", description()).add("PresignedUrlConfig", presignedUrlConfig())
                .add("JobExecutionsRolloutConfig", jobExecutionsRolloutConfig()).add("CreatedAt", createdAt())
                .add("LastUpdatedAt", lastUpdatedAt()).add("CompletedAt", completedAt())
                .add("JobProcessDetails", jobProcessDetails()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "jobArn":
            return Optional.ofNullable(clazz.cast(jobArn()));
        case "jobId":
            return Optional.ofNullable(clazz.cast(jobId()));
        case "targetSelection":
            return Optional.ofNullable(clazz.cast(targetSelectionAsString()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "forceCanceled":
            return Optional.ofNullable(clazz.cast(forceCanceled()));
        case "comment":
            return Optional.ofNullable(clazz.cast(comment()));
        case "targets":
            return Optional.ofNullable(clazz.cast(targets()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "presignedUrlConfig":
            return Optional.ofNullable(clazz.cast(presignedUrlConfig()));
        case "jobExecutionsRolloutConfig":
            return Optional.ofNullable(clazz.cast(jobExecutionsRolloutConfig()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "lastUpdatedAt":
            return Optional.ofNullable(clazz.cast(lastUpdatedAt()));
        case "completedAt":
            return Optional.ofNullable(clazz.cast(completedAt()));
        case "jobProcessDetails":
            return Optional.ofNullable(clazz.cast(jobProcessDetails()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        JobMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Job> {
        /**
         * <p>
         * An ARN identifying the job with format "arn:aws:iot:region:account:job/jobId".
         * </p>
         * 
         * @param jobArn
         *        An ARN identifying the job with format "arn:aws:iot:region:account:job/jobId".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobArn(String jobArn);

        /**
         * <p>
         * The unique identifier you assigned to this job when it was created.
         * </p>
         * 
         * @param jobId
         *        The unique identifier you assigned to this job when it was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobId(String jobId);

        /**
         * <p>
         * Specifies whether the job will continue to run (CONTINUOUS), or will be complete after all those things
         * specified as targets have completed the job (SNAPSHOT). If continuous, the job may also be run on a thing
         * when a change is detected in a target. For example, a job will run on a device when the thing representing
         * the device is added to a target group, even after the job was completed by all things originally in the
         * group.
         * </p>
         * 
         * @param targetSelection
         *        Specifies whether the job will continue to run (CONTINUOUS), or will be complete after all those
         *        things specified as targets have completed the job (SNAPSHOT). If continuous, the job may also be run
         *        on a thing when a change is detected in a target. For example, a job will run on a device when the
         *        thing representing the device is added to a target group, even after the job was completed by all
         *        things originally in the group.
         * @see TargetSelection
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetSelection
         */
        Builder targetSelection(String targetSelection);

        /**
         * <p>
         * Specifies whether the job will continue to run (CONTINUOUS), or will be complete after all those things
         * specified as targets have completed the job (SNAPSHOT). If continuous, the job may also be run on a thing
         * when a change is detected in a target. For example, a job will run on a device when the thing representing
         * the device is added to a target group, even after the job was completed by all things originally in the
         * group.
         * </p>
         * 
         * @param targetSelection
         *        Specifies whether the job will continue to run (CONTINUOUS), or will be complete after all those
         *        things specified as targets have completed the job (SNAPSHOT). If continuous, the job may also be run
         *        on a thing when a change is detected in a target. For example, a job will run on a device when the
         *        thing representing the device is added to a target group, even after the job was completed by all
         *        things originally in the group.
         * @see TargetSelection
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetSelection
         */
        Builder targetSelection(TargetSelection targetSelection);

        /**
         * <p>
         * The status of the job, one of <code>IN_PROGRESS</code>, <code>CANCELED</code>, or <code>COMPLETED</code>.
         * </p>
         * 
         * @param status
         *        The status of the job, one of <code>IN_PROGRESS</code>, <code>CANCELED</code>, or
         *        <code>COMPLETED</code>.
         * @see JobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the job, one of <code>IN_PROGRESS</code>, <code>CANCELED</code>, or <code>COMPLETED</code>.
         * </p>
         * 
         * @param status
         *        The status of the job, one of <code>IN_PROGRESS</code>, <code>CANCELED</code>, or
         *        <code>COMPLETED</code>.
         * @see JobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobStatus
         */
        Builder status(JobStatus status);

        /**
         * <p>
         * Will be <code>true</code> if the job was canceled with the optional <code>force</code> parameter set to
         * <code>true</code>.
         * </p>
         * 
         * @param forceCanceled
         *        Will be <code>true</code> if the job was canceled with the optional <code>force</code> parameter set
         *        to <code>true</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forceCanceled(Boolean forceCanceled);

        /**
         * <p>
         * If the job was updated, describes the reason for the update.
         * </p>
         * 
         * @param comment
         *        If the job was updated, describes the reason for the update.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder comment(String comment);

        /**
         * <p>
         * A list of IoT things and thing groups to which the job should be sent.
         * </p>
         * 
         * @param targets
         *        A list of IoT things and thing groups to which the job should be sent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<String> targets);

        /**
         * <p>
         * A list of IoT things and thing groups to which the job should be sent.
         * </p>
         * 
         * @param targets
         *        A list of IoT things and thing groups to which the job should be sent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(String... targets);

        /**
         * <p>
         * A short text description of the job.
         * </p>
         * 
         * @param description
         *        A short text description of the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * Configuration for pre-signed S3 URLs.
         * </p>
         * 
         * @param presignedUrlConfig
         *        Configuration for pre-signed S3 URLs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder presignedUrlConfig(PresignedUrlConfig presignedUrlConfig);

        /**
         * <p>
         * Configuration for pre-signed S3 URLs.
         * </p>
         * This is a convenience that creates an instance of the {@link PresignedUrlConfig.Builder} avoiding the need to
         * create one manually via {@link PresignedUrlConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link PresignedUrlConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #presignedUrlConfig(PresignedUrlConfig)}.
         * 
         * @param presignedUrlConfig
         *        a consumer that will call methods on {@link PresignedUrlConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #presignedUrlConfig(PresignedUrlConfig)
         */
        default Builder presignedUrlConfig(Consumer<PresignedUrlConfig.Builder> presignedUrlConfig) {
            return presignedUrlConfig(PresignedUrlConfig.builder().applyMutation(presignedUrlConfig).build());
        }

        /**
         * <p>
         * Allows you to create a staged rollout of a job.
         * </p>
         * 
         * @param jobExecutionsRolloutConfig
         *        Allows you to create a staged rollout of a job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobExecutionsRolloutConfig(JobExecutionsRolloutConfig jobExecutionsRolloutConfig);

        /**
         * <p>
         * Allows you to create a staged rollout of a job.
         * </p>
         * This is a convenience that creates an instance of the {@link JobExecutionsRolloutConfig.Builder} avoiding the
         * need to create one manually via {@link JobExecutionsRolloutConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link JobExecutionsRolloutConfig.Builder#build()} is called immediately
         * and its result is passed to {@link #jobExecutionsRolloutConfig(JobExecutionsRolloutConfig)}.
         * 
         * @param jobExecutionsRolloutConfig
         *        a consumer that will call methods on {@link JobExecutionsRolloutConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #jobExecutionsRolloutConfig(JobExecutionsRolloutConfig)
         */
        default Builder jobExecutionsRolloutConfig(Consumer<JobExecutionsRolloutConfig.Builder> jobExecutionsRolloutConfig) {
            return jobExecutionsRolloutConfig(JobExecutionsRolloutConfig.builder().applyMutation(jobExecutionsRolloutConfig)
                    .build());
        }

        /**
         * <p>
         * The time, in milliseconds since the epoch, when the job was created.
         * </p>
         * 
         * @param createdAt
         *        The time, in milliseconds since the epoch, when the job was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The time, in milliseconds since the epoch, when the job was last updated.
         * </p>
         * 
         * @param lastUpdatedAt
         *        The time, in milliseconds since the epoch, when the job was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedAt(Instant lastUpdatedAt);

        /**
         * <p>
         * The time, in milliseconds since the epoch, when the job was completed.
         * </p>
         * 
         * @param completedAt
         *        The time, in milliseconds since the epoch, when the job was completed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completedAt(Instant completedAt);

        /**
         * <p>
         * Details about the job process.
         * </p>
         * 
         * @param jobProcessDetails
         *        Details about the job process.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobProcessDetails(JobProcessDetails jobProcessDetails);

        /**
         * <p>
         * Details about the job process.
         * </p>
         * This is a convenience that creates an instance of the {@link JobProcessDetails.Builder} avoiding the need to
         * create one manually via {@link JobProcessDetails#builder()}.
         *
         * When the {@link Consumer} completes, {@link JobProcessDetails.Builder#build()} is called immediately and its
         * result is passed to {@link #jobProcessDetails(JobProcessDetails)}.
         * 
         * @param jobProcessDetails
         *        a consumer that will call methods on {@link JobProcessDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #jobProcessDetails(JobProcessDetails)
         */
        default Builder jobProcessDetails(Consumer<JobProcessDetails.Builder> jobProcessDetails) {
            return jobProcessDetails(JobProcessDetails.builder().applyMutation(jobProcessDetails).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String jobArn;

        private String jobId;

        private String targetSelection;

        private String status;

        private Boolean forceCanceled;

        private String comment;

        private List<String> targets = DefaultSdkAutoConstructList.getInstance();

        private String description;

        private PresignedUrlConfig presignedUrlConfig;

        private JobExecutionsRolloutConfig jobExecutionsRolloutConfig;

        private Instant createdAt;

        private Instant lastUpdatedAt;

        private Instant completedAt;

        private JobProcessDetails jobProcessDetails;

        private BuilderImpl() {
        }

        private BuilderImpl(Job model) {
            jobArn(model.jobArn);
            jobId(model.jobId);
            targetSelection(model.targetSelection);
            status(model.status);
            forceCanceled(model.forceCanceled);
            comment(model.comment);
            targets(model.targets);
            description(model.description);
            presignedUrlConfig(model.presignedUrlConfig);
            jobExecutionsRolloutConfig(model.jobExecutionsRolloutConfig);
            createdAt(model.createdAt);
            lastUpdatedAt(model.lastUpdatedAt);
            completedAt(model.completedAt);
            jobProcessDetails(model.jobProcessDetails);
        }

        public final String getJobArn() {
            return jobArn;
        }

        @Override
        public final Builder jobArn(String jobArn) {
            this.jobArn = jobArn;
            return this;
        }

        public final void setJobArn(String jobArn) {
            this.jobArn = jobArn;
        }

        public final String getJobId() {
            return jobId;
        }

        @Override
        public final Builder jobId(String jobId) {
            this.jobId = jobId;
            return this;
        }

        public final void setJobId(String jobId) {
            this.jobId = jobId;
        }

        public final String getTargetSelection() {
            return targetSelection;
        }

        @Override
        public final Builder targetSelection(String targetSelection) {
            this.targetSelection = targetSelection;
            return this;
        }

        @Override
        public final Builder targetSelection(TargetSelection targetSelection) {
            this.targetSelection(targetSelection.toString());
            return this;
        }

        public final void setTargetSelection(String targetSelection) {
            this.targetSelection = targetSelection;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(JobStatus status) {
            this.status(status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final Boolean getForceCanceled() {
            return forceCanceled;
        }

        @Override
        public final Builder forceCanceled(Boolean forceCanceled) {
            this.forceCanceled = forceCanceled;
            return this;
        }

        public final void setForceCanceled(Boolean forceCanceled) {
            this.forceCanceled = forceCanceled;
        }

        public final String getComment() {
            return comment;
        }

        @Override
        public final Builder comment(String comment) {
            this.comment = comment;
            return this;
        }

        public final void setComment(String comment) {
            this.comment = comment;
        }

        public final Collection<String> getTargets() {
            return targets;
        }

        @Override
        public final Builder targets(Collection<String> targets) {
            this.targets = JobTargetsCopier.copy(targets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(String... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        public final void setTargets(Collection<String> targets) {
            this.targets = JobTargetsCopier.copy(targets);
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final PresignedUrlConfig.Builder getPresignedUrlConfig() {
            return presignedUrlConfig != null ? presignedUrlConfig.toBuilder() : null;
        }

        @Override
        public final Builder presignedUrlConfig(PresignedUrlConfig presignedUrlConfig) {
            this.presignedUrlConfig = presignedUrlConfig;
            return this;
        }

        public final void setPresignedUrlConfig(PresignedUrlConfig.BuilderImpl presignedUrlConfig) {
            this.presignedUrlConfig = presignedUrlConfig != null ? presignedUrlConfig.build() : null;
        }

        public final JobExecutionsRolloutConfig.Builder getJobExecutionsRolloutConfig() {
            return jobExecutionsRolloutConfig != null ? jobExecutionsRolloutConfig.toBuilder() : null;
        }

        @Override
        public final Builder jobExecutionsRolloutConfig(JobExecutionsRolloutConfig jobExecutionsRolloutConfig) {
            this.jobExecutionsRolloutConfig = jobExecutionsRolloutConfig;
            return this;
        }

        public final void setJobExecutionsRolloutConfig(JobExecutionsRolloutConfig.BuilderImpl jobExecutionsRolloutConfig) {
            this.jobExecutionsRolloutConfig = jobExecutionsRolloutConfig != null ? jobExecutionsRolloutConfig.build() : null;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        public final Instant getLastUpdatedAt() {
            return lastUpdatedAt;
        }

        @Override
        public final Builder lastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
            return this;
        }

        public final void setLastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
        }

        public final Instant getCompletedAt() {
            return completedAt;
        }

        @Override
        public final Builder completedAt(Instant completedAt) {
            this.completedAt = completedAt;
            return this;
        }

        public final void setCompletedAt(Instant completedAt) {
            this.completedAt = completedAt;
        }

        public final JobProcessDetails.Builder getJobProcessDetails() {
            return jobProcessDetails != null ? jobProcessDetails.toBuilder() : null;
        }

        @Override
        public final Builder jobProcessDetails(JobProcessDetails jobProcessDetails) {
            this.jobProcessDetails = jobProcessDetails;
            return this;
        }

        public final void setJobProcessDetails(JobProcessDetails.BuilderImpl jobProcessDetails) {
            this.jobProcessDetails = jobProcessDetails != null ? jobProcessDetails.build() : null;
        }

        @Override
        public Job build() {
            return new Job(this);
        }
    }
}
