/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iot.model;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.services.iot.transform.NonCompliantResourceMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the resource that was non-compliant with the audit check.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NonCompliantResource implements StructuredPojo,
        ToCopyableBuilder<NonCompliantResource.Builder, NonCompliantResource> {
    private final String resourceType;

    private final ResourceIdentifier resourceIdentifier;

    private final Map<String, String> additionalInfo;

    private NonCompliantResource(BuilderImpl builder) {
        this.resourceType = builder.resourceType;
        this.resourceIdentifier = builder.resourceIdentifier;
        this.additionalInfo = builder.additionalInfo;
    }

    /**
     * <p>
     * The type of the non-compliant resource.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of the non-compliant resource.
     * @see ResourceType
     */
    public ResourceType resourceType() {
        return ResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The type of the non-compliant resource.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of the non-compliant resource.
     * @see ResourceType
     */
    public String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * <p>
     * Information identifying the non-compliant resource.
     * </p>
     * 
     * @return Information identifying the non-compliant resource.
     */
    public ResourceIdentifier resourceIdentifier() {
        return resourceIdentifier;
    }

    /**
     * <p>
     * Additional information about the non-compliant resource.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Additional information about the non-compliant resource.
     */
    public Map<String, String> additionalInfo() {
        return additionalInfo;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resourceIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(additionalInfo());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NonCompliantResource)) {
            return false;
        }
        NonCompliantResource other = (NonCompliantResource) obj;
        return Objects.equals(resourceTypeAsString(), other.resourceTypeAsString())
                && Objects.equals(resourceIdentifier(), other.resourceIdentifier())
                && Objects.equals(additionalInfo(), other.additionalInfo());
    }

    @Override
    public String toString() {
        return ToString.builder("NonCompliantResource").add("ResourceType", resourceTypeAsString())
                .add("ResourceIdentifier", resourceIdentifier()).add("AdditionalInfo", additionalInfo()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "resourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "resourceIdentifier":
            return Optional.ofNullable(clazz.cast(resourceIdentifier()));
        case "additionalInfo":
            return Optional.ofNullable(clazz.cast(additionalInfo()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        NonCompliantResourceMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, NonCompliantResource> {
        /**
         * <p>
         * The type of the non-compliant resource.
         * </p>
         * 
         * @param resourceType
         *        The type of the non-compliant resource.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The type of the non-compliant resource.
         * </p>
         * 
         * @param resourceType
         *        The type of the non-compliant resource.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(ResourceType resourceType);

        /**
         * <p>
         * Information identifying the non-compliant resource.
         * </p>
         * 
         * @param resourceIdentifier
         *        Information identifying the non-compliant resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceIdentifier(ResourceIdentifier resourceIdentifier);

        /**
         * <p>
         * Information identifying the non-compliant resource.
         * </p>
         * This is a convenience that creates an instance of the {@link ResourceIdentifier.Builder} avoiding the need to
         * create one manually via {@link ResourceIdentifier#builder()}.
         *
         * When the {@link Consumer} completes, {@link ResourceIdentifier.Builder#build()} is called immediately and its
         * result is passed to {@link #resourceIdentifier(ResourceIdentifier)}.
         * 
         * @param resourceIdentifier
         *        a consumer that will call methods on {@link ResourceIdentifier.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceIdentifier(ResourceIdentifier)
         */
        default Builder resourceIdentifier(Consumer<ResourceIdentifier.Builder> resourceIdentifier) {
            return resourceIdentifier(ResourceIdentifier.builder().applyMutation(resourceIdentifier).build());
        }

        /**
         * <p>
         * Additional information about the non-compliant resource.
         * </p>
         * 
         * @param additionalInfo
         *        Additional information about the non-compliant resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalInfo(Map<String, String> additionalInfo);
    }

    static final class BuilderImpl implements Builder {
        private String resourceType;

        private ResourceIdentifier resourceIdentifier;

        private Map<String, String> additionalInfo = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(NonCompliantResource model) {
            resourceType(model.resourceType);
            resourceIdentifier(model.resourceIdentifier);
            additionalInfo(model.additionalInfo);
        }

        public final String getResourceType() {
            return resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(ResourceType resourceType) {
            this.resourceType(resourceType.toString());
            return this;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        public final ResourceIdentifier.Builder getResourceIdentifier() {
            return resourceIdentifier != null ? resourceIdentifier.toBuilder() : null;
        }

        @Override
        public final Builder resourceIdentifier(ResourceIdentifier resourceIdentifier) {
            this.resourceIdentifier = resourceIdentifier;
            return this;
        }

        public final void setResourceIdentifier(ResourceIdentifier.BuilderImpl resourceIdentifier) {
            this.resourceIdentifier = resourceIdentifier != null ? resourceIdentifier.build() : null;
        }

        public final Map<String, String> getAdditionalInfo() {
            return additionalInfo;
        }

        @Override
        public final Builder additionalInfo(Map<String, String> additionalInfo) {
            this.additionalInfo = StringMapCopier.copy(additionalInfo);
            return this;
        }

        public final void setAdditionalInfo(Map<String, String> additionalInfo) {
            this.additionalInfo = StringMapCopier.copy(additionalInfo);
        }

        @Override
        public NonCompliantResource build() {
            return new NonCompliantResource(this);
        }
    }
}
