/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iot.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.services.iot.transform.TopicRulePayloadMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a rule.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TopicRulePayload implements StructuredPojo, ToCopyableBuilder<TopicRulePayload.Builder, TopicRulePayload> {
    private final String sql;

    private final String description;

    private final List<Action> actions;

    private final Boolean ruleDisabled;

    private final String awsIotSqlVersion;

    private final Action errorAction;

    private TopicRulePayload(BuilderImpl builder) {
        this.sql = builder.sql;
        this.description = builder.description;
        this.actions = builder.actions;
        this.ruleDisabled = builder.ruleDisabled;
        this.awsIotSqlVersion = builder.awsIotSqlVersion;
        this.errorAction = builder.errorAction;
    }

    /**
     * <p>
     * The SQL statement used to query the topic. For more information, see <a
     * href="http://docs.aws.amazon.com/iot/latest/developerguide/iot-rules.html#aws-iot-sql-reference">AWS IoT SQL
     * Reference</a> in the <i>AWS IoT Developer Guide</i>.
     * </p>
     * 
     * @return The SQL statement used to query the topic. For more information, see <a
     *         href="http://docs.aws.amazon.com/iot/latest/developerguide/iot-rules.html#aws-iot-sql-reference">AWS IoT
     *         SQL Reference</a> in the <i>AWS IoT Developer Guide</i>.
     */
    public String sql() {
        return sql;
    }

    /**
     * <p>
     * The description of the rule.
     * </p>
     * 
     * @return The description of the rule.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The actions associated with the rule.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The actions associated with the rule.
     */
    public List<Action> actions() {
        return actions;
    }

    /**
     * <p>
     * Specifies whether the rule is disabled.
     * </p>
     * 
     * @return Specifies whether the rule is disabled.
     */
    public Boolean ruleDisabled() {
        return ruleDisabled;
    }

    /**
     * <p>
     * The version of the SQL rules engine to use when evaluating the rule.
     * </p>
     * 
     * @return The version of the SQL rules engine to use when evaluating the rule.
     */
    public String awsIotSqlVersion() {
        return awsIotSqlVersion;
    }

    /**
     * <p>
     * The action to take when an error occurs.
     * </p>
     * 
     * @return The action to take when an error occurs.
     */
    public Action errorAction() {
        return errorAction;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(sql());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(actions());
        hashCode = 31 * hashCode + Objects.hashCode(ruleDisabled());
        hashCode = 31 * hashCode + Objects.hashCode(awsIotSqlVersion());
        hashCode = 31 * hashCode + Objects.hashCode(errorAction());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TopicRulePayload)) {
            return false;
        }
        TopicRulePayload other = (TopicRulePayload) obj;
        return Objects.equals(sql(), other.sql()) && Objects.equals(description(), other.description())
                && Objects.equals(actions(), other.actions()) && Objects.equals(ruleDisabled(), other.ruleDisabled())
                && Objects.equals(awsIotSqlVersion(), other.awsIotSqlVersion())
                && Objects.equals(errorAction(), other.errorAction());
    }

    @Override
    public String toString() {
        return ToString.builder("TopicRulePayload").add("Sql", sql()).add("Description", description()).add("Actions", actions())
                .add("RuleDisabled", ruleDisabled()).add("AwsIotSqlVersion", awsIotSqlVersion())
                .add("ErrorAction", errorAction()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "sql":
            return Optional.ofNullable(clazz.cast(sql()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "actions":
            return Optional.ofNullable(clazz.cast(actions()));
        case "ruleDisabled":
            return Optional.ofNullable(clazz.cast(ruleDisabled()));
        case "awsIotSqlVersion":
            return Optional.ofNullable(clazz.cast(awsIotSqlVersion()));
        case "errorAction":
            return Optional.ofNullable(clazz.cast(errorAction()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        TopicRulePayloadMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, TopicRulePayload> {
        /**
         * <p>
         * The SQL statement used to query the topic. For more information, see <a
         * href="http://docs.aws.amazon.com/iot/latest/developerguide/iot-rules.html#aws-iot-sql-reference">AWS IoT SQL
         * Reference</a> in the <i>AWS IoT Developer Guide</i>.
         * </p>
         * 
         * @param sql
         *        The SQL statement used to query the topic. For more information, see <a
         *        href="http://docs.aws.amazon.com/iot/latest/developerguide/iot-rules.html#aws-iot-sql-reference">AWS
         *        IoT SQL Reference</a> in the <i>AWS IoT Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sql(String sql);

        /**
         * <p>
         * The description of the rule.
         * </p>
         * 
         * @param description
         *        The description of the rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The actions associated with the rule.
         * </p>
         * 
         * @param actions
         *        The actions associated with the rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actions(Collection<Action> actions);

        /**
         * <p>
         * The actions associated with the rule.
         * </p>
         * 
         * @param actions
         *        The actions associated with the rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actions(Action... actions);

        /**
         * <p>
         * The actions associated with the rule.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Action>.Builder} avoiding the need to
         * create one manually via {@link List<Action>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Action>.Builder#build()} is called immediately and its
         * result is passed to {@link #actions(List<Action>)}.
         * 
         * @param actions
         *        a consumer that will call methods on {@link List<Action>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #actions(List<Action>)
         */
        Builder actions(Consumer<Action.Builder>... actions);

        /**
         * <p>
         * Specifies whether the rule is disabled.
         * </p>
         * 
         * @param ruleDisabled
         *        Specifies whether the rule is disabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleDisabled(Boolean ruleDisabled);

        /**
         * <p>
         * The version of the SQL rules engine to use when evaluating the rule.
         * </p>
         * 
         * @param awsIotSqlVersion
         *        The version of the SQL rules engine to use when evaluating the rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awsIotSqlVersion(String awsIotSqlVersion);

        /**
         * <p>
         * The action to take when an error occurs.
         * </p>
         * 
         * @param errorAction
         *        The action to take when an error occurs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorAction(Action errorAction);

        /**
         * <p>
         * The action to take when an error occurs.
         * </p>
         * This is a convenience that creates an instance of the {@link Action.Builder} avoiding the need to create one
         * manually via {@link Action#builder()}.
         *
         * When the {@link Consumer} completes, {@link Action.Builder#build()} is called immediately and its result is
         * passed to {@link #errorAction(Action)}.
         * 
         * @param errorAction
         *        a consumer that will call methods on {@link Action.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #errorAction(Action)
         */
        default Builder errorAction(Consumer<Action.Builder> errorAction) {
            return errorAction(Action.builder().applyMutation(errorAction).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String sql;

        private String description;

        private List<Action> actions = DefaultSdkAutoConstructList.getInstance();

        private Boolean ruleDisabled;

        private String awsIotSqlVersion;

        private Action errorAction;

        private BuilderImpl() {
        }

        private BuilderImpl(TopicRulePayload model) {
            sql(model.sql);
            description(model.description);
            actions(model.actions);
            ruleDisabled(model.ruleDisabled);
            awsIotSqlVersion(model.awsIotSqlVersion);
            errorAction(model.errorAction);
        }

        public final String getSql() {
            return sql;
        }

        @Override
        public final Builder sql(String sql) {
            this.sql = sql;
            return this;
        }

        public final void setSql(String sql) {
            this.sql = sql;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<Action.Builder> getActions() {
            return actions != null ? actions.stream().map(Action::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder actions(Collection<Action> actions) {
            this.actions = ActionListCopier.copy(actions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actions(Action... actions) {
            actions(Arrays.asList(actions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actions(Consumer<Action.Builder>... actions) {
            actions(Stream.of(actions).map(c -> Action.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setActions(Collection<Action.BuilderImpl> actions) {
            this.actions = ActionListCopier.copyFromBuilder(actions);
        }

        public final Boolean getRuleDisabled() {
            return ruleDisabled;
        }

        @Override
        public final Builder ruleDisabled(Boolean ruleDisabled) {
            this.ruleDisabled = ruleDisabled;
            return this;
        }

        public final void setRuleDisabled(Boolean ruleDisabled) {
            this.ruleDisabled = ruleDisabled;
        }

        public final String getAwsIotSqlVersion() {
            return awsIotSqlVersion;
        }

        @Override
        public final Builder awsIotSqlVersion(String awsIotSqlVersion) {
            this.awsIotSqlVersion = awsIotSqlVersion;
            return this;
        }

        public final void setAwsIotSqlVersion(String awsIotSqlVersion) {
            this.awsIotSqlVersion = awsIotSqlVersion;
        }

        public final Action.Builder getErrorAction() {
            return errorAction != null ? errorAction.toBuilder() : null;
        }

        @Override
        public final Builder errorAction(Action errorAction) {
            this.errorAction = errorAction;
            return this;
        }

        public final void setErrorAction(Action.BuilderImpl errorAction) {
            this.errorAction = errorAction != null ? errorAction.build() : null;
        }

        @Override
        public TopicRulePayload build() {
            return new TopicRulePayload(this);
        }
    }
}
