/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iot.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpdateStreamRequest extends IotRequest implements
        ToCopyableBuilder<UpdateStreamRequest.Builder, UpdateStreamRequest> {
    private final String streamId;

    private final String description;

    private final List<StreamFile> files;

    private final String roleArn;

    private UpdateStreamRequest(BuilderImpl builder) {
        super(builder);
        this.streamId = builder.streamId;
        this.description = builder.description;
        this.files = builder.files;
        this.roleArn = builder.roleArn;
    }

    /**
     * <p>
     * The stream ID.
     * </p>
     * 
     * @return The stream ID.
     */
    public String streamId() {
        return streamId;
    }

    /**
     * <p>
     * The description of the stream.
     * </p>
     * 
     * @return The description of the stream.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The files associated with the stream.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The files associated with the stream.
     */
    public List<StreamFile> files() {
        return files;
    }

    /**
     * <p>
     * An IAM role that allows the IoT service principal assumes to access your S3 files.
     * </p>
     * 
     * @return An IAM role that allows the IoT service principal assumes to access your S3 files.
     */
    public String roleArn() {
        return roleArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(streamId());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(files());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateStreamRequest)) {
            return false;
        }
        UpdateStreamRequest other = (UpdateStreamRequest) obj;
        return Objects.equals(streamId(), other.streamId()) && Objects.equals(description(), other.description())
                && Objects.equals(files(), other.files()) && Objects.equals(roleArn(), other.roleArn());
    }

    @Override
    public String toString() {
        return ToString.builder("UpdateStreamRequest").add("StreamId", streamId()).add("Description", description())
                .add("Files", files()).add("RoleArn", roleArn()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "streamId":
            return Optional.ofNullable(clazz.cast(streamId()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "files":
            return Optional.ofNullable(clazz.cast(files()));
        case "roleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends IotRequest.Builder, CopyableBuilder<Builder, UpdateStreamRequest> {
        /**
         * <p>
         * The stream ID.
         * </p>
         * 
         * @param streamId
         *        The stream ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamId(String streamId);

        /**
         * <p>
         * The description of the stream.
         * </p>
         * 
         * @param description
         *        The description of the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The files associated with the stream.
         * </p>
         * 
         * @param files
         *        The files associated with the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder files(Collection<StreamFile> files);

        /**
         * <p>
         * The files associated with the stream.
         * </p>
         * 
         * @param files
         *        The files associated with the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder files(StreamFile... files);

        /**
         * <p>
         * The files associated with the stream.
         * </p>
         * This is a convenience that creates an instance of the {@link List<StreamFile>.Builder} avoiding the need to
         * create one manually via {@link List<StreamFile>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<StreamFile>.Builder#build()} is called immediately and its
         * result is passed to {@link #files(List<StreamFile>)}.
         * 
         * @param files
         *        a consumer that will call methods on {@link List<StreamFile>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #files(List<StreamFile>)
         */
        Builder files(Consumer<StreamFile.Builder>... files);

        /**
         * <p>
         * An IAM role that allows the IoT service principal assumes to access your S3 files.
         * </p>
         * 
         * @param roleArn
         *        An IAM role that allows the IoT service principal assumes to access your S3 files.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends IotRequest.BuilderImpl implements Builder {
        private String streamId;

        private String description;

        private List<StreamFile> files = DefaultSdkAutoConstructList.getInstance();

        private String roleArn;

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateStreamRequest model) {
            super(model);
            streamId(model.streamId);
            description(model.description);
            files(model.files);
            roleArn(model.roleArn);
        }

        public final String getStreamId() {
            return streamId;
        }

        @Override
        public final Builder streamId(String streamId) {
            this.streamId = streamId;
            return this;
        }

        public final void setStreamId(String streamId) {
            this.streamId = streamId;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<StreamFile.Builder> getFiles() {
            return files != null ? files.stream().map(StreamFile::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder files(Collection<StreamFile> files) {
            this.files = StreamFilesCopier.copy(files);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder files(StreamFile... files) {
            files(Arrays.asList(files));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder files(Consumer<StreamFile.Builder>... files) {
            files(Stream.of(files).map(c -> StreamFile.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setFiles(Collection<StreamFile.BuilderImpl> files) {
            this.files = StreamFilesCopier.copyFromBuilder(files);
        }

        public final String getRoleArn() {
            return roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public UpdateStreamRequest build() {
            return new UpdateStreamRequest(this);
        }
    }
}
