/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iot.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpdateAuthorizerRequest extends IotRequest implements
        ToCopyableBuilder<UpdateAuthorizerRequest.Builder, UpdateAuthorizerRequest> {
    private static final SdkField<String> AUTHORIZER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(UpdateAuthorizerRequest::authorizerName)).setter(setter(Builder::authorizerName))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("authorizerName").build()).build();

    private static final SdkField<String> AUTHORIZER_FUNCTION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(UpdateAuthorizerRequest::authorizerFunctionArn)).setter(setter(Builder::authorizerFunctionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authorizerFunctionArn").build())
            .build();

    private static final SdkField<String> TOKEN_KEY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(UpdateAuthorizerRequest::tokenKeyName)).setter(setter(Builder::tokenKeyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tokenKeyName").build()).build();

    private static final SdkField<Map<String, String>> TOKEN_SIGNING_PUBLIC_KEYS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(UpdateAuthorizerRequest::tokenSigningPublicKeys))
            .setter(setter(Builder::tokenSigningPublicKeys))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tokenSigningPublicKeys").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(UpdateAuthorizerRequest::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AUTHORIZER_NAME_FIELD,
            AUTHORIZER_FUNCTION_ARN_FIELD, TOKEN_KEY_NAME_FIELD, TOKEN_SIGNING_PUBLIC_KEYS_FIELD, STATUS_FIELD));

    private final String authorizerName;

    private final String authorizerFunctionArn;

    private final String tokenKeyName;

    private final Map<String, String> tokenSigningPublicKeys;

    private final String status;

    private UpdateAuthorizerRequest(BuilderImpl builder) {
        super(builder);
        this.authorizerName = builder.authorizerName;
        this.authorizerFunctionArn = builder.authorizerFunctionArn;
        this.tokenKeyName = builder.tokenKeyName;
        this.tokenSigningPublicKeys = builder.tokenSigningPublicKeys;
        this.status = builder.status;
    }

    /**
     * <p>
     * The authorizer name.
     * </p>
     * 
     * @return The authorizer name.
     */
    public String authorizerName() {
        return authorizerName;
    }

    /**
     * <p>
     * The ARN of the authorizer's Lambda function.
     * </p>
     * 
     * @return The ARN of the authorizer's Lambda function.
     */
    public String authorizerFunctionArn() {
        return authorizerFunctionArn;
    }

    /**
     * <p>
     * The key used to extract the token from the HTTP headers.
     * </p>
     * 
     * @return The key used to extract the token from the HTTP headers.
     */
    public String tokenKeyName() {
        return tokenKeyName;
    }

    /**
     * Returns true if the TokenSigningPublicKeys property was specified by the sender (it may be empty), or false if
     * the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasTokenSigningPublicKeys() {
        return tokenSigningPublicKeys != null && !(tokenSigningPublicKeys instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The public keys used to verify the token signature.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTokenSigningPublicKeys()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The public keys used to verify the token signature.
     */
    public Map<String, String> tokenSigningPublicKeys() {
        return tokenSigningPublicKeys;
    }

    /**
     * <p>
     * The status of the update authorizer request.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AuthorizerStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the update authorizer request.
     * @see AuthorizerStatus
     */
    public AuthorizerStatus status() {
        return AuthorizerStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the update authorizer request.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AuthorizerStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the update authorizer request.
     * @see AuthorizerStatus
     */
    public String statusAsString() {
        return status;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(authorizerName());
        hashCode = 31 * hashCode + Objects.hashCode(authorizerFunctionArn());
        hashCode = 31 * hashCode + Objects.hashCode(tokenKeyName());
        hashCode = 31 * hashCode + Objects.hashCode(tokenSigningPublicKeys());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateAuthorizerRequest)) {
            return false;
        }
        UpdateAuthorizerRequest other = (UpdateAuthorizerRequest) obj;
        return Objects.equals(authorizerName(), other.authorizerName())
                && Objects.equals(authorizerFunctionArn(), other.authorizerFunctionArn())
                && Objects.equals(tokenKeyName(), other.tokenKeyName())
                && Objects.equals(tokenSigningPublicKeys(), other.tokenSigningPublicKeys())
                && Objects.equals(statusAsString(), other.statusAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("UpdateAuthorizerRequest").add("AuthorizerName", authorizerName())
                .add("AuthorizerFunctionArn", authorizerFunctionArn()).add("TokenKeyName", tokenKeyName())
                .add("TokenSigningPublicKeys", tokenSigningPublicKeys()).add("Status", statusAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "authorizerName":
            return Optional.ofNullable(clazz.cast(authorizerName()));
        case "authorizerFunctionArn":
            return Optional.ofNullable(clazz.cast(authorizerFunctionArn()));
        case "tokenKeyName":
            return Optional.ofNullable(clazz.cast(tokenKeyName()));
        case "tokenSigningPublicKeys":
            return Optional.ofNullable(clazz.cast(tokenSigningPublicKeys()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UpdateAuthorizerRequest, T> g) {
        return obj -> g.apply((UpdateAuthorizerRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends IotRequest.Builder, SdkPojo, CopyableBuilder<Builder, UpdateAuthorizerRequest> {
        /**
         * <p>
         * The authorizer name.
         * </p>
         * 
         * @param authorizerName
         *        The authorizer name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorizerName(String authorizerName);

        /**
         * <p>
         * The ARN of the authorizer's Lambda function.
         * </p>
         * 
         * @param authorizerFunctionArn
         *        The ARN of the authorizer's Lambda function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorizerFunctionArn(String authorizerFunctionArn);

        /**
         * <p>
         * The key used to extract the token from the HTTP headers.
         * </p>
         * 
         * @param tokenKeyName
         *        The key used to extract the token from the HTTP headers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tokenKeyName(String tokenKeyName);

        /**
         * <p>
         * The public keys used to verify the token signature.
         * </p>
         * 
         * @param tokenSigningPublicKeys
         *        The public keys used to verify the token signature.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tokenSigningPublicKeys(Map<String, String> tokenSigningPublicKeys);

        /**
         * <p>
         * The status of the update authorizer request.
         * </p>
         * 
         * @param status
         *        The status of the update authorizer request.
         * @see AuthorizerStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AuthorizerStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the update authorizer request.
         * </p>
         * 
         * @param status
         *        The status of the update authorizer request.
         * @see AuthorizerStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AuthorizerStatus
         */
        Builder status(AuthorizerStatus status);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends IotRequest.BuilderImpl implements Builder {
        private String authorizerName;

        private String authorizerFunctionArn;

        private String tokenKeyName;

        private Map<String, String> tokenSigningPublicKeys = DefaultSdkAutoConstructMap.getInstance();

        private String status;

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateAuthorizerRequest model) {
            super(model);
            authorizerName(model.authorizerName);
            authorizerFunctionArn(model.authorizerFunctionArn);
            tokenKeyName(model.tokenKeyName);
            tokenSigningPublicKeys(model.tokenSigningPublicKeys);
            status(model.status);
        }

        public final String getAuthorizerName() {
            return authorizerName;
        }

        @Override
        public final Builder authorizerName(String authorizerName) {
            this.authorizerName = authorizerName;
            return this;
        }

        public final void setAuthorizerName(String authorizerName) {
            this.authorizerName = authorizerName;
        }

        public final String getAuthorizerFunctionArn() {
            return authorizerFunctionArn;
        }

        @Override
        public final Builder authorizerFunctionArn(String authorizerFunctionArn) {
            this.authorizerFunctionArn = authorizerFunctionArn;
            return this;
        }

        public final void setAuthorizerFunctionArn(String authorizerFunctionArn) {
            this.authorizerFunctionArn = authorizerFunctionArn;
        }

        public final String getTokenKeyName() {
            return tokenKeyName;
        }

        @Override
        public final Builder tokenKeyName(String tokenKeyName) {
            this.tokenKeyName = tokenKeyName;
            return this;
        }

        public final void setTokenKeyName(String tokenKeyName) {
            this.tokenKeyName = tokenKeyName;
        }

        public final Map<String, String> getTokenSigningPublicKeys() {
            return tokenSigningPublicKeys;
        }

        @Override
        public final Builder tokenSigningPublicKeys(Map<String, String> tokenSigningPublicKeys) {
            this.tokenSigningPublicKeys = PublicKeyMapCopier.copy(tokenSigningPublicKeys);
            return this;
        }

        public final void setTokenSigningPublicKeys(Map<String, String> tokenSigningPublicKeys) {
            this.tokenSigningPublicKeys = PublicKeyMapCopier.copy(tokenSigningPublicKeys);
        }

        public final String getStatusAsString() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(AuthorizerStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public UpdateAuthorizerRequest build() {
            return new UpdateAuthorizerRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
