/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iot.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Send messages to an Amazon Managed Streaming for Apache Kafka (Amazon MSK) or self-managed Apache Kafka cluster.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class KafkaAction implements SdkPojo, Serializable, ToCopyableBuilder<KafkaAction.Builder, KafkaAction> {
    private static final SdkField<String> DESTINATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("destinationArn").getter(getter(KafkaAction::destinationArn)).setter(setter(Builder::destinationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("destinationArn").build()).build();

    private static final SdkField<String> TOPIC_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("topic")
            .getter(getter(KafkaAction::topic)).setter(setter(Builder::topic))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("topic").build()).build();

    private static final SdkField<String> KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("key")
            .getter(getter(KafkaAction::key)).setter(setter(Builder::key))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("key").build()).build();

    private static final SdkField<String> PARTITION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("partition").getter(getter(KafkaAction::partition)).setter(setter(Builder::partition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("partition").build()).build();

    private static final SdkField<Map<String, String>> CLIENT_PROPERTIES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("clientProperties")
            .getter(getter(KafkaAction::clientProperties))
            .setter(setter(Builder::clientProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clientProperties").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DESTINATION_ARN_FIELD,
            TOPIC_FIELD, KEY_FIELD, PARTITION_FIELD, CLIENT_PROPERTIES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String destinationArn;

    private final String topic;

    private final String key;

    private final String partition;

    private final Map<String, String> clientProperties;

    private KafkaAction(BuilderImpl builder) {
        this.destinationArn = builder.destinationArn;
        this.topic = builder.topic;
        this.key = builder.key;
        this.partition = builder.partition;
        this.clientProperties = builder.clientProperties;
    }

    /**
     * <p>
     * The ARN of Kafka action's VPC <code>TopicRuleDestination</code>.
     * </p>
     * 
     * @return The ARN of Kafka action's VPC <code>TopicRuleDestination</code>.
     */
    public String destinationArn() {
        return destinationArn;
    }

    /**
     * <p>
     * The Kafka topic for messages to be sent to the Kafka broker.
     * </p>
     * 
     * @return The Kafka topic for messages to be sent to the Kafka broker.
     */
    public String topic() {
        return topic;
    }

    /**
     * <p>
     * The Kafka message key.
     * </p>
     * 
     * @return The Kafka message key.
     */
    public String key() {
        return key;
    }

    /**
     * <p>
     * The Kafka message partition.
     * </p>
     * 
     * @return The Kafka message partition.
     */
    public String partition() {
        return partition;
    }

    /**
     * Returns true if the ClientProperties property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasClientProperties() {
        return clientProperties != null && !(clientProperties instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Properties of the Apache Kafka producer client.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasClientProperties()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Properties of the Apache Kafka producer client.
     */
    public Map<String, String> clientProperties() {
        return clientProperties;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(destinationArn());
        hashCode = 31 * hashCode + Objects.hashCode(topic());
        hashCode = 31 * hashCode + Objects.hashCode(key());
        hashCode = 31 * hashCode + Objects.hashCode(partition());
        hashCode = 31 * hashCode + Objects.hashCode(hasClientProperties() ? clientProperties() : null);
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof KafkaAction)) {
            return false;
        }
        KafkaAction other = (KafkaAction) obj;
        return Objects.equals(destinationArn(), other.destinationArn()) && Objects.equals(topic(), other.topic())
                && Objects.equals(key(), other.key()) && Objects.equals(partition(), other.partition())
                && hasClientProperties() == other.hasClientProperties()
                && Objects.equals(clientProperties(), other.clientProperties());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("KafkaAction").add("DestinationArn", destinationArn()).add("Topic", topic()).add("Key", key())
                .add("Partition", partition()).add("ClientProperties", hasClientProperties() ? clientProperties() : null).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "destinationArn":
            return Optional.ofNullable(clazz.cast(destinationArn()));
        case "topic":
            return Optional.ofNullable(clazz.cast(topic()));
        case "key":
            return Optional.ofNullable(clazz.cast(key()));
        case "partition":
            return Optional.ofNullable(clazz.cast(partition()));
        case "clientProperties":
            return Optional.ofNullable(clazz.cast(clientProperties()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<KafkaAction, T> g) {
        return obj -> g.apply((KafkaAction) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, KafkaAction> {
        /**
         * <p>
         * The ARN of Kafka action's VPC <code>TopicRuleDestination</code>.
         * </p>
         * 
         * @param destinationArn
         *        The ARN of Kafka action's VPC <code>TopicRuleDestination</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationArn(String destinationArn);

        /**
         * <p>
         * The Kafka topic for messages to be sent to the Kafka broker.
         * </p>
         * 
         * @param topic
         *        The Kafka topic for messages to be sent to the Kafka broker.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder topic(String topic);

        /**
         * <p>
         * The Kafka message key.
         * </p>
         * 
         * @param key
         *        The Kafka message key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder key(String key);

        /**
         * <p>
         * The Kafka message partition.
         * </p>
         * 
         * @param partition
         *        The Kafka message partition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partition(String partition);

        /**
         * <p>
         * Properties of the Apache Kafka producer client.
         * </p>
         * 
         * @param clientProperties
         *        Properties of the Apache Kafka producer client.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientProperties(Map<String, String> clientProperties);
    }

    static final class BuilderImpl implements Builder {
        private String destinationArn;

        private String topic;

        private String key;

        private String partition;

        private Map<String, String> clientProperties = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(KafkaAction model) {
            destinationArn(model.destinationArn);
            topic(model.topic);
            key(model.key);
            partition(model.partition);
            clientProperties(model.clientProperties);
        }

        public final String getDestinationArn() {
            return destinationArn;
        }

        @Override
        public final Builder destinationArn(String destinationArn) {
            this.destinationArn = destinationArn;
            return this;
        }

        public final void setDestinationArn(String destinationArn) {
            this.destinationArn = destinationArn;
        }

        public final String getTopic() {
            return topic;
        }

        @Override
        public final Builder topic(String topic) {
            this.topic = topic;
            return this;
        }

        public final void setTopic(String topic) {
            this.topic = topic;
        }

        public final String getKey() {
            return key;
        }

        @Override
        public final Builder key(String key) {
            this.key = key;
            return this;
        }

        public final void setKey(String key) {
            this.key = key;
        }

        public final String getPartition() {
            return partition;
        }

        @Override
        public final Builder partition(String partition) {
            this.partition = partition;
            return this;
        }

        public final void setPartition(String partition) {
            this.partition = partition;
        }

        public final Map<String, String> getClientProperties() {
            if (clientProperties instanceof SdkAutoConstructMap) {
                return null;
            }
            return clientProperties;
        }

        @Override
        public final Builder clientProperties(Map<String, String> clientProperties) {
            this.clientProperties = ClientPropertiesCopier.copy(clientProperties);
            return this;
        }

        public final void setClientProperties(Map<String, String> clientProperties) {
            this.clientProperties = ClientPropertiesCopier.copy(clientProperties);
        }

        @Override
        public KafkaAction build() {
            return new KafkaAction(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
