/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iot.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The Timestream rule action writes attributes (measures) from an MQTT message into an Amazon Timestream table. For
 * more information, see the <a
 * href="https://docs.aws.amazon.com/iot/latest/developerguide/timestream-rule-action.html">Timestream</a> topic rule
 * action documentation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TimestreamAction implements SdkPojo, Serializable,
        ToCopyableBuilder<TimestreamAction.Builder, TimestreamAction> {
    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("roleArn").getter(getter(TimestreamAction::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("roleArn").build()).build();

    private static final SdkField<String> DATABASE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("databaseName").getter(getter(TimestreamAction::databaseName)).setter(setter(Builder::databaseName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("databaseName").build()).build();

    private static final SdkField<String> TABLE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("tableName").getter(getter(TimestreamAction::tableName)).setter(setter(Builder::tableName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tableName").build()).build();

    private static final SdkField<List<TimestreamDimension>> DIMENSIONS_FIELD = SdkField
            .<List<TimestreamDimension>> builder(MarshallingType.LIST)
            .memberName("dimensions")
            .getter(getter(TimestreamAction::dimensions))
            .setter(setter(Builder::dimensions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dimensions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<TimestreamDimension> builder(MarshallingType.SDK_POJO)
                                            .constructor(TimestreamDimension::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<TimestreamTimestamp> TIMESTAMP_FIELD = SdkField
            .<TimestreamTimestamp> builder(MarshallingType.SDK_POJO).memberName("timestamp")
            .getter(getter(TimestreamAction::timestamp)).setter(setter(Builder::timestamp))
            .constructor(TimestreamTimestamp::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("timestamp").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ROLE_ARN_FIELD,
            DATABASE_NAME_FIELD, TABLE_NAME_FIELD, DIMENSIONS_FIELD, TIMESTAMP_FIELD));

    private static final long serialVersionUID = 1L;

    private final String roleArn;

    private final String databaseName;

    private final String tableName;

    private final List<TimestreamDimension> dimensions;

    private final TimestreamTimestamp timestamp;

    private TimestreamAction(BuilderImpl builder) {
        this.roleArn = builder.roleArn;
        this.databaseName = builder.databaseName;
        this.tableName = builder.tableName;
        this.dimensions = builder.dimensions;
        this.timestamp = builder.timestamp;
    }

    /**
     * <p>
     * The ARN of the role that grants permission to write to the Amazon Timestream database table.
     * </p>
     * 
     * @return The ARN of the role that grants permission to write to the Amazon Timestream database table.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * The name of an Amazon Timestream database.
     * </p>
     * 
     * @return The name of an Amazon Timestream database.
     */
    public final String databaseName() {
        return databaseName;
    }

    /**
     * <p>
     * The name of the database table into which to write the measure records.
     * </p>
     * 
     * @return The name of the database table into which to write the measure records.
     */
    public final String tableName() {
        return tableName;
    }

    /**
     * Returns true if the Dimensions property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasDimensions() {
        return dimensions != null && !(dimensions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Metadata attributes of the time series that are written in each measure record.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDimensions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Metadata attributes of the time series that are written in each measure record.
     */
    public final List<TimestreamDimension> dimensions() {
        return dimensions;
    }

    /**
     * <p>
     * Specifies an application-defined value to replace the default value assigned to the Timestream record's timestamp
     * in the <code>time</code> column.
     * </p>
     * <p>
     * You can use this property to specify the value and the precision of the Timestream record's timestamp. You can
     * specify a value from the message payload or a value computed by a substitution template.
     * </p>
     * <p>
     * If omitted, the topic rule action assigns the timestamp, in milliseconds, at the time it processed the rule.
     * </p>
     * 
     * @return Specifies an application-defined value to replace the default value assigned to the Timestream record's
     *         timestamp in the <code>time</code> column.</p>
     *         <p>
     *         You can use this property to specify the value and the precision of the Timestream record's timestamp.
     *         You can specify a value from the message payload or a value computed by a substitution template.
     *         </p>
     *         <p>
     *         If omitted, the topic rule action assigns the timestamp, in milliseconds, at the time it processed the
     *         rule.
     */
    public final TimestreamTimestamp timestamp() {
        return timestamp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(databaseName());
        hashCode = 31 * hashCode + Objects.hashCode(tableName());
        hashCode = 31 * hashCode + Objects.hashCode(hasDimensions() ? dimensions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(timestamp());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TimestreamAction)) {
            return false;
        }
        TimestreamAction other = (TimestreamAction) obj;
        return Objects.equals(roleArn(), other.roleArn()) && Objects.equals(databaseName(), other.databaseName())
                && Objects.equals(tableName(), other.tableName()) && hasDimensions() == other.hasDimensions()
                && Objects.equals(dimensions(), other.dimensions()) && Objects.equals(timestamp(), other.timestamp());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TimestreamAction").add("RoleArn", roleArn()).add("DatabaseName", databaseName())
                .add("TableName", tableName()).add("Dimensions", hasDimensions() ? dimensions() : null)
                .add("Timestamp", timestamp()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "roleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "databaseName":
            return Optional.ofNullable(clazz.cast(databaseName()));
        case "tableName":
            return Optional.ofNullable(clazz.cast(tableName()));
        case "dimensions":
            return Optional.ofNullable(clazz.cast(dimensions()));
        case "timestamp":
            return Optional.ofNullable(clazz.cast(timestamp()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TimestreamAction, T> g) {
        return obj -> g.apply((TimestreamAction) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TimestreamAction> {
        /**
         * <p>
         * The ARN of the role that grants permission to write to the Amazon Timestream database table.
         * </p>
         * 
         * @param roleArn
         *        The ARN of the role that grants permission to write to the Amazon Timestream database table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The name of an Amazon Timestream database.
         * </p>
         * 
         * @param databaseName
         *        The name of an Amazon Timestream database.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseName(String databaseName);

        /**
         * <p>
         * The name of the database table into which to write the measure records.
         * </p>
         * 
         * @param tableName
         *        The name of the database table into which to write the measure records.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tableName(String tableName);

        /**
         * <p>
         * Metadata attributes of the time series that are written in each measure record.
         * </p>
         * 
         * @param dimensions
         *        Metadata attributes of the time series that are written in each measure record.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Collection<TimestreamDimension> dimensions);

        /**
         * <p>
         * Metadata attributes of the time series that are written in each measure record.
         * </p>
         * 
         * @param dimensions
         *        Metadata attributes of the time series that are written in each measure record.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(TimestreamDimension... dimensions);

        /**
         * <p>
         * Metadata attributes of the time series that are written in each measure record.
         * </p>
         * This is a convenience that creates an instance of the {@link List<TimestreamDimension>.Builder} avoiding the
         * need to create one manually via {@link List<TimestreamDimension>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<TimestreamDimension>.Builder#build()} is called immediately
         * and its result is passed to {@link #dimensions(List<TimestreamDimension>)}.
         * 
         * @param dimensions
         *        a consumer that will call methods on {@link List<TimestreamDimension>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dimensions(List<TimestreamDimension>)
         */
        Builder dimensions(Consumer<TimestreamDimension.Builder>... dimensions);

        /**
         * <p>
         * Specifies an application-defined value to replace the default value assigned to the Timestream record's
         * timestamp in the <code>time</code> column.
         * </p>
         * <p>
         * You can use this property to specify the value and the precision of the Timestream record's timestamp. You
         * can specify a value from the message payload or a value computed by a substitution template.
         * </p>
         * <p>
         * If omitted, the topic rule action assigns the timestamp, in milliseconds, at the time it processed the rule.
         * </p>
         * 
         * @param timestamp
         *        Specifies an application-defined value to replace the default value assigned to the Timestream
         *        record's timestamp in the <code>time</code> column.</p>
         *        <p>
         *        You can use this property to specify the value and the precision of the Timestream record's timestamp.
         *        You can specify a value from the message payload or a value computed by a substitution template.
         *        </p>
         *        <p>
         *        If omitted, the topic rule action assigns the timestamp, in milliseconds, at the time it processed the
         *        rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamp(TimestreamTimestamp timestamp);

        /**
         * <p>
         * Specifies an application-defined value to replace the default value assigned to the Timestream record's
         * timestamp in the <code>time</code> column.
         * </p>
         * <p>
         * You can use this property to specify the value and the precision of the Timestream record's timestamp. You
         * can specify a value from the message payload or a value computed by a substitution template.
         * </p>
         * <p>
         * If omitted, the topic rule action assigns the timestamp, in milliseconds, at the time it processed the rule.
         * </p>
         * This is a convenience that creates an instance of the {@link TimestreamTimestamp.Builder} avoiding the need
         * to create one manually via {@link TimestreamTimestamp#builder()}.
         *
         * When the {@link Consumer} completes, {@link TimestreamTimestamp.Builder#build()} is called immediately and
         * its result is passed to {@link #timestamp(TimestreamTimestamp)}.
         * 
         * @param timestamp
         *        a consumer that will call methods on {@link TimestreamTimestamp.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #timestamp(TimestreamTimestamp)
         */
        default Builder timestamp(Consumer<TimestreamTimestamp.Builder> timestamp) {
            return timestamp(TimestreamTimestamp.builder().applyMutation(timestamp).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String roleArn;

        private String databaseName;

        private String tableName;

        private List<TimestreamDimension> dimensions = DefaultSdkAutoConstructList.getInstance();

        private TimestreamTimestamp timestamp;

        private BuilderImpl() {
        }

        private BuilderImpl(TimestreamAction model) {
            roleArn(model.roleArn);
            databaseName(model.databaseName);
            tableName(model.tableName);
            dimensions(model.dimensions);
            timestamp(model.timestamp);
        }

        public final String getRoleArn() {
            return roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        public final String getDatabaseName() {
            return databaseName;
        }

        @Override
        public final Builder databaseName(String databaseName) {
            this.databaseName = databaseName;
            return this;
        }

        public final void setDatabaseName(String databaseName) {
            this.databaseName = databaseName;
        }

        public final String getTableName() {
            return tableName;
        }

        @Override
        public final Builder tableName(String tableName) {
            this.tableName = tableName;
            return this;
        }

        public final void setTableName(String tableName) {
            this.tableName = tableName;
        }

        public final List<TimestreamDimension.Builder> getDimensions() {
            List<TimestreamDimension.Builder> result = TimestreamDimensionListCopier.copyToBuilder(this.dimensions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder dimensions(Collection<TimestreamDimension> dimensions) {
            this.dimensions = TimestreamDimensionListCopier.copy(dimensions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(TimestreamDimension... dimensions) {
            dimensions(Arrays.asList(dimensions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(Consumer<TimestreamDimension.Builder>... dimensions) {
            dimensions(Stream.of(dimensions).map(c -> TimestreamDimension.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setDimensions(Collection<TimestreamDimension.BuilderImpl> dimensions) {
            this.dimensions = TimestreamDimensionListCopier.copyFromBuilder(dimensions);
        }

        public final TimestreamTimestamp.Builder getTimestamp() {
            return timestamp != null ? timestamp.toBuilder() : null;
        }

        @Override
        public final Builder timestamp(TimestreamTimestamp timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public final void setTimestamp(TimestreamTimestamp.BuilderImpl timestamp) {
            this.timestamp = timestamp != null ? timestamp.build() : null;
        }

        @Override
        public TimestreamAction build() {
            return new TimestreamAction(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
