/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iot.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the date and time that a job will begin the rollout of the job document to all devices in the target group.
 * Additionally, you can specify the end behavior for each job execution when it reaches the scheduled end time.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SchedulingConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<SchedulingConfig.Builder, SchedulingConfig> {
    private static final SdkField<String> START_TIME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("startTime").getter(getter(SchedulingConfig::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build()).build();

    private static final SdkField<String> END_TIME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("endTime").getter(getter(SchedulingConfig::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endTime").build()).build();

    private static final SdkField<String> END_BEHAVIOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("endBehavior").getter(getter(SchedulingConfig::endBehaviorAsString)).setter(setter(Builder::endBehavior))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endBehavior").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(START_TIME_FIELD,
            END_TIME_FIELD, END_BEHAVIOR_FIELD));

    private static final long serialVersionUID = 1L;

    private final String startTime;

    private final String endTime;

    private final String endBehavior;

    private SchedulingConfig(BuilderImpl builder) {
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.endBehavior = builder.endBehavior;
    }

    /**
     * <p>
     * The time a job will begin rollout of the job document to all devices in the target group for a job. The
     * <code>startTime</code> can be scheduled up to a year in advance and must be scheduled a minimum of thirty minutes
     * from the current time.
     * </p>
     * 
     * @return The time a job will begin rollout of the job document to all devices in the target group for a job. The
     *         <code>startTime</code> can be scheduled up to a year in advance and must be scheduled a minimum of thirty
     *         minutes from the current time.
     */
    public final String startTime() {
        return startTime;
    }

    /**
     * <p>
     * The time a job will stop rollout of the job document to all devices in the target group for a job. The
     * <code>endTime</code> must take place no later than two years from the current time and be scheduled a minimum of
     * thirty minutes from the current time. The minimum duration between <code>startTime</code> and
     * <code>endTime</code> is thirty minutes. The maximum duration between <code>startTime</code> and
     * <code>endTime</code> is two years.
     * </p>
     * 
     * @return The time a job will stop rollout of the job document to all devices in the target group for a job. The
     *         <code>endTime</code> must take place no later than two years from the current time and be scheduled a
     *         minimum of thirty minutes from the current time. The minimum duration between <code>startTime</code> and
     *         <code>endTime</code> is thirty minutes. The maximum duration between <code>startTime</code> and
     *         <code>endTime</code> is two years.
     */
    public final String endTime() {
        return endTime;
    }

    /**
     * <p>
     * Specifies the end behavior for all job executions after a job reaches the selected <code>endTime</code>. If
     * <code>endTime</code> is not selected when creating the job, then <code>endBehavior</code> does not apply.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #endBehavior} will
     * return {@link JobEndBehavior#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #endBehaviorAsString}.
     * </p>
     * 
     * @return Specifies the end behavior for all job executions after a job reaches the selected <code>endTime</code>.
     *         If <code>endTime</code> is not selected when creating the job, then <code>endBehavior</code> does not
     *         apply.
     * @see JobEndBehavior
     */
    public final JobEndBehavior endBehavior() {
        return JobEndBehavior.fromValue(endBehavior);
    }

    /**
     * <p>
     * Specifies the end behavior for all job executions after a job reaches the selected <code>endTime</code>. If
     * <code>endTime</code> is not selected when creating the job, then <code>endBehavior</code> does not apply.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #endBehavior} will
     * return {@link JobEndBehavior#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #endBehaviorAsString}.
     * </p>
     * 
     * @return Specifies the end behavior for all job executions after a job reaches the selected <code>endTime</code>.
     *         If <code>endTime</code> is not selected when creating the job, then <code>endBehavior</code> does not
     *         apply.
     * @see JobEndBehavior
     */
    public final String endBehaviorAsString() {
        return endBehavior;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(endBehaviorAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SchedulingConfig)) {
            return false;
        }
        SchedulingConfig other = (SchedulingConfig) obj;
        return Objects.equals(startTime(), other.startTime()) && Objects.equals(endTime(), other.endTime())
                && Objects.equals(endBehaviorAsString(), other.endBehaviorAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SchedulingConfig").add("StartTime", startTime()).add("EndTime", endTime())
                .add("EndBehavior", endBehaviorAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "endTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "endBehavior":
            return Optional.ofNullable(clazz.cast(endBehaviorAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SchedulingConfig, T> g) {
        return obj -> g.apply((SchedulingConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SchedulingConfig> {
        /**
         * <p>
         * The time a job will begin rollout of the job document to all devices in the target group for a job. The
         * <code>startTime</code> can be scheduled up to a year in advance and must be scheduled a minimum of thirty
         * minutes from the current time.
         * </p>
         * 
         * @param startTime
         *        The time a job will begin rollout of the job document to all devices in the target group for a job.
         *        The <code>startTime</code> can be scheduled up to a year in advance and must be scheduled a minimum of
         *        thirty minutes from the current time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(String startTime);

        /**
         * <p>
         * The time a job will stop rollout of the job document to all devices in the target group for a job. The
         * <code>endTime</code> must take place no later than two years from the current time and be scheduled a minimum
         * of thirty minutes from the current time. The minimum duration between <code>startTime</code> and
         * <code>endTime</code> is thirty minutes. The maximum duration between <code>startTime</code> and
         * <code>endTime</code> is two years.
         * </p>
         * 
         * @param endTime
         *        The time a job will stop rollout of the job document to all devices in the target group for a job. The
         *        <code>endTime</code> must take place no later than two years from the current time and be scheduled a
         *        minimum of thirty minutes from the current time. The minimum duration between <code>startTime</code>
         *        and <code>endTime</code> is thirty minutes. The maximum duration between <code>startTime</code> and
         *        <code>endTime</code> is two years.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(String endTime);

        /**
         * <p>
         * Specifies the end behavior for all job executions after a job reaches the selected <code>endTime</code>. If
         * <code>endTime</code> is not selected when creating the job, then <code>endBehavior</code> does not apply.
         * </p>
         * 
         * @param endBehavior
         *        Specifies the end behavior for all job executions after a job reaches the selected
         *        <code>endTime</code>. If <code>endTime</code> is not selected when creating the job, then
         *        <code>endBehavior</code> does not apply.
         * @see JobEndBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobEndBehavior
         */
        Builder endBehavior(String endBehavior);

        /**
         * <p>
         * Specifies the end behavior for all job executions after a job reaches the selected <code>endTime</code>. If
         * <code>endTime</code> is not selected when creating the job, then <code>endBehavior</code> does not apply.
         * </p>
         * 
         * @param endBehavior
         *        Specifies the end behavior for all job executions after a job reaches the selected
         *        <code>endTime</code>. If <code>endTime</code> is not selected when creating the job, then
         *        <code>endBehavior</code> does not apply.
         * @see JobEndBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobEndBehavior
         */
        Builder endBehavior(JobEndBehavior endBehavior);
    }

    static final class BuilderImpl implements Builder {
        private String startTime;

        private String endTime;

        private String endBehavior;

        private BuilderImpl() {
        }

        private BuilderImpl(SchedulingConfig model) {
            startTime(model.startTime);
            endTime(model.endTime);
            endBehavior(model.endBehavior);
        }

        public final String getStartTime() {
            return startTime;
        }

        public final void setStartTime(String startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(String startTime) {
            this.startTime = startTime;
            return this;
        }

        public final String getEndTime() {
            return endTime;
        }

        public final void setEndTime(String endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(String endTime) {
            this.endTime = endTime;
            return this;
        }

        public final String getEndBehavior() {
            return endBehavior;
        }

        public final void setEndBehavior(String endBehavior) {
            this.endBehavior = endBehavior;
        }

        @Override
        public final Builder endBehavior(String endBehavior) {
            this.endBehavior = endBehavior;
            return this;
        }

        @Override
        public final Builder endBehavior(JobEndBehavior endBehavior) {
            this.endBehavior(endBehavior == null ? null : endBehavior.toString());
            return this;
        }

        @Override
        public SchedulingConfig build() {
            return new SchedulingConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
