/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iot.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpdateAuthorizerRequest extends IotRequest implements
        ToCopyableBuilder<UpdateAuthorizerRequest.Builder, UpdateAuthorizerRequest> {
    private static final SdkField<String> AUTHORIZER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("authorizerName").getter(getter(UpdateAuthorizerRequest::authorizerName))
            .setter(setter(Builder::authorizerName))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("authorizerName").build()).build();

    private static final SdkField<String> AUTHORIZER_FUNCTION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("authorizerFunctionArn").getter(getter(UpdateAuthorizerRequest::authorizerFunctionArn))
            .setter(setter(Builder::authorizerFunctionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authorizerFunctionArn").build())
            .build();

    private static final SdkField<String> TOKEN_KEY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("tokenKeyName").getter(getter(UpdateAuthorizerRequest::tokenKeyName))
            .setter(setter(Builder::tokenKeyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tokenKeyName").build()).build();

    private static final SdkField<Map<String, String>> TOKEN_SIGNING_PUBLIC_KEYS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tokenSigningPublicKeys")
            .getter(getter(UpdateAuthorizerRequest::tokenSigningPublicKeys))
            .setter(setter(Builder::tokenSigningPublicKeys))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tokenSigningPublicKeys").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(UpdateAuthorizerRequest::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Boolean> ENABLE_CACHING_FOR_HTTP_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("enableCachingForHttp").getter(getter(UpdateAuthorizerRequest::enableCachingForHttp))
            .setter(setter(Builder::enableCachingForHttp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("enableCachingForHttp").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AUTHORIZER_NAME_FIELD,
            AUTHORIZER_FUNCTION_ARN_FIELD, TOKEN_KEY_NAME_FIELD, TOKEN_SIGNING_PUBLIC_KEYS_FIELD, STATUS_FIELD,
            ENABLE_CACHING_FOR_HTTP_FIELD));

    private final String authorizerName;

    private final String authorizerFunctionArn;

    private final String tokenKeyName;

    private final Map<String, String> tokenSigningPublicKeys;

    private final String status;

    private final Boolean enableCachingForHttp;

    private UpdateAuthorizerRequest(BuilderImpl builder) {
        super(builder);
        this.authorizerName = builder.authorizerName;
        this.authorizerFunctionArn = builder.authorizerFunctionArn;
        this.tokenKeyName = builder.tokenKeyName;
        this.tokenSigningPublicKeys = builder.tokenSigningPublicKeys;
        this.status = builder.status;
        this.enableCachingForHttp = builder.enableCachingForHttp;
    }

    /**
     * <p>
     * The authorizer name.
     * </p>
     * 
     * @return The authorizer name.
     */
    public final String authorizerName() {
        return authorizerName;
    }

    /**
     * <p>
     * The ARN of the authorizer's Lambda function.
     * </p>
     * 
     * @return The ARN of the authorizer's Lambda function.
     */
    public final String authorizerFunctionArn() {
        return authorizerFunctionArn;
    }

    /**
     * <p>
     * The key used to extract the token from the HTTP headers.
     * </p>
     * 
     * @return The key used to extract the token from the HTTP headers.
     */
    public final String tokenKeyName() {
        return tokenKeyName;
    }

    /**
     * For responses, this returns true if the service returned a value for the TokenSigningPublicKeys property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasTokenSigningPublicKeys() {
        return tokenSigningPublicKeys != null && !(tokenSigningPublicKeys instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The public keys used to verify the token signature.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTokenSigningPublicKeys} method.
     * </p>
     * 
     * @return The public keys used to verify the token signature.
     */
    public final Map<String, String> tokenSigningPublicKeys() {
        return tokenSigningPublicKeys;
    }

    /**
     * <p>
     * The status of the update authorizer request.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AuthorizerStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the update authorizer request.
     * @see AuthorizerStatus
     */
    public final AuthorizerStatus status() {
        return AuthorizerStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the update authorizer request.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AuthorizerStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the update authorizer request.
     * @see AuthorizerStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * When <code>true</code>, the result from the authorizer’s Lambda function is cached for the time specified in
     * <code>refreshAfterInSeconds</code>. The cached result is used while the device reuses the same HTTP connection.
     * </p>
     * 
     * @return When <code>true</code>, the result from the authorizer’s Lambda function is cached for the time specified
     *         in <code>refreshAfterInSeconds</code>. The cached result is used while the device reuses the same HTTP
     *         connection.
     */
    public final Boolean enableCachingForHttp() {
        return enableCachingForHttp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(authorizerName());
        hashCode = 31 * hashCode + Objects.hashCode(authorizerFunctionArn());
        hashCode = 31 * hashCode + Objects.hashCode(tokenKeyName());
        hashCode = 31 * hashCode + Objects.hashCode(hasTokenSigningPublicKeys() ? tokenSigningPublicKeys() : null);
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(enableCachingForHttp());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateAuthorizerRequest)) {
            return false;
        }
        UpdateAuthorizerRequest other = (UpdateAuthorizerRequest) obj;
        return Objects.equals(authorizerName(), other.authorizerName())
                && Objects.equals(authorizerFunctionArn(), other.authorizerFunctionArn())
                && Objects.equals(tokenKeyName(), other.tokenKeyName())
                && hasTokenSigningPublicKeys() == other.hasTokenSigningPublicKeys()
                && Objects.equals(tokenSigningPublicKeys(), other.tokenSigningPublicKeys())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(enableCachingForHttp(), other.enableCachingForHttp());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UpdateAuthorizerRequest").add("AuthorizerName", authorizerName())
                .add("AuthorizerFunctionArn", authorizerFunctionArn()).add("TokenKeyName", tokenKeyName())
                .add("TokenSigningPublicKeys", hasTokenSigningPublicKeys() ? tokenSigningPublicKeys() : null)
                .add("Status", statusAsString()).add("EnableCachingForHttp", enableCachingForHttp()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "authorizerName":
            return Optional.ofNullable(clazz.cast(authorizerName()));
        case "authorizerFunctionArn":
            return Optional.ofNullable(clazz.cast(authorizerFunctionArn()));
        case "tokenKeyName":
            return Optional.ofNullable(clazz.cast(tokenKeyName()));
        case "tokenSigningPublicKeys":
            return Optional.ofNullable(clazz.cast(tokenSigningPublicKeys()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "enableCachingForHttp":
            return Optional.ofNullable(clazz.cast(enableCachingForHttp()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UpdateAuthorizerRequest, T> g) {
        return obj -> g.apply((UpdateAuthorizerRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends IotRequest.Builder, SdkPojo, CopyableBuilder<Builder, UpdateAuthorizerRequest> {
        /**
         * <p>
         * The authorizer name.
         * </p>
         * 
         * @param authorizerName
         *        The authorizer name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorizerName(String authorizerName);

        /**
         * <p>
         * The ARN of the authorizer's Lambda function.
         * </p>
         * 
         * @param authorizerFunctionArn
         *        The ARN of the authorizer's Lambda function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorizerFunctionArn(String authorizerFunctionArn);

        /**
         * <p>
         * The key used to extract the token from the HTTP headers.
         * </p>
         * 
         * @param tokenKeyName
         *        The key used to extract the token from the HTTP headers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tokenKeyName(String tokenKeyName);

        /**
         * <p>
         * The public keys used to verify the token signature.
         * </p>
         * 
         * @param tokenSigningPublicKeys
         *        The public keys used to verify the token signature.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tokenSigningPublicKeys(Map<String, String> tokenSigningPublicKeys);

        /**
         * <p>
         * The status of the update authorizer request.
         * </p>
         * 
         * @param status
         *        The status of the update authorizer request.
         * @see AuthorizerStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AuthorizerStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the update authorizer request.
         * </p>
         * 
         * @param status
         *        The status of the update authorizer request.
         * @see AuthorizerStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AuthorizerStatus
         */
        Builder status(AuthorizerStatus status);

        /**
         * <p>
         * When <code>true</code>, the result from the authorizer’s Lambda function is cached for the time specified in
         * <code>refreshAfterInSeconds</code>. The cached result is used while the device reuses the same HTTP
         * connection.
         * </p>
         * 
         * @param enableCachingForHttp
         *        When <code>true</code>, the result from the authorizer’s Lambda function is cached for the time
         *        specified in <code>refreshAfterInSeconds</code>. The cached result is used while the device reuses the
         *        same HTTP connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableCachingForHttp(Boolean enableCachingForHttp);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends IotRequest.BuilderImpl implements Builder {
        private String authorizerName;

        private String authorizerFunctionArn;

        private String tokenKeyName;

        private Map<String, String> tokenSigningPublicKeys = DefaultSdkAutoConstructMap.getInstance();

        private String status;

        private Boolean enableCachingForHttp;

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateAuthorizerRequest model) {
            super(model);
            authorizerName(model.authorizerName);
            authorizerFunctionArn(model.authorizerFunctionArn);
            tokenKeyName(model.tokenKeyName);
            tokenSigningPublicKeys(model.tokenSigningPublicKeys);
            status(model.status);
            enableCachingForHttp(model.enableCachingForHttp);
        }

        public final String getAuthorizerName() {
            return authorizerName;
        }

        public final void setAuthorizerName(String authorizerName) {
            this.authorizerName = authorizerName;
        }

        @Override
        public final Builder authorizerName(String authorizerName) {
            this.authorizerName = authorizerName;
            return this;
        }

        public final String getAuthorizerFunctionArn() {
            return authorizerFunctionArn;
        }

        public final void setAuthorizerFunctionArn(String authorizerFunctionArn) {
            this.authorizerFunctionArn = authorizerFunctionArn;
        }

        @Override
        public final Builder authorizerFunctionArn(String authorizerFunctionArn) {
            this.authorizerFunctionArn = authorizerFunctionArn;
            return this;
        }

        public final String getTokenKeyName() {
            return tokenKeyName;
        }

        public final void setTokenKeyName(String tokenKeyName) {
            this.tokenKeyName = tokenKeyName;
        }

        @Override
        public final Builder tokenKeyName(String tokenKeyName) {
            this.tokenKeyName = tokenKeyName;
            return this;
        }

        public final Map<String, String> getTokenSigningPublicKeys() {
            if (tokenSigningPublicKeys instanceof SdkAutoConstructMap) {
                return null;
            }
            return tokenSigningPublicKeys;
        }

        public final void setTokenSigningPublicKeys(Map<String, String> tokenSigningPublicKeys) {
            this.tokenSigningPublicKeys = PublicKeyMapCopier.copy(tokenSigningPublicKeys);
        }

        @Override
        public final Builder tokenSigningPublicKeys(Map<String, String> tokenSigningPublicKeys) {
            this.tokenSigningPublicKeys = PublicKeyMapCopier.copy(tokenSigningPublicKeys);
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(AuthorizerStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Boolean getEnableCachingForHttp() {
            return enableCachingForHttp;
        }

        public final void setEnableCachingForHttp(Boolean enableCachingForHttp) {
            this.enableCachingForHttp = enableCachingForHttp;
        }

        @Override
        public final Builder enableCachingForHttp(Boolean enableCachingForHttp) {
            this.enableCachingForHttp = enableCachingForHttp;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public UpdateAuthorizerRequest build() {
            return new UpdateAuthorizerRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
