/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iot.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the resource that was noncompliant with the audit check.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NonCompliantResource implements SdkPojo, Serializable,
        ToCopyableBuilder<NonCompliantResource.Builder, NonCompliantResource> {
    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceType").getter(getter(NonCompliantResource::resourceTypeAsString))
            .setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceType").build()).build();

    private static final SdkField<ResourceIdentifier> RESOURCE_IDENTIFIER_FIELD = SdkField
            .<ResourceIdentifier> builder(MarshallingType.SDK_POJO).memberName("resourceIdentifier")
            .getter(getter(NonCompliantResource::resourceIdentifier)).setter(setter(Builder::resourceIdentifier))
            .constructor(ResourceIdentifier::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceIdentifier").build())
            .build();

    private static final SdkField<Map<String, String>> ADDITIONAL_INFO_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("additionalInfo")
            .getter(getter(NonCompliantResource::additionalInfo))
            .setter(setter(Builder::additionalInfo))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("additionalInfo").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESOURCE_TYPE_FIELD,
            RESOURCE_IDENTIFIER_FIELD, ADDITIONAL_INFO_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String resourceType;

    private final ResourceIdentifier resourceIdentifier;

    private final Map<String, String> additionalInfo;

    private NonCompliantResource(BuilderImpl builder) {
        this.resourceType = builder.resourceType;
        this.resourceIdentifier = builder.resourceIdentifier;
        this.additionalInfo = builder.additionalInfo;
    }

    /**
     * <p>
     * The type of the noncompliant resource.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of the noncompliant resource.
     * @see ResourceType
     */
    public final ResourceType resourceType() {
        return ResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The type of the noncompliant resource.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of the noncompliant resource.
     * @see ResourceType
     */
    public final String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * <p>
     * Information that identifies the noncompliant resource.
     * </p>
     * 
     * @return Information that identifies the noncompliant resource.
     */
    public final ResourceIdentifier resourceIdentifier() {
        return resourceIdentifier;
    }

    /**
     * For responses, this returns true if the service returned a value for the AdditionalInfo property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAdditionalInfo() {
        return additionalInfo != null && !(additionalInfo instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Other information about the noncompliant resource.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAdditionalInfo} method.
     * </p>
     * 
     * @return Other information about the noncompliant resource.
     */
    public final Map<String, String> additionalInfo() {
        return additionalInfo;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resourceIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(hasAdditionalInfo() ? additionalInfo() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NonCompliantResource)) {
            return false;
        }
        NonCompliantResource other = (NonCompliantResource) obj;
        return Objects.equals(resourceTypeAsString(), other.resourceTypeAsString())
                && Objects.equals(resourceIdentifier(), other.resourceIdentifier())
                && hasAdditionalInfo() == other.hasAdditionalInfo() && Objects.equals(additionalInfo(), other.additionalInfo());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("NonCompliantResource").add("ResourceType", resourceTypeAsString())
                .add("ResourceIdentifier", resourceIdentifier())
                .add("AdditionalInfo", hasAdditionalInfo() ? additionalInfo() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "resourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "resourceIdentifier":
            return Optional.ofNullable(clazz.cast(resourceIdentifier()));
        case "additionalInfo":
            return Optional.ofNullable(clazz.cast(additionalInfo()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("resourceType", RESOURCE_TYPE_FIELD);
        map.put("resourceIdentifier", RESOURCE_IDENTIFIER_FIELD);
        map.put("additionalInfo", ADDITIONAL_INFO_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<NonCompliantResource, T> g) {
        return obj -> g.apply((NonCompliantResource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NonCompliantResource> {
        /**
         * <p>
         * The type of the noncompliant resource.
         * </p>
         * 
         * @param resourceType
         *        The type of the noncompliant resource.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The type of the noncompliant resource.
         * </p>
         * 
         * @param resourceType
         *        The type of the noncompliant resource.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(ResourceType resourceType);

        /**
         * <p>
         * Information that identifies the noncompliant resource.
         * </p>
         * 
         * @param resourceIdentifier
         *        Information that identifies the noncompliant resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceIdentifier(ResourceIdentifier resourceIdentifier);

        /**
         * <p>
         * Information that identifies the noncompliant resource.
         * </p>
         * This is a convenience method that creates an instance of the {@link ResourceIdentifier.Builder} avoiding the
         * need to create one manually via {@link ResourceIdentifier#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResourceIdentifier.Builder#build()} is called immediately and its
         * result is passed to {@link #resourceIdentifier(ResourceIdentifier)}.
         * 
         * @param resourceIdentifier
         *        a consumer that will call methods on {@link ResourceIdentifier.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceIdentifier(ResourceIdentifier)
         */
        default Builder resourceIdentifier(Consumer<ResourceIdentifier.Builder> resourceIdentifier) {
            return resourceIdentifier(ResourceIdentifier.builder().applyMutation(resourceIdentifier).build());
        }

        /**
         * <p>
         * Other information about the noncompliant resource.
         * </p>
         * 
         * @param additionalInfo
         *        Other information about the noncompliant resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalInfo(Map<String, String> additionalInfo);
    }

    static final class BuilderImpl implements Builder {
        private String resourceType;

        private ResourceIdentifier resourceIdentifier;

        private Map<String, String> additionalInfo = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(NonCompliantResource model) {
            resourceType(model.resourceType);
            resourceIdentifier(model.resourceIdentifier);
            additionalInfo(model.additionalInfo);
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(ResourceType resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        public final ResourceIdentifier.Builder getResourceIdentifier() {
            return resourceIdentifier != null ? resourceIdentifier.toBuilder() : null;
        }

        public final void setResourceIdentifier(ResourceIdentifier.BuilderImpl resourceIdentifier) {
            this.resourceIdentifier = resourceIdentifier != null ? resourceIdentifier.build() : null;
        }

        @Override
        public final Builder resourceIdentifier(ResourceIdentifier resourceIdentifier) {
            this.resourceIdentifier = resourceIdentifier;
            return this;
        }

        public final Map<String, String> getAdditionalInfo() {
            if (additionalInfo instanceof SdkAutoConstructMap) {
                return null;
            }
            return additionalInfo;
        }

        public final void setAdditionalInfo(Map<String, String> additionalInfo) {
            this.additionalInfo = StringMapCopier.copy(additionalInfo);
        }

        @Override
        public final Builder additionalInfo(Map<String, String> additionalInfo) {
            this.additionalInfo = StringMapCopier.copy(additionalInfo);
            return this;
        }

        @Override
        public NonCompliantResource build() {
            return new NonCompliantResource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
